function [theta, Ft, Ghat] = EstPanel(Y, X, T, h, J, delta)
% This function implements the Estimation Procedure I of CGPX (2020) 
% 
% Input:
% Y           The dependent variable    (N*T)-by-1
% X           The regressors            (N*T)-by-d_x
% T           The number of time periods
% h           The bandwidth
% J           A user pre-specified number for intitial guess on the
%             total number of factors
% delta       The point at which we evaluate beta_0(.)
% 
% Output:
% theta       The esitmate of theta_0
% Ft          The esitmate of F_0   
% Ghat        The estimate of Gamma_0
  
[NT, d] = size(X);
NT1     = size(Y, 1);
  
if NT ~= NT1
    error('The dimensions of X and Y do not match')
end
      
if rem(NT, T) ~= 0
    error('The number of individuals is not correct')
end
  
N = NT/T;  % The number of individuals

% Generate the matrix associated to the kernel function
K_d = K_delta(T, h, delta);
 
% Randomly generate the factor matrix as the initial estimate to start the
% iteration
Ft = randn(T, J);

% Start the iteration 
MF = eye(T) - Ft/(Ft'*Ft)*(Ft');
d_matrix = zeros(d, d);
n_matrix = zeros(d, 1);
for i = 1:N
    ind = (i-1)*T + (1:T);
    d_matrix = d_matrix + X(ind, :)' * K_d * MF * K_d * X(ind, :);
    n_matrix = n_matrix + X(ind, :)' * K_d * MF * K_d * Y(ind, :);
end
theta1 = d_matrix \ n_matrix;
theta2 = zeros(d, 1);

% Set a criterion for the iteration to stop, i.e. stopping the iteration 
% when ||theta1 - theta2|| <= crit. Let no_iter record the number of 
% iteration. Break the iteration when no_iter > 100
crit    = 10^(-2);
no_iter = 0;
  
while sqrt((theta1 - theta2)'*(theta1 - theta2)) > crit
    no_iter = no_iter + 1;
    if no_iter > 100 
        break 
    end
    
    % Update the estimate of F_o given theta1
    [ve, ~] = EstF(Y, X, K_d, theta1, T);
    Ft      = ve(:, 1:J);
    
    % Update the estimate of theta_o with new Ft, and store it in theta2
    MF = eye(T) - (Ft*(Ft'))/T;
    d_matrix = zeros(d, d);
    n_matrix = zeros(d, 1);
    for i = 1:N
        ind = (i-1)*T + (1:T);
        d_matrix = d_matrix + X(ind, :)' * K_d * MF * K_d * X(ind, :);
        n_matrix = n_matrix + X(ind, :)' * K_d * MF * K_d * Y(ind, :);
    end
    theta2 = d_matrix \ n_matrix;

    % Update the estimate of F_o given theta2
    [ve, ~] = EstF(Y, X, K_d, theta2, T);
    Ft      = ve(:, 1:J);
    
    % Update the estimate of theta_o with new Ft, and store it in theta1
    MF = eye(T) - (Ft*(Ft'))/T;
    d_matrix = zeros(d, d);
    n_matrix = zeros(d, 1);
    for i = 1:N
        ind = (i-1)*T + (1:T);
        d_matrix = d_matrix + X(ind, :)' * K_d * MF * K_d * X(ind, :);
        n_matrix = n_matrix + X(ind, :)' * K_d * MF * K_d * Y(ind, :);
    end
    theta1 = d_matrix \ n_matrix;
end

% Output for theta_o
theta = theta1;

% Output for the esitmate of F_o
[ve, ~] = EstF(Y, X, K_d, theta, T);
Ft      = ve(:, 1:J);

% Output for the esitmate of G_0 and est
Ghat = zeros(N, J); 
for i = 1:N
    ind = (i-1)*T + (1:T);
    Ghat(i, :) = (Y(ind) - X(ind, :) * theta)'*Ft;
end
Ghat = Ghat/T;

end