/*
AUTHOR:   Todd Clark, Federal Reserve Bank of Cleveland, todd.clark@researchfed.org
Code associated with Andrea Carriero, Todd Clark and Massimiliano Marcellino, 
"Nowcasting Tail Risks to Economic Activity at a Weekly Frequency"

Note:  The paper includes results for an equally weighted average forecast based on the BMF-SV, BQR, and PQR specifications
with the base M-F variable set.  In the evaluation of that forecast, for the CRPS measures that require a complete
predictive density, we obtain the statistics by linear pooling of the underlying densities.  (For the other metrics in
the paper, we directly average the forecasts and compute statistics using these forecast averages.
This includes the quantile forecasts and ES.  We take averages of these and then compute the QS and VaR-ES using
the quantile and ES averages.)  This program reads in the underlying forecast draws, computes their densities, forms the
pdf for the average forecast using linear pooling, and then takes draws from that pdf to compute the CRPS and qwcrps
for this average forecast.  The result is written to an Excel file.

In this case, the average forecast is computed using the outlier-adjusted BMF-SVO in lieu of BMF-SV,
limited to the 2019-2020 period.

Note also that, for reading in draws, BQR and PQR inputs and code differ a bit from the SV case because we have draws for multiple quantiles, unlike the SV case.
*/

dis %dateandtime()
dis %ratsversion()
comp overallstart = %cputime()

*********************
********************* BASIC SETUP & PARAMETER ENTRY
*********************
comp usesvo = 1          ;* 1 to use SVO, 0 to use SV

******* case-specific setup
comp nfc = 3             ;* 3 underlying forecasts to be combined
comp nsv = 1             ;* first nsv forecasts are from BMF-SV spec, remaining (nfc-nsv) from either BQR or PQR
comp ndraws =  5000	     ;*total number of Gibbs draws

******** setup of sample
comp useactual = 1          ;* adjustment for obs lost of last vintage relative to eval sample end date (used to set this at 2)
comp styr = 1947
cal(q) styr:1
comp stsmpl = styr:1	;*earliest period with data

comp stvint = 1985:1       ;* starting vintage for base M-F case.  Change if specs with small wkly or large wkly variable sets are used
comp endvint = 2021:1      ;* last quarter of data vintages considered
comp endsmpl = endvint - useactual

all endvint
smpl stsmpl endvint

comp startrange = 2019:1  ;* earliest is stvint
comp endrange = endvint-1
dis %datelabel(startrange) %datelabel(endrange)
comp seedval = 444*startrange
seed seedval
dis seedval

/*
comp startrange = 1995:1  ;* earliest is stvint
comp endrange = 2004:4     ;* latest is endvint-1

comp startrange = 2005:1  ;* earliest is stvint
comp endrange = 2013:4     ;* latest is endvint-1

comp startrange = 2014:1     ;* earliest is stvint
comp endrange = endvint-1    ;* latest is endvint-1
*/
******** setup of forecast specs and actual GDP series
comp npredmo = 4                 ;* number of months at which we form forecasts: months 1-3 of quarter t plus month 1 of quarter t+1
comp nweek = 4                   ;* number of weeks used in month
comp totweek = npredmo*nweek-1     ;* total number of predictions.  -1 because we don't form a forecast for a 16th week, by which time gdp(t+1) becomes available

** read in GDP estimates used as actuals in evaluation of forecasts:  these are 2nd avail in the quarterly RTDSM
open data ../data/GDPactuals.secondrelease.xls
data(format=xls,org=col) / actualGDP
close

******** setup of forecast input options among models and variable sets, as well as label
comp [vec[str]] filelabel = ||"SV_MF","BQR_MF","PQR_MF"||
if usesvo==1
 comp filelabel(1) = "SVO_MF"
endif
comp [vec[str]] methlabel = ||"base M-F:  BMF-SV","base M-F:  BQR","base M-F:  PQR"||
comp [str] modeldesc = "average base M-F forecast"

comp [vec] wghts = %fill(nfc,1,1./nfc)  ;* combo weights = 1/n
dis wghts

********************************
******************************** stuff for forecast processing and storage
********************************
sou(noecho) ../procedures/fcmoments.src

comp nquant = 19
dec vec pctiles(nquant) 
ewise pctiles(i) = float(i)/(nquant+1)

dec vec[ser] crpsseries(totweek) qwcrpsseries(totweek) 
do mm = 1,totweek                   ;* create series labels to facilitate later reading results from Excel file created below
 labels crpsseries(mm)
 # 'crps_m'+%string(mm)
 labels qwcrpsseries(mm)
 # 'qwcrps_m'+%string(mm)
end do mm
clear crpsseries qwcrpsseries

*** for reading in forecast draws
comp skipint = 5            ;* we will thin samples for BQR and PQR for computational tractability
comp ndraws2=ndraws/skipint
dis ndraws2 (ndraws2*nquant)
comp maxdraws = %imax(ndraws,ndraws2*nquant)
dec vec[ser] rawdraws(nquant) fcdraws(nfc)
clear(length=maxdraws) fcdraws 
clear(length=ndraws) svrawdraws rawdraws

dec vec[ser] ForecastRes(ndraws2*nquant)

***** stuff for density or CRPS calcs
comp ngridpts = 500  ;* we compute the densities using a grid of 500 points
clear(length=ngridpts) avggrid estdensity
dec vec gridvec(ngridpts) densityvec(ngridpts)

********************************
******************************** now loop over time to read draws and do computations
********************************

do time = startrange,endrange
 dis ''
 dis '*******************************************************'
 dis '******************************************************* forecast date = ' %datelabel(time)
 dis '*******************************************************'
 dis ''
 comp quarter = %month(time)
 comp originst = %cputime()

 do ww = 1,totweek
  smpl stsmpl time
  comp eststpt = stsmpl
  comp estendpt = time
  ****
  comp lastmoposs = ((ww-1)/4)
  comp nn = ww - lastmoposs*4
  comp yy = %year(time)
  comp month = lastmoposs+1+(quarter-1)*3
  comp mml = 1+(ww-1)/4     ;* month number in quarter (1-3 in quarter and 4 in next quarter)
  comp nn = ww - (mml-1)*4  ;* week number in month (1-4)
  if ww<=3
   comp horz = 2
  else
   comp horz = 1

  dis ""
  dis "****************************** totweek, month, week = " ww mml nn
  dis 'year, quarter, week = ' (%string(yy)+"qtr"+quarter+"wk"+ww)

  *** loop to pull in draws for different forecasts, and check min and max as part of this loop
  comp lbound=0.0, ubound=0.0
  do n = 1,nfc
   comp [str] rootfilename = 'fcdraws/' 
   comp outfilename = rootfilename + filelabel(n)+"/"+"draws."+%string(yy)+"qtr"+quarter+"wk"+ww+".csv"
   smpl 1 ndraws
   if n<=nsv
    {
     open data &outfilename
     data(org=col,for=cdf,nolabels,left=2) 1 ndraws svrawdraws
     close data
     set fcdraws(n) 1 ndraws = svrawdraws(t)
    }
   else
    {
     open data &outfilename
     data(org=col,for=cdf,skiplines=1,nolabels) 1 ndraws rawdraws
     close data
     do q = 1,nquant
      set fcdraws(n) (q-1)*ndraws2+1 q*ndraws2 = rawdraws(q)((t-(q-1)*ndraws2)*skipint)
     end do q
    }
   *
   if n<=nsv
    smpl 1 ndraws
   else
    smpl 1 ndraws2*nquant
   extremum(noprint) fcdraws(n)
   if n==1
    comp lbound=%minimum, ubound=%maximum
   else
    comp lbound=%min(lbound,%minimum), ubound=%max(ubound,%maximum)
  end do n

  *** determine common grid
  comp span2 = ubound-lbound
  set avggrid 2 ngridpts = lbound + (t-2)*span2/(ngridpts-2.)
  comp avggrid(1) = actualgdp(time)
  order avggrid 1 ngridpts

  *** compute and combine densities
  clear(zeros,length=ngridpts) avgdensity
  do n = 1,nfc
   if n<=nsv
    smpl 1 ndraws
   else
    smpl 1 ndraws2*nquant
   set(scratch) oneforecast = fcdraws(n){0}
   density(grid=input,type=gaussian) oneforecast / avggrid estdensity 
   set avgdensity 1 ngridpts = avgdensity{0} + wghts(n)*estdensity{0}
  end do n

  *** take draws from empirical density (combined)
  ewise gridvec(i) = avggrid(i)
  ewise densityvec(i) = avgdensity(i)
  set oneforecast 1 ndraws2*nquant = %rangrid(gridvec,densityvec)  ;* take draws from combined forecast density
  do i = 1,ndraws2*nquant
   set ForecastRes(i) time time = oneforecast(i)
  end do i
  
  *** compute CRPS from those forecast draws
  @FCMOMENTS ForecastRes actualGDP time 1 pctiles meanres pctileres crps qwcrps qscoreres esres fzgscoreres
  comp crpsseries(ww)(time) = crps
  comp qwcrpsseries(ww)(time) = qwcrps

 end do ww

 dis 'total time for this forecast origin = ' (%cputime()-originst)/60.
 
end do time

dis %dateandtime()
dis 'total loop time in minutes = ' (%cputime()-overallstart)/60.

********************************
******************************** write time series CRPS to Excel file
********************************
comp filename = %unitfnroot("input")+"_empiricaldensity.xls"
open copy &filename
copy(for=xls,dates,org=col) startrange endrange crpsseries qwcrpsseries
close copy
