%% pssiv.m
%
% Calculates Powell, Stock, and Stoker's density-weighted average derivatives.
%
% Written by Jeff Campbell

%% Set up the required inputs; a vector dependent variable, a matrix regressor, and a bandwidth.

if exist('pssivARG','var')      %The structure pssivARG is defined. Take the inputs from it.
    y=pssivARG.y;
    x=pssivARG.x;
    h=pssivARG.h;
else                            %Dummy arguments used to test the procedure.
    x=randn(1000,8);
    y=x*ones(size(x,2),1)+0.4*randn(1000,1);
    h=1;
end

%% Check inputs for validity and set up basic variables. (See Bierens (1987), p 112 for details.)
    

n = length(y');         %Sample size
k = length(x(1,:));     %Number of regressors

% Check for the conformity of x with y.

	if n~=length(x(:,1));
       error('Error in pss. y and x must have the same row dimension');
    end

% Choose the order of the kernel.

	if floor(k/2)==k/2;
		p= (k+4)/2;
	else
		p= (k+3)/2;
    end
m=ceil(p/2+0.5);

% Calculate the kernel's dispersion coefficients and weights.
% Defined by by Bierens line (2.2.37 & 2.2.38)
sigma = 1:1:m;
A=vander(sigma)';
b=[zeros(m-1,1); 1];
w=(inv(A)*b);

%% Define an in-line procedure for the bias reducing kernel.
% Input, x (nxk) mattrix
% Output, @(x) (nx1) vector of the multivariate kernel evaluated at the rows of x.


% Kernal defined by Bierens line (2.2.36)
 K = @(x) ((ones(size(x,1),1)*(2*pi*sigma).^(-k/2)).*exp(-0.5*sum(x.^2,2)*(sigma.^(-1))))*w;
 
% Plot the kernel if the program is being tested.
if exist('pssivARG','var')-1;
    xx=[-6:0.01:6]';
    yy=K(xx);
    
    figure(1)
    set(gcf,'color',[1 1 1]);
    plot(xx,yy,'-k','LineWidth',2)
    hold
    plot(xx,zeros(size(xx)),':k','LineWidth',1);
    box off      
    title(['Bias Reducing Kernel, $M=' num2str(m) '$'],'Interpreter','latex','FontSize',16)
end

%% Define an in-line procedure for the bias reducing kernel's derivative.
% Input, x (nxk) matrix
% Output @(x) (nxk) matrix of the k deriviatives of the multivariate kernel evaluated at the rows of x.

dK = @(x) -x.*((((ones(size(x,1),1)*(2*pi*sigma).^(-k/2)).*exp(-0.5*sum(x.^2,2)*(sigma.^(-1))))*(w./sigma'))*ones(1,size(x,2)));

%Plot the derivative if the program is being tested;
if exist('pssivARG','var')-1;
    xx=[-6:0.1:6]';
    yy=dK(xx);
    min(yy)
    max(yy)
    figure(2)
    set(gcf,'color',[1 1 1]);
    plot(xx,yy,'-k','LineWidth',2)
    hold
    plot(xx,zeros(size(xx)),':k','LineWidth',1);
    yy=K(xx);
    plot(xx,yy,'-k','LineWidth',2)
    box off
    title(['Derivative of Bias Reducing Kernel, $M=' num2str(m) '$'],'Interpreter','latex','FontSize',16)
end
    
%% Calculate the IV estimator.

%Evaluate the bias-reducing kernel's derivative at all possible x_i-x_j values. Sum them to get dfhat(x_i)/dx
dx = kron(ones(n,1),x)-kron(x,ones(n,1));
summat=sparse(kron(ones(n,1),sparse(eye(n))))*((1/(n-1))*(1/h)^(k+1));
dfhat = summat'*dK(dx/h);

%Use dfhat as a matrix of instrumental variables.
ideltaxn=inv(-(2/n)*dfhat'*x);
dhat=ideltaxn*(-(2/n)*dfhat'*y);


%% Calculate the IV estimator's variance-covariance matrix.

% Calculate residuals.
u = y-x*dhat;

%Calculate the vector rhat (equation 4.10)
du=kron(ones(n,1),u)-kron(u,ones(n,1));
r=-summat'*(dK(dx/h).*(du*ones(1,k)));
r=r*ideltaxn';
sigmahatd=4*r'*r/n;

Wmat = sigmahatd/n;

%% Report test results
%
% If we are testing the program, report the coefficient estimates, the associated least-squares estimates, and their estimated standard errors.

if exist('pssivARG','var')-1
    
    %Calculate OLS estimates
    betahat=inv(x'*x)*(x'*y);
    resids=y-x*betahat;
    sigmahat=(1/(n-k))*sum(resids.^2);
    capSigmahat=sigmahat*inv(x'*x);
    
    %Report results
    dhat
    sqrt(diag(Wmat))
    betahat
    sqrt(diag(capSigmahat))
            
end

