#include <oxstd.h>
#include <oxdraw.h>
#include <oxprob.h>
#include "rq.ox"

nhstat(const mX, const cTrend, const cBandWidth)
// performs the multivariate generalization of KPSS test
// see Nyblom & Harvey (2000) ET 16, pp.176-199
// mX (cT x cN) data matrix
// bTrend =	0 nothing
//		  =	1 de-mean
//        = 2 de-trend (linear trend)
// cBandWidth =	negative -> automatic
//            = integer -> bandwith of Bartlett window
{
	decl mY;
	decl cT = rows(mX);
	decl cN = columns(mX);
	decl cM = (cBandWidth < 0)? int(4*(cT/100)^(2/9)) : cBandWidth;
	if (cTrend == 0)
	{
		mY = mX;
	}
	else if (cTrend == 1)
	{
		mY = mX - meanc(mX);
	}
	else
	{
		decl mBeta;
		decl mZ = ones(cT,1)~range(1,cT)';
		olsc(mX, mZ, &mBeta);
		mY = mX - mZ*mBeta;
	}

	decl mS = mY'mY/cT; // long run covariance matrix
	decl mGamma;
	for (decl i=1; i<=cM; ++i) // Bartlett Kernel
	{
		mGamma = mY[i:][]'mY[0:cT-1-i][]/cT;
		mS += (1 - i/(cM+1))*(mGamma + mGamma');
	}

	decl mYcum = cumsum(mY,1);	 // cumulative sums
	decl mC = mYcum'mYcum / cT^2;// numerator of the statistic
	decl vEigVals, mEigVecs;
	eigensymgen(mC, mS, &vEigVals, &mEigVecs);
//	eigensym(invertsym(mS)*mC, &vEigVals, &mEigVecs);
	return reversec(cumsum(sortr(vEigVals)',1));
}

inhstat(const mX, const cTrend, const cBandWidth)
// performs the multivariate generalization of KPSS test
// see Nyblom & Harvey (2000) ET 16, pp.176-199
// mX (cT x cN) data matrix
// bTrend =	0 nothing
//		  =	1 de-mean
//        = 2 de-trend (linear trend)
// cBandWidth =	negative -> automatic
//            = integer -> bandwith of Bartlett window
{
	decl mY;
	decl cT = rows(mX);
	decl cN = columns(mX);
	decl cM = (cBandWidth < 0)? int(4*(cT/100)^(2/9)) : cBandWidth;
	if (cTrend == 0)
	{
		mY = mX;
	}
	else if (cTrend == 1)
	{
		mY = mX - quantilec(mX);
	}
	else
	{
		RCI = 0;
		decl aBeta;
		decl mZ = ( ones(cT,1) ~ range(1,cT)' );
		mY = new matrix[cT][cN];
		for (decl i=0; i<cN; ++i)
		{
			aBeta = rq(mZ[][1], mX[][i], 0.5); // LAD regression
			mY[][i] = mX[][i] - mZ*aBeta[0]; // LAD regr. residuals
		}
	}

	// sign maker (for zero, in case of odd number of obs., set a tollerance)
	if (cT/2 == int(cT/2)) mY = (mY .> 0) - (mY .< 0);
	else mY = (mY .> 0.000001) - (mY .< -0.000001);

	decl mS = mY'mY/cT; // long run covariance matrix
	decl mGamma;
	for (decl i=1; i<=cM; ++i)
	{
		mGamma = mY[i:][]'mY[0:cT-1-i][]/cT;
		mS += (1 - i/(cM+1))*(mGamma + mGamma');
	}
	decl mYcum = cumsum(mY,1);	 // cumulative sums
	decl mC = mYcum'mYcum / cT^2;// numerator of the statistic
	decl vEigVals, mEigVecs;
	eigensymgen(mC, mS, &vEigVals, &mEigVecs);   // failed many times
//	eigensym(invertgen(mS,3)*mC, &vEigVals, &mEigVecs);
	return reversec(cumsum(sortr(vEigVals)',1));
}

//mnhstat(const mX, const cTrend, const cBandWidth)
//// performs the multivariate generalization of KPSS test
//// see Nyblom & Harvey (2000) ET 16, pp.176-199
//// mX (cT x cN) data matrix
//// bTrend =	0 nothing
////		  =	1 de-mean
////        = 2 de-trend (linear trend)
//// cBandWidth =	negative -> automatic
////            = integer -> bandwith of Bartlett window
//{
//	decl mY, mDX, mSignDX;
//	decl cT = rows(mX);
//	decl cN = columns(mX);
//	decl cM = (cBandWidth < 0)? int(4*(cT/100)^(2/9)) : cBandWidth;
//
//	mDX = diff0(mX,1);
//	mSignDX	= (mDX .> 0) - (mDX .< 0);
//	mY = cumsum(mSignDX,1);
//	
//	if (cTrend == 0)
//	{
//		mY = mX;
//	}
//	else if (cTrend == 1)
//	{
//		mY = mX - meanc(mX);
//	}
//	else
//	{
//		decl mBeta;
//		decl mZ = ones(cT,1)~range(1,cT)';
//		olsc(mX, mZ, &mBeta);
//		mY = mX - mZ*mBeta;
//	}
//
//	decl mS = mY'mY/cT; // long run covariance matrix
//	decl mGamma;
//	for (decl i=1; i<=cM; ++i)
//	{
//		mGamma = mY[i:][]'mY[0:cT-1-i][]/cT;
//		mS += (1 - i/(cM+1))*(mGamma + mGamma');
//	}
//	decl mYcum = cumsum(mY,1);	 // cumulative sums
//	decl mC = mYcum'mYcum / cT^2;// numerator of the statistic
//	decl vEigVals, mEigVecs;
//	eigensymgen(mC, mS, &vEigVals, &mEigVecs);   // failed many times
////	eigensym(invertgen(mS,3)*mC, &vEigVals, &mEigVecs);
//	return reversec(cumsum(sortr(vEigVals)',1));
//}

nhstatsim(const cT, const cSim, const cNmax, const cTrend, const cBandWidth)
{
	decl vSim    = new matrix[1][cSim];
	decl mPerc01 = new matrix[cNmax-1][cNmax];
	decl mPerc05 = new matrix[cNmax-1][cNmax];
	decl mPerc10 = new matrix[cNmax-1][cNmax];
	decl asCols  = new array[cNmax];
	decl asRows  = new array[cNmax-1];

	for (decl i=2; i<=cNmax; ++i)  // number of series
	{
		for (decl j=0; j<i; ++j)   // number of trends
		{
			for (decl k=0; k<cSim; ++k)
			{
				vSim[k] = nhstat( (cumsum(rann(cT,j),1)~rann(cT,i-j)), cTrend, cBandWidth)[j];
			}
			mPerc01[i-2][j] = quantiler(vSim, .99);
			mPerc05[i-2][j] = quantiler(vSim, .95);
			mPerc10[i-2][j] = quantiler(vSim, .90);
		}
	}

	for (decl i=0; i < cNmax-1; ++i)
	{
		asRows[i] = sprint(i+2);
	}
	for (decl j=0; j < cNmax; ++j)
	{
		asCols[j] = sprint(j);
	}

	println("Number of observations = ", int(cT));
	println("Number of simulations  = ", int(cSim));
	println("Trend type (0 nothing, 1 mean, 2 linear): ", cTrend);
	println("Bandwidth: ", (cBandWidth<0)? int(4*(cT/100)^(2/9)) : cBandWidth);
	println("--------------------------");
	println("Size 0.10","%r",asRows,"%c",asCols,mPerc10);
	println("--------------------------");
	println("Size 0.05","%r",asRows,"%c",asCols,mPerc05);
	println("--------------------------");
	println("Size 0.01","%r",asRows,"%c",asCols,mPerc01);
}

sign(const mX)
{
	return (mX .> 0) - (mX .< 0);
}

inhstatsim(const cT, const cSim, const cNmax, const cTrend, const cBandWidth)
{
	decl vSim    = new matrix[1][cSim];
	decl mPerc01 = new matrix[cNmax-1][cNmax];
	decl mPerc05 = new matrix[cNmax-1][cNmax];
	decl mPerc10 = new matrix[cNmax-1][cNmax];
	decl asCols  = new array[cNmax];
	decl asRows  = new array[cNmax-1];
	decl mProcess;
	
	for (decl i=2; i<=cNmax; ++i)
	{
		for (decl j=0; j<i; ++j)
		{
			for (decl k=0; k<cSim; ++k)
			{
				vSim[k] = inhstat( (cumsum(rann(cT,j),1)~(rann(cT,i-j))), cTrend, cBandWidth)[j];
			}
			mPerc01[i-2][j] = quantiler(vSim, .99);
			mPerc05[i-2][j] = quantiler(vSim, .95);
			mPerc10[i-2][j] = quantiler(vSim, .90);
		}
	}

	for (decl i=0; i < cNmax-1; ++i)
	{
		asRows[i] = sprint(i+2);
	}
	for (decl j=0; j < cNmax; ++j)
	{
		asCols[j] = sprint(j);
	}

	println("Number of observations = ", int(cT));
	println("Number of simulations  = ", int(cSim));
	println("Trend type (0 nothing, 1 mean, 2 linear): ", cTrend);
	println("Bandwidth: ", (cBandWidth<0)? int(4*(cT/100)^(2/9)) : cBandWidth);
	println("--------------------------");
	println("Size 0.10","%r",asRows,"%c",asCols,mPerc10);
	println("--------------------------");
	println("Size 0.05","%r",asRows,"%c",asCols,mPerc05);
	println("--------------------------");
	println("Size 0.01","%r",asRows,"%c",asCols,mPerc01);
}

//mnhstatsim(const cT, const cSim, const cNmax, const cTrend, const cBandWidth)
//{
//	decl vSim    = new matrix[1][cSim];
//	decl mPerc01 = new matrix[cNmax-1][cNmax];
//	decl mPerc05 = new matrix[cNmax-1][cNmax];
//	decl mPerc10 = new matrix[cNmax-1][cNmax];
//	decl asCols  = new array[cNmax];
//	decl asRows  = new array[cNmax-1];
//	decl mProcess;
//
//	for (decl i=2; i<=cNmax; ++i)
//	{
//		for (decl j=0; j<i; ++j)
//		{
//			for (decl k=0; k<cSim; ++k)
//			{
//				vSim[k] = mnhstat( (cumsum(rann(cT,j),1)~(rann(cT,i-j)) ), cTrend, cBandWidth)[j];
//			}
//			mPerc01[i-2][j] = quantiler(vSim, .99);
//			mPerc05[i-2][j] = quantiler(vSim, .95);
//			mPerc10[i-2][j] = quantiler(vSim, .90);
//		}
//	}
//
//	for (decl i=0; i < cNmax-1; ++i)
//	{
//		asRows[i] = sprint(i+2);
//	}
//	for (decl j=0; j < cNmax; ++j)
//	{
//		asCols[j] = sprint(j);
//	}
//
//	println("Number of observations = ", int(cT));
//	println("Number of simulations  = ", int(cSim));
//	println("Trend type (0 nothing, 1 mean, 2 linear): ", cTrend);
//	println("Bandwidth: ", (cBandWidth<0)? int(4*(cT/100)^(2/9)) : cBandWidth);
//	println("--------------------------");
//	println("Size 0.10","%r",asRows,"%c",asCols,mPerc10);
//	println("--------------------------");
//	println("Size 0.05","%r",asRows,"%c",asCols,mPerc05);
//	println("--------------------------");
//	println("Size 0.01","%r",asRows,"%c",asCols,mPerc01);
//}

mhpowers(const cT, const cSim, const cN, const cTrend, const cBandWidth, const fnDistRW, const fnDistNoise, const dVarRatio)
// cT = obs number
// cSim = simulations number
// cN = number of series (>= 2)
// cTrend = 0 (nothing), 1 (de-mean), 2 (de-trend)
// cBandWidth = Bartlett Kernel Bandwidth
// fnDist = function generating rnd numbers, must have two arguments: rows and cols
// dVarRatio = ratio (variance rw shock)/(variance noise)
{
	// 5% critical values
	decl mCritNH = <
		0.7271,  0.2058,  0.0000,  0.0000,  0.0000,  0.0000;
		0.9524,  0.3429,  0.1097,  0.0000,  0.0000,  0.0000;
		 1.165,  0.4615,  0.1833, 0.07170,  0.0000,  0.0000;
		 1.368,  0.5719,  0.2486,  0.1206, 0.05102,  0.0000;
		 1.562,  0.6744,  0.3082,  0.1635, 0.08742, 0.03948>;
	decl mCritINH = <
		0.7204,  0.2078,  0.0000,  0.0000,  0.0000,  0.0000;
		0.9614,  0.3456,  0.1066,  0.0000,  0.0000,  0.0000;
		 1.176,  0.4623,  0.1774, 0.06790,  0.0000,  0.0000;
		 1.363,  0.5750,  0.2396,  0.1132, 0.04856,  0.0000;
		 1.567,  0.6708,  0.2954,  0.1559, 0.08286, 0.03686>;
	decl mCritMNH = <
		0.7167,  0.2078,  0.0000,  0.0000,  0.0000,  0.0000;
		0.9556,  0.3443,  0.1110,  0.0000,  0.0000,  0.0000;
		 1.170,  0.4627,  0.1860, 0.07157,  0.0000,  0.0000;
		 1.369,  0.5702,  0.2495,  0.1204, 0.05151,  0.0000;
		 1.562,  0.6718,  0.3061,  0.1643, 0.08766, 0.03975>;
	// decision counters: rows-number of trends, cols-estimated number of trends
	decl mPowerNH  =	zeros(cN+1, cN+1);
	decl mPowerINH =	zeros(cN+1, cN+1);
	decl mPowerMNH =	zeros(cN+1, cN+1);
	// stdev of rw shocks
	decl dStDev = sqrt(dVarRatio);
	decl mRW, mNoise, mX, vStat;
	
	for (decl i=0; i<cSim; ++i)	 // simulation iterations
	{
		mRW = cumsum(dStDev*fnDistRW(cT, cN), 1);  // RW's
		mNoise = fnDistNoise(cT, cN);			   // WN's
		mX = mNoise;							   // series to be tested
		for (decl j=0; j<=cN; ++j) // iterations on number of trends
		{
			if (j > 0) mX[][:j-1] += mRW[][:j-1]; // add j trends to noise

			vStat = nhstat(mX, cTrend, cBandWidth);   // vector of nh stats
			for (decl k=0; k<=cN; ++k)
			{
				if (k==cN)	// if statistic > all critical values then there are cN trends
				{			// the value k==cN is only reached if all stats are > criticals
					++mPowerNH[j][k];
					break;
				}
				if (vStat[k] < mCritNH[cN-2][k])
				{
					++mPowerNH[j][k];
					break;
				}
			}
			vStat = inhstat(mX, cTrend, cBandWidth);   // vector of inh stats
			for (decl k=0; k<=cN; ++k)
			{
				if (k==cN)	// if statistic > all critical values then there are cN trends
				{			// the value k==cN is only reached if all stats are > criticals
					++mPowerINH[j][k];
					break;
				}
				if (vStat[k] < mCritINH[cN-2][k])
				{
					++mPowerINH[j][k];
					break;
				}
			}
			vStat = mnhstat(mX, cTrend, cBandWidth);   // vector of mnh stats
			for (decl k=0; k<=cN; ++k)
			{
				if (k==cN)	// if statistic > all critical values then there are cN trends
				{			// the value k==cN is only reached if all stats are > criticals
					++mPowerMNH[j][k];
					break;
				}
				if (vStat[k] < mCritMNH[cN-2][k])
				{
					++mPowerMNH[j][k];
					break;
				}
			}
		}
	}
	println("Size-corrected powers of NH, INH and MNH tests");
	println("Number of observations = ", int(cT));
	println("Number of simulations  = ", int(cSim));
	println("Number of time series = ", int(cN));
	println("Signal-noise ratio = ", dVarRatio);
	println("RW shock distribution: ", fnDistRW);
	println("Noise distribution: ", fnDistNoise);
	println("Trend type (0 nothing, 1 mean, 2 linear): ", cTrend);
	println("Bandwidth: ", (cBandWidth<0)? int(4*(cT/100)^(2/9)) : cBandWidth);
	println("-----------------------------------------------");
	println("Test NH");
	println(mPowerNH/cSim);
	println("-----------------------------------------------");
	println("Test INH");
	println(mPowerINH/cSim);
	println("-----------------------------------------------");
	println("Test MNH");
	println(mPowerMNH/cSim);
}

rant5(const r, const c)
{
	return rant(r,c,5);
}

rant2(const r, const c)
{
	return rant(r,c,2);
}

rant1(const r, const c)
{
	return rancauchy(r,c);
}

nhsizecheck(const cT, const cSim, const cNmax, const cTrend, const cBandWidth,
		    const distRW, const distNoise)
{
	decl vSim    = new matrix[1][cSim];
	decl mPerc05 = new matrix[cNmax-1][cNmax];
	decl asCols  = new array[cNmax];
	decl asRows  = new array[cNmax-1];
	decl mCriticals;
	if (cTrend == 1)
	{
		mCriticals =
		<
			0.72710,	0.20580;				
			0.95240,	0.34290,	0.10970;			
			1.16500,	0.46150,	0.18330,	0.07170;		
			1.36800,	0.57190,	0.24860,	0.12060,	0.05102;	
			1.56200,	0.67440,	0.30820,	0.16350,	0.08742,	0.03948
		>;

	}
	if (cTrend == 2)
	{
		mCriticals =
		<
			0.24440,	0.10450;				
			0.33610,	0.17720,	0.07462;			
			0.42140,	0.24330,	0.12860,	0.05541;		
			0.50660,	0.30620,	0.17640,	0.09506,	0.04289;	
			0.59060,	0.37640,	0.22600,	0.13250,	0.07476,	0.03482
		>;
	}
	
	for (decl i=2; i<=cNmax; ++i)  // number of series
	{
		for (decl j=0; j<i; ++j)   // number of trends
		{
			for (decl k=0; k<cSim; ++k)
			{
				vSim[k] = nhstat( (cumsum(distRW(cT,j),1)~distNoise(cT,i-j)), cTrend, cBandWidth)[j];
			}
			mPerc05[i-2][j] = meanr(vSim .> mCriticals[i-2][j]); //quantiler(vSim, .95);
		}
	}

	for (decl i=0; i < cNmax-1; ++i)
	{
		asRows[i] = sprint(i+2);
	}
	for (decl j=0; j < cNmax; ++j)
	{
		asCols[j] = sprint(j);
	}

	println("NH test");
	println("Number of observations = ", int(cT));
	println("Number of simulations  = ", int(cSim));
	println("Trend type (0 nothing, 1 mean, 2 linear): ", cTrend);
	println("Bandwidth: ", (cBandWidth<0)? int(4*(cT/100)^(2/9)) : cBandWidth);
	println("Random Walk distribution: ", distRW);
	println("White Noise distribution: ", distNoise);
	println("--------------------------");
	println("Nominal Size 0.05","%r",asRows,"%c",asCols,mPerc05);
}

inhsizecheck(const cT, const cSim, const cNmax, const cTrend, const cBandWidth,
		    const distRW, const distNoise)
{
	decl vSim    = new matrix[1][cSim];
	decl mPerc05 = new matrix[cNmax-1][cNmax];
	decl asCols  = new array[cNmax];
	decl asRows  = new array[cNmax-1];
	decl mCriticals;
	if (cTrend == 1)
	{
		mCriticals =
		<
			0.72040,	0.20780;				
			0.96140,	0.34560,	0.10660;			
			1.17600,	0.46230,	0.17740,	0.06790;		
			1.36300,	0.57500,	0.23960,	0.11320,	0.04856;	
			1.56700,	0.67080,	0.29540,	0.15590,	0.08286,	0.03686
		>;

	}
	if (cTrend == 2)
	{
		mCriticals =
		<
			0.25130,	0.10940;				
			0.34850,	0.18780,	0.07868;			
			0.43870,	0.25930,	0.13700,	0.05997;		
			0.52940,	0.32820,	0.18990,	0.10420,	0.04804;	
			0.61610,	0.39460,	0.24140,	0.14470,	0.08351,	0.04011
		>;
	}
	
	for (decl i=2; i<=cNmax; ++i)  // number of series
	{
		for (decl j=0; j<i; ++j)   // number of trends
		{
			for (decl k=0; k<cSim; ++k)
			{
				vSim[k] = inhstat( (cumsum(distRW(cT,j),1)~distNoise(cT,i-j)), cTrend, cBandWidth)[j];
			}
			mPerc05[i-2][j] = meanr(vSim .> mCriticals[i-2][j]); //quantiler(vSim, .95);
		}
	}

	for (decl i=0; i < cNmax-1; ++i)
	{
		asRows[i] = sprint(i+2);
	}
	for (decl j=0; j < cNmax; ++j)
	{
		asCols[j] = sprint(j);
	}

	println("INH test");
	println("Number of observations = ", int(cT));
	println("Number of simulations  = ", int(cSim));
	println("Trend type (0 nothing, 1 mean, 2 linear): ", cTrend);
	println("Bandwidth: ", (cBandWidth<0)? int(4*(cT/100)^(2/9)) : cBandWidth);
	println("Random Walk distribution: ", distRW);
	println("White Noise distribution: ", distNoise);
	println("--------------------------");
	println("Nominal Size 0.05","%r",asRows,"%c",asCols,mPerc05);
	println("--------------------------");
}

//main()
//{
//	format("%#8.4g");
//	println("Simulation on raw data");
//	nhstatsim( 5000, 50000, 6, 1, 0);	 // cT, cSim, cNmax, 1=de-mean, -1=auto-bandwidth
//	println("Simulation on signs (IHS statistic)");
//	inhstatsim(5000, 50000, 6, 1, -1);	 // cT, cSim, cNmax, 1=de-mean, -1=auto-bandwidth
//	println("Simulation on integrated signs of increments (MHS statistic)");
//	mnhstatsim(5000, 50000, 6, 1, -1);	 // cT, cSim, cNmax, 1=de-mean, -1=auto-bandwidth
//}
