#include <oxstd.h>
#include <oxdraw.h>
#include <oxfloat.h>
#include "rq.ox"

enum {QUADRATIC, BARTLETT};
enum {ANDREWS, NEWEYWEST};

static decl M_KERNEL = BARTLETT;
static decl M_BANDWIDTH = NEWEYWEST;
static decl M_VERB=1;

quadratic(const mX)	// quadratic spectral M_KERNEL Andrews (1991) p.821
{
	decl mOut;
	mOut = ( sin(mX*6*M_PI/5) ./ (mX*6*M_PI/5) - cos(mX*6*M_PI/5) )*25
			    ./ (12*sqr(M_PI*mX));
//	mOut = mOut .* (fabs(mX) .>= 0.5) + (fabs(mX) .< 0.5 );
//	mOut = mOut.*(!isdotnan(mOut)) + isdotnan(mOut);
	return mOut;
}

bartlett(const mX)
{
	return (fabs(mX) .< 1) .* (1 - fabs(mX));
}

quadraticbw(const mX)
{
	decl cK = columns(mX);
	decl cT = rows(mX);
	decl mY = mX - meanc(mX);

	decl temp;
	decl rho  = zeros(1,cK);
	decl sig2 = zeros(1,cK);
	for (decl i=0; i<cK; ++i)
	{
		olsc(mX[:cT-2][i],mX[1:][i], &temp);
		rho[i] = (temp>0.99)? 0.99 : temp;
		sig2[i] = varc(mX[:cT-2][i] - rho[][i]*mX[1:][i]);
	}
	decl dAlpha2 = sumr( 4*(rho.*sig2).^2 ./ (1 - rho).^8 ) /
	               sumr( sig2.^2 ./ (1 - rho).^4 );
	decl dBandwidth = min(1.3221*(dAlpha2*cT)^(1/5), cT/8);
	return dBandwidth;
}

bartlettbw(const mX)
{
	decl cK = columns(mX);
	decl cT = rows(mX);
	decl mY = mX - meanc(mX);

	decl temp;
	decl rho  = zeros(1,cK);
	decl sig2 = zeros(1,cK);
	for (decl i=0; i<cK; ++i)
	{
		olsc(mX[:cT-2][i],mX[1:][i], &temp);
		rho[i] = (temp>0.99)? 0.99 : temp;
		sig2[i] = varc(mX[:cT-2][i] - rho[][i]*mX[1:][i]);
	}
	decl dAlpha1 = sumr( 4*(rho.*sig2).^2 ./ ((1 - rho).^6 .* (1 + rho).^2) ) /
	               sumr( sig2.^2 ./ (1 - rho).^4 );
	decl dBandwidth = min(1.1447*(dAlpha1*cT)^(1/3), cT/2);
	return int(dBandwidth);
}

neweywest(decl cT)
{
	if (M_KERNEL == BARTLETT) return int(4*(cT/100)^(2/9));
	if (M_KERNEL == QUADRATIC) return 4*(cT/100)^(2/25);
}

ikpss(const vY, const bTrend, const cTrunc)
// bTrend: if 0 constant, otherwise linear
// cTrunc: if -1 automatic, otherwise as specified
{
	decl cT = rows(vY);
	decl vEps;
	
	if (bTrend)
	{
		RCI = 0;	  // rq parameter
		decl mX = range(1,cT)';
		decl aOut = rq(mX, vY, 0.5); // LAD regression
		vEps = vY - (ones(cT,1)~mX)*aOut[0]; // LAD regr. residuals
	}
	else  vEps =  vY - quantilec(vY);	// median centered data
	
	decl vIEps, dVar, dS;

	// sign maker (for zero, in case of even number, set a tollerance)
	if (cT/2 == int(cT)/2) vIEps = (vEps .> 0) - (vEps .< 0);
	else vIEps = (vEps .> 0.000001) - (vEps .< -0.000001);

	decl vS = cumsum(vIEps,1); // cumulated residuals
	if (cTrunc == -1)
	{
		if (M_BANDWIDTH == ANDREWS)
		{
			if (M_KERNEL == QUADRATIC) dS = quadraticbw(vIEps);
			if (M_KERNEL == BARTLETT ) dS =  bartlettbw(vIEps);
		}
		if (M_BANDWIDTH == NEWEYWEST)
		{
			if (M_KERNEL == QUADRATIC) dS = neweywest(cT);
			if (M_KERNEL == BARTLETT ) dS = neweywest(cT);
		}
	}
	else dS = cTrunc;

	if (cTrunc == 0) dVar = meanc(fabs(vIEps));
	else
	{
		decl mCross, mKer;
		if (M_KERNEL == QUADRATIC)
		{
			mCross = vIEps*vIEps';
			mKer = toeplitz(quadratic(range(0,cT-1)/dS+.000001));
			dVar = meanc(sumr(mKer .* mCross));
		}
		if (M_KERNEL == BARTLETT)
		{
			dVar = meanc(fabs(vIEps));
			for (decl j=1; j<dS; ++j)
			{
				dVar += 2*bartlett(j/dS)*(vIEps[0:cT-1-j]'vIEps[j:])/cT;
			}
		}
	}
	
	decl dKPSS = sumsqrc(vS)/(dVar*cT^2);
	if (M_VERB)
	{
		println("Observations: ", cT);
		println("Kernel: ", (M_KERNEL == BARTLETT)? "Bartlett" : "Quadratic Spectral");
		println("Bandwidth = ", dS);
		println("Long-run var = ", double(dVar));
		println("KPSS = ", double(dKPSS));
		println("Critical values:");
		println("%c",{"90%","95%","99%"},
			bTrend? <0.119, 0.146, 0.216>:
					<0.347, 0.463, 0.739>);
		println("---------------------------------------");
	}
	return dKPSS;
}

kpss(const vY, const bTrend, const cTrunc)
// bTrend: if 0 constant, otherwise linear
// cTrunc: if -1 automatic, otherwise as specified
{
	decl cT = rows(vY);
	decl vEps;
	
	if (bTrend)
	{
		decl mX = ones(cT,1)~range(1,cT)';
		decl vBeta; olsc(vY,mX,&vBeta); // regression
		vEps = vY - mX*vBeta; // regr. residuals
	}
	else  vEps =  vY - meanc(vY);	// mean centered data
	
	decl dVar, dS;

	decl vS = cumsum(vEps,1); // cumulated residuals
	if (cTrunc == -1)
	{
		if (M_BANDWIDTH == ANDREWS)
		{
			if (M_KERNEL == QUADRATIC) dS = quadraticbw(vEps);
			if (M_KERNEL == BARTLETT ) dS =  bartlettbw(vEps);
		}
		if (M_BANDWIDTH == NEWEYWEST)
		{
			if (M_KERNEL == QUADRATIC) dS = neweywest(cT);
			if (M_KERNEL == BARTLETT ) dS = neweywest(cT);
		}
	}
	else dS = cTrunc;

	if (cTrunc == 0) dVar = sumsqrc(vEps);
	else
	{
		decl mCross, mKer;
		if (M_KERNEL == QUADRATIC)
		{
			mCross = vEps*vEps';
			mKer = toeplitz(quadratic(range(0,cT-1)/dS+.000001));
			dVar = meanc(sumr(mKer .* mCross));
		}
		if (M_KERNEL == BARTLETT)
		{
			dVar = meanc(fabs(vEps));
			for (decl j=1; j<dS; ++j)
			{
				dVar += 2*bartlett(j/dS)*(vEps[0:cT-1-j]'vEps[j:])/cT;
			}
		}
	}
	
	decl dKPSS = sumsqrc(vS)/(dVar*cT^2);
	if (M_VERB)
	{
		println("Observations: ", cT);
		println("Kernel: ", (M_KERNEL == BARTLETT)? "Bartlett" : "Quadratic Spectral");
		println("Bandwidth = ", dS);
		println("Long-run var = ", double(dVar));
		println("KPSS = ", double(dKPSS));
		println("Critical values:");
		println("%c",{"90%","95%","99%"},
			bTrend? <0.119, 0.146, 0.216>:
					<0.347, 0.463, 0.739>);
		println("---------------------------------------");
	}
	return dKPSS;
}

sim(decl cT, decl cSim, decl vBeta, decl dSig)
{
	decl bM_VERB = M_VERB;
	M_VERB = 0;
	decl vTrend = (ones(cT,1)~range(1,cT)')*vBeta;
	decl vY;
	decl vTests = new matrix[cSim][1];
	for (decl i=0; i<cSim; ++i)
	{
		vY = vTrend + rann(cT,1)*dSig;
		vTests[i] = ikpss(vY, TRUE, -1);
	}
	savemat("ikpss.in7", vTests);
	println("%c",{".90",".95",".99"},quantiler(vTests',<.90,.95,.99>));
	M_VERB =bM_VERB;
}


main0()
{
	decl asVarNames;
	decl mData = loadmat("Prices.xls",&asVarNames);																																										 
	decl avSeries = {<0,168>,<1,168>,<2,168>,<3,168>,<5,168>,<7,168>};
	decl vY, cT, dBw;
	M_KERNEL = QUADRATIC;

	decl mTests = new matrix[2][sizeof(avSeries)];

	for (decl i=0; i<sizeof(avSeries); ++i)
	{
		println("\nSeries: ", asVarNames[avSeries[i][0]]);
		cT = rows(mData[avSeries[i][1]:][avSeries[i][0]]);
		dBw = neweywest(cT);
		mTests[0][i]=ikpss(mData[avSeries[i][1]:][avSeries[i][0]],FALSE, dBw);
		mTests[1][i]=ikpss(mData[avSeries[i][1]:][avSeries[i][0]],TRUE, dBw);
	}
	println(mTests);
//	sim(1000, 10000, <2.652709;0.003269>, .25);
}

//main1()
//{
//	decl asVarNames, vY;
//	decl mData = loadmat("Prices.xls",&asVarNames)[168:][];
//	decl mTests = new matrix[4][4];
//	for (decl i=0; i<4; ++i)
//	{
//		for (decl j=i+1; j<4; ++j)
//		{
//			vY=mData[][i]-mData[][j];
//			mTests[i][j]=ikpss(vY, FALSE, -1);
//		}
//	}
//	println("%c",asVarNames[:3],"%r",asVarNames[:3],mTests);
//}
//
//main()
//{
//	decl asVarNames;
//	decl mData = (loadmat("MeanPrices.xls",&asVarNames));																																										
//	decl avSeries = {<4,168>,<5,168>,<6,168>,<9,168>};
//	decl vY, cT, dBw;
//	M_KERNEL = QUADRATIC;
//	M_BANDWIDTH = NEWEYWEST;
//
//	decl mTests = new matrix[2][sizeof(avSeries)];
//
//	for (decl i=0; i<sizeof(avSeries); ++i)
//	{
//		println("\nSeries: ", asVarNames[avSeries[i][0]]);
//		cT = rows(mData[avSeries[i][1]:][avSeries[i][0]]);
//		dBw = -1;//neweywest(cT);
//		mTests[0][i]=ikpss(mData[avSeries[i][1]:][avSeries[i][0]],FALSE, dBw);
//		mTests[1][i]=ikpss(mData[avSeries[i][1]:][avSeries[i][0]],TRUE, dBw);
//	}
//	println(mTests);
//}