/*
**  Program PostAnal.Ox
**
**  Purpose:
**    Analyse the posterior. Calculated are the mean, mode, standard
**    deviation and highest posterior density region of the posterior,
**    together with the signal-to-noise ratio (defined as the
**    variance of the observation equation divided by the unconditional
**    variance of the transition equation) and Harvey's q, which is the
**    variance of the observation equation disturbance divided by the variance 
**    of the transition equation disturbance.
**
**    For each model, a graph with extension uah.plb is written with
**    the marginal prior, posterior and HPD region.
**
**  Author:
**    Charles S. Bos
**
**  Date:
**    7/11/2000
*/
#include <oxstd.h>
#include <oxfloat.h>
#include <oxprob.h>
#include <include/gnudraw_jae.h>
#include "include/oxprobig.ox"
#include "include/size.ox"

#include "simox.dec"

AnalPostI(const amPostI, const vParIndex, const avPrior, 
          const aParNames, const dAlpha, const sInbase);
PrintTable(const mPost, const aTexNames);
CalcHPD(const mDens, const dAlpha, const bVerbose);
CalcPrior(const vX, const iInd, const vPrior);

main()
{
  decl dAlpha, aParNames, aTexNames, aMatNames, aParIndex, aPrior,
       nPars, nMat, i, j, mPost, mPostI, vIndex, aLine, vMultFact,
       vOrder, iInd, sInd, asMat, dTime;

  dTime= timer();
  dAlpha= 0.95;
  aParNames= {"mu", "rho", "seta", "seps", "delta", "alpha", "nu",
              "muH", "phi", "sxi"};
  asMat= {"ma", "mb", "mc", "md", "me", "mf", "mg"};
  aTexNames= {"$\\mu \\times 100$", "$\\rho$", "$\\sigma_\\eta \\times 10$", "$\\sigma_\\epsilon$", 
              "$\\delta$", "$\\alpha \\times 10$", "$\\nu$", "$\\mu_h$", "$\\phi$", 
              "$\\sigma_\\xi$"};

  // Simulation filenames are of format excl/XX/XXYY.fmt with
  //   XX = model indication
  //   YY = country indication, usually du=DM/USD
  sInd= g_OutDir[sizeof(g_OutDir)-2:];
  for (i= 0, aMatNames= {}; i < 7; ++i)
    aMatNames~= {sprint("excl/", asMat[i], sInd, "/", asMat[i])};

  // What parameters are used in which models?
  aParIndex= {<0, 3>, <2, 3>, <1, 2, 3>, <1, 2, 3, 4, 5>,
              <1, 2, 3, 8, 9, 7>, <1, 2, 3, 6>, <1, 2, 3, 4, 5, 6>};
  aPrior= {<0, .02>, <.8, .2>, <2.25, 100>, <2.5, 4/3>, <>, <>, <>, 
           <-1, 1>, <.5, .3>, <2.5, 4/3>};
  // Multiplication factors for usage in tables
  vMultFact= <100; 1; 10; 1; 1; 10; 1; 1; 1; 1; 100>;
  // Order of the presentation of the models
  vOrder= <0, 1, 2, 3, 4, 5, 6>;

  nPars= sizeof(aParNames);
  nMat= sizerc(vOrder);

  fopen("excl/postanal.out", "l");
  println ("PostAnal");
  println ("--------");

  mPost= new matrix [nPars+1][5*nMat];
  for (i= 0; i < nMat; ++i)
    { 
      iInd= vOrder[i];
      AnalPostI(&mPostI, aParIndex[iInd], aPrior[aParIndex[iInd]], aParNames, 
                dAlpha, aMatNames[iInd]);
      mPost[][5*i:5*i+4]= mPostI;
    }
  mPost = mPost .* vMultFact;
  
  println("Table with models WN, LL, GLL: ");
  PrintTable(mPost[:3][:3*5-1]|mPost[nPars][:3*5-1], aTexNames);
  println("Table with models GLL-GARCH, -SV, -Student t and -GARCH-Student t: ");
  PrintTable(mPost[1:][3*5:], aTexNames[1:]);

  println("Time elapsed:\n ", "%10s", timespan(dTime));
}

/*
**  AnalPostI(const amPostI, const vParIndex, const avPrior, 
**            const aParNames, const dAlpha, const sInbase)
**
**  Purpose:
**    Calculate mean, mode, sdev and HPD for a model
*/
AnalPostI(const amPostI, const vParIndex, const avPrior, 
          const aParNames, const dAlpha, const sInbase)
{
  decl mTheta, nDim, nPars, i, iInd, mRet, vHPD, vPrior, vX, vSEps, mS, mN;

  mTheta = loadmat(sprint(sInbase, ".fmt"), 1);
  if (!(vParIndex != 9))         // Replace MuH by constructed seps in the SV model
    {
      vSEps= sqrt(exp(mTheta[][2]+0.5*sqr(mTheta[][4])./(1-sqr(mTheta[][3]))));
      mTheta= mTheta[][:1]~vSEps~mTheta[][3:]~mTheta[][2];
    }

  if (sizeof(mTheta) > 0)
    {
      println ("Analysing posterior of ", sInbase, " based on ", 
               rows(mTheta), " drawings");

      nDim= sizerc(vParIndex);
      nPars= sizeof(aParNames);
      amPostI[0]= new matrix [nPars+1][5];
      amPostI[0]*= M_NAN;
      if (!(vParIndex != 1))  // If rho included in model
        {
          mS= sqr(mTheta[][1])./(1-sqr(mTheta[][0]));
          mN= sqr(mTheta[][2]);

          // Signal-to-noise ratio var(mu)/var(eps)
          amPostI[0][nPars][0]= meanc(mS./mN);
        }

      // Harvey's signal-to-noise ratio q
      if (!(vParIndex != 0))   // If mu included in model
        amPostI[0][nPars][1]= 0;
      else 
        {
          // Harvey's signal-to-noise ratio q
          i= (!(vParIndex != 1));        // Skip rho, if needed
          amPostI[0][nPars][1]= meanc(sqr(mTheta[][0+i]./mTheta[][1+i]));
        }

      for (i= 0; i < nDim; ++i)
        {
          iInd= vParIndex[i];
          println("Working on parameter ", aParNames[iInd]);

          amPostI[0][iInd][0]= meanc(mTheta[][i]);
          amPostI[0][iInd][2]= sqrt(varc((mTheta[][i])));

          // Check out the mode, forget histogram
          mRet= DensEst(mTheta[][i], 0, 0, 0, 128)[<0, 2>][];
          amPostI[0][iInd][1]= mRet[0][ limits(mRet[1][]')[3][0] ];

          // Calculate the highest contiguous posterior density region
          vHPD= CalcHPD(mRet, dAlpha, FALSE);
          amPostI[0][iInd][3:4]= vHPD;

          vX= mRet[0][];
          vPrior= CalcPrior(vX, iInd, avPrior[i]);
          DrawXMatrix(i, vPrior, "Prior", vX, "", 0, 3);
          DrawXMatrix(i, mRet[1][], "Posterior", mRet[0][], "");
          DrawXMatrix(i, 0~dAlpha*ones(1, 2)/(vHPD[1]-vHPD[0])~0, "HPD", 
    	  			     vHPD ** ones(1, 2), "", 0, 4);
          DrawAdjust(ADJ_LABEL, aParNames[iInd]);
        }

      SaveDrawWindow(sprint(sInbase, "uah.plb"));
      CloseDrawWindow();

      vX= acf(mTheta, 60);
      DrawMatrix(0, vX[1:][]', aParNames[vParIndex], 1, 1);
      SaveDrawWindow(sprint(sInbase, "acf.plb"));
      CloseDrawWindow();

      print ("Results ");
      print ("%r", aParNames~{"S/N, q"}, "%c", 
             {"mean", "mode", "sdev", "L", "U"}, amPostI[0]); 
    }
  else
    println ("Matrix ", sInbase, ".fmt not found");
}

/*
**  PrintTable(const mPost, const aTexNames)
**
**  Purpose:
**    Print a table in a format close to the LaTeX format
*/
PrintTable(const mPost, const aTexNames)
{
  decl aLine, i, j, nPars, nMat;

  nMat= columns(mPost)/5;
  nPars= rows(mPost)-1;
  for (i= 0; i < nPars; ++i)
    {
      println(aTexNames[i]);
      aLine= "";
      for (j= 0; j < nMat; ++j)
        if (isnan(mPost[i][j*5]))
          aLine= aLine~" & &";
        else
          aLine= aLine~sprint(" & ", "%.2f", mPost[i][j*5],
                              " & (", "%.2f", mPost[i][2+j*5], ")");
      println (aLine, " \\\\");

      aLine= "";
      for (j= 0; j < nMat; ++j)
        if (isnan(mPost[i][1+j*5]))
          aLine= aLine~" & &";
        else
          aLine= aLine~sprint(" & ", "%.2f", mPost[i][1+j*5], 
                              " & [", "%.2f,", mPost[i][3+j*5], "%.2f", mPost[i][4+j*5], "]");
      println (aLine, " \\\\");
      println ("\\hline");
    }

  println("S/N $\\times$ 100");
  aLine= "";
  for (j= 0; j < nMat; ++j)
    if (isnan(mPost[nPars][j*5]))
      aLine= aLine~" & &";
    else
      aLine= aLine~sprint(" & \\multicolumn{2}{c}{", "%.2f", mPost[nPars][j*5], "}");
  println (aLine, " \\\\");

  println("q(Harvey) $\\times$ 100");
  aLine= "";
  for (j= 0; j < nMat; ++j)
    if (isnan(mPost[nPars][j*5+1]))
      aLine= aLine~" & &";
    else
      aLine= aLine~sprint(" & \\multicolumn{2}{c}{", "%.2f", mPost[nPars][j*5+1], "}");

  println (aLine, " \\\\");
  println ("\\hline");
}

/*
**  CalcHPD(const mDens, const dAlpha, const bVerbose)
**
**  Purpose:
**    Calculate the highest posterior density region for the density
**    given, of percentage dAlpha. Only unimodal HPD's can be
**    calculated.
**
**  Inputs:
**    mDens       2 x N matrix with X and f(X). It is assumed that f(X)
**                integrates to one, and that X is equidistant
**    dAlpha      Coverage probability
**
**  Outputs:
**    r.v.        1 x 2 matrix with HPD borders
*/
CalcHPD(const mDens, const dAlpha, const bVerbose)
{
  decl vX, vdX, vCDF, iN, mCov, dCov, vHPD, i, j, dC0, dC1, dY, 
  	   idCov, iwCov, mHPD, vSDens;

  iN= columns(mDens);
  vHPD= new matrix [1][2];
  vX= mDens[0][];
  vdX= diff0(vX', 1)';
  if (!isfeq(max(vdX[1:]), min(vdX[1:])))
    println("Warning: Not equidistant? vX, vdX= ", vX|vdX);

  // vCDF is a row vector
  vCDF= cumulate(mDens[1][]')' .* vdX[1];
  mCov= vCDF' - vCDF;
  // Save in each column a one where ij < dalpha and ij+1 > dalpha
  idCov= (mCov[:iN-2][] .< dAlpha) .&&
  		 (mCov[1:][] .> dAlpha);
  // Find in endpoints of intervals, this is row index
  idCov= limits(idCov)[3][];
  iwCov= idCov - range(0, iN-1);
  
  // Negative widths of intervals are not allowed
  iwCov[vecrindex(iwCov .< 0)]= M_INF;

  // Still multiple possibilities  
  mHPD= vecrindex(iwCov .== min(iwCov))~
  	    idCov[vecrindex(iwCov .== min(iwCov))]';
  vSDens= fabs(mDens[1][mHPD[][0]]-mDens[1][mHPD[][1]])';
  
  // Choose mHPD where distance between densities is smallest
  vHPD= mHPD[limits(vSDens)[2][0]][];
  
  dCov= vCDF[vHPD[1]]-vCDF[vHPD[0]];
				
  // Calculate approximate rates of change at the corners
  dC0= M_INF;
  if (vHPD[0] > 0)
    dC0= (mDens[1][vHPD[0]]-mDens[1][vHPD[0]-1])/vdX[1];

  dC1= M_INF;
  if (vHPD[1] < iN-1)
    dC1= (mDens[1][vHPD[1]]-mDens[1][vHPD[1]+1])/vdX[1];

  dY= sqrt(2*(dAlpha-dCov)/(1/dC0 + 1/dC1));
  
  if (bVerbose)
    {
      print   ("Initial HPD-i                     ", vHPD);
      print   ("Change in dens, left-right:       ", dC0~dC1);
      println ("Minimal change in Y:              ", dY);
      println ("Resulting change in X left-right: ", dY./(dC0~dC1)); 
      println ("Resulting change in cov prob:     ", .5*dY.*dY./(dC0~dC1));
      println ("Previous and total cov prob:      ",
               dCov~(dCov+(.5*dY*dY*(1/dC0+1/dC1))));
      println ("Left and right CDF:               ",
      		   vCDF[vHPD]~(vCDF[vHPD[1]]-vCDF[vHPD[0]]));
    }

  vHPD= vX[vHPD];
  if (dC0 < M_INF)
    vHPD[0]-= vdX[1]*dY/dC0;
  if (dC1 < M_INF)
    vHPD[1]+= vdX[1]*dY/dC1;

  return vHPD;
}


/*
**  CalcPrior()
*/
CalcPrior(const vX, const iInd, const vPrior)
{
  decl vY;

  vY= M_NAN;
  if (!(iInd != <0, 1, 7, 8>))
    vY= densn((vX-vPrior[0])/vPrior[1])/vPrior[1];
  else if (!(iInd != <2, 3, 9>))
    vY= densigamma(vX.*vX, vPrior[0], vPrior[1]) .* vX * 2;
  else if (!(iInd != <4, 5>))
    vY= ones(vX)*2;
  else if (iInd == 6)
    vY= denst(vX, 1)/probt(-2, 1);
  else
    println ("iInd ", iInd, " not understood...");

  return vY;
}

