/*
**  Program NumAcc.ox
**
**  Purpose:
**    Calculate the tapered covariance, and numerical accuracy
**
**  Note:
**    Based on MixAnal in mixox.zip
**
**  Version:
**    3     Using Newey-West, see Hamilton, p 281., univariate
**
**  Author:
**    Charles Bos
**
**  Date:
**    21/6/2000
**
*/
#include <oxstd.h> // include the Ox standard library header
#include <oxfloat.h> // M_2PI declaration
#include "include/gnudraw_jae.h"
#include "include/printmat.ox"
#include "include/info.ox"
#include "include/size.ox"
#include "include/setseed.ox"

/* Include declarations of the model  */
#include "simox.dec"

// Function declarations
CalcAutoCov(mTheta, const dTaper, const sSimbase);
CalcVariance(const mGamma, const vTaper, const sSimbase);

main()                // function main is the starting point
{
  decl 
    nMax, vTaper, sSimbase, sSimfile, mTheta, vTheta_Mean, vTheta_Var,
    mVarTheta, i, j, dFrac,
    dS2Fac, nDim, nLagSh, nLag, fmt, mVTap, mGamma, mVariance,
    mX, acLabs, arLabs, aPrFmt, aTexNames, aParNames, vParIndex,
    iInd, iM, dTime;

  /* Initialize */
  dTime= timer();
  aTexNames= {"\\mu", "\\rho", "\\sigma_\\eta", "\\sigma_\\epsilon", 
              "\\delta", "\\alpha", "\\nu", "\\mu_h", "\\phi", 
              "\\sigma_\\xi"};
  aParNames= {"mu", "rho", "seta", "seps", 
              "delta", "alpha", "nu", "muH", "phi", 
              "sxi"};

  nLagSh= 5;             // Number of AC matrices to show
  nMax= 10000;           /* maximum number of Theta's used in calculation
                            of autocovariances */
  vTaper= <0; .04; .08; .15>;
  sSimbase= sprint(g_OutDir, "/", g_VersFile);
  sSimfile = sprint(sSimbase, ".fmt");

  fopen(sSimbase~"na3.out", "l");
  println ("NumAcc");
  println ("------");
  println ("Calculating posterior mean and variance from ", sSimfile);

  dFrac= 1;
  mTheta = loadmat(sSimfile, 1);
  if (dFrac <= 1)
    dFrac *= rows(mTheta);
  mTheta= thinr(mTheta, dFrac);
  nDim= columns(mTheta);
  iM= rows(mTheta);
  vTheta_Mean= meanc(mTheta)';
  vTheta_Var= varc(mTheta);
  vParIndex= zeros(1, nDim);
  for (j= 0; j < nDim; ++j)
    for (i= 0; i < sizeof(aParNames); ++i)
      if (aParNames[i] == g_VarNames[j])
        vParIndex[j]= i;

  mX= constant(M_NAN, rows(vTaper), sizeof(aTexNames));

  mGamma= CalcAutoCov(mTheta, max(vTaper), sSimbase);
  mVariance= CalcVariance(mGamma, ceil(iM*vTaper), sSimbase);

  print ("Variance of mean Theta, normally calculated: ",
         vTheta_Var); 
  print ("Variance of mean Theta, matrix calculated: ",
         diagonal(variance(mTheta)));

  for (i= 0; i < rows(vTaper); ++i)
    {
      mVTap= mVariance[i];

      print ("Using tapering ", vTaper[i], ":", mVTap);

      mX[i][vParIndex]= vTheta_Var ./ mVTap;
      print ("Relative numerical accuracy: ");
      print (mX[i][vParIndex]);
      println("");
    }

  aPrFmt= acLabs= new array[sizeof(aTexNames)];
  for (i= 0; i < sizeof(aTexNames); ++i)
    {
      aPrFmt[i]= "%5.2f";
      acLabs[i]= aTexNames[i];
    }
  arLabs= new array[rows(mX)];
  for (i= 0; i < rows(mX); ++i)
    arLabs[i]= sprint("RNA, $\\tau=$", vTaper[i]);
  PrintMatrix(0, g_Vers, aPrFmt, acLabs, arLabs, mX, FALSE);

  fmt= new array[nDim+1];
  fmt[0]= "  AC %2.0f: ";
  for (i= 1; i <= nDim; ++i)
    fmt[i]= "%10.4f";

  print("Autocorrelations: ");
  println("%cf", fmt, range(0,nLagSh)'~acf(mTheta, nLagSh));

  println("Time elapsed:\n ", "%10s", timespan(dTime));
}

/*
**  CalcAutoCov(mTheta, const dTaper, const sSimbase)
**
**  Purpose:
**    Calculate the autocovariances
**
**  Inputs:
**    mTheta      M x nDim matrix of Theta's
**    vTaper      Maximum percentage of tapering to use
**
**  Output:
**    mGamma      Array with autocovariances.
**
*/
CalcAutoCov(mTheta, const dTaper, const sSimbase)
{
  decl
    nDim, iM, iL, mGamma, i, mAutoCov;

nDim= columns(mTheta);
iM= rows(mTheta);
iL= ceil(dTaper*iM);
mTheta= mTheta-meanc(mTheta);
if (iL > 0)
  mTheta= zeros(iL, nDim)|mTheta;

mAutoCov= 0;
for (i= 0; i < iM; ++i)
  {
    if (imod(i, 500) == 0)
      info (i, iM);
    mAutoCov= mAutoCov + mTheta[i:i+iL][] .* mTheta[i+iL][];
      // Covariance of size iL+1 x nDim, with Cov(t, t) in last row
  }
info (iM, iM);

/*
DrawMatrix(0, (mAutoCov[range(iL, 0, -1)][]./iM)', "", 0, 1);
DrawMatrix(1, (mAutoCov[range(iL-1, 0, -1)][]./iM)', "", 1, 1);
SaveDrawWindow(sprint(sSimbase, "nacf.plb"));
// ShowDrawWindow();
CloseDrawWindow();
*/

mGamma= new array [iL+1];
for (i= 0; i < iL+1; ++i)
  mGamma[i]= mAutoCov[iL-i][]/iM;

return mGamma;
}

/*
**  CalcVariance(const mGamma, const vTaper, const sSimbase)
**
**  Purpose:
**    Calculate the variance adjusted for correlation
**
**  Inputs:
**    mGamma      array with autocovariances
**    vTaper      nTap vector, percentages of tapering to use
**
**  Output:
**    mVariance   array of size nTap with Newey-West variance adjusted for correlation
**
*/
CalcVariance(const mGamma, const vTaper, const sSimbase)
{
  decl
    iL, i, j, nL, mVariance, mX;

iL= vTaper;
nL= sizerc(iL);
mVariance= new array[nL];

mVariance[0]= mGamma[0];
for (i= 0; i < nL; ++i)
  {
    mVariance[i]= mGamma[0];
    mX= zeros(iL[i]+1, columns(mGamma[0]));
    mX[0][]= mGamma[0];
    for (j= 1; j < iL[i]+1; ++j)
      {
        mVariance[i]+= (1-j/iL[i])*(2*mGamma[j]);
        mX[j][]= ((1-j/iL[i])*(2*mGamma[j]));
      }
    mX= mX ./ mX[0][];
  }

return mVariance;
}


