/*
**
**  Program MLWN.ox
**
**  Purpose:
**    Run Gibbs sampling on the White Noise
**    model, in order to get the marginal likelihood. 
**
**  Author:
**    Charles S. Bos
**
**  Date:
**    12/5/2000
**
**  Version:
**    1     Based on MLGL2.ox
**
**  The model:
**    We sample from the model
**          y(t)= mu + sigma(eps) eps(t)
**
**  The marginal likelihood is calculated as
**           f(y|theta, mu) pi(theta, mu)
**    m(y) = ----------------------------------
**                   P(theta, mu|y)
**  in a point (theta= mean posterior, mu= filtered mean)
*/
#include <oxstd.h>
#include <oxfloat.h>    // M_INF_NEG
#include <oxprob.h>     // RanGamma
#include <arma.h>       // armagen
#include <ssfpack.h>
#include "include/info.ox"      // Information on lenght of calc
#include "include/setseed.ox"   // Reset the seed?
#include "include/gnudraw_jae.h"
#include "include/oxprobig.ox"
#include "include/tractime.ox"
#include "include/size.ox"

#include "simox.dec"    // Declaration of the model

// Function declarations
RunMLChain(dMu, dSEps, 
           const mYtEst, const nReps, const nSkip, const iSel);
SampleMu(const mYt, const dSEps, const vPriorMu);
LnDensMu(const dMu, const mYt, const dSEps, const vPriorMu);
SampleS2(const ve, const vPriorS2AB);
LnDensS2(const dS2, const ve, const vPriorS2AB);
InitGibWN(const vInitVP, const sSimFile, const amYtEst, const adMu,
          const adSEps);

main()                // function main is the starting point
{
  decl 
    sSimBase, sSimFile, sOutFile, 
    nReps, nRot, nSkip, nPeriods, iT, mPost, dTime,
    dMu, dSEps, mYtEst, mDMY, ir, dLnPrior, dLnLikl, dLnML, i;

  /* Initialize */
  if (g_Kalm_UseModel != "WN")
    {
      print ("Error: Incorrect declarations file?");
      exit(1);
    }

  dTime= timer();
  nRot= sizerc(g_Flex_nMH);
  nReps= g_Flex_nMH[nRot-1]/g_Flex_nFact;
  nSkip= floor(g_Flex_nSkip/g_Flex_nFact);

  sSimBase= sprint(g_OutDir, "/", g_VersFile);
  sSimFile= sprint(sSimBase, ".fmt");
  sOutFile= sprint(sSimBase, "ml.out");

  fopen(sOutFile, "l");
  println ("Writing output to ", sSimFile, " and ", sOutFile);
  InitGibWN(g_InitVP, sSimFile, &mYtEst, &dMu, &dSEps);
  iT= columns(mYtEst);

  mPost= zeros(2, 2);
  for (i= 0; i < 2; ++i)
    mPost[][i]= 
      RunMLChain(dMu, dSEps, mYtEst, 
                 nReps, nSkip, g_FreePars[i]);

  dLnPrior= LnPrior_Shell(dMu|dSEps);
  // LnPdf_Shell calculates mean posterior, not only the likelihood
  ir= LnPdf_Shell(dMu|dSEps, &dLnLikl, 0, 0);
  dLnLikl= dLnLikl*iT - dLnPrior;
  dLnML= dLnLikl + dLnPrior - sumr(log(mPost[0][])+mPost[1][]);

  print ("Posteriors:                 ", mPost);
  print ("Log Posteriors:             ", "%10.2f", log(mPost[0][])+mPost[1][]);
  println ("Sum log Posteriors:         ", "%10.2f", double(sumr(log(mPost[0][])+mPost[1][])));
  println ("Log conditional likelihood: ", "%10.2f", double(dLnLikl));
  println ("Log prior:                  ", "%10.2f", double(dLnPrior));
  println ("Log Marginal likelihood:    ", "%10.2f", double(dLnML));
  println ("Marginal likelihood:        ", "%10g", double(exp(dLnML)));

  println("Time elapsed:\n ", "%10s", timespan(dTime));
}

/*
**
**  Function RunChain(const sSimFile, const vP, const nReps)
**
**  Purpose:
**    Run the chain of the Gibbs sampler. 
**
**  Inputs:
**    vP          vector, starting point of the chain
**    nReps       scalar, number of repetitions
**    iSel        scalar, indicating which parameters to fix, which to
**                sample
**
**  Outputs:
**    dPost       scalar, with reduced complete conditional posterior 
**                estimate for element iSel
*/
RunMLChain(dMu, dSEps, const mYtEst, const nReps, const nSkip, const iSel)
{
  decl
    nChain, nX, nHist, nPeriods, i, iOut, vMu, ve, 
    ir, vLnPost, dLnPostMax, vPars, vParsOld,
    dS2Eps, aPars;

  /* Run the chain */
  vPars= {"Mu", "S2Eps"};
  println ("Running the chain for parameter ", vPars[iSel], "...");
  TrackTime(vPars);

  iOut= 0;
  vLnPost= new matrix [1][ceil(nReps/(nSkip+1))];

  aPars= {dMu, dSEps};
  vParsOld= 0;

  for (i= 0; i < nReps; ++i)
    {
    if (imod(i, g_Flex_InfoRep) .== 0)
      {
        info(i, nReps);
        TrackReport();
      }

    TrackTime(0);
    if (iSel <= 0)
      {
        dMu= SampleMu(mYtEst, dSEps, g_Kalm_PriorMu);
        if ((imod(i, nSkip+1) == 0) && (iSel == 0))
          {
            vLnPost[iOut]= LnDensMu(aPars[0], mYtEst, dSEps, g_Kalm_PriorMu);
//            print (vLnPost[iOut]~aPars[0]~dMu~dSEps);
            ++iOut;
          }
      }

    TrackTime(1);
    if (iSel <= 1)
      {
        ve= (mYtEst - dMu);
        dS2Eps= SampleS2(ve, g_Kalm_PriorS2EpsAB);
        dSEps= sqrt(dS2Eps);
        if ((imod(i, nSkip+1) == 0) && (iSel == 1))
          {
            // Add density of SEps, calculated from S2Eps and jacobian
            vLnPost[iOut]= LnDensS2(aPars[1]*aPars[1], ve, g_Kalm_PriorS2EpsAB) 
                          + log(2*aPars[1]);
            ++iOut;
          }
      }
    TrackTime(-1);
    }
  info(nReps, nReps);

  dLnPostMax= max(vLnPost);
  return (meanr(exp(vLnPost[0][:iOut-1]-dLnPostMax)))|dLnPostMax;
}


/*
**  SampleMu(const mYt, const dSEps, const vPriorMu)
**
**  Purpose:
**    Sample Mu, conditional on the data and the
**    standard deviations of epsilon
**
**  Inputs:
**    mYt         1 x n row vector of data
**    dSEps       scalar, sdev of observation equation
**
**  Outputs:
**    dMu   Return value, scalar with sampled value of Mu
*/
SampleMu(const mYt, const dSEps, const vPriorMu)
{
  decl
    dMu, iN, dMean, dVar;

  iN= columns(mYt);
  dVar= sqr(dSEps*vPriorMu[1])/(iN*sqr(vPriorMu[1]) + sqr(dSEps));
  dMean= (sqr(vPriorMu[1])*sumr(mYt) + sqr(dSEps)*vPriorMu[0])/
                  (iN*sqr(vPriorMu[1]) + sqr(dSEps));
//  print (dMean~sqrt(dVar)~dVar~sumr(mYt));

  return rann(1, 1)*sqrt(dVar)+dMean;
}

/*
**  LnDensMu(const dMu, const mYt, const dSEps, const vPriorMu)
**
**  Purpose:
**    Sample Mu, conditional on the data and the
**    standard deviations of epsilon
**
**  Inputs:
**    mYt         1 x n row vector of data
**    dSEps       scalar, sdev of observation equation
**
**  Outputs:
**    dMu   Return value, scalar with sampled value of Mu
*/
LnDensMu(const dMu, const mYt, const dSEps, const vPriorMu)
{
  decl
    iN, dMean, dVar;

  iN= columns(mYt);
  dVar= sqr(dSEps*vPriorMu[1])/(iN*sqr(vPriorMu[1]) + sqr(dSEps));
  dMean= (sqr(vPriorMu[1])*sumr(mYt) + sqr(dSEps)*vPriorMu[0])/
                  (iN*sqr(vPriorMu[1]) + sqr(dSEps));

  return log(densn((dMu-dMean)/sqrt(dVar)))-0.5*log(dVar);
}

/*
**  SampleS2(const ve, const vPriorS2AB)
**
**  Purpose:
**    Sample a variance, conditional on the errors ve,
**    applying the IG prior
**
**  Inputs:
**    ve          row vector with disturbances
**    vPriorS2AB  alpha and beta of the IG(alpha, beta) prior
**
**  Outputs:
**    dS2         Return value, scalar, variance
*/
SampleS2(const ve, const vPriorS2AB)
{
  decl
    iT, alpha, beta, ds, dS2;

  iT= columns(ve);
  alpha= iT/2 + vPriorS2AB[0];
  ds= sumsqrr(ve) + 2/vPriorS2AB[1];
  beta= 2/ds;
  dS2= ranigamma(1, 1, alpha, beta);

  return dS2;
}

/*
**  LnDensS2(const dS2, const ve, const vPriorS2AB)
**
**  Purpose:
**    Calculate density of a variance, conditional on the errors ve,
**    applying the IG prior
**
**  Remark:
**    See SampleS2
*/
LnDensS2(const dS2, const ve, const vPriorS2AB)
{
  decl
    iT, alpha, beta, ds, dLnDensS2;

  iT= columns(ve);
  alpha= iT/2 + vPriorS2AB[0];
  ds= sumsqrr(ve) + 2/vPriorS2AB[1];
  beta= 2/ds;
  dLnDensS2= lndensigamma(dS2, alpha, beta);

  return dLnDensS2;
}

/*
**  InitGibWN
**
**  Purpose:
**    Initialize the static variables in this module, using settings in
**    simox.dec
**
**  Inputs:
**    InitVP      Initial values for vP. Might have been
**                initialized in Kalman.DEC
**    amY         Address to matrix with data
**
**  Outputs:
**    amY         Row vector of data
*/
InitGibWN(const vInitVP, const sSimFile, const amYtEst, const adMu, 
          const adSEps)
{
  decl 
    vP, iT, mInter, mTheta, mYt, mDMY;

  SetSeed(g_Seed);

  println("Gibbs sampling over White Noise model");
  vP= vInitVP;     // Mu, sEps
  PdfInit(&vP, &iT);

  // Look for starting values 
  GetData(&mYt, &mDMY, &mInter);
  amYtEst[0]= mYt[:g_Kalm_FracEst-1];

  mTheta= loadmat(sSimFile);
  vP= meanc(mTheta)';

  adMu[0]= vP[0];
  adSEps[0]= vP[1];
}


