/*
**  Program MLGSV.ox
**
**  Purpose:
**    Run Gibbs sampling on the generalized local level-stochastic volatility
**    model, in order to get the marginal likelihood. 
**
**  Author:
**    Charles S. Bos
**
**  Date:
**    1/6/2000
**
**  Version:
**    1     Based on MlGAS3.ox and GibSV16.ox
**    2     Using sampling routines from incsv4
**
**  The model:
**    We sample from the model
**          y(t)= mu(t) + eps(t)e^(h(t)/2)
**          mu(t)= rho mu(t-1) + eta(t)
**          h(t)= muh + phi (h(t) - muh) + xi(t)
**          eps(t) ~ norm(0, 1)
**          eta(t) ~ norm(0, s2eta)
**          xi(t)  ~ norm(0, s2xi)
**
**    The SV is approximated as in Kim, Shephard and Chib:
**          ln(y-mu)^2 = z(t) + h(t)
**    with z(t) being a mixture of 7 normals,
**          z(t)= ln(eps(t)^2)
**          f(z(t)) approx sum qi f_Norm(z|mi-1.2704, vi^2)
**                i= 1, .., 7
*/
#include <oxstd.h>
#import <maximize>      // The optimization routines 
#include <oxfloat.h>    // M_INF_NEG
#include <oxprob.h>     // RanGamma
#include <ssfpack.h>
#include "include/info.ox"      // Information on lenght of calc
#include "include/setseed.ox"   // Reset the seed?
#include "include/gnudraw_jae.h"
#include "include/libkern.ox"
#include "include/oxprobig.ox"
#include "include/size.ox"
#include "include/tractime.ox"

#include "simox.dec"    // Declaration of the model

// Function declarations
RunMLChain(dRho, dSEta, dMuH, dPhi, dSXi, vh, vis,
           const mYt, const nReps, const nSkip, const iSel);
LnDensRho(const dRho, const vMu, const dSEta, const vPriorRho);
LnDensS2(const dS2, const ve, const vPriorS2AB);
LnDensS(const vis, const vYStar, const vh);
LnDensH(const vH, const vYStar, const vis, const dPhi, const dMuH, const dSXi);
LnDensPhi(const dPhi, const vh, const dMuH, const dSXi, 
          const vPriorPhi);
LnDensMuH(const dMuH, const vh, const dPhi, const dSXi, const vPriorMuH);
LnDensS2Xi(const dS2Xi, const dPhi, const dMuH, const vh, const vPriorS2AB);
LnPriorH(vh, const dMuH, const dPhi, const dSXi);
LnPdfSVCond(const dRho, const dSEta, const dMuH, const dPhi, const dSXi, 
            const vh, const vis, const mYt, const adLnLikl);
InitGibSV(const vInitVP, const sSimFile, const amYtEst, 
          const adRho, const adSEta, const adMuH, const adPhi, 
          const adSXi, const avh, const avis);

/*
**  Static declarations, on dimension of problem, index of present
**  parameter, and the full vector of parameters
*/
static decl s_GS_mYStar,
            s_GS_MaxPars, s_GS_MaxLL= M_INF_NEG,
            s_GS_nOut= 1000, 
            s_SV_Filter= FALSE;

main()                // function main is the starting point
{
  decl 
    sSimBase, sSimFile, sOutFile, sDRetFile, 
    nReps, nRot, nSkip, nPeriods, iT, ir,  
    dRho, dSEta, dMuH, dPhi, dSXi, vh, vis, vX, vHist, 
    mYt, mDMY, i, mPost, dLnLikl, dLnPrior, vLnPrior, dLnML, dTime;

  /* Initialize */
  if (g_Seed != 0)
    SetSeed(g_Seed);

  if (g_Kalm_UseModel != "GLLSV")
    {
      print ("Error: Incorrect declarations file?");
      exit(1);
    }

  dTime= timer();
  nRot= sizerc(g_Flex_nMH);
  nReps= g_Flex_nMH[nRot-1]/g_Flex_nFact;
  nSkip= floor(g_Flex_nSkip/g_Flex_nFact);

  sSimBase= sprint(g_OutDir, "/", g_VersFile);
  sSimFile= sprint(sSimBase, ".fmt");
  sOutFile= sprint(sSimBase, "ml.out");

  fopen(sOutFile, "l");
  println ("Writing output to ", sOutFile);
  InitGibSV(g_InitVP, sSimFile, &mYt, &dRho, &dSEta, &dMuH, &dPhi, 
            &dSXi, &vh, &vis);

  iT= columns(mYt);
  mPost= new matrix [2][6];
  for (i= 0; i < 6; ++i)
    mPost[][i]= 
      RunMLChain(dRho, dSEta, dMuH, dPhi, dSXi, vh, vis, mYt, 
                 nReps, nSkip, i);

  ir= LnPdfSVCond(dRho, dSEta, dMuH, dPhi, dSXi, vh, vis,  
                  mYt, &dLnLikl);

  // Prior of the normal parameters, plus the startup of h[0], plus the 
  // uniform discrete prior on s, which is t times 1/7.
  [dLnPrior, vLnPrior]= LnPrior_Shell(dRho|dSEta|dMuH|dPhi|dSXi, TRUE);
  vLnPrior= LnPriorH(vh, dMuH, dPhi, dSXi)|vLnPrior;
  dLnML= dLnLikl + sumc(vLnPrior) - sumr(log(mPost[0][])+mPost[1][]);

  print ("Posteriors:                   ", 
         "%c", {"h", "Rho", "SEta", "MuH", "Phi", "SXi"}, mPost);
  print ("Log Posteriors:               ", log(mPost[0][])+mPost[1][]);
  println ("Sum log Posteriors:         ", "%10.2f", double(sumr(log(mPost[0][])+mPost[1][])));
  println ("Log conditional likelihood: ", "%10.2f", double(dLnLikl));
  println ("Log prior:                  ", "%10.2f", vLnPrior');
  println ("Log prior:                  ", "%10.2f", double(sumc(vLnPrior)));
  println ("Log Marginal likelihood:    ", "%10.2f", double(dLnML));
  println ("Marginal likelihood:        ", "%10.2f", double(exp(dLnML)));

  println("Time elapsed:\n ", "%10s", timespan(dTime));
}

/*
**  RunMLChain(const sSimFile, const vP, const nReps)
**
**  Purpose:
**    Run the chain of the Gibbs sampler. 
**
**  Inputs:
**    vP          vector, starting point of the chain, containing the 
**                free parameters from the sequence Rho, SEta, MuH,
**                Phi, SXi
**    nReps       scalar, number of repetitions
**    iSel        scalar, indicating which parameters to fix, which to
**                sample
**
**  Outputs:
**    dPost       scalar, with reduced complete conditional posterior 
**                estimate for element iSel
*/
RunMLChain(dRho, dSEta, dMuH, dPhi, dSXi, vh, vis, const mYt, const nReps, 
           const nSkip, const iSel)
{
  decl
    i, j, vMu, vS, vYStar, dS2Eta, dS2Xi,
    iOut, vLnPost, dLnPostMax, 
    ve, iT, aPars, vPars, vParsOld, mPars;

  vPars= {"Mu", "iMixt", "vh", "Rho", "SEta", "MuH", "Phi", "SXi"};
  println ("Running the chain for parameter ", vPars[iSel+2], "...");
  TrackTime(vPars);

  // Run the chain
  iOut= 0;
  mPars= new matrix [nReps][6];
  vLnPost= new matrix [1][ceil(nReps/(nSkip+1))];

  vParsOld= vMu= vS= 0;
  aPars= {vh, dRho, dSEta, dMuH, dPhi, dSXi};
  iT= columns(mYt);

  for (i= 0; i < nReps; ++i)
    {
    if (imod(i, g_Flex_InfoRep) .== 0)
      {
        info(i, nReps);
        TrackReport();
      }

    TrackTime(0);
    vPars= dRho~dSEta~sumr(vh);
    if (!(vPars == vParsOld))
      { 
        vParsOld= vPars;
        vMu= SampleMu(vh, dRho, dSEta, mYt);
      }

    vYStar= log(sqr(mYt-vMu));
    TrackTime(1);
    vis= SampleS(vYStar, vh);

    TrackTime(2);
    if (iSel <= 0)
      {
        vh= SampleH(vYStar, vis, dPhi, dMuH, dSXi);
        if ((imod(i, nSkip+1) == 0) && (iSel == 0))
          {
            vLnPost[iOut]= LnDensH(aPars[0], vYStar, vis, dPhi, dMuH, dSXi);
            ++iOut;
          }
      }

    TrackTime(3);
    if (iSel <= 1)
      {
        SampleRho(&dRho, vMu, dSEta, g_Kalm_PriorRho);
        if ((imod(i, nSkip+1) == 0) && (iSel == 1))
          {
            vLnPost[iOut]= LnDensRho(aPars[1], vMu, dSEta, g_Kalm_PriorRho);
            ++iOut;
          }
      }

    TrackTime(4);
    if (iSel <= 2)
      {
        ve= (vMu - dRho * lag0(vMu', 1)')[1:];
        dS2Eta= SampleS2(ve, g_Kalm_PriorS2EtaAB);
        dSEta= sqrt(dS2Eta);
        if ((imod(i, nSkip+1) == 0) && (iSel == 2))
          {
            // Add density of SEta, calculated from S2Eta and jacobian
            vLnPost[iOut]= LnDensS2(aPars[2]*aPars[2], ve, g_Kalm_PriorS2EtaAB) 
                          + log(2*aPars[2]);
            ++iOut;
          }
      }

    TrackTime(5);
    if (iSel <= 3)
      {
        dMuH= SampleMuH(vh, dPhi, dSXi, g_Kalm_PriorMuH);
        if ((imod(i, nSkip+1) == 0) && (iSel == 3))
          {
            vLnPost[iOut]= LnDensMuH(aPars[3], vh, dPhi, dSXi, g_Kalm_PriorMuH);
            ++iOut;
          }
      }

    TrackTime(6);
    if (iSel <= 4)
      {
        dPhi= SamplePhi(vh, dMuH, dSXi, g_Kalm_PriorPhi);
        if ((imod(i, nSkip+1) == 0) && (iSel == 4))
          {
            vLnPost[iOut]= LnDensPhi(aPars[4], vh, dMuH, dSXi, g_Kalm_PriorPhi);
            ++iOut;
          }
      }

    TrackTime(7);
    if (iSel <= 5)
      {
        dS2Xi= SampleS2Xi(dPhi, dMuH, vh, g_Kalm_PriorS2XiAB);
        dSXi= sqrt(dS2Xi);
        if ((imod(i, nSkip+1) == 0) && (iSel == 5))
          {
            // Add density of SEta, calculated from S2Eta and jacobian
            vLnPost[iOut]= LnDensS2Xi(aPars[5]*aPars[5], dPhi, dMuH, vh, g_Kalm_PriorS2XiAB);
                          + log(2*aPars[5]);
            ++iOut;
          }
      }
    TrackTime(-1);
	
    mPars[i][]= meanr(vh)~dRho~dSEta~dMuH~dPhi~dSXi;
    }

  info(nReps, nReps);
  print ("Mean pars: ", meanc(mPars));
  vPars= {"vh", "Rho", "SEta", "MuH", "Phi", "SXi"};
/*
  DrawDensity(0, mPars[][iSel:]', vPars[iSel:],
  			  TRUE, FALSE, FALSE);
  SaveDrawWindow(sprint("excl/mlp", iSel, ".plb"));
  ShowDrawWindow();
*/
  
  vLnPost= deletec(vLnPost[0][:iOut-1]);
  dLnPostMax= max(vLnPost);

  return (meanr(exp(vLnPost-dLnPostMax)))|dLnPostMax;
}

/*
**  LnDensRho(const dRho, const vMu, const dSEta, const vPriorRho)
**
**  Purpose:
**    Calculate density of Rho, conditional on the state Mu and the variance of the
**    equation
**
**  Remark:
**    See SampleRho
*/
LnDensRho(const dRho, const vMu, const dSEta, const vPriorRho)
{
  decl
    iN, invXX, dMuRho, dS2Rho, dRhoHat, dS2RhoHat, dLnDensRho;

  if (columns(vPriorRho) == 2)
    {
      println("Error: Incorrect non-normal prior on Rho");
      exit(1);
    }
  iN= columns(vMu);
  olsr(vMu[1:], vMu[:iN-2], &dRhoHat, &invXX);
  dS2RhoHat= dSEta*dSEta*invXX;

  dMuRho= (dRhoHat*vPriorRho[1]^2 + vPriorRho[0]*dS2RhoHat)/
           (vPriorRho[1]^2+dS2RhoHat);
  dS2Rho= dS2RhoHat * vPriorRho[1]^2 / (vPriorRho[1]^2+dS2RhoHat);
  dLnDensRho= -0.5*(log(M_2PI*dS2Rho)+sqr(dRho-dMuRho)/dS2Rho);

  return dLnDensRho;
}

/*
**  LnDensS2(const dS2, const ve, const vPriorS2AB)
**
**  Purpose:
**    Calculate density of a variance, conditional on the errors ve,
**    applying the IG prior
**
**  Remark:
**    See SampleS2
*/
LnDensS2(const dS2, const ve, const vPriorS2AB)
{
  decl
    iT, alpha, beta, ds, dLnDensS2;

  iT= columns(ve);
  alpha= iT/2 + vPriorS2AB[0];
  ds= sumsqrr(ve) + 2/vPriorS2AB[1];
  beta= 2/ds;
  dLnDensS2= lndensigamma(dS2, alpha, beta);

  return dLnDensS2;
}

/*
**  LnDensS(const vis, const vyStar, const vh)
**
**  Purpose:
**    Calculate logdensity of S, the indicator into the mixture of normals used to
**    approximate the log squared normal distribution
**
**  Inputs:
**    vis         row vector with indices
**    vyStar      row vector with log((y(t)-mu(t))^2)
**    vh          row vector with sampled values of h(t)
**
**  Outputs:
**    r.v.        scalar, log density of vector vis.
*/
LnDensS(const vis, const vYStar, const vh)
{
  decl
    vQi, vMui, vSi, mfti, mftia, vLnDensS, iT, j;

    // Size 7 x 1
  vQi= s_GS_qmv[][0];
    // Size 7 x 1
  vMui= s_GS_qmv[][1]-1.2704;
    // Size 7 x 1
  vSi= sqrt(s_GS_qmv[][2]);

    // Calculate q_i * f_Norm(Ystar - h - mui, S2i), size 7 x N
//  mfti= (vQi ./ vSi) .* densn((vYStar - vh - vMui)./vSi);
  mfti= log(vQi) - log(vSi) - 0.5* sqr((vYStar - vh - vMui)./vSi);

//  print (exp(mfti[][:4]) ./ mftia[][:4]);
  
    // Normalize
  mfti= exp(mfti-limits(mfti)[1][]);
  mfti= mfti ./ sumc(mfti);

    // Size 1 x N
  iT= columns(vYStar);
  vLnDensS= zeros(vYStar);
  for (j= 0; j < iT; ++j)
    {
      vLnDensS[0][j]= mfti[vis[j]][j];
      if (vLnDensS[0][j] < 1e-80)
        print (mfti[][j]',
               vLnDensS[0][j]~vis[j]~vYStar[j]~vh[j]~vMui[vis[j]]);
    }
//  print (sumr(log(vLnDensS))~min(vLnDensS)~max(vLnDensS));

  return sumr(log(vLnDensS));
}

/*
**  LnDensH(const vh, const vYStar, const vis, const dPhi, 
**          const dMuH, const dSXi)
**
**  Purpose:
**    Calculate logdensity of H, conditional on (a function of) the data, 
**    the values of indices vis, Phi, MuH, and the variance of Xi. See
**    note of 5/6/2000 for density
**
**  Inputs:
**    vh          Stochastic variance elements
**    vYStar      row vector, function of the data
**
**  Outputs:
**    r.v.        Logdensity of vh
*/
LnDensH(const vh, const vYStar, const vis, const dPhi, const dMuH, const dSXi)
{
  decl
    vMui, vS2i, vLnDensH, ve, vMuH, vS2H;

  // Calculate base mean and variance 
  vMui= s_GS_qmv[][1]-1.2704;
  vS2i= s_GS_qmv[][2];

  // Relate it to the vector of indices
  vMui= vMui[vis]';
  vS2i= vS2i[vis]';

  vMuH= (dSXi*dSXi * (vYStar-vMui) + vS2i.*(dMuH+dPhi*lag0(vh'-dMuH, 1)'))./
               (vS2i+dSXi*dSXi);
  vS2H= (vS2i*dSXi*dSXi)./(vS2i+dSXi*dSXi);

  // Initial condition
  vMuH[0]= (dSXi*dSXi * (vYStar[0]-vMui[0]) + (1-sqr(dPhi))*vS2i[0]*dMuH)./
               ((1-sqr(dPhi))*vS2i[0]+dSXi*dSXi);
  vS2H[0]= (vS2i[0]*dSXi*dSXi)./((1-sqr(dPhi))*vS2i[0]+dSXi*dSXi);

  // Loglikelihood consists of likelihood contribution
  ve= vh-vMuH;
  vLnDensH= -0.5*(log(M_2PI*vS2H)+sqr(ve)./vS2H);
  
//  print (meanr(ve)~meanr(sqr(ve)./vS2H));
/*
  Draw(0, ve);
  Draw(1, sqr(ve)|vS2H);
  ShowDrawWindow();
  exit(1);
*/
  return sumr(vLnDensH);
}

/*
**  LnDensPhi(const dPhi, const vh, const dMuH, const dSXi, 
**            const vPriorPhi)
**
**  Purpose:
**    Calculate logdensity of Phi, conditional on the values of h, the
**    const dMuH and the variance of Xi. 
**
**  Inputs:
**    vh          row vector, values of h(t)
**    dMuH        scalar, value of MuH
**    dSXi        scalar, sdev of Xi
**    vPriorPhi   vector, mean and sdev of normal prior
**
**  Outputs:
**    r.v.        scalar with logdensity
*/
LnDensPhi(const dPhi, const vh, const dMuH, const dSXi, const vPriorPhi)
{
  decl
    iN, XX, vhMu, dS2PhiHat, dPhiHat, dMuPhi, dSPhi;

  iN= columns(vh);
  vhMu= vh-dMuH;
  XX= sumsqrr(vhMu[:iN-2]);
  dS2PhiHat= dSXi*dSXi/XX;
  dPhiHat= vhMu[:iN-2]*vhMu[1:]'/XX;

  dMuPhi= (dPhiHat*vPriorPhi[1]^2 + vPriorPhi[0]*dS2PhiHat)/
            (vPriorPhi[1]^2+dS2PhiHat);
  dSPhi= sqrt(dS2PhiHat * vPriorPhi[1]^2 / (vPriorPhi[1]^2+dS2PhiHat));

  return LnPriorNorm(dPhi, dMuPhi, dSPhi);
}

/*
**  LnDensMuH(const dMuH, const vh, const dPhi, const dSXi)
**
**  Purpose:
**    Calculate logdensity of MuH, conditional on the state h and the
**    variance of the equation
**
**  Inputs:
**    vh          row vector with sampled values of H
**    dPhi        scalar, ather constant
**    dSXi        scalar, sdev of Xi
**    vPriorMuH   row vector, mean and sDev of normal prior on MuH
**
**  Outputs:
**    dMuH        Return value, scalar, value of MuH
*/
LnDensMuH(const dMuH, const vh, const dPhi, const dSXi, const vPriorMuH)
{
  decl
    iN, dS2MuH, dMuHHat, dN, dLnDensMuH;

  iN= columns(vh);
  dS2MuH= dSXi*dSXi/((iN-1)*(1-dPhi)^2 + (1-dPhi^2));
  dMuHHat= (dS2MuH/(dSXi*dSXi))
             *((1-dPhi^2)*vh[0] + (1-dPhi)*sumr(vh[1:]-dPhi*vh[:iN-2]));

  // Equivalent number of observations in prior information
  dN= vPriorMuH[1] * vPriorMuH[1] / dS2MuH;
  dMuHHat= (dN * dMuHHat + vPriorMuH[0])/(dN+1);
  dS2MuH= dS2MuH * dN/(dN+1);

  dLnDensMuH= -0.5*(log(M_2PI*dS2MuH)+sqr(dMuH-dMuHHat)/dS2MuH);

  return dLnDensMuH;
}

/*
**  LnDensS2Xi(const dS2Xi, const dPhi, const dMuH, const vh, 
**             const vPriorS2Xi)
**
**  Purpose:
**    Calculate logdensity of S2Xi, conditional on the state h, mean MuH and 
**    the AR parameter Phi
**
**  Inputs:
**    dPhi        scalar, value of Phi
**    dMuH        scalar, value of MuH
**    vh          row vector with sampled values of h
**    vPriorS2XiAB 2 x 1 vector with hyperparameters alpha and beta for IG-prior
**
**  Outputs:
**    dLnDensS2Xi    Return value, scalar, logdensity of S2Xi
*/
LnDensS2Xi(const dS2Xi, const dPhi, const dMuH, const vh, const vPriorS2AB)
{
  decl
    iT, alpha, beta, ds, vhMu, dLnDensS2;

  iT= columns(vh);
  alpha= iT/2 + vPriorS2AB[0];
  vhMu= vh-dMuH;
  ds= (1-dPhi^2)*vhMu[0]^2 + sumsqrr(vhMu[1:]-dPhi*vhMu[:iT-2]) 
        + 2/vPriorS2AB[1];
  beta= 2/ds;
  dLnDensS2= lndensigamma(dS2Xi, alpha, beta);

  return dLnDensS2;
}

/*
**  LnPriorH(vh, const dMuH, const dPhi, const dSXi)
**
**  Purpose:
**    Calculate the log-prior of h
*/
LnPriorH(vh, const dMuH, const dPhi, const dSXi)
{
  decl i, vLnPriorH;

  vh= vh - dMuH;
  vLnPriorH= zeros(vh);
  vLnPriorH[0]= LnPriorNorm(vh[0], 0, dSXi/sqrt(1-dPhi^2));
  for (i= 1; i < columns(vh); ++i)
    vLnPriorH[i]= LnPriorNorm(vh[i], dPhi*vh[i-1], dSXi);

//  print (vLnPriorH);

  return sumr(vLnPriorH);
}

/*
**  LnPdfSVCond(const dRho, const dSEta, const dMuH, const dPhi, const dSXi, 
**              const vh, const vis, const mYt, const adLnLikl)
**
**  Purpose:
**    Calculate the likelihood of the SV model, for given values of
**    H and S.
**
**  Remark:
**    Note that vis is not used; likelihood does not depend on vis once
**    h is given.
*/
LnPdfSVCond(const dRho, const dSEta, const dMuH, const dPhi, const dSXi, 
            const vh, const vis, const mYt, const adLnLikl)
{
  decl
    mPhi, mOmega, mSigma, mDelta, mJ_Phi, mJ_Omega, mJ_Delta, vS2Eps,
    ir, dVar;

  mPhi= dRho|1;
  mOmega= diag(dSEta*dSEta|1);
  mSigma= <-1; 0>;
  if (dRho < 1)
    mSigma= dSEta*dSEta/(1-dRho*dRho)|0;
  mJ_Phi= mDelta= mJ_Delta= <>;

  // Indicate that the variance of epsilon is time varying
  mJ_Omega= <-1, -1; -1, 0>;

  // Create a vector with the variances of epsilon
  vS2Eps= exp(vh);
  ir= SsfLik(adLnLikl, &dVar, mYt, mPhi, mOmega, mSigma, mDelta, mJ_Phi,
             mJ_Omega, mJ_Delta, vS2Eps);

  return ir;
}

/*
**
**  Procedure InitGibSV
**
**  Purpose:
**    Initialize the static variables in this module, using settings in
**    GibSV.Dec
**
**  Inputs:
**    InitVP      Initial values for vP. Might have been
**                initialized in Kalman.DEC
**    amY         Address to matrix with data
**
**  Outputs:
**    amY         Row vector of data
**
*/
InitGibSV(const vInitVP, const sSimFile, const amYtEst, 
          const adRho, const adSEta, const adMuH, const adPhi, 
          const adSXi, const avh, const avis)
{
  decl 
    dLnPdf, vP, mTheta, mPhi, mSigma, mDelta, mOmega, nData, ir, 
    vMuHat, vh, vis, mKF, dFrac, mInter, mDMY, mYt, vYStar, vMu, nK,
    mivis, mvh, i, j, iT;

  println("Gibbs sampling over Local level-Stochastic Volatility model");
  println("Parameters Rho, SEta, MuH, Phi, SXi are used");
  adRho[0]= vInitVP[0];
  adSEta[0]= vInitVP[1];
  adMuH[0]= vInitVP[2];
  adPhi[0]= vInitVP[3];
  adSXi[0]= vInitVP[4];

  vP= vInitVP;
  PdfInit(&vP, &g_nData);
  GetData(&mYt, &mDMY, &mInter);
  amYtEst[0]= mYt[:g_Kalm_FracEst-1];
  mDMY= mDMY[][:g_Kalm_FracEst-1];
  iT= columns(amYtEst[0]);

  mTheta= loadmat(sSimFile);
  vP= meanc(mTheta);

  adRho[0]= vP[0];
  adSEta[0]= vP[1];
  adMuH[0]= vP[2];
  adPhi[0]= vP[3];
  adSXi[0]= vP[4];

  // Calculate the smoothed mean of the ll model
  mPhi= adRho[0]|1;
  mOmega= diag(adSEta[0]*adSEta[0]~exp(adMuH[0]));
  /* Choose a diffuse initial state */
  mSigma=<-1; 0>;
  if (adRho[0] < 1)
    mSigma= adSEta[0]*adSEta[0]/(1-adRho[0]^2)|0;
  mKF= KalmanFil(amYtEst[0], mPhi, mOmega, mSigma);
  vMuHat= amYtEst[0]-mKF[0][];

  // Look for starting values of the SV part
  s_GS_mYStar= log(sqr(amYtEst[0]-vMuHat));

  // Calculate the smoothed mean of the SV model
  mPhi= adPhi[0]|1;
  mDelta= adMuH[0]*(1-adPhi[0])| -1.2704; // vP[3];
  mOmega= diag(adSXi[0]*adSXi[0]~(M_PI^2/2)); //vP[4]*vP[4]);
  /* Choose a diffuse initial state */
  mSigma= -1|(M_PI^2/2);
  if (adPhi[0] < 1)
    mSigma= adSXi[0]*adSXi[0]/(1-adPhi[0]^2)|(M_PI^2/2);

  vh= SsfCondDens(ST_SMO, s_GS_mYStar, mPhi, mOmega, mSigma, mDelta);

  nK= 10;
  vh= vh[0][];
  mvh= zeros(amYtEst[0]);
  mivis= zeros(7, iT);
  for (i= 0; i < 3*nK; ++i)
    {
      vMu= SampleMu(vh, adRho[0], adSEta[0], amYtEst[0]);
      vYStar= log(sqr(amYtEst[0]-vMu));
      if (i < nK)
        {
          vis= SampleS(vYStar, vh);
          for (j= 0; j < iT; ++j)
            ++mivis[vis[j]][j];
        }
      if (i == nK)
        vis= limits(mivis)[3][];

      vh= SampleH(vYStar, vis, adPhi[0], adMuH[0], adSXi[0]);
      if (i >= 2*nK)
        mvh+= vh;
    }
  avis[0]= vis;
  avh[0]= mvh/nK;
  avh[0]= vh;
}

