/*
**
**  Program MLGAS.ox
**
**  Purpose:
**    Run Gibbs sampling on the generalized local level-GARCH-Student t
**    model, in order to get the marginal likelihood. 
**
**  Author:
**    Charles S. Bos
**
**  Date:
**    12/5/2000
**
**  Version:
**    2     Including Richard's remarks: Normal prior on Rho, changing
**          order etc.
**    4     Using sampling routines from incglgs4.ox
**    5     Using freepars
**    6     Further refinements
**
**  The model:
**    We sample from the model
**          y(t)= mu(t) + sigma(eps(t)) eps(t)
**          mu(t)= rho mu(t-1) + sigma(eta) eta(t)
**          h(t)= delta h(t-1) + (1-delta-alpha) + alpha eps(t-1)^2
**          eps(t) ~ t(0, (nu-2)/nu h(t), nu)   (with var(eps(t))= h(t),
**                                               var(eps)= E(h)= 1)
**          eta(t) ~ norm(0, 1)
**    To incorporate student t errors, effectively we sample
**          eps(t) ~ norm(0, z(t))
**          z(t) ~ IG(alpha= nu/2, beta= 2/(nu-2))
**
**  The marginal likelihood is calculated as
**           f(y|theta, z, mu) pi(theta, z, mu)
**    m(y) = ----------------------------------
**                   P(theta, z, mu|y)
**  in a point (theta= mean posterior, mu= filtered mean, z= some mean
**  of simulated z's.
**  
*/
#include <oxstd.h>
#include <oxfloat.h>    // M_INF_NEG
#include <oxprob.h>     // RanGamma
#include <quadpack.h>   // QAGS
#include <arma.h>       // armagen
#include <ssfpack.h>
#include "include/info.ox"      // Information on lenght of calc
#include "include/setseed.ox"   // Reset the seed?
#include "include/gnudraw_jae.h"
#include "include/libkern.ox"
#include "include/oxprobig.ox"
#include "include/tractime.ox"
#include "include/size.ox"

#include "simox.dec"    // Declaration of the model

// Function declarations
RunMLChain(dRho, dSEta, dSEps, dDelta, dAlpha, 
           dNu, vz, const mYtEst, const nReps, 
           const nSkip, const iSel);
LnDensRho(const dRho, const vMu, const dSEta, const vPriorRho);
LnDensS2(const dS2, const ve, const vPriorS2AB);
LnDensZ(const vz, const dSEps, const dDelta, const dAlpha, const dNu, const vv2);
LnDensNu(const dNu, const vz);
KernelDensNu(const dNu);
LnDensGARCH(const dSEps, const dDelta, const dAlpha, const vz, 
            const vv2, const vPriorS2Eps);
InitGibStGa(const vInitVP, const sSimFile, const amYtEst, const adRho, const adSEta,
          const adSEps, const adDelta, const adAlpha, const adNu, const avz);

/*
**  Static declarations, on dimension of problem, index of present
**  parameter, and the full vector of parameters
*/
static decl s_GS_Nu_DrawnAcc= <0; 0>,
            s_GS_Garch_DrawnAcc= <0; 0>, 
            s_GS_OldNu= 0, s_GS_OldGARCH= 0, 
            s_GS_vz= -1, s_GS_vv2, s_GS_CandGARCH, 
            s_GS_KernMax, s_GS_LnDensNu= M_NAN, s_GS_LnDensGARCH= M_NAN, 
            s_GS_vGARCH, s_GS_nRepGARCH= <100, 1000>;

main()                // function main is the starting point
{
  decl 
    sSimBase, sSimFile, sOutFile, 
    nReps, nRot, nSkip, nPeriods, nPars, iT, mPost,
    dRho, dSEta, dSEps, dDelta, dAlpha, dNu, vz, 
    mYtEst, mDMY, ir, dLnPrior, vLnPrior, dLnLikl, dLnML, i, dTime;

  /* Initialize */
  if ((g_Kalm_UseModel != "GLLGAS") && (g_Kalm_UseModel != "GLLGA") &&
      (g_Kalm_UseModel != "GLLGS") && (g_Kalm_UseModel != "GLL"))
    {
      print ("Error: Incorrect declarations file?");
      exit(1);
    }

  dTime= timer();
  nRot= sizerc(g_Flex_nMH);
  nReps= g_Flex_nMH[nRot-1]/g_Flex_nFact;
  nSkip= floor(g_Flex_nSkip/g_Flex_nFact);

  sSimBase= sprint(g_OutDir, "/", g_VersFile);
  sSimFile= sprint(sSimBase, ".fmt");
  sOutFile= sprint(sSimBase, "ml.out");

  fopen(sOutFile, "l");
  println ("Writing output to ", sOutFile);
  InitGibStGa(g_InitVP, sSimFile, &mYtEst, &dRho, &dSEta, &dSEps, &dDelta, &dAlpha,
              &dNu, &vz);
  iT= columns(mYtEst);

  mPost= new matrix [2][5];
  for (i= 0; i < 5; ++i)
    mPost[][i]= 
      RunMLChain(dRho, dSEta, dSEps, dDelta, dAlpha, dNu, vz, mYtEst, 
                 nReps, nSkip, i);

  mPost= deletec(mPost);

  ir= LnPdfGSStudCond(dRho, dSEta, dSEps, dDelta, dAlpha, vz, 
                      mYtEst, &dLnLikl);
  dLnLikl*= iT;
  [dLnPrior, vLnPrior]= LnPrior_Shell(dRho|dSEta|dSEps|dDelta|dAlpha|dNu, TRUE);
  if (!(g_FreePars != 5))
    vLnPrior |= sumr(lndensigamma(vz, dNu/2, 2/(dNu-2)));
  dLnML= dLnLikl + sumc(vLnPrior) - sumr(log(mPost[0][])+mPost[1][]);

  print ("Posteriors:                   ", mPost);
  print ("Log Posteriors:               ", log(mPost[0][])+mPost[1][]);
  println ("Sum log Posteriors:         ", "%10.2f", double(sumr(log(mPost[0][])+mPost[1][])));
  println ("Log conditional likelihood: ", "%10.2f", double(dLnLikl));
  println ("Log prior:                  ", "%10.2f", vLnPrior');
  println ("Sum log prior:                  ", "%10.2f", double(sumc(vLnPrior)));
  println ("Log Marginal likelihood:    ", "%10.2f", double(dLnML));
  println ("Marginal likelihood:        ", "%10.2f", double(exp(dLnML)));

  println("Time elapsed:\n ", "%10s", timespan(dTime));
}

/*
**  RunMLChain(const sSimFile, const vP, const nReps)
**
**  Purpose:
**    Run the chain of the Gibbs sampler. 
**
**  Inputs:
**    vP          vector, starting point of the chain
**    nReps       scalar, number of repetitions
**    iSel        scalar, indicating which parameters to fix, which to
**                sample
**
**  Outputs:
**    dPost       scalar, with reduced complete conditional posterior 
**                estimate for element iSel
*/
RunMLChain(dRho, dSEta, dSEps, dDelta, dAlpha, dNu, vz, 
           const mYtEst, const nReps, const nSkip, const iSel)
{
  decl
    nChain, nX, nHist, nPeriods, i, iOut, vMu, ve, vv2,
    ir, vLnPost, dLnPostMax, vPars, vParsOld,
    dS2Eta, dS2Eps, aPars;

  // Check if this parameter is part of the free parameters
  if (((iSel == 0) && (g_FreePars != 0)) ||
      ((iSel == 1) && (g_FreePars != 1)) ||
      ((iSel == 2) && (g_FreePars != 5)) ||
      ((iSel == 3) && (g_FreePars != 5)) ||
      ((iSel == 4) && (g_FreePars != 2)))
    return M_NAN|M_NAN;

  /* Run the chain */
  vPars= {"Mu", "Rho", "SEta", "Z", "Nu", "GARCH"};
  println ("Running the chain for parameter ", vPars[iSel+1], "...");
  TrackTime(vPars~"VGarch"~"DNu"~"DGarch");

  iOut= 0;
  vLnPost= new matrix [1][ceil(nReps/(nSkip+1))];

  aPars= {dRho, dSEta, vz, dNu, dSEps|dDelta|dAlpha};
  vParsOld= 0;

  for (i= 0; i < nReps; ++i)
    {
    if (imod(i, g_Flex_InfoRep) .== 0)
      {
        info(i, nReps);
		if (s_GS_Garch_DrawnAcc[0] > 0)
          println("Garch accepted, drawn, fraction: ", "%5.0i", 
                  s_GS_Garch_DrawnAcc[1], "%5.0i", s_GS_Garch_DrawnAcc[0], "%6.2f",
                  s_GS_Garch_DrawnAcc[1]/s_GS_Garch_DrawnAcc[0]);
		if (s_GS_Nu_DrawnAcc[0] > 0)
          println("Nu accepted, drawn, fraction:    ", "%5.0i", 
                  s_GS_Nu_DrawnAcc[1], "%5.0i", s_GS_Nu_DrawnAcc[0], "%6.2f",
                  s_GS_Nu_DrawnAcc[1]/s_GS_Nu_DrawnAcc[0]);
        TrackReport();
      }

    TrackTime(0);
    vPars= dRho~dSEta~dSEps~sumr(vz);
    if (!(vPars == vParsOld))
      { 
        vParsOld= vPars;
        vMu= SampleMu(mYtEst, dRho, dSEta, dSEps, dDelta, dAlpha, vz);
      }

    TrackTime(1);
    if (iSel <= 0)
      {
        if (!(g_FreePars != 0))
          SampleRho(&dRho, vMu, dSEta, g_Kalm_PriorRho);
        if ((imod(i, nSkip+1) == 0) && (iSel == 0))
          {
            vLnPost[iOut]= LnDensRho(aPars[0], vMu, dSEta, g_Kalm_PriorRho);
            ++iOut;
          }
      }

    TrackTime(2);
    if ((iSel <= 1) && !(g_FreePars != 1))
      {
        ve= (vMu - dRho * lag0(vMu', 1)')[1:];
        dS2Eta= SampleS2(ve, g_Kalm_PriorS2EtaAB);
        dSEta= sqrt(dS2Eta);
        if ((imod(i, nSkip+1) == 0) && (iSel == 1))
          {
            // Add density of SEta, calculated from S2Eta and jacobian
            vLnPost[iOut]= LnDensS2(aPars[1]*aPars[1], ve, g_Kalm_PriorS2EtaAB) 
                          + log(2*aPars[1]);
            ++iOut;
          }
      }

    TrackTime(3);
    if ((iSel <= 2) && !(g_FreePars != 5))
      {
        vv2= sqr(mYtEst - vMu);
        SampleZ(&vz, dSEps, dDelta, dAlpha, dNu, vv2);
        if ((imod(i, nSkip+1) == 0) && (iSel == 2))
          {
            vLnPost[iOut]= LnDensZ(aPars[2], dSEps, dDelta, dAlpha, dNu, vv2);
            ++iOut;
          }
      }

    TrackTime(4);
    if ((iSel <= 3) && !(g_FreePars != 5))
      {
        ++s_GS_Nu_DrawnAcc[0];
        s_GS_Nu_DrawnAcc[1]+=
          SampleNu(&dNu, vz, g_Kalm_CandNuRW);
        if ((imod(i, nSkip+1) == 0) && (iSel == 3))
          {
            TrackTime(7);
            vLnPost[iOut]= LnDensNu(aPars[3], vz);
            ++iOut;
          }
      }

    TrackTime(5);
    if (iSel <= 4)
      if (!(g_FreePars != 2) && !(g_FreePars != 3) && 
          !(g_FreePars != 4))
        {
          if (i == 0)
            println ("Using GARCH sampling");
          // Sample the GARCH parameters here
          vv2= sqr(mYtEst - vMu);
          ++s_GS_Garch_DrawnAcc[0];
          s_GS_Garch_DrawnAcc[1]+=
            SampleGARCH(&dSEps, &dDelta, &dAlpha, vz, vv2, 
                        dSEps|dDelta|dAlpha, s_GS_CandGARCH[][1],
                        g_Kalm_PriorS2EpsAB);

          if ((imod(i, nSkip+1) == 0) && (iSel == 4))
            {
              TrackTime(8);
              // Add density of GARCH parameters
              vLnPost[iOut]= LnDensGARCH(aPars[4][0], aPars[4][1], aPars[4][2], 
                                         vz, vv2, g_Kalm_PriorS2EpsAB);
              ++iOut;
            }
      }
    else if (!(g_FreePars != 2))
      {
        if (i == 0)
          println ("Using S2Eps sampling");
        ve= mYtEst - vMu;
        dS2Eps= SampleS2(ve, g_Kalm_PriorS2EpsAB);
        dSEps= sqrt(dS2Eps);
        if ((imod(i, nSkip+1) == 0) && (iSel == 4))
          {
            // Add density of SEps, calculated from S2Eps and jacobian
            vLnPost[iOut]= LnDensS2(aPars[4][0]*aPars[4][0], ve, g_Kalm_PriorS2EpsAB) 
                          + log(2*aPars[4][0]);
            ++iOut;
          }
      }
    TrackTime(-1);

    }
  info(nReps, nReps);

  vLnPost= deletec(vLnPost[0][:iOut-1]);
  dLnPostMax= max(vLnPost);
  print ("Limits vLnPost: ", limits(vLnPost')');
  return (meanr(exp(vLnPost-dLnPostMax)))|dLnPostMax;
}

/*
**  LnDensRho(const dRho, const vMu, const dSEta, const vPriorRho)
**
**  Purpose:
**    Calculate density of Rho, conditional on the state Mu and the variance of the
**    equation
**
**  Remark:
**    See SampleRho
*/
LnDensRho(const dRho, const vMu, const dSEta, const vPriorRho)
{
  decl
    iN, invXX, dMuRho, dS2Rho, dRhoHat, dS2RhoHat, dLnDensRho;

  iN= columns(vMu);
  olsr(vMu[1:], vMu[:iN-2], &dRhoHat, &invXX);
  dS2RhoHat= dSEta*dSEta*invXX;

  dMuRho= (dRhoHat*vPriorRho[1]^2 + vPriorRho[0]*dS2RhoHat)/
           (vPriorRho[1]^2+dS2RhoHat);
  dS2Rho= dS2RhoHat * vPriorRho[1]^2 / (vPriorRho[1]^2+dS2RhoHat);
  dLnDensRho= log(densn((dRho-dMuRho)/sqrt(dS2Rho)))-0.5*log(dS2Rho);

  return dLnDensRho;
}

/*
**  LnDensS2(const dS2, const ve, const vPriorS2AB)
**
**  Purpose:
**    Calculate density of a variance, conditional on the errors ve,
**    applying the IG prior
**
**  Remark:
**    See SampleS2
*/
LnDensS2(const dS2, const ve, const vPriorS2AB)
{
  decl
    iT, alpha, beta, ds, dLnDensS2;

  iT= columns(ve);
  alpha= iT/2 + vPriorS2AB[0];
  ds= sumsqrr(ve) + 2/vPriorS2AB[1];
  beta= 2/ds;
  dLnDensS2= lndensigamma(dS2, alpha, beta);

  return dLnDensS2;
}

/*
**  LnDensZ(const vz, const dSEps, const dDelta, const dAlpha, const dNu, const vv2)
**
**  Purpose:
**    Calculate density of Z, conditional on the data, and the parameters. 
**
**  Remark:
**    See SampleZ
*/
LnDensZ(const vz, const dSEps, const dDelta, const dAlpha, const dNu, const vv2)
{
  decl
    iT, alpha, beta, ve2, vh, dLnDensZ;

  iT= columns(vv2);
  CalcVarianceGARCH(&vh, dSEps, dDelta, dAlpha, vv2);

  ve2= vv2 ./ ((dSEps * dSEps) * vh);  

  alpha= (dNu+1)/2;
  beta= 2.0  ./ (dNu-2 + ve2);
  dLnDensZ= sumr(lndensigamma(vz.*beta, alpha, 1)+log(beta));

  return dLnDensZ;
}

/*
**  LnDensNu(const dNu, const vz)
**
**  Purpose:
**    Calculate density of Nu, the degrees of freedom parameter, given z.
**
**  Remark:
**    See SampleNu
*/
LnDensNu(const dNu, const vz)
{
  decl ir, dKern, dIntKern, dAbsErr, vNu, vDNu, i;

  if (!(dNu~sumr(vz) == s_GS_OldNu))
    {
      s_GS_OldNu= dNu~sumr(vz);

      s_GS_vz= vz;
      s_GS_KernMax= M_NAN;

      dKern= KernelDensNu(dNu);
      ir= QAGS(KernelDensNu, 2, 20, &dIntKern, &dAbsErr);
      if (ir > 0)
        print ("Warning in LnDensNu: ", ir~dIntKern~dAbsErr);
      s_GS_LnDensNu= log(dKern)-log(dIntKern);
    }

  return s_GS_LnDensNu;
}

/*
**  KernelDensNu(const dNu)
**
**  Purpose:
**    Calculate the kernel of the density of Nu, the degrees of freedom 
**    parameter, given z.
**
**  Remark:
**    See SampleNu
*/
KernelDensNu(const dNu)
{
  decl dKern, dalpha, dbeta;

  dalpha= dNu/2;
  dbeta= 2/(dNu-2);
  dKern= sumr(lndensigamma(s_GS_vz, dalpha, dbeta));

  // Adjust for Cauchy density, truncated at two from below
  dKern= dKern + LnPriorNu(dNu);

  if (isnan(s_GS_KernMax))
    s_GS_KernMax= dKern;

  return exp(dKern-s_GS_KernMax);
}

/*
**  LnDensGARCH(const adSEps, const adDelta, const adAlpha, const vz, 
**              const vv2, const vPriorS2Eps)
**
**  Purpose:
**    Calculate density of mGamma= (dDelta, dAlpha), conditional on the
**    other parameters
**
**  Remark:
**    This routine takes a lot of time, as the density is approximated
**    using a kernel method.
**
**  Inputs:
**
**  Outputs:
**    adSEps, adDelta, adAlpha
**          pointers to new dSEps, dDelta, dAlpha
**
*/
LnDensGARCH(const dSEps, const dDelta, const dAlpha, const vz, 
            const vv2, const vPriorS2Eps)
{
  decl ir, dIntKern, mGARCH, i, dSEps0, dDelta0, dAlpha0, 
  dLnDensGARCH, dVar, vDA, mCandGARCH;

  dIntKern= 0;
  if (!(dSEps~dDelta~dAlpha~sumr(vz)~sumr(vv2) == s_GS_OldGARCH))
    {
      s_GS_OldGARCH= dSEps~dDelta~dAlpha~sumr(vz)~sumr(vv2);

      ir= CondLik_Garch(&dLnDensGARCH, &dVar, dSEps, 
                        dDelta, dAlpha, vz, vv2);

      dLnDensGARCH += 
        LnPriorGARCH(dSEps, dDelta, dAlpha, 
                     g_Kalm_PriorS2EpsAB[0], g_Kalm_PriorS2EpsAB[1]);

      mGARCH= new matrix [3][s_GS_nRepGARCH[0][1]];
      dSEps0= dSEps;
      dDelta0= dDelta;
      dAlpha0= dAlpha;
      vDA= s_GS_Garch_DrawnAcc;
      s_GS_Garch_DrawnAcc= 0~0;
      mCandGARCH= (dSEps0|dDelta0|dAlpha0)~s_GS_CandGARCH[][1];
      for (i= 0; i < s_GS_nRepGARCH[1]; ++i)
        {
          ++s_GS_Garch_DrawnAcc[0];
          s_GS_Garch_DrawnAcc[1]+=
            SampleGARCH(&dSEps0, &dDelta0, &dAlpha0, vz, vv2, 
                        mCandGARCH[][0], mCandGARCH[][1],
                        g_Kalm_PriorS2EpsAB);
          
          mGARCH[][i]= dSEps0|dDelta0|dAlpha0;
          if (i== s_GS_nRepGARCH[0])
            {
              mCandGARCH= meanr(mGARCH[][:i])~sqrt(varr(mGARCH[][:i]));
              s_GS_Garch_DrawnAcc= 0~0;
            }
        }
      ir= ir && s_KernelMult(dSEps|dDelta|dAlpha, mGARCH, &dIntKern);
  
      s_GS_Garch_DrawnAcc= vDA;
      s_GS_LnDensGARCH= log(dIntKern);
    }  
  else
    {
//    println("Same pars, returning old LnDens: ", double(s_GS_LnDensGARCH));
    }

  return s_GS_LnDensGARCH;
}

/*
**  InitGibSt
**
**  Purpose:
**    Initialize the static variables in this module, using settings in
**    simox.dec
**
**  Inputs:
**    InitVP      Initial values for vP. Might have been
**                initialized in Kalman.DEC
**    amY         Address to matrix with data
**
**  Outputs:
**    amY         Row vector of data
*/
InitGibStGa(const vInitVP, const sSimFile, const amYtEst, const adRho, const adSEta, 
            const adSEps, const adDelta, const adAlpha, const adNu, const avz)
{
  decl 
    dLnPdf, vP, vS, ir, iT, mInter, mTheta, mSigma,
    vS2Eps, asvz, i, j, vMuHat, mKFh, vv2, mYt, mDMY;

  SetSeed(g_Seed);

  println("Gibbs sampling over Local level-Garch-Student t model");
  println("Using parameters ", g_VarNames[g_FreePars]);

  vP= vInitVP;     // Rho, sEta, sEps, Nu
  PdfInit(&vP, &iT);

  // Look for starting values 
  GetData(&mYt, &mDMY, &mInter);
  amYtEst[0]= mYt[:g_Kalm_FracEst-1];

  mTheta= loadmat(sSimFile);
  vP[g_FreePars]= meanc(mTheta);

  adRho[0]= vP[0];
  adSEta[0]= vP[1];
  adSEps[0]= vP[2];
  adDelta[0]= vP[3];
  adAlpha[0]= vP[4];
  adNu[0]= vP[5];
  // Candidate density for GARCH parameters 
  s_GS_CandGARCH= M_NAN;
  if (!(g_FreePars != 3))
    {
      // Check for restrictions on rho and sEps
      ir= !(g_FreePars != 0) + !(g_FreePars != 1);
      s_GS_CandGARCH= vP[2:4]~(.3*sqrt(varc(mTheta[][ir:ir+2])'));
    }

  // Choose a diffuse initial state
  mSigma=<-1; 0>;
  if (adRho[0] < 1)
    mSigma[0][0]= adSEta[0]*adSEta[0]/(1-adRho[0]^2);

  avz[0]= ones(amYtEst[0]);
  if (adNu[0] > 0)
    {
      j= 0;
      asvz= zeros(avz[0]);
      for (i= 0; i < 100; ++i)
        {
          // Calculate the smoothed mean of the ll model
          vS2Eps= adSEps[0]*adSEps[0]*avz[0];

          // Approximation: Filter instead of smoother. 
          //   Smoothing with GARCH disturbances is different...
          mKFh= KalmanFil_Shell(amYtEst[0], adRho[0], adSEta[0], adSEps[0], 
                                adDelta[0], adAlpha[0], avz[0], mSigma);
          vMuHat= amYtEst[0]-mKFh[0][];
          vv2= sqr(mKFh[0][]);
          SampleZ(avz, adSEps[0], adDelta[0], adAlpha[0], adNu[0], vv2);
          if (i >= 80)
            {
              asvz += avz[0];
              ++j;
            }
        }
      asvz/= j;

/*
      DrawT(0, vMuHat[0][], mDMY[][:g_Kalm_FracEst-1], 0, 0);
      DrawTMatrix(1, avz[0]|asvz, {"Last", "Mean"}, mDMY[][:g_Kalm_FracEst-1], 0, 0);
      SaveDrawWindow("excl/zmugas.plb");
      ShowDrawWindow();
      CloseDrawWindow();
*/

      // Do not reset avz[0], leading to using the last sampled set of Z's.
      // avz[0]= asvz;
	}
}
