/*
**
**  Program MHFlex.OX
**
**  Purpose:
**    Run Metropolis-Hastings sampling. 
**
**  Author:
**    Charles S. Bos
**
**  Date:
**    2 March 1999, in Ox
**
**  Version:
**    1     Based on GGFlex2.Ox
**    2     Allow free/fixed parameters
**    3     Extra checking for maximum in LnPdf_Loc
**    6     Only using g_InitVP, g_nData, g_FreePars, g_Flex_MH,
**          g_Flex_nBurnin
**
**  Uses:
**    gnudraw package, see http://www2.tinbergen.nl/~cbos/gnudraw.html
**
*/
#include <oxstd.h>
#import <maximize>      // The optimization routines 
#include <oxfloat.h>    // M_INF_NEG
#include "include/info.ox"      // Information on lenght of calc
#include "include/setseed.ox"   // Reset the seed?
#include "include/size.ox" 
#include "include/gnudraw_jae.h"    // Just in case

// Function declarations
Initialize(const avInitVP, const amInitS2, const ainData, const avFreePars, 
           const dSeed);
MaxIt(const avP, const amS2, const adLnPdf, const nData);
CalcInvHess(const vP, const nData, const amS2);
Rotation(const aMuS2, const aAccRate, const aFM, 
         const nMh, const nBurnin, const nSkip, const pForceMove, 
         const SimFile);
DrawThetaCand(const aTheta, const aLnPdf, const vMu, const mC, 
              const dLnDetC);
LnPdfTarg_Loc(const Theta, const aLnPdf);
LnPdf_Loc(const vP, const adFunc, const avScore, const amHessian);
InitOutFile(const OutFile, const Vers, const VarNames, 
            const vMu, const mS2, const nMH, const nBurnin);
GiveResults(const OutFile, const SimFile, const iRot, const aResults, 
            const vMuOld, const nMH, const nBurnIn, const nLag, 
            const nData, const dTime);

#include "simox.dec"    // Declaration of the model

/*
**  Static declarations, on dimension of problem, index of present
**  parameter, and the full vector of parameters
*/
static decl s_MH_nDim, s_MH_iPar, s_MH_dPars, s_MH_nFunc= 0, 
            s_MH_MaxPars, s_MH_MaxLL= M_INF_NEG, s_MH_nOut= 1000;

main()                // function main is the starting point
{
  decl dLnPdf, vMu, mS2, aResults, nRot, nLag, nMH, nSkip, OutFile, SimFile, 
       ir, iRot, AccRate, nFunc, FM, pForceMove, Theta, Succ, dTime;

  /* Initialize */
  dTime= timer();
  vMu= g_InitVP; mS2= g_InitS2;
  Initialize(&vMu, &mS2, &g_nData, &g_FreePars, 
             g_Seed);
//  g_InitVP= vMu;

  ir= 1;
  if (g_Optim == 1)
    ir= MaxIt(&vMu, &mS2, &dLnPdf, g_nData);
  if (g_Optim == -1)
    ir= CalcInvHess(vMu, g_nData, &mS2);
  if (ir == 0)
    println ("Warning: Problem in initialization with Maxit or InvHess");

  nRot= sizerc(g_Flex_nMH);
  nLag= 20;
  pForceMove= 100;
  nMH= g_Flex_nMH;
  OutFile= sprint(g_OutDir, "/", g_VersFile, "mh.out");

  println ("Writing output to ", OutFile);
  InitOutFile(OutFile, g_Vers, g_VarNames[g_FreePars], vMu, mS2, 
              nMH, g_Flex_nBurnin);

  aResults= new array [nRot+1];
  aResults[0]= new array[3];        // Contains Mu, S2 and acc rate
  aResults[0][0]= vMu;              // Initial Mu
  aResults[0][1]= mS2;              // Initial S2
  aResults[0][2]= zeros(2, 1);      // Initial acc rate, number of
                                    // function evaluations and forced moves

  SimFile= sprint(g_OutDir, "/", g_VersFile, ".fmt");

  nSkip= 0;
  for (iRot= 0; iRot < nRot; ++iRot)
    {
      FM= 0;
      if (iRot== nRot-1)
        nSkip= g_Flex_nSkip;
      Succ= Rotation(aResults[iRot][0:1], &AccRate, &FM, nMH[iRot], 
                     g_Flex_nBurnin[iRot], nSkip, pForceMove, SimFile);

      Theta= loadmat(SimFile);

      aResults[iRot+1]= new array[3];
      aResults[iRot+1][0]= meanc(Theta)';
      aResults[iRot+1][1]= variance(Theta);
      aResults[iRot+1][2]= AccRate|(s_MH_nFunc-aResults[iRot][2][1])|FM;

      GiveResults(OutFile, SimFile, iRot+1, aResults[iRot+1], 
                  aResults[iRot][0], nMH[iRot], g_Flex_nBurnin[iRot],
                  nLag, g_nData, dTime);
    }
}

/*
**
**  Procedure Initialize
**
**  Purpose:
**    Call the PdfInit, set the bounds free etc.
**
*/
Initialize(const avInitVP, const amInitS2, const ainData, const avFreePars, 
           const dSeed)
{
  decl vP, f;

  SetSeed(dSeed);

  PdfInit(avInitVP, ainData);

  f= avFreePars[0];
  if (!(size(f) == size(g_FreePars)))
    f= g_FreePars;
  if (f == 0)
    f= range(0, rows(avInitVP[0])-1)';
  avFreePars[0]= f;

  vP= avInitVP[0][f];
  if (!(size(amInitS2[0]) == size(g_InitS2)))
    amInitS2[0]= g_InitS2;
  if (rows(amInitS2[0]) < rows(vP))
    amInitS2[0]= amInitS2[0] .* unit(rows(vP));

  avInitVP[0]= vP;
}

/*
**
**  Function MaxIt(const avP, const adLnPdf)
**
**  Call the maximizing routines, gives the output
**
**  Inputs:
**    avP         Address of vector of initial parameters
**    amS2        Address of matrix for returning the covariance matrix
**    adLnPdf     Address of function value to be returned
**    nData       Number of datapoints
**
**  Outputs:
**    avP         vector of optimal parameters
**    amS2        matrix returning the covariance matrix
**    adLnPdf     Optimal function value
**
**  Return value:
**    ir          Message number of MaxBFGS
**
*/
MaxIt(const avP, const amS2, const adLnPdf, const nData)
{
  decl dTime, ir;

  dTime= timer();
  println("\nStarting values used: ", avP[0]');
  ir= MaxBFGS(LnPdf_Loc, avP, adLnPdf, amS2, TRUE);

  println("\n", MaxConvergenceMsg(ir),
          " using numerical derivatives",
          "\nLog-likelihood = ", 
          "%.8g", double(adLnPdf[0]*g_nData),
                  "; n = ", g_nData);
  ir= (ir == 0) && CalcInvHess(avP[0], nData, amS2);

  println ("Time elapsed: ", timespan(dTime));
  print("Parameters with standard errors:",
        "%cf", {"%12.5g", "  (%7.5f)"}, 
        avP[0] ~ sqrt(diagonal(amS2[0])'));

  return ir;
}

/*
**
**  CalcInvHess(const vP, const nData, const amS2)
**
**  Purpose:
**    Calculate a numerical approximation to the variance
**
**  Inputs:
**    vP    Vector of parameters
**    nData Number of datapoints
**    amS2  Input: Address of initial value of variance estimate
**
**  Outputs:
**    r.v.  1 if succeeded, 0 otherwise
**    amS2  Inverse hessian if all went well, not changed otherwise
*/
CalcInvHess(const vP, const nData, const amS2)
{
  decl
    ir, mHess, miHess;

  ir= Num2Derivative(LnPdf_Loc, vP, &mHess);
  if (!ir)
    println("Warning: Num2Derivative failed in numerical second derivatives");
  else
    {
      miHess= invertgen(-mHess, 30);
      if (!isnan(miHess))
        amS2[0]=miHess/nData;
      else
        ir= 0;
    }
  return ir;
}


/*
**
**  Rotation(const aMuS2, const aAccRate, const anFunc, const aFM, 
**           const nMH, const nBurnin, const pForceMove, 
**           const SimFile)
**
**  Purpose:
**    Run one rotation of the MH algorithm. Drawn Theta's are written
**    in Simfile, after an initial number of nBurnin were drawn.
**
**  Inputs:
**
**  Return value:
**    ir    If succeeded, ir= 1, else ir= 0
**
*/
Rotation(const aMuS2, const aAccRate, const aFM, const nMH,
		 const nBurnin, const nSkip, const pForceMove, 
         const SimFile)
{
  decl 
    nDim, C, LnDetC, Theta, LnPdfCand, LnPdfTarg, ThetaC, LnPdfCandC, 
    LnPdfTargC, fh, i, LnAlpha, U, iForceMove, nForceMove, SuccAcc,
    Succ, iReject, s, iOut, mOut;

  nDim= rows(aMuS2[0]);
  C= choleski(aMuS2[1]);
  LnDetC= log(prodr(diagonal(C)));
  iReject= 0;
  nForceMove= 0;
  iOut= 0;
  mOut= zeros(s_MH_nOut, nDim);
  // Open the Simfile
  fh= fopen(SimFile, "wbf");

  // Draw a candidate which has positive target probability
  Succ= 0;
  while (!Succ)
    Succ= DrawThetaCand(&Theta, &LnPdfCand, aMuS2[0], C, LnDetC) 
          && LnPdfTarg_Loc(Theta, &LnPdfTarg);

  println ("Init LnPdf: ", LnPdfTarg);

  for (i= 0; i < nMH+nBurnin; ++i)
    {  
      if (imod(i+iReject, g_Flex_InfoRep) .== 0)
        {
          println("Theta accepted, drawn, rate: ", 
                  "%10i", i, "%10i", i+iReject,
                  "%10.4f", (i/(i+iReject+ (i+iReject == 0))));
          info(i, nMH+nBurnin); 
        }

      SuccAcc= 0;                   // Only count accepted drawings
      iForceMove= 0;
      while (SuccAcc == 0)
        {
          // Draw a candidate which can be which has positive target probability
          Succ= 0;
          while (!Succ)
            Succ= DrawThetaCand(&ThetaC, &LnPdfCandC, aMuS2[0], C, LnDetC) 
                  && LnPdfTarg_Loc(ThetaC, &LnPdfTargC);

          LnAlpha= LnPdfTargC  - LnPdfTarg + LnPdfCand - LnPdfCandC;
          LnAlpha= min (0, LnAlpha);

          U= ranu(1, 1);
          if (log(U) < LnAlpha)
            {  // Accept
              Theta= ThetaC;
              LnPdfTarg= LnPdfTargC;
              LnPdfCand= LnPdfCandC;
              SuccAcc= 1;
            }
          else if (iForceMove == pForceMove)
            { // Accept as well
              println ("Warning: Forcing acceptance after ", 
                       pForceMove, " rejections.");
              Theta= ThetaC;
              LnPdfTarg= LnPdfTargC;
              LnPdfCand= LnPdfCandC;
              SuccAcc= 1;
              ++nForceMove;
            }
          else
            { // Reject
              ++iForceMove;
              ++iReject;
            }

          if ((i >= nBurnin) && (imod(i-nBurnin, nSkip+1) == 0))
            {
              mOut[iOut][]= Theta';
              ++iOut;
            }
          if (((iOut == s_MH_nOut) || (i == nMH+nBurnin-1)) && (iOut > 0))
            {
              Succ= fwrite(fh, mOut[:iOut-1][]);
              iOut= 0;
            }
            
      }  // End of do-loop, running until acception

    }  // End of for-loop, running until nMH acceptions  
  fh= fclose(fh);

  aAccRate[0]= (nMH+nBurnin)/(nMH+nBurnin+iReject);
  aFM[0]= nForceMove;

  return 1;
}

/*
**  DrawThetaCand(const aTheta, const aLnPdf, 
**                const vMu, const mC, const dLnDetC) 
**
**  Purpose:
**    Draw a Theta from the normal candidate, and calculate the 
**    loglikelihood in that parameter vector.
**
**  Inputs:
**    aTheta      Address of vector Theta
**    aLnPdf      Address of loglikelihood
**    vMu         Mean vector
**    mC          Matrix of Cholesky decomposition of variance
**    dLnDetC     Scalar, logarithm of determinant of C
**
**  Outputs:
**    aTheta      Sampled Theta-vector
**    aLnPdf      Loglikelihood in Theta
**
**  Return value:
**    ir          1
*/
DrawThetaCand(const aTheta, const aLnPdf, 
              const vMu, const mC, const dLnDetC) 
{
  decl nDim, U; 

nDim= rows(vMu);
U= rann(nDim, 1);
aTheta[0]= vMu + mC * U;
aLnPdf[0]= -nDim/2*log(M_2PI) - dLnDetC - U'U/2;

return 1;
}    

/*
**
**  Procedure LnPdfTarg_Loc(const Theta, const aLnPdf)
**
**  Purpose:
**    Return the loglikelihood, taking the number of datapoints into
**    account. Also tracks the maximum likelihood estimator
**
**  Inputs:
**    Theta       nDim vector of parameters
**    aLnPdf      Address of loglikelihood
**    g_nData     Global, number of datapoints
**    s_MH_nFunc  Static, number of function evaluations
**    s_MH_MaxLL  Static, maximum LL found
**    s_MH_MaxPars Static, parameters at maximum
**
**  Outputs:
**    aLnPdf      Loglikelihood, not the MEAN loglikelihood
**    s_MH_nFunc  Static, number of function evaluations
**    s_MH_MaxLL  Static, maximum LL found
**    s_MH_MaxPars Static, parameters at maximum
**
**  Return value:
**    1           if succeeded, 0 otherwise
**
*/
LnPdfTarg_Loc(const vP, const adLnPdf)
{
  decl Succ;

  Succ= LnPdf_Loc(vP, adLnPdf, 0, 0);
  adLnPdf[0]*= g_nData;
  return Succ;
}

/*
**
**  Function LnPdf_Loc
**
**  Purpose:
**    Enlarge the parameter vector with the elements
**    that are not free. Do not change likelihood value (so return
**    the MEAN loglikelihood.
**
*/
LnPdf_Loc(const vP, const adLnPdf, const avScore, const amHessian)
{
  decl vP_Loc, Succ;

  vP_Loc= g_InitVP;
  vP_Loc[g_FreePars]= vP;

  Succ= LnPdf_Shell(vP_Loc, adLnPdf, avScore, amHessian);
  if (adLnPdf[0] > s_MH_MaxLL)
    {
      print ("Maximum LL: ", adLnPdf[0]*g_nData, " at parameters");
      print (vP');
      s_MH_MaxLL= adLnPdf[0];
      s_MH_MaxPars= vP;
    }

  ++s_MH_nFunc;
  return Succ;
}

/*
**  InitOutFile
**
**  Purpose:
**    Do the general initializations
*/
InitOutFile(const OutFile, const Vers, const VarNames, 
            const vMu, const mS2, const nMH, const nBurnin)
{
    
    decl fh, i, dS2Fac;

    fh= fopen(OutFile, "w");    
    fprintln(fh, "Metropolis-Hastings sampling");
    fprintln(fh, " ");
    fprintln(fh, "Version ");
    fprintln(fh, "  ", Vers);
    fprintln(fh, "Free parameters ");
    for (i= 0; i < rows(VarNames); ++i)
      fprint(fh, "  ", VarNames[i]);
    fprintln(fh, " ");
    fprintln(fh, "Number of rotations: ");
    fprintln(fh, "%10i", rows(nMH));
    fprint(fh, "Length of burn-in period: ");
    fprintln(fh, "%10i", nBurnin');
    fprint(fh, "Length of sample: ");
    fprintln(fh, "%10i", nMH');
    fprint(fh, "Initial Mu: ");
    fprintln(fh, "%10.4f", vMu');
    dS2Fac= floor(min(log10(diagonal(mS2))));
    fprint(fh, "Initial S2 (x 10^", "%1.0f", dS2Fac, "): ");
    fprintln(fh, "%10.4f", mS2/(10^dS2Fac));
    fprintln(fh, "Date of run: ");
    fprintln(fh, "%10s", date());
    fprintln(fh, "Time of run: ");
    fprintln(fh, "%10s", time());

    fh= fclose(fh);
    
}

/*
**
**  Procedure GiveResults
**
**  Purpose:
**    Give the results of this rotation
**
**  Inputs:
**
*/
GiveResults(const OutFile, const SimFile, const iRot, const aResults, 
            const vMuOld, const nMH, const nBurnin, const nLag,
            const nData, const dTime)
{
    decl fh, nDim, Theta, i, fmt, Mah, dS2Fac;
    
    fh= fopen(OutFile, "a");    
    fprintln(fh, "-----------------------------------------------------");
    fprintln(fh, "Results of rotation: ");
    fprintln(fh, "%10i", iRot);
    
    fprintln(fh, "Number of sampled Theta's: ");
    fprintln(fh, "%10i", nMH);
    fprintln(fh, "Number of burn-in Theta's thrown away: ");
    fprintln(fh, "%10i", nBurnin);

    fprint(fh, "Mean of sampled Theta's: ");
    fprintln(fh, "%10.4f", aResults[0]');
    dS2Fac= floor(min(log10(diagonal(aResults[1]))));
    fprint(fh, "Covariance of sampled Theta's (x 10^", "%1.0f", dS2Fac, "): ");
    fprintln(fh, "%10.4f", aResults[1]/(10^dS2Fac));

    fprintln(fh, "Acceptance rate: ");
    fprintln(fh, "%10.4f", aResults[2][0]);
    fprint(fh, "Mahalanobis distance: ");
    Mah= (aResults[0]-vMuOld)'*invert(aResults[1])*(aResults[0]-vMuOld);
    fprintln(fh, "%10.4f", Mah);
    fprintln(fh, "Number of function evaluations in this rotation: ");
    fprintln(fh, "%10i", aResults[2][1]);
    fprintln(fh, "Number of forced moves in this rotation: ");
    fprintln(fh, "%10i", aResults[2][2]);
    fprintln(fh, "Total number of function evaluations: ");
    fprintln(fh, "%10i", s_MH_nFunc);
    fprint(fh, "Autocorrelations: ");

    Theta= loadmat(SimFile);
    nDim= columns(Theta);
    fmt= new array[nDim+1];
    fmt[0]= "  AC %2.0f: ";
    for (i= 1; i <= nDim; ++i)
      fmt[i]= "%10.4f";

    fprintln(fh, "%cf", fmt, range(0,nLag)'~acf(Theta, nLag));

    fprint(fh, "Maximum loglikelihood: ");
    fprintln(fh, "%10.4f", s_MH_MaxLL*nData);
    fprint(fh, "at parameters: ");
    fprintln(fh, "%10.4f", s_MH_MaxPars');

    fprintln(fh, "Ended at time: ");
    fprintln(fh, "%10s", time());
    fprintln(fh, "Time elapsed:\n ", "%10s", timespan(dTime));

    fh= fclose(fh);
}


