/*
**  Library probig
**
**  Purpose:
**    Calculate density and simulate from the inverted gamma and gamma
**    densities, with parameters as in Poirier (expressed in alpha and
**    beta)
**
**  Author:
**    Charles Bos
*/
#ifndef PROBIG_INCLUDED
#define PROBIG_INCLUDED

#include <oxprob.h>

/*
**  densigamma(const ma, const dalpha, const dbeta)
**
**  Purpose:
**    Calculate density of inverse gamma
**
**  Remark:
**    densigamma(ma, alpha, beta) = 
**      densigamma(ma*beta, alpha, 1) * beta
**    which is useful for incorporating matrix size beta's.  
**
*/
densigamma(const ma, const dalpha, const dbeta)
{
  decl dr, da, ddens;

dr= dalpha;
da= 1/dbeta;

ddens= densgamma(1.0 ./ ma, dr, da);
ddens= ddens./(ma.*ma);

return ddens;
}

/*
**  ranigamma(const r, const c, const dalpha, const dbeta)
**
**  Purpose:
**    Draw randoms from inverse gamma
**
**  Remark:
**    ranigamma(r, c, alpha, beta)= 
**      ranigamma(r, c, alpha, 1) / beta
**    which is useful for incorporating matrix size beta's.  
**
*/
ranigamma(const r, const c, const dalpha, const dbeta)
{
  decl dr, da, ma;

dr= dalpha;
da= 1/dbeta;

ma= rangamma(r, c, dr, da);
return 1.0 ./ ma;
}


/*
**  densgammaab(const ma, const dalpha, const dbeta)
**
**  Purpose:
**    Calculate density of gamma, with parameters alpha and beta
**
**  Remark:
**    densgammaab(ma, alpha, beta)= 
**      densgammaab(ma./beta, alpha, 1) ./ beta
**    which is useful for incorporating matrix size beta's.  
*/
densgammaab(const ma, const dalpha, const dbeta)
{
  decl dr, da, ddens;

dr= dalpha;
da= 1/dbeta;

ddens= densgamma(ma, dr, da);

return ddens;
}

/*
**  rangammaab(const r, const c, const dalpha, const dbeta)
**
**  Purpose:
**    Draw randoms from gamma, with parameters alpha and beta
**    which is useful for incorporating matrix size beta's.  
**
**  Remark:
**    rangammaab(r, c, alpha, beta)= 
**      rangammaab(r, c, alpha, 1) * beta
**    which is useful for incorporating matrix size beta's.  
*/
rangammaab(const r, const c, const dalpha, const dbeta)
{
  decl dr, da, ma;

dr= dalpha;
da= 1/dbeta;

ma= rangamma(r, c, dr, da);
return ma;
}

/*
**  lndensgammaab(const ma, const dalpha, const dbeta)
**
**  Purpose:
**    Calculate logdensity of gamma, with parameters alpha and beta
**
*/
lndensgammaab(const ma, const dalpha, const dbeta)
{
  decl dlndens;

dlndens= -dalpha.*log(dbeta) - loggamma(dalpha)
         + (dalpha-1).*log(ma) -ma./dbeta;

return dlndens;
}

/*
**  lndensigamma(const ma, const dalpha, const dbeta)
**
**  Purpose:
**    Calculate density of inverse gamma
**
**  Remark:
**    densigamma(ma, alpha, beta) = 
**      densigamma(ma*beta, alpha, 1) * beta
**    which is useful for incorporating matrix size beta's.  
**
*/
lndensigamma(const ma, const dalpha, const dbeta)
{
  decl dlndens;

dlndens= -dalpha.*log(dbeta) - loggamma(dalpha) 
         - (dalpha+1).*log(ma) - 1.0 ./ (ma .* dbeta);

// dlndens= log(densigamma(ma, dalpha, dbeta));

return dlndens;
}

#endif

