/*
**
**  Library IncWN
**
**  Purpose:
**    Contain routines for the WN model,
**    together with the likelihood routine.
**
**  Version:
**    1     Based on IncGL
**    2     Including UIP
**
**  Date:
**    8/8/2000
**
**  Author:
**    Charles Bos
**
*/
#include <oxfloat.h>    // M_NAN and M_INF_NEG
#include <oxprob.h>
#include "include/oxprobig.ox"
#include "include/incdata.ox"

PdfInit(const avP, const anData);
GenrWN(const aY, const dMu, const dSEps, const iT);
LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian);
LnPrior_Shell(const vP);
ChangeBounds(const amCBounds);
GetData(const amY, const amDMY, const amInter);
LnPriorMu(const dMu, const vPrior);
LnPriorS2(const mS2, const alpha, const beta);
GiveResModel(const vP, const OutBase, const VarNames);
CalcKalmanFilter(const vP, const mY, const aPredMean, const aPredVol);
CalcKalmanMVS(const vP, const mY, const nPeriods, const amPMSdS, ...);

/* Local static declarations, for the dataset */
static decl s_Kalm_mYt, s_Kalm_mYtEst, s_Kalm_mDMY, s_Kalm_mInter, 
            s_Kalm_mIntDiff;

/*
**
**  Procedure PdfInit
**
**  Purpose:
**    Initialize the static variables in this module, using settings in
**    Kalman.Dec
**
**  Inputs:
**    aInitVP     Address of initial values for vP. Might have been
**                initialized in Kalman.DEC
**    anData      Address of number of data points.
**
**  Outputs:
**    aInitVP     Initial values for VP
**    anData      Number of data points
**
*/
PdfInit(const avP, const anData)
{
  decl iT, dd;

  if (g_Kalm_UseModel == "WN")
    {
      println("White noise model");
      println("Parameters: mu, s(Eps)");
      if (rows(avP[0]) != 2)
        {  
          println ("Error: Size of InitVP not correct");
          println ("InitVP: ", avP[0]');
        }
    }
  else 
    println("Error: Model type not recognized.\n", 
            "Incorrect declarations file?");

  if (g_Kalm_Reload == 1)
    {
      LoadData(g_Kalm_DataFile, g_Kalm_DataFMT, g_Kalm_DataFrac, &s_Kalm_mYt, 
               &s_Kalm_mDMY, &s_Kalm_mInter, g_Kalm_Diff, g_Kalm_Invert);
      iT= columns(s_Kalm_mYt);
    }
  else
    {
      GenrWN(&s_Kalm_mYt, g_Kalm_Pars[0], g_Kalm_Pars[1], 
             max(g_Kalm_DataFrac, g_nData));

      iT= columns(s_Kalm_mYt);
      s_Kalm_mDMY= range(1, iT)|ones(2, iT);
      s_Kalm_mInter= zeros(2, iT);

      dd= s_Kalm_mDMY'~ones(iT, g_Kalm_DataFMT[0]-3);
      dd[][g_Kalm_DataFMT[1]]= s_Kalm_mYt';

      if (!savemat(g_Kalm_DataFile, dd, 1))
        println ("Error: Saving of datafile did not succeed");
    }

  // Calculate the interest rate differential
  s_Kalm_mIntDiff= (s_Kalm_mInter[0][]-s_Kalm_mInter[1][])/360;

  s_Kalm_mYtEst=  s_Kalm_mYt[:g_Kalm_FracEst];
  anData[0]= columns(s_Kalm_mYtEst);

  // If UIP is included in the model, use s*(t)= s(t) + rh - rf,
  //   and run model on this s*(t)
  if (g_Kalm_UseUIP == 1)
    {
      s_Kalm_mYtEst+= s_Kalm_mIntDiff[:g_Kalm_FracEst];
      println("Using the uncovered interest rate parity");
    }

  if (g_Kalm_UsePrior)
    println("Using a prior on the parameters");

  print ("Mean and sdev of y: ",
meanr(s_Kalm_mYtEst)~sqrt(varr(s_Kalm_mYtEst)));
}

/*
**
**  GenrWN(const aY, const dMu, const dSEps, const iT)
**
**  Purpose:
**    Generate data from the GLL model
**
*/
GenrWN(const aY, const dMu, const dSEps, const iT)
{
  print ("Generating new data using parameters Mu and sEps: ",
         dMu~dSEps);
  aY[0]= dMu + dSEps * rann(1,  iT);
}

/*
**
**  LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian)
**
**  Purpose:
**    Calculate the MEAN LogLikelihood for the WN model, using the global
**    data variables
**
**  Inputs:
**    vP          nDim vector of parameters, containing Mu
**                and s(epsilon)
**    adFunc      Address for function value
**    avScore     0 or address
**    amHessian   0 or address
**    s_Kalm_mYtEst  static, matrix with the data
**
**  Outputs:
**    adFunc      Function value at parameter vector(s)
**    avScore     Not changed
**    amHessian   Not changed
**    Return-value      1 if succeeded, 0 otherwise
**
*/
LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian)
{
  decl dMu, dSEps, ir, vLL, ve;

  dMu= vP[0];
  dSEps= vP[1];
  adFunc[0]= M_NAN;

  ir= 0;
  if (dSEps >= 0)
    {
      ir= 1;

      ve= (s_Kalm_mYtEst-dMu)/dSEps;
      vLL= -0.5*log(M_2PI) -log(dSEps) - 0.5* ve .* ve;

      adFunc[0]= meanr(vLL);

      if (g_Kalm_UsePrior)
        adFunc[0]= adFunc[0] + LnPrior_Shell(vP)/columns(s_Kalm_mYtEst);
    }

  return ir;
}

/*
**  LnPrior_Shell(const vP)
**
**  Purpose:
**    Calculate the Log prior for the WN model, using the global
**    data variables
**
**  Inputs:
**    vP          nDim vector of parameters, containing Mu, s(epsilon)
**
**  Outputs:
**    Return-value  Log prior
*/
LnPrior_Shell(const vP)
{
  decl Retval, dMu, dSEps;

  dMu= vP[0];
  dSEps= vP[1];

  Retval= M_NAN;
  if (vP[1] >= 0)
    {
      // Prior is IG(a,b) on S2Eps, thus 2*SEps*IG(a, b) on SEps
      Retval= 
          LnPriorMu(dMu, g_Kalm_PriorMu)
          + LnPriorS2(dSEps^2, g_Kalm_PriorS2EpsAB[0], g_Kalm_PriorS2EpsAB[1])
          + log(2*dSEps);
    }
  else 
    println ("Error: Vector out of bounds");

  return Retval;
}

/*
**
**  ChangeBounds(const amCBounds)
**
**  Purpose:
**    Check the bounds that are going to be used, e.g. against non-
**    stationarity of the parameters
**
**  Inputs:
**    mCBounds    Address of matrix of proposed bounds
**
**  Output:
**    mCBounds    Address of matrix of changed proposed bounds
**
*/
ChangeBounds(const amCBounds)
{
  println ("Warning: Using changebounds? Not implemented");
}

/*
**
**  Procedure GetData(const amY, const amDMY, const amInter, ...);
**
**  Purpose:
**    Read the dataset from the static variable, and return it
**
**  Inputs:
**    amY    Address of vector of data
**
**  Outputs:
**    amY    Vector of data
**
*/
GetData(const amY, const amDMY, const amInter)
{
  amY[0]= s_Kalm_mYt;
  amDMY[0]= s_Kalm_mDMY;
  amInter[0]= s_Kalm_mInter;
}

/*
**  LnPriorMu(const dMu, const vPrior)
**
**  Purpose:
**    Calculate the Normal prior on Mu, E(mu) and sdev in
**    the first two elements of vPrior
*/
LnPriorMu(const dMu, const vPrior)
{
  decl dLnPrior;

  dLnPrior= log(densn((dMu-vPrior[0])/vPrior[1])) - log(vPrior[1]);

  return dLnPrior;
}

/*
**
**  LnPriorS2(const mS2, const alpha, const beta)
**
**  Purpose:
**    Calculate the Inv gamma prior on S2. Do not apply the prior
**    if S2 is identical to zero.
**
*/
LnPriorS2(const mS2, const alpha, const beta)
{
  if (mS2 > 0)
    return log(densigamma(mS2, alpha, beta));
  else
    return 0;
}

/*
**
**  Procedure GiveResModel(vP, OutBase);
**
**  Purpose:
**    Prepare some results
**
*/
GiveResModel(const vP, const OutBase, const VarNames)
{
  // Does not do a thing.
}

/*
**
**  CalcKalmanFilter(const vP, const mY, const aPredMean, const aPredVol)
**
**  Purpose:
**    Calculate the filter, return predictions for mean and variance
**    of the observation
**
**  Inputs:
**    vP          Vector of parameters
**    mY          1 x T row with dataset; if wanted, the original dataset may be
**                shortened to include less info, as only the last
**                100 observations are effectively used...
**
**  Outputs:
**    aPredMean   T+1 x 1 vector, predicted mean of the series,
**    aPredVol    T+1 x 1 vector, predicted variance of the series,
**    Ret. value  Indicates if filtering succeeded
*/
CalcKalmanFilter(const vP, const mY, const aPredMean, const aPredVol)
{
  decl mPVS, ir;

  ir= CalcKalmanMVS(vP, mY, columns(mY)+1, &mPVS);
  if (ir)
    {
      aPredMean[0]= mPVS[0][]';
      aPredVol[0]= mPVS[1][]';
    }
  return ir;
}

/*
**  CalcKalmanMVS(const vP, const mY, const nPeriods, const amPMSdS)
**
**  Purpose:
**    Calculate the filter, return predictions for mean and variance
**    of the observation and a sample, over the last nPeriods
**
**  Inputs:
**    vP          Vector of parameters
**    mY          1 x T row with dataset; if wanted, the original dataset may be
**                shortened to include less info, as only the last
**                100 observations are effectively used...
**    nPeriods    Integer, number of periods at end of sample to use
**    nRep        Number of repetitions for sample, default= 1
**
**  Outputs:
**    amPMSdS     2+nRep x nPeriods vector, with predicted mean, predicted sdev and
**                a sample from the predicted density of the series,
**    Ret. value  Indicates if filtering succeeded
*/
CalcKalmanMVS(const vP, const mY, const nPeriods, const amPMSdS, ...)
{
  decl dMu, dSEps, ir, vMu, vSd, vSamp, iT, va, nRep;

  nRep= 1;
  va= va_arglist();
  if (sizeof(va) > 0)
    nRep= va[0];

  dMu= vP[0];
  dSEps= vP[1];
  amPMSdS[0]= M_NAN;

  ir= 0;
  if (vP[1] >= 0)
    {
      ir= 1;
      iT= columns(mY);
      vMu= dMu .* ones(1, nPeriods);
      vSd= dSEps .* ones(1, nPeriods);
      vSamp= vMu + vSd .* rann(nRep, nPeriods);
      amPMSdS[0]= vMu|vSd|vSamp;
    }

  return ir;
}

