/*
**
**  Library IncSV
**
**  Purpose:
**    Contain routines for the GLL-SV model,
**    together with the likelihood routine.
**
**  Version:
**    1     Based on IncGL
**
**  Date:
**    13/3/2000
**
**  Author:
**    Charles Bos
**
*/
#include <oxfloat.h>    // M_NAN and M_INF_NEG
#include <oxprob.h>
#include <arma.h>
#include "ssfpack.h" 
#include "include/oxprobig.ox"
#include "include/incdata.ox"

PdfInit(const avP, const anData);
LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian);
LnPrior_Shell(const vP, ...);
ChangeBounds(const amCBounds);
GetData(const amY, const amDMY, const amInter);
LnPriorRho(const dRho, const vPrior);
LnPriorS2(const mS2, const alpha, const beta);
LnPriorNorm(const dP, const dMu, const dSDev);
GiveResModel(const vP, const OutBase, const VarNames);
CalcKalmanFilter(const vP, const mY, const aPredMean, const aPredVol);
CalcKalmanMVS(const vP, const mYt, const nPeriods, const amPMSdS);
SampleMu(const vh, const dRho, const dSEta, const mYt);
SampleRho(const adRho, const vMu, const dSEta, const vPriorRho);
SampleS2(const ve, const vPriorS2AB);
SampleS(const vYStar, const vh);
SampleH(const vYStar, const vis, const dPhi, const dMuH, const dSXi);
SamplePhi(const vh, const dMuH, const dSXi, const vPriorPhi);
SampleMuH(const vh, const dPhi, const dSXi, const vPriorMuH);
SampleS2Xi(const dPhi, const dMuH, const vh, const vPriorS2AB);

/* Local static declarations, for the dataset */
static decl s_Kalm_mYt, s_Kalm_mYtEst, s_Kalm_mDMY, s_Kalm_mInter, s_Kalm_mIntDiff, 
            s_Kalm_Pars, s_Kalm_mKFaP, s_Kalm_Warn;

/* Local fixed parameters */
// Table 4.
static decl s_GS_qmv= <.00730, -10.12999, 5.79596;
                       .10556,  -3.97281, 2.61369;
                       .00002,  -8.56686, 5.17950;
                       .04395,   2.77786, 0.16735;
                       .34001,   0.61942, 0.64009;
                       .24566,   1.79518, 0.34023;
                       .25750,  -1.08819, 1.26261>;

/*
**
**  Procedure PdfInit
**
**  Purpose:
**    Initialize the static variables in this module, using settings in
**    Kalman.Dec
**
**  Inputs:
**    aInitVP     Address of initial values for vP. Might have been
**                initialized in Kalman.DEC
**    anData      Address of number of data points.
**
**  Outputs:
**    aInitVP     Initial values for VP
**    anData      Number of data points
**
*/
PdfInit(const avP, const anData)
{
  decl iT, dd;

  if (g_Kalm_UseModel == "GLLSV")
    {
      println("Generalized Local Level model-Stochastic Volatility model");
      println("Parameters: Rho, s(Eta), MuH, Phi, Sigma Xi");
      if (rows(avP[0]) != 5)
        {  
          println ("Error: Size of InitVP not correct");
          println ("InitVP: ", avP[0]');
        }
    }
  else 
    println("Error: Model type not recognized.\n", 
            "Incorrect declarations file?");

  if (g_Kalm_Reload == 1)
    {
      LoadData(g_Kalm_DataFile, g_Kalm_DataFMT, g_Kalm_DataFrac, &s_Kalm_mYt, 
               &s_Kalm_mDMY, &s_Kalm_mInter, g_Kalm_Diff, g_Kalm_Invert);
      iT= columns(s_Kalm_mYt);
    }
  else
    {
      println ("Error: Generating data with LL-SV model not implemented");
      exit(1);
    }

  // Calculate the interest rate differential
  s_Kalm_mIntDiff= (s_Kalm_mInter[0][]-s_Kalm_mInter[1][])/360;

  s_Kalm_mYtEst=  s_Kalm_mYt[:g_Kalm_FracEst];
  anData[0]= columns(s_Kalm_mYtEst);

  // If UIP is included in the model, use s*(t)= s(t) + rh - rf,
  //   and run model on this s*(t)
  if (g_Kalm_UseUIP == 1)
    {
      s_Kalm_mYtEst+= s_Kalm_mIntDiff[:g_Kalm_FracEst];
      println("Using the uncovered interest rate parity");
    }

  if (g_Kalm_UsePrior)
    println("Using a prior on the parameters");
  if (sizeof(g_Kalm_PriorRho) == 2)
    println("Using Beta prior on Rho");
  else
    println("Using Normal prior on Rho");
}

/*
**
**  LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian)
**
**  Purpose:
**    Calculate the MEAN LogLikelihood for the GLL model, using the global
**    data variables
**
**  Inputs:
**    vP          nDim vector of parameters, containing Rho, s(eta)
**                and s(epsilon)
**    adFunc      Address for function value
**    avScore     0 or address
**    amHessian   0 or address
**    s_Kalm_mYtEst    static, matrix with the data
**
**  Outputs:
**    adFunc      Function value at parameter vector(s)
**    avScore     Not changed
**    amHessian   Not changed
**    Return-value      1 if succeeded, 0 otherwise
**
*/
LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian)
{
  decl dRho, dSEta, dSEps, ir, vLL, mKF, mPhi, mOmega, mSigma;

  if (!s_Kalm_Warn)
    {
      println ("Warning: Quasi-likelihood is calculated");
      s_Kalm_Warn= 1;
    }

  dRho= vP[0];
  dSEta= vP[1];
  dSEps= vP[2];
  adFunc[0]= M_NAN;

  ir= 0;
  if ((vP[0] <= 1) && (vP[1:2] >= 0))
    {
      ir= 1;
      mPhi= dRho|1;
      mOmega= diag(dSEta*dSEta~dSEps*dSEps);
      mSigma= 1e6|0;
      if (dRho < 1)
        mSigma= (dSEta*dSEta/(1-dRho))|0;
      mKF= KalmanFil(s_Kalm_mYtEst, mPhi, mOmega, mSigma);

      vLL= 0.5*log(mKF[2][])
           -0.5*mKF[0][].*mKF[0][].*mKF[2][]
           -0.5*log(M_2PI);

      adFunc[0]= meanr(vLL);

      if (g_Kalm_UsePrior)
        adFunc[0]= adFunc[0] + LnPrior_Shell(vP)/columns(s_Kalm_mYtEst);
    }

  return ir;
}

/*
**
**  LnPrior_Shell(const vP)
**
**  Purpose:
**    Calculate the Log prior for the GLL model, using the global
**    data variables
**
**  Inputs:
**    vP          nDim vector of parameters, containing Rho, s(epsilon), 
**                s(eta). 
**
**  Outputs:
**    Return-value  Log prior
**
*/
LnPrior_Shell(const vP, ...)
{
  decl Retval, dRho, dSEps, dSEta, dMuH, dPhi, dSXi, va;

  dRho= vP[0];
  dSEta= vP[1];
  dSEps= vP[2];
  dMuH= dPhi= dSXi= 0;
  if (sizeof(vP) == 5)
    {
      dSEps= 0;
      dMuH= vP[2];
      dPhi= vP[3];
      dSXi= vP[4];
    }

  Retval= M_NAN;

  // Prior is IG(a,b) on S2Eps, thus SEps*IG(a, b) on SEps
  Retval= LnPriorRho(dRho, g_Kalm_PriorRho);
  if (dSEta > 0)
    Retval|= (LnPriorS2(dSEta^2, g_Kalm_PriorS2EtaAB[0], g_Kalm_PriorS2EtaAB[1])
              + log(2*dSEta));
  if (dSEps > 0)
    Retval|= (LnPriorS2(dSEps^2, g_Kalm_PriorS2EpsAB[0], g_Kalm_PriorS2EpsAB[1])
              + log(2*dSEps));
  if (!(dMuH == 0))
    Retval|= LnPriorNorm(dMuH, g_Kalm_PriorMuH[0], g_Kalm_PriorMuH[1]);
  if (!(dPhi == 0))
    Retval|= LnPriorNorm(dPhi, g_Kalm_PriorPhi[0], g_Kalm_PriorPhi[1]);
  if (!(dSXi == 0))
    Retval|= (LnPriorS2(dSXi^2, g_Kalm_PriorS2XiAB[0], g_Kalm_PriorS2XiAB[1])
              + log(2*dSXi));

  va= va_arglist();
  if (sizeof(va) == 0)
    return sumc(Retval);
  else
    return {sumc(Retval), Retval};
}

/*
**
**  ChangeBounds(const amCBounds)
**
**  Purpose:
**    Check the bounds that are going to be used, e.g. against non-
**    stationarity of the parameters
**
**  Inputs:
**    mCBounds    Address of matrix of proposed bounds
**
**  Output:
**    mCBounds    Address of matrix of changed proposed bounds
**
*/
ChangeBounds(const amCBounds)
{
  println ("Warning: Using changebounds? Not implemented");
}

/*
**
**  Procedure GetData(const amY, const amDMY, const amInter);
**
**  Purpose:
**    Read the dataset from the static variable, and return it
**
**  Inputs:
**    amY    Address of vector of data
**
**  Outputs:
**    amY    Vector of data
**
*/
GetData(const amY, const amDMY, const amInter)
{
  amY[0]= s_Kalm_mYt;
  amDMY[0]= s_Kalm_mDMY;
  amInter[0]= s_Kalm_mInter;
}

/*
**  LnPriorRho(const dRho, const vPrior);
**
**  Purpose:
**    Calculate the Beta prior on Rho, only if it is truly different
**    from zero and 1.
**    Calculate the normal prior if vPrior is of size 3; mu and sdev in
**    the first to elements of vPrior
*/
LnPriorRho(const dRho, const vPrior)
{
  decl dLnPrior;

  dLnPrior= 0;
  if ((dRho != 0) && (dRho != 1))
    if (sizerc(vPrior) == 2)
      dLnPrior= log(densbeta(dRho, vPrior[0], vPrior[1]));
    else
      dLnPrior= log(densn((dRho-vPrior[0])/vPrior[1])) - log(vPrior[1]);
  return dLnPrior;
}

/*
**  LnPriorS2(const mS2, const alpha, const beta)
**
**  Purpose:
**    Calculate the Inv gamma prior on S2. Do not apply the prior
**    if S2 is identical to zero.
*/
LnPriorS2(const mS2, const alpha, const beta)
{
  if (mS2 > 0)
    return log(densigamma(mS2, alpha, beta));
  else
    return 0;
}

/*
**  LnPriorNorm(const dP, const dMu, const dSDev)
**
**  Purpose:
**    Calculate the normal prior on dP, with mean and sdev given
*/
LnPriorNorm(const dP, const dMu, const dSDev)
{
  return -0.5*(log(M_2PI*sqr(dSDev)) + sqr((dP-dMu)/dSDev));
}

/*
**
**  Procedure GiveResModel(vP, OutBase);
**
**  Purpose:
**    Prepare some results
**
*/
GiveResModel(const vP, const OutBase, const VarNames)
{
  // Does not do a thing.
}

/*
**
**  CalcKalmanFilter(const vP, const mY, const aPredMean, const aPredVol)
**
**  Purpose:
**    Calculate the filter, return predictions for mean and variance
**    of the observation
**
**  Inputs:
**    vP          Vector of parameters
**    mY          1 x T row with dataset; if wanted, the original dataset may be
**                shortened to include less info, as only the last
**                100 observations are effectively used...
**
**  Outputs:
**    aPredMean   T+1 x 1 vector, predicted mean of the series,
**    aPredVol    T+1 x 1 vector, predicted variance of the series,
**    Ret. value  Indicates if filtering succeeded
*/
CalcKalmanFilter(const vP, const mY, const aPredMean, const aPredVol)
{
  println ("CalcKalmanFilter not implemented for SV model");

  return 0;
}

CalcKalmanMVS(const vP, const mYt, const nPeriods, const amPMSdS)
{
  println("CalcKalmanMVS not implemented, \n",
          "filtering should be done during sampling");
  return 0;
}

/*
**  SampleMu(const vh, const dRho, const dSEta, const mYt)
**
**  Purpose:
**    Sample Mu, conditional on the data, the values of h, rho and the
**    variance of eta
**
**  Inputs:
**    vh          row vector, values of h(t)
**    dRho        scalar, value of Rho
**    dSEta       scalar, sdev of Eta
**    mYt          row vector of data
**
**  Outputs:
**    vMu   Return value, row vector with sampled values of Mu
*/
SampleMu(const vh, const dRho, const dSEta, const mYt)
{
  decl
    mPhi, mOmega, mSigma, mDelta, mJ_Phi, mJ_Omega, mJ_Delta, vS2Eps,
    vMu;

  mPhi= dRho|1;
  mOmega= diag(dSEta*dSEta|1);
  mSigma= <-1; 0>;
  if (dRho < 1)
    mSigma= dSEta*dSEta/(1-dRho*dRho)|0;
  mJ_Phi= mDelta= mJ_Delta= <>;

  // Indicate that the variance of epsilon is time varying
  mJ_Omega= <-1, -1; -1, 0>;

  // Create a vector with the variances of epsilon
  vS2Eps= exp(vh);
//  println ("l S2Eps: ", limits(vS2Eps')'~dRho~dSEta~limits(vh')'[:2]);

  vMu= SsfCondDens(ST_SIM, mYt, mPhi, mOmega, mSigma, mDelta, mJ_Phi,
                   mJ_Omega, mJ_Delta, vS2Eps);
  vMu= vMu[0][];

  return vMu;
}

/*
**  SampleRho(const adRho, const vMu, const dSEta, const vPriorRho)
**
**  Purpose:
**    Sample Rho, conditional on the state Mu and the variance of the
**    equation
**
**  Remark:
**    The likelihood of Rho is normal, with the OLS estimates for
**    mean and variance. Prior is Beta, which limits Rho between 
**    0 and 1. Candidate is the normal density, such that 
**      alpha= (L(RhoNew) pi(RhoNew) f_c(RhoOld)) / 
**             (L(RhoOld) pi(RhoOld) f_c(RhoNew)) =
**             pi(RhoNew) / pi(RhoOld)
**    as f_c(Rho) = L(Rho)
**
**  Inputs:
**    adRho       pointer to present value of rho
**    vMu         row vector with sampled values of Mu
**    dSEta       scalar, sdev of Eta
**
**  Outputs:
**    adRho       pointer to new value of Rho
*/
SampleRho(const adRho, const vMu, const dSEta, const vPriorRho)
{
  decl
    iN, invXX, dS2RhoHat, dRhoHat, dMuRho, dS2Rho;

  iN= columns(vMu);
  olsr(vMu[1:], vMu[:iN-2], &dRhoHat, &invXX);
  dS2RhoHat= dSEta*dSEta*invXX;

  if (sizerc(vPriorRho) == 2)
    {
      println("Error: Only normal prior on rho implemented");
      exit(1);
    }
  else
    {
      adRho[0]= 2;
      dMuRho= (dRhoHat*vPriorRho[1]^2 + vPriorRho[0]*dS2RhoHat)/
              (vPriorRho[1]^2+dS2RhoHat);
      dS2Rho= dS2RhoHat * vPriorRho[1]^2 / (vPriorRho[1]^2+dS2RhoHat);
      adRho[0]= rann(1, 1)*sqrt(dS2Rho) + dMuRho;
    }
}

/*
**
**  SampleS2(const ve, const vPriorS2AB)
**
**  Purpose:
**    Sample a variance, conditional on the errors ve,
**    applying the IG prior
**
**  Inputs:
**    ve          row vector with disturbances
**    vPriorS2AB  alpha and beta of the IG(alpha, beta) prior
**
**  Outputs:
**    dS2         Return value, scalar, variance
**
*/
SampleS2(const ve, const vPriorS2AB)
{
  decl
    iT, alpha, beta, ds, dS2;

  iT= columns(ve);
  alpha= iT/2 + vPriorS2AB[0];
  ds= sumsqrr(ve) + 2/vPriorS2AB[1];
  beta= 2/ds;
  dS2= ranigamma(1, 1, alpha, beta);

  return dS2;
}

/*
**
**  Procedure SampleS(const vyStar, const vh)
**
**  Purpose:
**    Sample S, the indicator into the mixture of normals used to
**    approximate the log squared normal distribution
**
**  Inputs:
**    vyStar      row vector with log((y(t)-mu(t))^2)
**    vh          row vector with sampled values of h(t)
**
**  Outputs:
**    vis         Return value, vector with integer indices
**
*/
SampleS(const vYStar, const vh)
{
  decl
    vis, vQi, vMui, vSi, mfti, u;

    // Size 7 x 1
  vQi= s_GS_qmv[][0];
    // Size 7 x 1
  vMui= s_GS_qmv[][1]-1.2704;
    // Size 7 x 1
  vSi= sqrt(s_GS_qmv[][2]);

    // Calculate q_i * f_Norm(Ystar - h - mui, S2i), size 7 x N
  mfti= (vQi ./ vSi) .* densn((vYStar - vh - vMui)./vSi);
    // Calculate cumsum, but normalize u instead of mfti
  mfti= cumulate(mfti);

    // Size 1 x N, and normalize here
  u= ranu(1, columns(vYStar)) .* mfti[6][];
  vis= limits(u.<mfti)[3][];  

  return vis;
}

/*
**  SampleH(const vYStar, const vis, const dPhi, 
**          const dMuH, const dSXi)
**
**  Purpose:
**    Sample H, conditional on (a function of) the data, the values of 
**    indices vis, Phi, MuH, and the variance of Xi
**
**  Inputs:
**    vYStar      row vector, function of the data
**
**  Outputs:
**    vh   Return value, row vector with sampled values of Mu
*/
SampleH(const vYStar, const vis, const dPhi, const dMuH, const dSXi)
{
  decl
    mPhi, mOmega, mSigma, mDelta, mJ_Phi, mJ_Omega, mJ_Delta, vMui, vS2i,
    vh;

  mPhi= dPhi|1;
  mOmega= diag(dSXi*dSXi|1);
  mDelta= ((1-dPhi)*dMuH)|1;
  mSigma= -1|(meanr(vYStar)+1.2704);
  if (dPhi < 1)
    mSigma[0]= dSXi*dSXi/(1-dPhi^2);
  mJ_Phi= <>;

  // Indicate that the mean disturbance in the observation equation is time varying
  mJ_Delta= <-1; 0>;

  // Indicate that the variance in the observation equation is time varying
  mJ_Omega= <-1, -1; -1, 1>;

  // Calculate base mean and variance 
  vMui= s_GS_qmv[][1]-1.2704;
  vS2i= s_GS_qmv[][2];

  // Relate it to the vector of indices
  vMui= vMui[vis]';
  vS2i= vS2i[vis]';

  vh= SsfCondDens(ST_SIM, vYStar, mPhi, mOmega, mSigma, mDelta, mJ_Phi,
                  mJ_Omega, mJ_Delta, vMui|vS2i);
  vh= vh[0][];

  return vh;
}

/*
**  SamplePhi(const vh, const dMuH, const dSXi, const vPriorPhi)
**
**  Purpose:
**    Sample Phi, conditional on the values of h, the const dMuH and the 
**    variance of Xi. Use normal prior, no rejections.
**
**  Inputs:
**    vh          row vector, values of h(t)
**    dMuH        scalar, value of MuH
**    dSXi        scalar, sdev of Xi
**    dPhiOld     scalar, old value of Phi
**
**  Outputs:
**    dPhi  Return value, scalar with sampled value of Phi
*/
SamplePhi(const vh, const dMuH, const dSXi, const vPriorPhi)
{
  // Declaration of maximum number of tries for getting a stationary dPhi
  decl
    iMaxTry= <100>;

  decl
    iN, XX, vhMu, dS2PhiHat, dPhiHat, dPhi, dMuPhi, dS2Phi, gPhi, gPhiOld, dAlpha, iTry;

  iN= columns(vh);
  vhMu= vh-dMuH;
  XX= sumsqrr(vhMu[:iN-2]);
  dS2PhiHat= dSXi*dSXi/XX;
  dPhiHat= vhMu[:iN-2]*vhMu[1:]'/XX;

  if (sizerc(vPriorPhi) == 2)
    {
      println ("Error: Only normal prior on Phi implemented");
      exit(1);
    }
  else
    {
      dMuPhi= (dPhiHat*vPriorPhi[1]^2 + vPriorPhi[0]*dS2PhiHat)/
                (vPriorPhi[1]^2+dS2PhiHat);
      dS2Phi= dS2PhiHat * vPriorPhi[1]^2 / (vPriorPhi[1]^2+dS2PhiHat);
      dPhi= rann(1, 1)*sqrt(dS2Phi) + dMuPhi;
    }

  return dPhi;
}

/*
**  SampleMuH(const vh, const dPhi, const dSXi)
**
**  Purpose:
**    Sample MuH, conditional on the state h and the variance of the
**    equation
**
**  Inputs:
**    vh          row vector with sampled values of H
**    dPhi        scalar, ather constant
**    dSXi        scalar, sdev of Xi
**    vPriorMuH   row vector, mean and sDev of normal prior on MuH
**
**  Outputs:
**    dMuH        Return value, scalar, value of MuH
*/
SampleMuH(const vh, const dPhi, const dSXi, const vPriorMuH)
{
  decl
    iN, dS2MuH, dMuHHat, dMuH, dN;

  iN= columns(vh);
  dS2MuH= dSXi*dSXi/((iN-1)*(1-dPhi)^2 + (1-dPhi^2));
  dMuHHat= (dS2MuH/(dSXi*dSXi))
             *((1-dPhi^2)*vh[0] + (1-dPhi)*sumr(vh[1:]-dPhi*vh[:iN-2]));

  // Equivalent number of observations in prior information
  dN= vPriorMuH[1] * vPriorMuH[1] / dS2MuH;
  dMuHHat= (dN * dMuHHat + vPriorMuH[0])/(dN+1);
  dS2MuH= dS2MuH * dN/(dN+1);

  dMuH= rann(1, 1)*sqrt(dS2MuH) + dMuHHat;

  return dMuH;
}

/*
**  SampleS2Xi(const dPhi, const dMuH, const vh, 
**                       const vPriorS2Xi)
**
**  Purpose:
**    Sample S2Xi, conditional on the state Mu and the AR parameter
**    Rho
**
**  Inputs:
**    dPhi        scalar, value of Phi
**    dMuH        scalar, value of MuH
**    vh          row vector with sampled values of h
**    vPriorS2XiAB 2 x 1 vector with hyperparameters alpha and beta for IG-prior
**
**  Outputs:
**    dS2Xi       Return value, scalar, variance of Xi
*/
SampleS2Xi(const dPhi, const dMuH, const vh, const vPriorS2AB)
{
  decl
    iT, alpha, beta, ds, dS2Xi, vhMu;

  iT= columns(vh);
  alpha= iT/2 + vPriorS2AB[0];
  vhMu= vh-dMuH;
  ds= (1-dPhi^2)*vhMu[0]^2 + sumsqrr(vhMu[1:]-dPhi*vhMu[:iT-2]) 
        + 2/vPriorS2AB[1];
  beta= 2/ds;
  dS2Xi= ranigamma(1, 1, alpha, beta);

  return dS2Xi;
}


