/*
**
**  CalcHPD(const amR, const amXL, const amXU, const avCov, 
**          const vX, const vF, const vAlpha, const bScal)
**
**  Purpose:
**    Find the highest posterior density region with coverage equal
**    to vAlpha. Region is indicated by a vector vR of 0's and 1's,
**    indicating the vX's that are included in the region, and by
**    (adjusted) lower and upper bounds.
**
**  Inputs:
**    vX          1 x n vector with equidistant x-grid
**    vF          1 x n vector with heights of density above the 
**                grid. 
**    vAlpha      nA x 1 vector, coverage probabilities
**    bScal       Boolean, indicating if scaling should be applied if
**                necessary.
**    
**  Outputs:
**    amR         Pointer to nA x n matrix of 0/1 variables indicating
**                if corresponding element is included in HPD
**    amXL        Pointer to nA x 1 matrix with first 
**                element belonging to HPD region. 
**    amXU        Pointer to nA x 1 matrix with last
**                element belonging to HPD region. 
**    avCov       Pointer to nA x 1 vector, effective coverage probability
**    r.v.        Boolean, true if (vX, vF) is found to integrate to
**                1, false otherwise. 
**
**  Date:
**    2/9/2000
**
**  Author:
**    Charles Bos
*/
// static decl s_HPD_Warn= FALSE;

CalcHPD(const amR, const amXL, const amXU, const avCov, 
        const vX, const vF, const vAlpha, const bScal)
{
  decl nX, nA, vdX, vFF, ir, mXF, vHPD, i, miLU, mXLU, 
       dC0, dC1, dY, dCov;

  nX= columns(vX);
  vdX= diff0((vX[0]~vX~vX[nX-1])', 1)[1:]';
  // Endpoints get half their weight
  vdX= (vdX[:nX-1]+vdX[1:])/2;

  // Warning not necessary?
  if (!isfeq(min(diff0(vX', 1)[1:]), max(diff0(vX', 1)[1:])))
    println("Warning: Non-equidistant grid in incHPD", 
            min(diff0(vX', 1)[1:])~max(diff0(vX', 1)[1:]));

  vFF= vdX .* vF;
  ir= 1;
  if (!isfeq(sumr(vFF), 1.0))
    {
      ir= 0;
      if (bScal)
        {
//          println("Warning: Scaling density by ", double(sumr(vFF)));
          vFF/= sumr(vFF);
        }
      else
        println("Warning: Scaling density by ", double(sumr(vFF)), " advised.");
    }

  // Sort by FF, largest first
  mXF= reverser(sortbyr(range(0, nX-1)|vFF, 1));

  mXF|= cumulate(mXF[1][]')';
  mXF= sortbyr(mXF, 0);

  nA= sizerc(vAlpha);
  amR[0]= constant(M_NAN, nA, nX);
  avCov[0]= constant(M_NAN, nA, 1);
  mXLU= (vX[0]~vX[nX-1]).*ones(nA, 2);

  for (i= 0; i < nA; ++i)
    {
      vHPD= (mXF[2][] .<= vAlpha[i]);

      amR[0][i][]= vHPD;

      if (max(vHPD) == 1)
        {
          dCov= max(selectifc(mXF[2][], vHPD));
          miLU= limits(selectifc(range(0, nX-1), vHPD)')[0:1]';

          // Calculate approximate rates of change at the corners
          dC0= dC1= M_INF;
          if (miLU[0] > 0)
            dC0= (vFF[miLU[0]]-vFF[miLU[0]-1])/vdX[miLU[0]];
          if (miLU[1] < nX-1)
            dC1= (vFF[miLU[1]]-vFF[miLU[1]+1])/vdX[miLU[1]];
            
          // Change in height which can be made:
          dY= sqrt(2*(vAlpha[i]-dCov)/(1/dC0 + 1/dC1));

          mXLU[i][]= vX[miLU];
          if (dC0 < M_INF)
            mXLU[i][0]-= vdX[miLU[0]]*dY/dC0;
          if (dC1 < M_INF)
            mXLU[i][1]+= vdX[miLU[1]]*dY/dC1;
          avCov[0][i]= vAlpha[i];
//          print (dC0~dC1~dY~dCov~vX[miLU]~mXLU[i][]);
        }
      else
        println("Trouble in IncHPD; no coverage?");
    }

  amXL[0]= mXLU[][0];
  amXU[0]= mXLU[][1];
  
  return ir;
}

/*
**  CalcHPDOnesided(const amR, const amXL, const amXU,  
**                  const vX, const vF, const dAlpha, const bScal, 
**                  const bLeft)
**
**  Purpose:
**    Find the one-sided region with coverage equal
**    to vAlpha. Region is indicated by a vector vR of 0's and 1's,
**    indicating the vX's that are included in the region, and by
**    the (adjusted) lower or upper bounds.
**
**  Inputs:
**    vX          1 x n vector with equidistant x-grid
**    vF          1 x n vector with heights of density above the 
**                grid. 
**    vAlpha      nA x 1 vector, coverage probabilities
**    bScal       Boolean, indicating if scaling should be applied if
**                necessary.
**    bLeft       Boolean, indicating if left part is to be cut off
**    
**  Outputs:
**    amR         Pointer to nA x n matrix of 0/1 variables indicating
**                if corresponding element is included in HPD
**    amXB        Pointer to nA x 1 matrix with bound of region. 
**    r.v.        Boolean, true if (vX, vF) is found to integrate to
**                1, false otherwise. 
**
**  Date:
**    2/9/2000
**
**  Author:
**    Charles Bos
*/
CalcHPDOnesided(const amR, const amXB, const vX, const vF, 
                const vAlpha, const bScal, const bLeft)
{
  decl nX, nA, vdX, vFF, ir, mXF, vHPD, i, miB, mXB, 
       dC, dY, dCov, iB, iBB;

  nX= columns(vX);
  vdX= diff0((vX[0]~vX~vX[nX-1])', 1)[1:]';
  // Endpoints get half their weight
  vdX= (vdX[:nX-1]+vdX[1:])/2;

  // Warning not necessary?
  if (!isfeq(min(diff0(vX', 1)[1:]), max(diff0(vX', 1)[1:])))
    println("Warning: Non-equidistant grid in incHPD", 
            min(diff0(vX', 1)[1:])~max(diff0(vX', 1)[1:]));

  vFF= vdX .* vF;
  ir= 1;
  if (!isfeq(sumr(vFF), 1.0))
    {
      ir= 0;
      if (bScal)
        {
//          println("Warning: Scaling density by ", double(sumr(vFF)));
          vFF/= sumr(vFF);
        }
      else
        println("Warning: Scaling density by ", double(sumr(vFF)), " advised.");
    }

  mXF= range(0, nX-1)|vFF;
  if (bLeft)    // Reverse if left part should be cut off.
    mXF= reverser(mXF);

  mXF|= cumulate(mXF[1][]')';

  nA= sizerc(vAlpha);
  amR[0]= constant(M_NAN, nA, nX);
  mXB= constant(M_NAN, nA, 1);

  for (i= 0; i < nA; ++i)
    {
      vHPD= (mXF[2][] .<= vAlpha[i]);

      amR[0][i][]= vHPD;

      if (max(vHPD) == 1)
        {
          dCov= max(selectifc(mXF[2][], vHPD));
          miB= limits((mXF[2][].*vHPD)')[3][];
          iB= iBB= mXF[0][miB];
          mXB[i]= vX[iB];
          if (miB < nX-1)
            {
              iBB= mXF[0][miB+1];
              mXB[i]+= (vX[iBB]-vX[iB]) * (vAlpha[i]-dCov)/
                          (mXF[2][miB+1]-mXF[2][miB]);
            }

//          print (iB~iBB~dCov~vX[iB]~vX[iBB]~mXB[i]);
//          print (mXF'~vHPD'~vX'~vFF');
//          exit(1);
        }
      else
        println("Trouble in IncHPD; no coverage?");
    }

  amXB[0]= mXB;
  
  return ir;
}
