/*
**
**  Library IncGLGAS
**
**  Purpose:
**    Contain routines for the GLL-GARCH-Student t model,
**    together with the likelihood routine.
**
**  Version:
**    1     Based on IncGLGA
**    4     Including sampling routines
**    5     Including UIP
**
**  Date:
**    8/8/2000
**
**  Author:
**    Charles Bos
**
*/
#include <oxfloat.h>    // M_NAN and M_INF_NEG
#include <oxprob.h>
#include <arma.h>
#include "ssfpack.h"
#include "include/oxprobig.ox"
#include "include/incdata.ox"

#ifdef OX_Linux
  #define INCFILGS_INCLUDED
  extern "include/incfilgs_lin,FnKalmanFil_GarchSt" 
    FnKalmanFil_GarchSt(const mYt, const dRho, const dSEta, const dSEps, 
                        const dDelta, const dAlpha, const vz, const mSigma);
#endif
#ifdef OX_Sun
  #define INCFILGS_INCLUDED
  extern "include/incfilgs_sun,FnKalmanFil_GarchSt" 
    FnKalmanFil_GarchSt(const mYt, const dRho, const dSEta, const dSEps, 
                        const dDelta, const dAlpha, const vz, const mSigma);
#endif
#ifdef OX_Windows
  #define INCFILGS_INCLUDED
  extern "include/incfilgs,_FnKalmanFil_GarchSt"
    FnKalmanFil_GarchSt(const mYt, const dRho, const dSEta, const dSEps, 
                        const dDelta, const dAlpha, const vz, const mSigma);
#endif



PdfInit(const avP, const anData);
GenrKalmanGARCHSt(const aY, const dRho, const dSEta, const dSEps, 
                  const dDelta, const dAlpha, const dNu, 
                  const dPastMu, const dPastH, const nData);
LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian);
LnPdfGSStudCond(const dRho, const dSEta, const dSEps, const dDelta, 
                const dAlpha, const vz, const mYt, const adFunc);
KalmanFil_Shell(const mYt, const dRho, const dSEta, const dSEps, 
                const dDelta, const dAlpha, vz, const mSigma);
KalmanFil_GarchSt(const mYt, const dRho, const dSEta, const dSEps, 
                  const dDelta, const dAlpha, const vz, const mSigma);
SsfMomentEstPred_GarchSt(const amPred, const mYt, const dRho, const dSEta, 
                         const dSEps, const dDelta, const dAlpha, const vz);
LnPrior_Shell(const vP, ...);
ChangeBounds(const amCBounds);
GetData(const amY, const amDMY, const amInter);
LnPriorRho(const dRho, const vPrior);
LnPriorS2(const mS2, const s, const nu);
LnPriorS(const dS, const alpha, const beta);
LnPriorGARCH(const dSEps, const dDelta, const dAlpha, 
             const dIGalpha, const dIGbeta);
LnPriorNu(const dNu);
GiveResModel(const vP, const OutBase, const VarNames);
CalcKalmanFilter(const vP, const mYt, const aPredMean, const aPredVol);
CalcKalmanMVS(const vP, const mYt, const nPeriods, const amPMSdS, ...);
SampleMu(const mYt, const dRho, const dSEta, const dSEps, 
         const dDelta, const dAlpha, const vz);
SampleRho(const adRho, const vMu, const dSEta, const vPriorRho);
SampleS2(const ve, const vPriorS2AB);
SampleGARCH(const adSEps, const adDelta, const adAlpha, const vz, 
            const vv2, const vMean_Cand, const vS_Cand, const vPriorS2EpsAB);
SampleZ(const avz, const dSEps, const dDelta, const dAlpha, 
        const dNu, const vv2);
SampleNu(const adNu, const vz, const vCandNu);
SampleCandGarch(const vGammaOld, const vSGamma);
CondLik_Garch(const adLL, const adVar, const dSEps, 
              const dDelta, const dAlpha, const vz, const vv2);
CalcVarianceGARCH(const avh, const dSEps, const dDelta, const dAlpha, const vv2);
RanLnCandNuRW(const adNu, const dNuOld, const vCand, const bDraw);
LnPdfPostNu(const vz, const dNu);

/* Local static declarations, for the dataset */
static decl s_Kalm_mYt, s_Kalm_mYtEst, s_Kalm_mDMY, s_Kalm_mInter, s_Kalm_mIntDiff, 
            s_Kalm_Pars, s_Kalm_mKFh, s_Kalm_Cauchy= 0,
            s_Kalm_Filter= 0, s_Kalm_Warn= 0,
            s_Kalm_yPred= 0, s_Kalm_yVPred= -1,
            s_Kalm_GPars, s_Kalm_vh;

/*
**
**  Procedure PdfInit
**
**  Purpose:
**    Initialize the static variables in this module, using settings in
**    Kalman.Dec
**
**  Inputs:
**    aInitVP     Address of initial values for vP. Might have been
**                initialized in Kalman.DEC
**    anData      Address of number of data points.
**
**  Outputs:
**    aInitVP     Initial values for VP
**    anData      Number of data points
**
*/
PdfInit(const avP, const anData)
{
  decl iT, dd, vP;

  vP= avP[0];
  if (rows(vP) != 6)
    {  
      println ("Error: Size of InitVP not correct, all parameters");
      println ("Rho, s(Eta), s(Eps), Delta, Alpha and dNu should be specified");
      println ("InitVP: ", vP');
      exit(1);
    }
  if (g_Kalm_UseModel == "GLLGAS")
    {
      println("Generalized Local Level-GARCH Student t model");
      println("Parameters: Rho, s(Eta), s(Eps), Delta, Alpha, dNu");
    }
  else if (g_Kalm_UseModel == "GLLGS")
    {
      println("Generalized Local Level-Student t model");
      println("Parameters: Rho, s(Eta), s(Eps), dNu");
    }
  else if (g_Kalm_UseModel == "GLL")
    {
      println("Generalized Local Level model");
      println("Parameters: Rho, s(Eta), s(Eps)");
    }
  else if (g_Kalm_UseModel == "GLLGA")
    {
      println("Generalized Local Level-GARCH model");
      println("Parameters: Rho, s(Eta), s(Eps), Delta, Alpha");
    }
  else 
    println("Error: Model type not recognized.\n", 
            "Incorrect declarations file?");

  if (g_Kalm_Reload == 1)
    {
      LoadData(g_Kalm_DataFile, g_Kalm_DataFMT, g_Kalm_DataFrac, &s_Kalm_mYt, 
               &s_Kalm_mDMY, &s_Kalm_mInter, g_Kalm_Diff, g_Kalm_Invert);
      iT= columns(s_Kalm_mYt);
    }
  else
    {
      println("Warning: Generating only checked for GLL-Student t-GARCH");
      GenrKalmanGARCHSt(&s_Kalm_mYt, vP[0], vP[1], vP[2], 
                        vP[3], vP[4], vP[5],
                        0, 1, max(g_Kalm_DataFrac, g_nData));

      iT= columns(s_Kalm_mYt);
      s_Kalm_mDMY= range(1, iT)|ones(2, iT);
      s_Kalm_mInter= zeros(2, iT);

      dd= s_Kalm_mDMY'~ones(iT, g_Kalm_DataFMT[0]-3);
      dd[][g_Kalm_DataFMT[1]]= s_Kalm_mYt';

      if (!savemat(g_Kalm_DataFile, dd, 1))
        println ("Error: Saving of datafile did not succeed");
    }

  // Calculate the interest rate differential
  s_Kalm_mIntDiff= (s_Kalm_mInter[0][]-s_Kalm_mInter[1][])/360;

  s_Kalm_mYtEst=  s_Kalm_mYt[:g_Kalm_FracEst];
  anData[0]= columns(s_Kalm_mYtEst);

  // If UIP is included in the model, use s*(t)= s(t) + rh - rf,
  //   and run model on this s*(t)
  if (g_Kalm_UseUIP == 1)
    {
      s_Kalm_mYtEst+= s_Kalm_mIntDiff[:g_Kalm_FracEst];
      println("Using the uncovered interest rate parity");
    }

  if (g_Kalm_UsePrior == 1)
    println("Using a prior on the parameters");
  if (sizeof(g_Kalm_PriorRho) == 2)
    println("Using Beta prior on Rho");
  else
    println("Using Normal prior on Rho");
}

/*
**
**  GenrKalmanGARCHSt(const aY, const dRho, const dSEta, const dSEps, 
**                    const dDelta, const dAlpha, const dNu,
**                    const dPastMu, const dPastH, const iT)
**
**  Purpose:
**    Generate data from the GLL-GARCH Student t model
**
*/
GenrKalmanGARCHSt(const aY, const dRho, const dSEta, const dSEps, 
                  const dDelta, const dAlpha, const dNu, 
                  const dPastMu, const dPastH, const iT)
{
  decl u, eps, eta, mu, h, i, dOmega;

  aY[0]= zeros(1, iT);
  if (dNu > 2)
    u= (sqrt((dNu-2)/dNu)*rant(iT, 1, dNu))~rann(iT, 1);
  else
    u= rann(iT, 2);

  eta= u[][1]*dSEta;
  h= dPastH;
  dOmega= (1-dDelta-dAlpha);
  if (h < 0)
    h= dOmega/(1-dDelta-dAlpha);
  mu= dPastMu;

  for (i= 0; i< iT; ++i)
    {
      eps= u[i][0] * sqrt(h);
      mu= dRho * mu + eta[i];
      aY[0][i]= mu + dSEps * eps;
      h= dDelta * h + dOmega + dAlpha * eps * eps;
    }
}

/*
**
**  LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian)
**
**  Purpose:
**    Calculate the MEAN LogLikelihood for the GLL-Garch model, using
**    the global data variables
**    Note that this returns the likelihood for the GLL-GARCH model,
**    which is only an approximation for the GLL-GARCH Student t model
**
**  Inputs:
**    vP          nDim vector of parameters, containing Rho, s(eta), 
**                s(epsilon), delta and alpha. 
**    adFunc      Address for function value
**    avScore     0 or address
**    amHessian   0 or address
**    s_Kalm_mYtEst    static, matrix with the data
**
**  Outputs:
**    adFunc      Function value at parameter vector(s)
**    avScore     Not changed
**    amHessian   Not changed
**    Return-value      1 if succeeded, 0 otherwise
**
*/
LnPdf_Shell(const vP, const adFunc, const avScore, const amHessian)
{
  decl dRho, dSEps, dSEta, dDelta, dAlpha, ir, vLL, mKFh, vz, mSigma;

  if (!s_Kalm_Warn && (g_Kalm_UseModel != "GLLGA") && (g_Kalm_UseModel != "GLL"))
    {
      println ("Warning: Quasi-likelihood GLL-GARCH is calculated");
      s_Kalm_Warn= 1;
    }
  dRho= vP[0];
  dSEta= vP[1];
  dSEps= vP[2];
  dDelta= dAlpha= 0;
  if (rows(vP) > 3)
    {
      dDelta= vP[3];
      dAlpha= vP[4];
    }
  adFunc[0]= M_NAN;
  vz= 1;

  ir= 0;
  mSigma= 1e6|0;
  if (dRho < 1)
    mSigma= dSEta*dSEta/(1-dRho)|0;
  mKFh= KalmanFil_Shell(s_Kalm_mYtEst, dRho, dSEta, dSEps, 
                        dDelta, dAlpha, vz, mSigma);
  if (!(mKFh == 0))
    {
      ir= 1;
      vLL= 0.5*log(mKFh[2][])
           -0.5*mKFh[0][].*mKFh[0][].*mKFh[2][]
           -0.5*log(M_2PI);

      adFunc[0]= meanr(vLL);

      if (g_Kalm_UsePrior)
        adFunc[0]= adFunc[0] + 
          LnPrior_Shell(vP[:2]|dDelta|dAlpha|0)/columns(s_Kalm_mYtEst);

    }
    
//  print (dRho~dSEta~dSEps~dDelta~dAlpha~adFunc[0]~ir);
  
  return ir;
}

/*
**  LnPdfGSStudCond(const dRho, const dSEta, const dSEps, const dDelta, 
**                  const dAlpha, const vz, const mYt, const adFunc)
**
**  Purpose:
**    Calculate the MEAN LogLikelihood for the GLL-Student t model, 
**    conditional on the value of the z-vector. Do NOT apply a 
**    prior.
**
**  Inputs:
**    mYtEst      matrix with the data
**    adFunc      Address for function value
**
**  Outputs:
**    adFunc      Function value at parameter vector(s)
**    Return-value      1 if succeeded, 0 otherwise
**
*/
LnPdfGSStudCond(const dRho, const dSEta, const dSEps, const dDelta, 
                const dAlpha, const vz, const mYt, const adFunc)
{
  decl ir, vLL, mKFh, mSigma;

  adFunc[0]= M_NAN;

  ir= 0;
  mSigma= 1e6|0;
  if (dRho < 1)
    mSigma= dSEta*dSEta/(1-dRho)|0;
  mKFh= KalmanFil_Shell(mYt, dRho, dSEta, dSEps, 
                        dDelta, dAlpha, vz, mSigma);
  if (!(mKFh == 0))
    {
      ir= 1;
      vLL= 0.5*log(mKFh[2][])
           -0.5*mKFh[0][].*mKFh[0][].*mKFh[2][]
           -0.5*log(M_2PI);

      adFunc[0]= meanr(vLL);

    }
  return ir;
}

/*
**
**  KalmanFil_Shell(const mYt, const dRho, const dSEta, const dSEps, 
**                  const dDelta, const dAlpha, const mSigma)
**
**  Purpose:
**    Provide a shell around the Kalman filter, possibly calling
**    c-routine
**
**  Inputs:
**    mYt               1 x n row vector of data
**    dRho..dAlpha      Parameters in the model
**    vz                1 x n row vector of variance factors
**    mSigma            Initial conditions
**
**  Output:
**    mKF               6 x T matrix, with v in first row, K in second,
**                      inv(F) in third row, h in fourth row, a in fifth and
**                      P in sixth row.
**                  or  0, if parameters incorrect
*/
KalmanFil_Shell(const mYt, const dRho, const dSEta, const dSEps, 
                const dDelta, const dAlpha, vz, const mSigma)
{
  decl vPars;

  if (columns(vz) < columns(mYt))
    vz= vz * ones(1, columns(mYt));
  vPars= dRho|dSEta|dSEps|dDelta|dAlpha|sumr(vz);
  if (!(s_Kalm_Pars == vPars))
    {
      s_Kalm_Pars= vPars;
      s_Kalm_mKFh= 0;
      if ((dSEta >= 0) && (dSEps >= 0) && (dDelta >= 0) && (dAlpha >= 0) 
          && (dDelta + dAlpha < 1))
        {
        // Use a trick: Only if this is Windows/Linux/Sun, UseSSF is checked
        #ifdef INCFILGS_INCLUDED
          if (g_Kalm_UseSSF > 0)
            s_Kalm_mKFh= FnKalmanFil_GarchSt(mYt, dRho, dSEta, dSEps, 
                                             dDelta, dAlpha, vz, mSigma);
          else
        #endif
            s_Kalm_mKFh= KalmanFil_GarchSt(mYt, dRho, dSEta, dSEps, 
                                           dDelta, dAlpha, vz, mSigma);
        }
    }
 
  return s_Kalm_mKFh;
}

/*
**
**  KalmanFil_GarchSt(const mYt, const dRho, const dSEta, const dSEps, 
**                    const dDelta, const dAlpha, const vz, const mSigma)
**
**  Purpose:
**    Run the Kalman filter, like in KalmanFil, now taking the GARCH
**    effect into account. Input and output similar. For the student-t
**    element, an extra time varying element vz is needed. It works like
**    a (time-varying) variance factor, multiplying h.
**
**  Inputs:
**    mYt               1 x n row vector of data
**    dRho..dAlpha      Parameters in the model
**    vz                1 x m row vector of variance increments
**    mSigma            Initial conditions
**
**  Output:
**    mKF               6 x T matrix, with v in first row, K in second,
**                      inv(F) in third row, h in fourth row, a in fifth and
**                      P in sixth row.
*/
KalmanFil_GarchSt(const mYt, const dRho, const dSEta, const dSEps, 
                  const dDelta, const dAlpha, const vz, const mSigma)
{
  decl iT, mHH, dS2Eps, dRho2, dOmega, 
       vh, vv, vF, vK, va, vP, i, mKFh, bGARCH;

  iT= columns(mYt);
  mHH= dSEta * dSEta;
  dS2Eps= dSEps * dSEps;
  dRho2= dRho * dRho;
  vv= vF= vK= va= vP= new matrix [1][iT+1];
  vh= ones(1, iT+1);
  va[0]= mSigma[1];
  vP[0]= mSigma[0];

  dOmega= 1-dDelta-dAlpha;
  bGARCH= (dDelta != 0) || (dAlpha != 0);

  for (i= 0; i < iT; ++i)
    {
      vv[i]= mYt[i] - va[i];
      vF[i]= vP[i] + dS2Eps * vh[i] * vz[i];
      vK[i]= dRho * vP[i] / vF[i];

      // Working for t+1
      va[i+1]= dRho * va[i] + vK[i] * vv[i];
      vP[i+1]= dRho2 * vP[i] + mHH - vK[i]*vK[i] * vF[i];
      if (bGARCH)
        vh[i+1]= dOmega + dDelta * vh[i] + dAlpha * vv[i]*vv[i]/dS2Eps;
    }
  mKFh= (vv[:iT-1]|vK[:iT-1]|(1.0 ./ vF[:iT-1])|
         vh[:iT-1]|va[:iT-1]|vP[:iT-1]);

  return mKFh;
}

/*
**  SsfMomentEstPred_GarchSt(const amPredH, const mYt, const dRho, 
**                           const dSEta, const dSEps, const dDelta, 
**                           const dAlpha, const vz)
**
**  Purpose:
**    Calculate the state predictions, as in SsfMomentEst(ST_PRED, ..). 
**
**  Inputs:
**    amPredH           0 or pointer to return matrix with predicted states
**                      a(t), y(t), P(t), F(t) and h(t)
**    mYt               1 x n row vector of data
**    dRho..dAlpha      Parameters in the model
**    vz                1 x m row vector of variance increments
**
**  Output:
**    mStateH           3 x 1 vector with P(T+1), a(T+1) and h(T+1)
**    amPred            if pointer to an address on input, 4 x T matrix
**                      with a(t), predicted y(t) (=a(t)), P(t), F(t) and h(t)
*/
SsfMomentEstPred_GarchSt(const amPred, const mYt, const dRho, const dSEta, 
                         const dSEps, const dDelta, const dAlpha, const vz)
{
  decl mKFh, mStateH, mSigma, iT, daPred, dPPred, dhPred;

  mSigma= 1e6|0;
  if (dRho < 1)
    mSigma= dSEta*dSEta/(1-dRho)|0;
  mKFh= KalmanFil_Shell(mYt, dRho, dSEta, dSEps, 
                        dDelta, dAlpha, vz, mSigma);

  if (!(mKFh == 0))
    {
      // mKFh= v K 1/F h a P

      if (isarray(amPred))
        // mPred contains a, y(=a), P, F and h
        amPred[0]= mKFh[<4; 4; 5>][]|(1.0 ./ mKFh[2][])|mKFh[3][];

      iT= columns(mKFh);
      daPred= dRho * mKFh[4][iT-1] + mKFh[1][iT-1]*mKFh[0][iT-1];
          // aPred= Rho * va[T] + K[T] * vv[t]
      dPPred= dRho*dRho * mKFh[5][iT-1]
                + dSEta*dSEta - mKFh[1][iT-1]*mKFh[1][iT-1]/mKFh[2][iT-1];
          // PPred= Rho^2 * vP[T] + dS2Eta - vK[T]*vK[T] * vF[T];
      dhPred= (1-dDelta-dAlpha) + dDelta * mKFh[3][iT-1] 
                + dAlpha * mKFh[0][iT-1] * mKFh[0][iT-1]/(dSEps * dSEps);
          // hPred= dOmega + dDelta * vh[T] + dAlpha * vv[T]*vv[T]/dS2Eps;

      return daPred|dPPred|dhPred;
    }
  else
    return 0;
}

/*
**  LnPrior_Shell(const vP, ...)
**
**  Purpose:
**    Calculate the Log prior for the GLL-GARCH model, using the global
**    data variables
**
**  Inputs:
**    vP          nDim vector of parameters, containing Rho, s(eta), 
**                s(epsilon), delta, alpha, nu. 
**
**  Outputs:
**    Return-value  Log prior
*/
LnPrior_Shell(const vP, ...)
{
  decl Retval, dRho, dSEps, dSEta, dDelta, dAlpha, dNu, va;

  dDelta= dAlpha= dNu= 0;
  dRho= vP[0];
  dSEta= vP[1];
  dSEps= vP[2];
  if (sizeof(vP) > 4)
    {
      dDelta= vP[3];
      dAlpha= vP[4];
    }
  if (sizeof(vP) > 5)
    dNu= vP[5];

  Retval= M_NAN;
  if ((dRho >= 0) && (dSEta >= 0) && (dSEps >= 0))
    {
      // Prior is IG(a,b) on S2Eps, thus SEps*IG(a, b) on SEps
      Retval= 
        LnPriorRho(dRho, g_Kalm_PriorRho)
          |LnPriorS(dSEta, g_Kalm_PriorS2EtaAB[0], g_Kalm_PriorS2EtaAB[1])
          |LnPriorS(dSEps, g_Kalm_PriorS2EpsAB[0], g_Kalm_PriorS2EpsAB[1]);
    }
  else 
    println ("Error: Rho, SEta or SEps out of bounds", vP');

  if ((dDelta >= 0) && (dAlpha >= 0) && (dDelta + dAlpha < 1))
    {
      // Skip prior SEps
      Retval= Retval[:1]
        |LnPriorGARCH(dSEps, dDelta, dAlpha, g_Kalm_PriorS2EpsAB[0], 
                      g_Kalm_PriorS2EpsAB[1]);
          
    }
  else if ((dDelta != 0) || (dAlpha != 0))
    println ("Error: Delta or Alpha out of bounds", dDelta~dAlpha);

  if (dNu > 1)
    Retval|= LnPriorNu(dNu);
  else if (dNu != 0)
    println ("Error: Nu out of bounds", dNu);

  va= va_arglist();
  if (sizeof(va) == 0)
    return sumc(Retval);
  else
    return {sumc(Retval), Retval};
}

/*
**
**  ChangeBounds(const amCBounds)
**
**  Purpose:
**    Check the bounds that are going to be used, e.g. against non-
**    stationarity of the parameters
**
**  Inputs:
**    mCBounds    Address of matrix of proposed bounds
**
**  Output:
**    mCBounds    Address of matrix of changed proposed bounds
**
**  Version:
**    Kalman      Only check on bounds on Delta and Alpha
**
*/
ChangeBounds(const amCBounds)
{
  decl vPL, vPH, dvP, Rho;

  println ("Warning: Using changebounds?");

  /* For the GARCH model */  
  /* Delta and Alpha, the squares of elements 3 and 4, have
     to be smaller than 1 in sum */
  vPL= amCBounds[0][<3, 4>][0].^2;
  vPH= amCBounds[0][<3, 4>][1].^2;
  dvP= vPH - vPL;
  Rho= min((1-sumc(vPL))/sumc(dvP), 1);
  if ( sumc(vPL) < 1 )
    amCBounds[0][<3, 4>][1]= (vPL + Rho.*dvP).^2;
  else
    amCBounds[0][<3, 4>][0]= (vPL + Rho.*dvP).^2;

}

/*
**
**  Procedure GetData(const amY, const amDMY, const amInter);
**
**  Purpose:
**    Read the dataset from the static variable, and return it
**
**  Inputs:
**    amY    Address of vector of data
**
**  Outputs:
**    amY    Vector of data
**
*/
GetData(const amY, const amDMY, const amInter)
{
  amY[0]= s_Kalm_mYt;
  amDMY[0]= s_Kalm_mDMY;
  amInter[0]= s_Kalm_mInter;
}

/*
**  LnPriorRho(const dRho, const vPrior);
**
**  Purpose:
**    Calculate the Beta prior on Rho, only if it is truly different
**    from zero and 1.
**    Calculate the normal prior if vPrior is of size 3; mu and sdev in
**    the first to elements of vPrior
*/
LnPriorRho(const dRho, const vPrior)
{
  decl dLnPrior;

  dLnPrior= 0;
  if ((dRho != 0) && (dRho != 1))
    if (sizerc(vPrior) == 2)
      dLnPrior= log(densbeta(dRho, vPrior[0], vPrior[1]));
    else
      dLnPrior= log(densn((dRho-vPrior[0])/vPrior[1])) - log(vPrior[1]);
  return dLnPrior;
}

/*
**  LnPriorS2(const mS2, const alpha, const beta)
**
**  Purpose:
**    Calculate the Inv gamma prior on S2. Do not apply the prior
**    if S2 is identical to zero.
*/
LnPriorS2(const mS2, const alpha, const beta)
{
  if (mS2 > 0)
    return log(densigamma(mS2, alpha, beta));
  else
    return 0;
}

/*
**  LnPriorS(const dS, const alpha, const beta)
**
**  Purpose:
**    Calculate the transformed Inv gamma prior on S. Do not apply the prior
**    if S2 is identical to zero.
*/
LnPriorS(const dS, const alpha, const beta)
{
  if (dS > 0)
    return lndensigamma(dS*dS, alpha, beta) + log(2*dS);
  else
    return 0;
}

/*
**  LnPriorGARCH(const dSEps, const dDelta, const dAlpha, 
**               const dIGalpha, const dIGbeta)
**
**  Purpose:
**    Calculate the Inv gamma prior on S2. Do not apply the prior
**    if S2 is identical to zero.
**
*/
LnPriorGARCH(const dSEps, const dDelta, const dAlpha, 
             const dIGalpha, const dIGbeta)
{
  decl dLnPrior;

  dLnPrior= M_INF_NEG;
  if (dSEps > 0)
    dLnPrior= LnPriorS(dSEps, dIGalpha, dIGbeta);
  if ((dDelta != 0) || (dAlpha != 0))
    if ((dDelta >= 0) && (dAlpha >= 0) && 
        (dDelta + dAlpha < 1))
      dLnPrior += log(2);
    else
      dLnPrior += M_INF_NEG;

  return dLnPrior;
}

/*
**
**  LnPriorNu(const dNu)
**
**  Purpose:
**    Calculate the prior on Nu. It is a Cauchy density, truncated
**    from below at 2.
*/
LnPriorNu(const dNu)
{
  decl dLnPriorNu;

  // Cauchy == t(df=1)
  if (s_Kalm_Cauchy == 0)
    s_Kalm_Cauchy= probt(-2, 1);

  dLnPriorNu= M_INF_NEG;
  if (dNu > 2)
    dLnPriorNu= log(denst(dNu, 1)/s_Kalm_Cauchy);

  return dLnPriorNu;
}
/*
**
**  Procedure GiveResModel(vP, OutBase);
**
**  Purpose:
**    Prepare some results
**
*/
GiveResModel(const vP, const OutBase, const VarNames)
{
  // Does not do a thing.
}

/*
**
**  Procedure CalcKalmanFilter(const vP, const mY, const aPredMean, const aPredVol)
**
**  Purpose:
**    Calculate the filter, return predictions for mean and variance
**    of the observation
**
**  Remark:
**    Not implemented, as a vector vz is needed to be able to make
**    variance prediction.
**
*/
CalcKalmanFilter(const vP, const mYt, const aPredMean, const aPredVol)
{
  println("CalcKalmanFilter not implemented, \n",
          "filtering should be done during sampling");

  return 0;
}

/*
**  CalcKalmanMVS(const vP, const mYt, const nPeriods, const amPMSdS)
**
**  Purpose:
**    Calculate the filter, return predictions for mean and variance
**    of the observation and a sample, over the last nPeriods
**
**  Inputs:
**    vP          Vector of parameters
**    mYt         1 x T row with dataset; if wanted, the original dataset may be
**                shortened to include less info, as only the last
**                100 observations are effectively used...
**    nPeriods    Integer, number of periods at end of sample to use
**    nRep        Number of repetitions for sample, default= 1
**
**  Outputs:
**    amPMSdS     2+nRep x nPeriods vector, with predicted mean, predicted sdev and
**                a sample from the predicted density of the series,
**    Ret. value  Indicates if filtering succeeded
*/
CalcKalmanMVS(const vP, const mYt, const nPeriods, const amPMSdS, ...)
{
  decl dFunc, mY_Org, mPred, dRho, dSEta, dSEps, dDelta, dAlpha, ir, mStateH,
       vMu, vSd, vSamp, iT, va, nRep;

  ir= 0;
  if ((g_Kalm_UseModel == "GLL") || (g_Kalm_UseModel == "GLLGA"))
    {
      nRep= 1;
      va= va_arglist();
      if (sizeof(va) > 0)
        nRep= va[0];

      dDelta= dAlpha= 0;
      dRho= vP[0];
      dSEta= vP[1];
      dSEps= vP[2];
      if (g_Kalm_UseModel == "GLLGA")
        {
          dDelta= vP[3];
          dAlpha= vP[4];
        }
//      print (dDelta~dAlpha);

      amPMSdS[0]= M_NAN;

      ir= 0;
      if ((vP[0] <= 1) && (vP[1:2] >= 0))
        {
          ir= 1;
          mStateH= 
            SsfMomentEstPred_GarchSt(&mPred, mYt, dRho, dSEta, 
                                     dSEps, dDelta, dAlpha, 1);

          iT= columns(mYt);
          vMu= (mPred[0][iT-nPeriods+1:]~mStateH[0]);
          vSd= sqrt(mPred[3][iT-nPeriods+1:]~(mStateH[1]+dSEps*dSEps));
          vSamp= vMu + vSd .* rann(nRep, nPeriods);
          amPMSdS[0]= vMu|vSd|vSamp;
        }
    }
  else
    println ("CalcKalmanMVS only implemented for GLL/GLLGA model");

  return ir;
}

/*
**  SampleMu(const mYt, const dRho, const dSEta, const dSEps, 
**           const dDelta, const dAlpha, const vz)
**
**  Purpose:
**    Sample Mu, conditional on the data, the values of h, rho and the
**    variance of eta
**
**  Inputs:
**    mYt         1 x n row vector of data
**    dRho        scalar, value of Rho
**    dSEta       scalar, sdev of Eta
**    dSEps       scalar, unconditional sdev of observation equation
**    dDelta      scalar, Garch parameter
**    dAlpha      scalar, Garch parameter
**    vz          1 x n row vector of variance factors
**
**  Outputs:
**    vMu   Return value, row vector with sampled values of Mu
**
**  Note:
**    Kalman model is
**      y(t)= mu(t) + sigma(eps) eps(t)
**      mu(t)= rho mu(t-1) + sigma(eta) eta(t)
**      eps(t) ~ N(0, h(t) z(t)) 
**      h(t)= delta h(t-1) + 1-delta-alpha + alpha eps(t-1)^2
**    thus (sigma(eps)^2 h(t) z(t)) is the variance in the observation
**    equation
*/
SampleMu(const mYt, const dRho, const dSEta, const dSEps, const dDelta, 
         const dAlpha, const vz)
{
  decl
    iT, mKFh, mWgt, mPi, mD, mGamma, mPhi, mOmega, mSigma, vMu;

  iT= columns(mYt);
  mSigma= 1e6|0;
  if (dRho < 1)
    mSigma= dSEta*dSEta/(1-dRho*dRho)|0;
  mKFh= KalmanFil_Shell(mYt, dRho, dSEta, dSEps, 
                        dDelta, dAlpha, vz, mSigma);

  mPi= rann(1, iT);
  mGamma= diag(<1, 0>);
  mPhi= dRho|1;
  mOmega= diag(dSEta*dSEta|1);

  // NB: SimSmoWgt, SimSmoDraw and SsfRecursion do not use observation variance; 
  // varying variances are not indicated, standard routines can be used.
  mWgt= SimSmoWgt(mGamma, mKFh[:2][], mPhi, mOmega, mSigma);
  mD= SimSmoDraw(mGamma, mPi, mWgt, mKFh[:2][], mPhi, mOmega, mSigma);
  vMu= SsfRecursion(mD, mPhi, mOmega, mSigma);
  vMu= vMu[0][:iT-1];
  
  return vMu;
}

/*
**  SampleRho(const adRho, const vMu, const dSEta, const vPriorRho)
**
**  Purpose:
**    Sample Rho, conditional on the state Mu and the variance of the
**    equation
**
**  Remark:
**    The likelihood of Rho is normal, with the OLS estimates for
**    mean and variance. Prior is Beta, which limits Rho between 
**    0 and 1. Candidate is the normal density, such that 
**      alpha= (L(RhoNew) pi(RhoNew) f_c(RhoOld)) / 
**             (L(RhoOld) pi(RhoOld) f_c(RhoNew)) =
**             pi(RhoNew) / pi(RhoOld)
**    as f_c(Rho) = L(Rho)
**
**  Inputs:
**    adRho       pointer to present value of rho
**    vMu         row vector with sampled values of Mu
**    dSEta       scalar, sdev of Eta
**
**  Outputs:
**    adRho       pointer to new value of Rho
**
*/
SampleRho(const adRho, const vMu, const dSEta, const vPriorRho)
{
  decl
    iN, invXX, dS2RhoHat, dRhoHat, dRhoNew, dDensRhoNew, dDensRhoOld, 
    dAlpha, dU, dMuRho, dS2Rho;

  iN= columns(vMu);
  olsr(vMu[1:], vMu[:iN-2], &dRhoHat, &invXX);
  dS2RhoHat= dSEta*dSEta*invXX;

  if (sizeof(vPriorRho) == 2)
    {
      println("Error: Beta prior Rho not implemented");
      exit(1);
    }
  else
    {
      dMuRho= (dRhoHat*vPriorRho[1]^2 + vPriorRho[0]*dS2RhoHat)/
                (vPriorRho[1]^2+dS2RhoHat);
      dS2Rho= dS2RhoHat * vPriorRho[1]^2 / (vPriorRho[1]^2+dS2RhoHat);
      adRho[0]= rann(1, 1)*sqrt(dS2Rho) + dMuRho;
    }
}

/*
**  SampleS2(const ve, const vPriorS2AB)
**
**  Purpose:
**    Sample a variance, conditional on the errors ve,
**    applying the IG prior
**
**  Inputs:
**    ve          row vector with disturbances
**    vPriorS2AB  alpha and beta of the IG(alpha, beta) prior
**
**  Outputs:
**    dS2         Return value, scalar, variance
**
*/
SampleS2(const ve, const vPriorS2AB)
{
  decl
    iT, alpha, beta, ds, dS2;

  iT= columns(ve);
  alpha= iT/2 + vPriorS2AB[0];
  ds= sumsqrr(ve) + 2/vPriorS2AB[1];
  beta= 2/ds;
  dS2= ranigamma(1, 1, alpha, beta);

  return dS2;
}

/*
**  SampleGARCH(const adSEps, const adDelta, const adAlpha, const vz, 
**              const vv2, const vMean_Cand, const vS_Cand, const vPriorS2Eps)
**
**  Purpose:
**    Sample mGamma= (dDelta, dAlpha), conditional on the other
**    parameters
**
**  Remark:
**
**  Inputs:
**
**  Outputs:
**    adSEps, adDelta, adAlpha
**          pointers to new dSEps, dDelta, dAlpha
**    r.v.  1 if GARCH accepted, 0 if not accepted
**
*/
SampleGARCH(const adSEps, const adDelta, const adAlpha, const vz, 
            const vv2, const vMean_Cand, const vS_Cand, const vPriorS2Eps)
{
  decl vGammaOld, vGammaNew, dLLOld, dLLNew, ir, dVar, iAcc,
       dLnDensOld, dLnDensNew, dLnPrOld, dLnPrNew, dLnAlphaMH, u, ve;

  vGammaNew= SampleCandGarch(vMean_Cand, vS_Cand);

  ir= CondLik_Garch(&dLLOld, &dVar, adSEps[0], 
                    adDelta[0], adAlpha[0], vz, vv2) &&
      CondLik_Garch(&dLLNew, &dVar, vGammaNew[0], 
                     vGammaNew[1], vGammaNew[2], vz, vv2);

  if (ir)
    {
      // Candidate density is normal, centered around vMean
      ve= (vGammaNew - vMean_Cand)./vS_Cand;
      dLnDensOld= -1.5*log(M_2PI)-log(prodc(vS_Cand)) - ve've/2;
      vGammaOld= adSEps[0]|adDelta[0]|adAlpha[0];
      ve= (vGammaOld - vMean_Cand)./vS_Cand;
      dLnDensNew= -1.5*log(M_2PI)-log(prodc(vS_Cand)) - ve've/2;

      // Prior on Delta, Alpha is uniform, on S2Eps is inverted gamma,
      //   on SEps it should be multiplied by the Jacobian SEps.
      dLnPrOld= LnPriorGARCH(adSEps[0], adDelta[0], adAlpha[0], 
                             vPriorS2Eps[0], vPriorS2Eps[1]);
      dLnPrNew= LnPriorGARCH(vGammaNew[0], vGammaNew[1], vGammaNew[2], 
                             vPriorS2Eps[0], vPriorS2Eps[1]);

      dLnAlphaMH= dLLNew + dLnPrNew + dLnDensOld
        - (dLLOld + dLnPrOld + dLnDensNew);

      iAcc= 0;
      u= ranu(1, 1);
      if (log(u) > dLnAlphaMH)
        // Reject
        vGammaNew= vGammaOld;
      else
        iAcc= 1; 

      adSEps[0]= vGammaNew[0];
      adDelta[0]= vGammaNew[1];
      adAlpha[0]= vGammaNew[2];
    }
  else
    println ("Error: Problems in SampleGARCH", 
             adSEps[0]~adDelta[0]~adAlpha[0]|vMean_Cand|vS_Cand);
 
  return iAcc;
}

/*
**  SampleZ(const avz, const dSEps, const dDelta, const dAlpha, 
**          const dNu, const vv2)
**
**  Purpose:
**    Sample Z, conditional on the data, and the parameters. 
**
**  Inputs:
**    avz               pointer to 1 x n row vector of old variance increments
**    dSEps..dAlpha     Parameters in the model
**    dNu               scalar, degrees of freedom
**    vv2               1 x n row vector of (data - state)^2
**
**  Outputs:
**    avz               pointer to 1 x n row vector, new variance factors
*/
SampleZ(const avz, const dSEps, const dDelta, const dAlpha, 
        const dNu, const vv2)
{
  decl
    iT, alpha, beta, ve, vh;

  iT= columns(vv2);
  CalcVarianceGARCH(&vh, dSEps, dDelta, dAlpha, vv2);

  ve= vv2 ./ ((dSEps * dSEps) * vh);

  alpha= (dNu+1)/2;
  beta= 2.0  ./ (dNu-2 + ve);
  avz[0]= ranigamma(1, iT, alpha, 1) ./ beta;
}

/*
**  SampleNu(const adNu, const vz, const vCandNu)
**
**  Purpose:
**    Sample Nu, the degrees of freedom parameter, given z.
**
**  Inputs:
**    adNu        On input, address of old value of nu
**    vz          vector with variance
**    vCandNu     1 x 4 vector, with lower, upper bound on region for
**                nu, and parameters of drifting RW density
**
**  Outputs:
**    adNu        On output, sampled degrees of freedom
**    r.v.        1 if accepted, 0 otherwise
*/
SampleNu(const adNu, const vz, const vCandNu)
{
  decl iT, dNuCand, dLnCandNew, dLLNew, dLnCandOld, dLLOld, 
       uu, dLnAlphaMH, x, y, i, dMu, iAcc;

  iT= columns(vz);

  // Calculations for the candidate value
  dLnCandNew= RanLnCandNuRW(&dNuCand, adNu[0], vCandNu, 1);
  dLLNew= LnPdfPostNu(vz, dNuCand);

  dLnCandOld= RanLnCandNuRW(adNu, adNu[0], vCandNu, 0);
  dLLOld= LnPdfPostNu(vz, adNu[0]);

  dLnAlphaMH= dLLNew + dLnCandOld - (dLLOld + dLnCandNew);

  iAcc= 0;  
  uu= ranu(1, 1);
  if (log(uu) < dLnAlphaMH)
    // Accept
    {
      iAcc= 1;
      adNu[0]= dNuCand;
    }
  return iAcc;
}


/*
**  SampleCandGarch(const vGammaMean, const vSGamma)
**
**  Purpose:
**    Sample a candidate set of GARCH parameters Delta and Alpha, 
**    from a normal density, checking on bounds etc.
**
**  Inputs:
**    vGammaMean  3 x 1 vector of SEps, Delta, and Alpha at the last
**                iteration, mean of candidate normal density
**    vSGamma     3 x 1 vector of diagonal elements of standard deviation of
**                candidate normal density
**
**  Outputs:
**    vGamma      Return value, vector with new Delta and Alpha,
**                with finite unconditional variance
*/
SampleCandGarch(const vGammaMean, const vSGamma)
{
  decl vGamma;

  vGamma= vGammaMean + vSGamma.*rann(3, 1);
  while ((vGamma[0] <= 0) || (vGamma[1] < 0)|| (vGamma[2] < 0) || 
         (vGamma[1] + vGamma[2] >= 1))
    vGamma= vGammaMean + vSGamma.*rann(3, 1);

  return vGamma;
}

/*
**  CondLik_Garch(const adLL, const adVar, const dSEps, 
**                const dDelta, const dAlpha, const vz, const vv2)
**
**  Purpose:
**    Calculate the loglikelihood of the Student t-Garch model,
**    conditional on the state and other parameters.
**
**
**  Inputs:
**    dSEps..dAlpha     Parameters of model
**    vz                1 x n row vecotr of variance factors
**    vv2               1 x n row vector of (data - state)^2
**
**  Outputs:
**    adLL              Scalar, pointer to value of loglikelhood
**    adVar             Scalar, pointer to value of scaling factor
**    ir                Return value, 1 if parameter values accepted
*/
CondLik_Garch(const adLL, const adVar, const dSEps, 
              const dDelta, const dAlpha, const vz, const vv2)
{
  decl iT, vh, vhz, ir;

  ir= 0;
  adLL[0]= adVar[0]= M_NAN;
  if ((dSEps > 0) && (dDelta > 0) && 
      (dAlpha > 0) && (dDelta + dAlpha < 1))
    {
      ir= 1;
      iT= columns(vv2);
      CalcVarianceGARCH(&vh, dSEps, dDelta, dAlpha, vv2);

      vhz= vh .* vz;
      adVar[0]= sumr(vv2 ./ vhz) / (dSEps*dSEps);
      adLL[0]= -iT/2*log(M_2PI) - 0.5 * sumr(log(vhz)) - iT*log(dSEps) 
               - 0.5*adVar[0];

      adVar[0]= adVar[0]/iT;
    }
  return ir;
}

/*
**  CalcVarianceGARCH(const avh, const dSEps, const dDelta, const dAlpha, const vv2)
**
**  Purpose:
**    Calculate the GARCH variance h according to
**      y(t)= mu(t) + sigma(eps) eps(t)
**      eps(t)= N(0, z(t) * h(t))
**      h(t)= delta(h(t-1)) + (1-delta-alpha) + alpha eps(t-1)^2
**
**  Inputs:
**    dSEps..dAlpha     Parameters in the model
**    vv2               1 x n vector, (y(t) - mu(t))^2
**
**  Output:
**    avh   1 x n vector with GARCH variances
**
*/
CalcVarianceGARCH(const avh, const dSEps, const dDelta, const dAlpha, const vv2)
{
  decl iT, ve, vX, vPars;

  avh[0]= 1;
  if ((dDelta != 0) || (dAlpha != 0))
    {
      vPars= dSEps|dDelta|dAlpha;
      iT= columns(vv2);
      if ((!(vPars == s_Kalm_GPars)) || (iT != columns(s_Kalm_vh)))
        { // Recalculate
          s_Kalm_GPars= vPars;
 
          ve= 0~(dAlpha*vv2/(dSEps*dSEps));
          vX= constant(1-dDelta-dAlpha, iT, 1);
          vX[0]= 1;
          s_Kalm_vh= armagen(vX, ve[:iT-1]', dDelta, 1, 0)';
        }
      avh[0]= s_Kalm_vh;
    }
}

/*
**  RanLnCandNuRW(const adNu, const vCand, const bDraw)
**
**  Purpose:
**    (Sample and) calculate logdensity of candidate drawing of Nu
**
**  Inputs:
**    adNu        Address of (old?) value of Nu
**    vCand       1 x 5 vector with bounds and parameters of drifting
**                random walk. Drifting parameters are mean, drifting 
**                speed and sdev.
**    bDraw       Boolean, 1 if adNu should be sampled, 0 if only
**                calculation of density is requested
**
**  Outputs:
**    R.v.        Double, logdensity of the candidate density in adNu
*/
RanLnCandNuRW(const adNu, const dNuOld, const vCand, const bDraw)
{
  decl dMu, de;

dMu= dNuOld - vCand[3]*(dNuOld-vCand[2]);
if (bDraw)
  {
    do 
      adNu[0]= rann(1, 1)*vCand[4] + dMu;
    while ((adNu[0] <= vCand[0]) || (adNu[0] >= vCand[1]));
  }

// logarithm of Normal density, excluding constants
if ((adNu[0] > vCand[0]) && (adNu[0] < vCand[1]))
  de= (adNu[0] - dMu) ./ vCand[4];
else
  de= 1e6;

return -de'de/2;
}

/*
**  LnPdfPostNu(const vz, const dNu)
**
**  Purpose:
**    Calculate the logarithm of the conditional posterior density of
**    nu|z.
**
**  Posterior of nu|z is a proportional to the (normal) likelihood times the
**    product of N IG-densities with parameters alpha=nu/2,
**    beta= 2/nu. As nu does not appear in the likelihood, the posterior
**    is (is exactly, not only proportional) to the product of the
**    priors of z, being the products of IG densities.
**    Truncated Cauchy prior is applied.
*/
LnPdfPostNu(const vz, const dNu)
{
  decl dLnPr, dalpha, dbeta;

  dalpha= dNu/2;
  dbeta= 2/(dNu-2);
  dLnPr= sumr(lndensigamma(vz, dalpha, dbeta));

  // Adjust for Cauchy density, truncated at two from below
  dLnPr+= LnPriorNu(dNu);

  return dLnPr;
}

