/*
**  Library IncFilGS
**
**  Purpose:
**    Contain the Kalman filter with GARCH and Student t 
**    extensions in C...
**
**  Version:
**    1     Based on IncFil
**
**  Date:
**    16/11/2000
**
**  Author:
**    Charles Bos
*/

#include "oxexport.h"

/*
**
**  KalmanFil_GarchSt(const mYt, const dRho, const dSEta, const dSEps, 
**                    const dDelta, const dAlpha, const vz, const mSigma)
**
**  Purpose:
**    Run the Kalman filter, like in KalmanFil, using a loop.
**    Input and output similar. 
**
**  Inputs:
**    mYt               1 x n row vector of data
**    dRho..dAlpha      Parameters in the model
**    vz                1 x n row vector of variance factors
**    mSigma            Initial conditions
**
**  Output:
**    mKF               6 x T matrix, with v in first row, K in second,
**                      inv(F) in third row, h in fourth row, a in fifth and
**                      P in sixth row.
*/
void OXCALL FnKalmanFil_GarchSt(OxVALUE *rtn, OxVALUE *pv, int cArg)
{
  int iT, i;
  double mHH, dS2Eps, dRho2, vv, vF, vK, va, vP, vh, dRho, dSEta, 
         dSEps, dDelta, dAlpha, dOmega;
  MATRIX mvKiFh, vz;

  if (!(cArg == 8))
    OxRunErrorMessage("FnKalmanFil_GarchSt(): Incorrect number of parameters");

  OxLibCheckType(OX_MATRIX, pv, 0, 0);  /* mYt */
  OxLibCheckType(OX_DOUBLE, pv, 1, 5);  /* Rho, SEta, SEps, Delta, Alpha */
  OxLibCheckType(OX_MATRIX, pv, 6, 6);  /* vz */
  OxLibCheckType(OX_MATRIX, pv, 7, 7);  /* mSigma */

  /* Recalculate */
  iT= OxMatc(pv, 0);
  dRho= OxDbl(pv, 1);
  dSEta= OxDbl(pv, 2);
  dSEps= OxDbl(pv, 3);
  dDelta= OxDbl(pv, 4);
  dAlpha= OxDbl(pv, 5);
  vz= OxMat(pv, 6);
  mHH= dSEta * dSEta; /* SEta * SEta */
  dS2Eps= dSEps * dSEps; /* dSEps * dSEps; */
  dRho2= dRho * dRho; /* dRho * dRho; */
  dOmega= 1-dDelta-dAlpha;

  OxLibValMatMalloc(rtn, 6, iT);
  mvKiFh= OxMat( rtn, 0 );

  vh= 1.0;
  va= OxMat(pv, 7)[1][0];  /* mSigma[1]; */
  vP= OxMat(pv, 7)[0][0];  /* mSigma[0]; */

  for (i= 0; i < iT; ++i)
    {
      vv= OxMat(pv, 0)[0][i] - va;   /* mYt - va */
      vF= vP + dS2Eps * vh * vz[0][i];
      vK= dRho * vP / vF;

      mvKiFh[0][i]= vv;
      mvKiFh[1][i]= vK;
      mvKiFh[2][i]= 1.0 / vF;
      mvKiFh[3][i]= vh;
      mvKiFh[4][i]= va;
      mvKiFh[5][i]= vP;

      /* Working for t+1 */
      va= dRho * va + vK * vv;
      vP= dRho2 * vP + mHH - vK*vK*vF;
      vh= dOmega + dDelta * vh + dAlpha * vv * vv / dS2Eps;
    }
}

