/*
**
**  Library file GnuDraw.OX
**
**  Purpose:
**    Contain routines for drawing plots, with high level of
**    compatibility with Ox drawing routines
**
**  Version:
**    Specialized for JAE: Not calling GnuPlot
**
**  Date:
**    6/11/2000
**
**  Author:
**    Charles Bos
**    email:cbos@few.eur.nl
**    http://www2.tinbergen.nl/~cbos/
**
**  Based on:
**    oxdraw plotting package by Jurgen Doornik.
**
**  To do:
**    - Extent possibilities of DrawAdjust
**    - Include DrawPText etc.
*/
#ifndef GNUDRAW_INCLUDED
  #include "include/gnudraw_jae.h"
#else

/*
** JAE version: Do not call gnuplot...
#ifdef OX_Linux
  #define EXECSYS_INCLUDED
  extern "incexec_lin,FnExecSys" 
    FnExecSys(const sCommand);
#endif
#ifdef OX_Sun
  #define EXECSYS_INCLUDED
  extern "incexec_sun,FnExecSys" 
    FnExecSys(const sCommand);
#endif
#ifdef OX_Windows
  #define EXECSYS_INCLUDED
  extern "incexec,_FnExecSys"
    FnExecSys(const sCommand);
#endif
*/

#include <oxprob.h>
#include <oxfloat.h>
#include "lib/densest.ox"
#include "libkern.ox"

// Static declaration for passing values within this file. 
static decl s_XYZ_Figure= {},
            s_XYZ_LastPlot= -1,
            s_XYZ_WarnPlb= FALSE,
            s_XYZ_asKeys= "", s_XYZ_Location= "top left",
            s_XYZ_TempFile="",
            s_XYZ_Align= 0;

/* 
**  Figure contains
**    Figure[0]: Plots = array [nPlots]
**    Figure[1]: nPlots = integer
**    Figure[2]: OutType = String
**    Figure[3]: Sizes= array with {Fontsize, XSize, YSize}
**
**  Plot contains
**    Plot[0]: Lines = array [nLines]
**    Plot[1]: nLines = integer
**    Plot[2]: Set = array of strings, extra set
**                commands. See enumeration for index into array.
**    Plot[3]: Border, 1 x 4 matrix with booleans for bottom, left,
**             top, right and z border. Default is bottom left z.
**    Plot[4]: Text = array of {Text, xLoc, yLoc}
**    Plot[5]: Type= matrix of booleans [useKeys, isDMY, isXYZ, isContour], 
**                indicating if keys are used, if dates are on x-axis, 
**                if this is an XYZ plot or if this is a contour plot
**
**  Line contains
**    Line[0]: XYZ= array [2] or array [3] with mX, mY (and mZ)
**    Line[1]: Type= array of string, integer, string with linestyle,
**                linetype
**    Line[2]: Key= string
**    Line[3]: Axes= string (or empty string)
*/

#ifdef OX_Windows
  static decl s_XYZ_Terminal= "windows";
#else
  static decl s_XYZ_Terminal= "x11";
#endif

enum {F_PLOT, F_NPLOTS, F_OUTTYPE, F_SIZES, F_LAST};
enum {P_LINE, P_NLINES, P_SET, P_BORDER, P_TEXT, P_TYPE, P_LAST};
enum {L_XYZ, L_TYPE, L_KEY, L_AXES, L_LAST};
// This enumeration is meant to provide for an index into aPlot[P_SET]
// where set- and unset commands for these gnuplot elements are given.
// Elements in the first element of aPlot[P_SET][] are executed before the 
// plot, in the second element before the next plot, if needed to reset the defaults.
enum {S_TITLE, S_XLABEL, S_YLABEL, S_ZLABEL, S_LABEL, S_XRANGE, S_YRANGE,
      S_XZEROAXIS, S_YZEROAXIS, S_ZEROAXIS, S_XDATA, S_FORMAT,
      S_TIMEFMT, S_KEY, S_TICSLEVEL, S_XTICS, S_YTICS, S_DGRID3D, S_HIDDEN3D, S_VIEW, 
      S_SURFACE, S_CLABEL, S_CONTOUR, S_LAST};
// Skipping P_Labs (Plot[2]), P_Location (Plot[6]) and P_range (Plot[7])

/*
**  s_ParseFilename(sFilename, const asBasename, const asOutfile, const asOuttype)
**
**  Purpose:
**    Parse the filename, splitting it up in a base filename, an output
**    filename, and a string indicating the output type
**
**  Output:
**    asDirectory Directory for output file
**    asBasename  Filename without extension, without directory
**    asOuttype   Extension indicating output type
**
**  Return value:
**    1     if filename could be parsed, zero otherwise
**
*/
s_ParseFilename(sFilename, const asDirectory, const asBasename, const asOuttype)
{
  decl iBar, ir, il;

  if (sizeof(sFilename) <= 4)
    sFilename ~= ".plb";
  il= sizeof(sFilename);

  // Output type is defined by 3-letter extension
  asOuttype[0]= strlwr(sFilename[il-3:]);
  if (asOuttype[0]== "tex")
    asOuttype[0]= "aux";
  ir= (strfind({"plb", "plt", "aux", "eps", "epc", "scr", "gif", "png", "pnc"}, asOuttype[0]) > -1);
  if (!ir)
    {
      if (!s_XYZ_WarnPlb)
        println("Warning in GnuDraw: Extension .plb added to filename ", sFilename);
      s_XYZ_WarnPlb= TRUE;
      sFilename ~= ".plb";
      asOuttype[0]= "plb";
      il += 4;
    }

  iBar= il-4; 
  while ((iBar >= 0) && (sFilename[iBar:iBar] != "/") && (sFilename[iBar:iBar] != "\\"))
    iBar= iBar - 1;
  asBasename[0]= sFilename[iBar+1:il-5];
  asDirectory[0]= "";
  if (iBar >= 0)
    asDirectory[0]= sFilename[:iBar];

  return 1;
}

/*
**
**  s_InitOutfile(const sBasename, const sOuttype, const afh)
**
**  Purpose:
**    Initialize the file
**
**  Inputs:
**    sBasename   Base for output file, excluding extension
**    sOuttype    String indicating output type
**
**  Outputs:
**    afh   Pointer to opened file
**
**  Return value:
**    1 if succesfully opened the file, 0 otherwise
**
*/
s_InitOutfile(const sBasename, const sOuttype, const afh)
{
  decl sOutfile, ir;

  ir= 0;
  sOutfile= sBasename~".plt";
  if (sOuttype == "plb")
    sOutfile= sBasename~".plb";

  if (sBasename != s_XYZ_TempFile)
    println("Writing file ", sOutfile, "...");
  afh[0]= fopen(sOutfile, "w");
  if (!isfile(afh[0]))
    println ("Error in GnuDraw: Opening file ", sOutfile, " failed");
  else
    {
      ir= 1;
      if (sOuttype == "scr")
        fprintln(afh[0], "# Plotting file on screen");
      else
        fprintln(afh[0], "# Writing file ", sBasename~"."~sOuttype);
    }

  return ir;
}

/*
**
**  s_OpenTerminal(const fh, const sBasename, const sOuttype, const aSizes)
**
**  Purpose:
**    Open the terminal in the output file
**
**  Input:
**    fh   Opened file handle
**    sBasename   Base (without extension) of output file name
**    sOuttype    String, output type
**    aSizes      Array, with fontsize, xsize and ysize
*/
s_OpenTerminal(const fh, const sBasename, const sOuttype, const aSizes)
{
  decl sOutfile, ir;

  sOutfile= sBasename~".plt";
  if (sOuttype == "plb")
    sOutfile= sBasename~".plb";

  ir= 1;
  if (sOuttype == "eps")
    {
      fprintln(fh, "reset");
      fprintln(fh, "set term postscript eps monochrome ", aSizes[0]);
      fprintln(fh, "set output '", sBasename, ".eps'");
    }
  else if (sOuttype == "epc")
    {
      fprintln(fh, "reset");
      fprintln(fh, "set term postscript eps color ", aSizes[0]);
      fprintln(fh, "set output '", sBasename, ".eps'");
    }
  else if (sOuttype == "aux")
    {
      fprintln(fh, "reset");
      fprintln(fh, "set term latex ", aSizes[0]);
      fprintln(fh, "set output '", sBasename, ".aux'");
    }
  else if (sOuttype == "gif")
    {
      fprintln(fh, "reset");
      fprintln(fh, "set term gif ", aSizes[0], " size ", aSizes[1], ", ", aSizes[2]);
      fprintln(fh, "set output '", sBasename, ".gif'");
    }
  else if (sOuttype == "png")
    {
      fprintln(fh, "reset");
      fprintln(fh, "set term png ", aSizes[0], " monochrome");
      fprintln(fh, "set output '", sBasename, ".png'");
    }
  else if (sOuttype == "pnc")
    {
      fprintln(fh, "reset");
      fprintln(fh, "set term png ", aSizes[0], " color");
      fprintln(fh, "set output '", sBasename, ".png'");
    }
  else if (sOuttype == "scr")
    {
      fprintln(fh, "reset");
      fprintln(fh, "set term ", s_XYZ_Terminal);
      fprintln(fh, "set output");
    }
  else if (sOuttype == "plb")
    { // Nothing done
      fprintln(fh, "# Include terminal and output file here");
    }
  else
    {
      println ("Warning in GnuDraw: Output type ", sOuttype, 
               " not recognized");
      ir= 0;
    }

  return ir;
}

/*
**  s_CloseOutfile
**
**  Purpose:
**    Close the disk file, possibly close the terminal as well
*/
s_CloseOutfile(const afh, const sBasename, const sOuttype)
{
  if (isfile(afh[0]))
    {
      fprintln(afh[0], "set nomultiplot");
      if ((sOuttype == "scr")
#ifndef OX_Windows
           && (sBasename != s_XYZ_TempFile)
#endif
         )
      fprintln(afh[0], "pause -1 'File ", sBasename, "'");

      if (sOuttype != "plb")
        {
//          fprintln(afh[0], "set term ", s_XYZ_Terminal);
          fprintln(afh[0], "set term unknown");
          fprintln(afh[0], "set output");
        }

      afh[0]=fclose(afh[0]);
    }
}

/*
**  s_InitFigure()
**
**  Purpose:
**    Initialize the figure
*/
s_InitFigure()
{
  s_XYZ_Figure= new array[F_LAST];
  // Indicate that no figure has been initialized yet
  s_XYZ_Figure[F_PLOT]= {};
  s_XYZ_Figure[F_NPLOTS]= 0;
  s_XYZ_Align= 0;
  s_XYZ_LastPlot= -1;
}

/*
**  s_InitPlot()
**
**  Purpose:
**    Initialize a new plot
*/
s_InitPlot()
{
  decl nP, aPlot, i;

  if (sizeof(s_XYZ_Figure) == 0)
    s_InitFigure();

  // Add a plot
  ++s_XYZ_Figure[F_NPLOTS];

  aPlot= new array [P_LAST];
  aPlot[P_LINE]= {};
  aPlot[P_NLINES]= 0;

  aPlot[P_SET]= new array [S_LAST];
  for (i= 0; i < S_LAST; ++i)
    aPlot[P_SET][i]= {""};
  aPlot[P_SET][S_KEY]= {"set nokey"};

  // Default bottom left border, and z-border
  aPlot[P_BORDER]= <TRUE, TRUE, FALSE, FALSE, TRUE>;

  aPlot[P_TEXT]= {}; 

  // Indicate that this probably will be a normal xy plot
  aPlot[P_TYPE]= <FALSE; FALSE; FALSE; FALSE>;

  s_XYZ_Figure[F_PLOT] |= {aPlot};
  if (sizeof(s_XYZ_Figure[F_PLOT]) != s_XYZ_Figure[F_NPLOTS])
    println ("Warning in InitPlot: nFigs <> size(Plots)", 
             sizeof(s_XYZ_Figure[F_PLOT])~s_XYZ_Figure[F_NPLOTS]);
}

/*
**  s_PrepareMultPlot(const fh, const iPlot, const inPlots, 
**                    const aSizes, const sOuttype)
**
**  Purpose:
**    Prepare the file for the next element of the multiplot
*/
s_PrepareMultPlot(const fh, const iPlot, const inPlots, 
                  const aSizes, const sOuttype)
{
  decl nx, ny, ix, iy, dSx, dSy;

  if (s_XYZ_Align == 0)
    {
      ny= ceil(sqrt(inPlots));
      nx= ceil(inPlots/ny);
    }
  else if (s_XYZ_Align > 0)
    {
      nx= ceil(s_XYZ_Align);
      ny= ceil(inPlots/nx);
    }
  else if (s_XYZ_Align < 0)
    {
      nx= ceil(-s_XYZ_Align);
      ny= ceil(inPlots/nx);
    }
  dSx= aSizes[1];
  dSy= aSizes[2];

  if (iPlot == 0)
    {
      if (((dSx != 1) || (dSy != 1) || (inPlots > 1)) && 
          (sOuttype != "gif"))
        fprintln(fh, "set size ", dSx, ", ", dSy);
      if (inPlots > 1)
        {
          fprintln(fh, "set multiplot");
          fprintln(fh, "set size ", "%4.2f", dSx/nx, ",", 
                                    "%4.2f", dSy/ny);
        }
    }

  ix= imod(iPlot, nx);
  iy= idiv(iPlot, nx);
  fprintln(fh, "set origin ", "%4.2f", dSx*ix/nx, ",", "%4.2f", 
               dSy*(1-(iy+1)/ny));
}

/*
**  s_CreateLine(const mX, const mY, const mZ, const sLineStyle, 
**               const sLineType, const sKey, const bDMY, const aaLine)
**
**  Purpose:
**    Create info on a line in appropriate array
**
**  Inputs:
**    mX          n x 1 vector with x coordinates, or n x 3 matrix when
**                linetype is yerrorbars
**    mY          n x 1 vector with y coordinates, or n x 3 matrix when
**                linetype is xerrorbars
**    mZ          matrix or column of z coordinates
**    sLineStyle  string with style of line (lines, points etc)
**    sLineType   integer with type (=color) of line
**    sKey        string with key
**
**  Output:
**    aaLine      Pointer to array of line type
**
**  Return value
**    1 if succeeded
*/
s_CreateLine(const mX, const mY, const mZ, const sLineStyle, 
             const sLineType, const sKey, const aaLine)
{
  decl mDum, ir;

  if (rows(mX) == rows(mY))
    mDum= mX~mY;
  else if (rows(mY) == rows(mZ))
    mDum= mY~mZ;
  else
    println("Warning: Incorrect sizes of X, Y and Z in CreateLine(): ", 
            (rows(mX)~rows(mY)~rows(mZ))|
            columns(mX)~columns(mY)~columns(mZ));

  ir= (sizeof(deleter(mDum)) > 0);

  if (ir)
    {
      aaLine[0]= new array[L_LAST];
      aaLine[0][L_XYZ]= new array[2];
      aaLine[0][L_XYZ][0]= mX;
      aaLine[0][L_XYZ][1]= mY;
      if (sizerc(mZ) > 1)
        aaLine[0][L_XYZ]|= {mZ};
      aaLine[0][L_TYPE]= {sLineStyle, sLineType};
      aaLine[0][L_KEY]= sKey;
      aaLine[0][L_AXES]= "";
    }

  return ir;
}

/*
**  s_AppendLine(const iArea, const mX, const mY, const mZ, const sLineStyle, 
**               const sLineType, const sKey)
**
**  Purpose:
**    Create info on a line in appropriate array
**
**  Inputs:
**    mX          n x 1 vector with x coordinates, or n x 3 matrix when
**                linetype is yerrorbars
**    mY          n x 1 vector with y coordinates, or n x 3 matrix when
**                linetype is xerrorbars
**    mZ          matrix or column of z coordinates
**    sLineStyle  string with style of line (lines, points etc)
**    sLineType   integer with type (=color) of line
**    sKey        string with key
**
**  Output:
**    aaLine      Pointer to array of line type
**
**  Return value
**    1 if succeeded
*/
s_AppendLine(const iArea, const mX, const mY, const mZ, const sLineStyle, 
             const sLineType, const sKey)
{
  decl ir, aLine, mDum; 

  while ((sizeof(s_XYZ_Figure) < 4) || (s_XYZ_Figure[F_NPLOTS] <= iArea))
    s_InitPlot();

  ir= s_CreateLine(mX, mY, mZ, sLineStyle, sLineType, sKey, &aLine);
  if (ir)
    {
      s_XYZ_Figure[F_PLOT][iArea][P_LINE] |= {aLine};
      ++s_XYZ_Figure[F_PLOT][iArea][P_NLINES];

      if (sizeof(sKey) > 0)
        {
          s_XYZ_Figure[F_PLOT][iArea][P_TYPE][0]= TRUE;
          if (s_XYZ_Figure[F_PLOT][iArea][P_SET][S_KEY][0] == "set nokey")
            s_XYZ_Figure[F_PLOT][iArea][P_SET][S_KEY]= 
              {"set key top left", "set nokey"};
        }
      if (s_XYZ_Figure[F_PLOT][iArea][P_NLINES] != sizeof(s_XYZ_Figure[F_PLOT][iArea][P_LINE]))
        {
          println("Warning in s_AppendLine: Size of plot ", iArea, 
                  " not equal to number of lines.",
                  s_XYZ_Figure[F_PLOT][iArea][P_NLINES]~sizeof(s_XYZ_Figure[F_PLOT][iArea][P_LINE]));
          ir= 0;
        }
    }
  
  return ir;
}

/*
**  s_SetSize(const vArgs, const sOuttype, const aaSizes)
**
**  Purpose:
**    Set the fontsize and the output size of the graph
**
**  Input:
**    vArgs       Array with arguments
**    sOuttype    String with output type
**
**  Output:
**    amSizes     Pointer to array with fontsize, xsize and ysize
**
**  Return value:
**    Always 1
*/
s_SetSize(const vArgs, const sOuttype, const aaSizes)
{
  decl ia;

  ia= sizeof(vArgs);
  aaSizes[0] = {14, 1, 1};
  if (sOuttype == "eps")
    {
      if ((ia > 0) && (vArgs[0] > 0))
        aaSizes[0][0]= vArgs[0];
    }
  if (sOuttype == "aux")
    {
      aaSizes[0][0]= 6;
      aaSizes[0][1]= 1;
      aaSizes[0][2]= 1;
      if ((ia > 0) && (vArgs[0] > 0))
        aaSizes[0][0]= vArgs[0];
    }
  if ((sOuttype == "gif") || (sOuttype == "png") || (sOuttype == "pnc"))
    {
      aaSizes[0][0]= "small";
      aaSizes[0][1]= 1;
      aaSizes[0][2]= 1;
      if ((ia > 0) && (vArgs[0] > 14))
        aaSizes[0][0]= "large";
    }
  if (sOuttype == "scr")
    aaSizes[0][0]= 10;

  if ((ia > 2) && (vArgs[1] > 0))
    aaSizes[0][1]= vArgs[1];
  if ((ia > 2) && (vArgs[2] > 0))
    aaSizes[0][2]= vArgs[2];

  if (sOuttype == "gif")
    {
	  if ((aaSizes[0][1] > 1) && (aaSizes[0][1] < 10))
	    aaSizes[0][1] *= 640;
	  if ((aaSizes[0][2] > 1) && (aaSizes[0][2] < 10))
	    aaSizes[0][2] *= 480;
	}

  return 1;
}

/*
**  s_WriteFileUni(const fh)
**
**  Purpose:
**    Actually write the data part of the output file, for a univariate
**    graph
**
**  Inputs:
**    fh          file handle of opened output file
**    aPlot       Array with information on a set of lines
*/
s_WriteFileUni(const fh, const aPlot)
{
  decl i, j, pl, mX, mY, asPrFmt, sRange;

  pl = "plot ";
  for (i= 0; i < aPlot[P_NLINES]; ++i)
    {
      mX= aPlot[P_LINE][i][L_XYZ][0];
      // If coordinate X is replaced by a string, we're plotting a 3-dim file
      if (isstring(mX))
        pl= sprint(pl, "'", mX, "'");
      else
        {
          pl= sprint(pl, "'-'");

          // if using dates, first columns contain dates, the first column
          //   after mX contains the y component
          if (columns(mX) > 1)
            pl= sprint(pl, " using 1:", columns(mX)+1);
        }

      // add the axes
      if (sizeof(aPlot[P_LINE][i][L_AXES]) > 0)
        pl= sprint(pl, " axes ", aPlot[P_LINE][i][L_AXES]);
      // add the key
      if (aPlot[P_SET][S_KEY][0][:8] != "set nokey")
        pl= sprint(pl, " title '", aPlot[P_LINE][i][L_KEY], "'");
      // add the linestyle (lines, points etc)
      if (aPlot[P_LINE][i][L_TYPE][0] != "")
        pl= sprint(pl, " with ", aPlot[P_LINE][i][L_TYPE][0]);
      // add the linetype (color: 1 less then Ox-linetype)
      if (aPlot[P_LINE][i][L_TYPE][1] > -1)
        if (aPlot[P_LINE][i][L_TYPE][0] != "points")
          pl= sprint(pl, " lt ", aPlot[P_LINE][i][L_TYPE][1]-1);
        else
          pl= sprint(pl, " pt ", aPlot[P_LINE][i][L_TYPE][1]-1);
      if (i < aPlot[P_NLINES]-1)
        fprintln(fh, pl, ", \\");
      else
        fprint(fh, pl);
      pl= "     ";
    }

  for (i = 0; i < aPlot[P_NLINES]; ++i)
    {
      mX= aPlot[P_LINE][i][L_XYZ][0];
      if (!isstring(mX))
        { 
          // Set up the format. Short format for the dates, long format 
          // for normal data.
          asPrFmt= {"%10.6g ", "%10.6g"};
          if (columns(mX) == 2)      // If MY date format
            asPrFmt= {"%1.0f ", "%1.0f ", "%10.6g"};
          else if (columns(mX) > 2)  // If DMY date format
            asPrFmt= {"%1.0f ", "%1.0f ", "%1.0f ", "%10.6g"};

          // Plot elements without missing values
          mY= aPlot[P_LINE][i][L_XYZ][1];
          fprint(fh, "%cf", asPrFmt, deleter(mX~mY));
          fprint(fh, "e");
        }
    }
  fprintln(fh, "");
}

/*
**  s_WriteFileBiv(const fh, const aPlot)
**
**  Purpose:
**    Actually write the data part of the output file, for a bivariate
**    graph
**
**  Inputs:
**    fh    file handle of opened output file
**    aPlot       Array with information on a set of lines
*/
s_WriteFileBiv(const fh, const aPlot)
{
  decl i, j, pl, mX, mY, mZ, mXYZ, sRange;

  pl = "splot '-'";
  for (i= 0; i < aPlot[P_NLINES]; ++i)
    {
      // use key
      if (aPlot[P_SET][S_KEY][0][:8] != "set nokey")
        pl= sprint(pl, " title '", aPlot[P_LINE][i][L_KEY], "'");
      // add the linestyle
      if (aPlot[P_LINE][i][L_TYPE][0] != "")
        pl= sprint(pl, " with ", aPlot[P_LINE][i][L_TYPE][0]);
      if (aPlot[P_LINE][i][L_TYPE][1] > -1)
        pl= sprint(pl, " lt ", aPlot[P_LINE][i][L_TYPE][1]-1);
      if (i < aPlot[P_NLINES]-1)
        fprintln(fh, pl, ", \\");
      else
        fprint(fh, pl);
      pl= "     '-'";
    }

  for (i = 0; i < aPlot[P_NLINES]; ++i)
    { 
      if (sizeof(aPlot[P_LINE][i][L_XYZ]) == 3)
        [mX, mY, mZ]= aPlot[P_LINE][i][L_XYZ];
      else
        { // Mixing univariate and bivariate graphs?
          [mX, mY]= aPlot[P_LINE][i][L_XYZ];
          mZ= zeros(rows(mY), 1);
        }

      if (columns(mZ) == 1)
        {
          mXYZ= deleter(mX~mY~mZ);
          if (sizerc(mXYZ) > 0)
            fprint(fh, "%10.4g ", mXYZ);
          else   // Only missings, write a placeholder
            fprintln(fh, "0 0 0");
        }
      else
        for (j= 0; j < rows(mX); ++j)
          {
            mXYZ= deleter((mX[j][0]*ones(mY))~mY~mZ[][j]);
            if (sizerc(mXYZ) > 0)
              fprintln(fh, "%10.4g ", mXYZ);
            else   // Only missings, write a placeholder
              fprintln(fh, "0 0 0");
          }
      fprint(fh, "e");
    }
  fprintln(fh, "");
}

/*
**  s_InitContour(const fh)
**
**  Purpose:
**    Initialize a file for writing a contour plot
**
**  Inputs:
**    fh    File handle
**
*/
s_InitContour(const fh, const sBasename)
{
  fprintln(fh, "# Plotting contour ", sBasename);

  fprintln(fh, "set terminal table");
  fprintln(fh, "set output '", sBasename, ".log'");
}  

/*
**  s_WriteFileContour(const fh, const sBasename, const aPlot)
**
**  Purpose:
**    Write final lines to a file for a contour plot
**
**  Inputs:
**    fh    File handle
**
*/
s_WriteFileContour(const fh, const sBasename, const aPlot)
{
  decl i, pl;

  pl= sprint("plot '", sBasename, ".log' ");
  if (aPlot[P_SET][S_KEY][0][:8] != "set nokey")
    pl= sprint(pl, " title '", aPlot[P_LINE][0][L_KEY], "'");
  fprintln(fh, pl, " with lines");
  
  fprintln(fh, "");
}  

/*
**  s_CheckDimUni(const mX, const mY, const mErr, 
**                const amlX, const amlY, const sFilename)
**
**  Purpose:
**    Check dimensions for a univariate plot
**
**    Returns two matrices mlX and mlY of the same size
*/
s_CheckDimUni(const mX, const mY, const mErr, 
              const amlX, const amlY, const sFilename)
{
  decl ErrStr, rx, ry, re, cx, cy, ce, ir, mlX, mlY;

  ir= 1;

  ErrStr= "";
  mlX= mX;
  if ((mlX == 0) && (rows(mlX) == 0))
    mlX= range(1, rows(mY))';
  mlY= mY;
  if ((mlY == 0) && (rows(mlY) == 0))
    mlY= range(1, rows(mX))';

  rx= rows(mlX);
  ry= rows(mlY);
  re= rows(mErr);
  cx= columns(mlX);
  cy= columns(mlY);
  ce= columns(mErr);
  
  if (rx != ry)
    ErrStr |= "Rows x should equal rows y\n";
  if (cx != cy)
    if (min(cx, cy) > 1)
      ErrStr |= "x or y should have one column, or the number of columns should be equal\n";
    else if (cx == 1)
      mlX= mlX .* ones(1, cy);
    else if (cy == 1)
      mlY= mlY .* ones(1, cx);

  if (!(mErr == 0))
    {
      if (cy > 1)
        ErrStr |= "With error bars, only one column y allowed\n";
      if (rx != re)
        ErrStr |= "Rows x not equal to number of error bars\n";
      if (ce > 2)
        ErrStr |= "Maximum of two columns allowed for error bars\n";
    }

  if (ErrStr != "")
    {
      println ("Error: Measures of matrices incorrect while drawing ", 
               sFilename);
      println (ErrStr);
      println ("x: ", "%5.0f", rows(mlX), "%5.0f", columns(mlX));
      println ("y: ", "%5.0f", rows(mlY), "%5.0f", columns(mlY));
      ir= 0;
    }
  if (isarray(amlX))
    amlX[0]= mlX;
  if (isarray(amlY))
    amlY[0]= mlY;

  return ir;
}

/*
**  s_CheckDimBiv(const vX, const vY, const mZ, 
**                const avlX, const avlY, const sFilename)
**
**  Purpose:
**    Check dimensions for a bivariate plot
**
**  Inputs (if correct):
**    vX    k x 1 vector
**    vY    n x 1 vector
**    mZ    n x k matrix
**  or
**    vX    n x 1 vector
**    vY    n x 1 vector
**    mZ    n x 1 vector
**
**
*/
s_CheckDimBiv(const vX, const vY, const mZ, 
              const avlX, const avlY, const sFilename)
{
  decl ErrStr, rx, ry, rz, cx, cy, cz, ir;

  ir= 1;
  ErrStr= "";
  avlX[0]= vX;
  if ((avlX[0] == 0) && (rows(avlX[0]) == 0))
    avlX[0]= range(1, columns(mZ))';
  avlY[0]= vY;
  if ((avlY[0] == 0) && (rows(avlY[0]) == 0))
    avlY[0]= range(1, rows(mZ))';

  rx= rows(avlX[0]);
  ry= rows(avlY[0]);
  rz= rows(mZ);
  cx= columns(avlX[0]);
  cy= columns(avlY[0]);
  cz= columns(mZ);
  
  if (max(cx, cy) > 1)
    ErrStr |= "Both x and y should have one column\n";
  if (cz == 1)
    {
      if (min(rx, ry, rz) != max(rx, ry, rz))
        ErrStr |= "X, y and z should the same number of rows for non-grid data\n";
    }
  else 
    {
      if (rx != cz)
        ErrStr |= "Rows x not equal to columns z\n";
      if (ry != rz)
        ErrStr |= "Rows y not equal to rows z\n";
    }

  if (ErrStr != "")
    {
      println ("Error: Measures of matrices incorrect while drawing ", 
               sFilename);
      println (ErrStr);
      ir= 0;
    }

  return ir;
}

/*
**  s_FileInitPlot(const fh, const aSetOld, const aSetNew, 
**                 const vBorder)
**
**  Purpose:
**    Print the unset-commands for the old plot, the set-commands
**    for the new plot, and the border information.
**
**  Input:
**    fh        File handle to opened file
**    aSetOld   Array aPlot[P_SET] with set-commands (disregarded)
**              and unset-commands, which applied as needed.
**    aSetNew   Array aPlot[P_SET] with set-commands 
**              for the new plot. If an option is not set explicitly
**              the unset command (if available) is applied.
**    vBorder   Vector with booleans indicating the borders to
**              apply in this graph.
*/
s_FileInitPlot(const fh, const aSetOld, const aSetNew, 
               const vBorder)
{
  decl i, iBorder;
  
  iBorder= 0;
  for (i= 0; i < 5; ++i)
    if (vBorder[i] == TRUE)
      iBorder += 2^i;
  fprintln(fh, sprint("set border ", double(iBorder)));
  
  if ((vBorder[0]==TRUE) && (vBorder[2]==TRUE))
    fprintln(fh, "set xtics nomirror\nset x2tics");
  else if ((vBorder[0]==TRUE) && (vBorder[2]==FALSE))
    fprintln(fh, "set xtics nomirror\nset nox2tics");
  else if ((vBorder[0]==FALSE) && (vBorder[2]==TRUE))
    fprintln(fh, "set noxtics\nset x2tics");
  else if ((vBorder[0]==FALSE) && (vBorder[2]==FALSE))
    fprintln(fh, "set noxtics\nset nox2tics");

  if ((vBorder[1]==TRUE) && (vBorder[3]==TRUE))
    fprintln(fh, "set ytics nomirror\nset y2tics");
  else if ((vBorder[1]==TRUE) && (vBorder[3]==FALSE))
    fprintln(fh, "set ytics nomirror\nset noy2tics");
  else if ((vBorder[1]==FALSE) && (vBorder[3]==TRUE))
    fprintln(fh, "set noytics\nset y2tics");
  else if ((vBorder[1]==FALSE) && (vBorder[3]==FALSE))
    fprintln(fh, "set noytics\nset noy2tics");


  for (i= 0; i < S_LAST; ++i)
    if (aSetNew[i][0] != "")
      fprintln(fh, aSetNew[i][0]);
    else if ((sizeof(aSetOld) > i) && 
             (sizeof(aSetOld[i]) == 2) && 
             (aSetOld[i][1] != ""))
      fprintln(fh, aSetOld[i][1]);
      
}

/*
**  SaveDrawWindow(const sFilename, ...)
**
**  sFilename 
**      in: valid file name 
**  iFontSize
**      in: integer, size of font
**  dXSize
**      in: double, indicating x-size
**  dYSize
**      in: double, indicating y-size
**
**  No return value. 
**  
**  Saves the current graph to a file with the same basename as sFilename,
**  and extension .plt (containing information on final output type) or .plb
**  (no information on output type included). The ASCII file is 
**  prepared to be translated by GnuPlot to a file with the specified
**  extension in the corresponding format. The following formats are supported: 
**      .eps : Encapsulated PostScript 
**      .gif : gif format, for inclusion on Web pages
**      .scr : No final output, except screen output
**      .aux : LaTeX output
**      .plb : General, output type can be specified later using a
**             script
*/
SaveDrawWindow(const sFilename, ...)
{
  decl ir, vArgs, fh, sDirectory, sBasename, sOuttype, aSizes, i,
       aSetOld, aPlot, sCommand;

  aSizes= s_XYZ_Figure[F_SIZES];
  vArgs= va_arglist();
  ir= s_ParseFilename(sFilename, &sDirectory, &sBasename, &sOuttype) &&
      s_SetSize(vArgs, sOuttype, &aSizes) && 
      s_InitOutfile(sDirectory~sBasename, sOuttype, &fh);

  if (ir)
    {
      s_XYZ_Figure[F_OUTTYPE]= sOuttype;
      s_XYZ_Figure[F_SIZES]= aSizes;

      // Check for contours, and prepare
      aSetOld= {};
      for (i= 0; i < s_XYZ_Figure[F_NPLOTS]; ++i)
        {
          aPlot= s_XYZ_Figure[F_PLOT][i];
          if ((aPlot[P_SET][S_CONTOUR][0]=="set contour")
              && (aPlot[P_SET][S_SURFACE][0]=="set nosurface"))
            {
              s_InitContour(fh, sprint(sBasename, i));
              s_FileInitPlot(fh, aSetOld, aPlot[P_SET], aPlot[P_BORDER]);
              s_WriteFileBiv(fh, aPlot);
              
              // Save resets
              aSetOld= aPlot[P_SET];
            }
        }
    }

  ir= ir && s_OpenTerminal(fh, sBasename, sOuttype, s_XYZ_Figure[F_SIZES]);

  if (ir)
    {
      for (i= 0; i < s_XYZ_Figure[F_NPLOTS]; ++i)
        {
          aPlot= s_XYZ_Figure[F_PLOT][i];
          s_PrepareMultPlot(fh, i, s_XYZ_Figure[F_NPLOTS],
                            s_XYZ_Figure[F_SIZES], sOuttype);

          s_FileInitPlot(fh, aSetOld, aPlot[P_SET], aPlot[P_BORDER]);

          if (aPlot[P_NLINES] > 0)
            {
              // Is this an XYZ graph?
              if (sizeof(aPlot[P_LINE][0][L_XYZ]) == 2)
                s_WriteFileUni(fh, aPlot);
              else
                {
                  if (aPlot[P_SET][S_SURFACE][0]=="set surface")
                    s_WriteFileBiv(fh, aPlot);
                  else
                    s_WriteFileContour(fh, sprint(sBasename, i), aPlot);
                }
            }
          // Save resets
          aSetOld= aPlot[P_SET];
        }

      s_CloseOutfile(&fh, sBasename, sOuttype);

    #ifdef EXECSYS_INCLUDED
      if ((sOuttype != "plb") && (sOuttype != "scr"))
        {
          // Run gnuplot without removing the file
          if (sDirectory == "")
            sDirectory= ".";
          sCommand= sprint("execgnu ", sDirectory, " ", sBasename);

      #ifdef OX_Windows  
          // Change slashes in backslashes
          for (i= 0; i < sizeof(sDirectory); ++i)
            if (sDirectory[i:i]=="/")
              sDirectory[i:i]= "\\";
          // On windows, spawn a new process 
          sCommand= sprint("start command /c ", sCommand);
      #endif
          FnExecSys(sCommand);
        }
    #endif
    }
}

/*
**  CloseDrawWindow()
**
**  Purpose:
**    Clean up settings on lines, keys and others
*/
CloseDrawWindow()
{
  s_InitFigure();
}

/*
**  #include <gnudraw.h>
**  Draw(const iArea, const mYt);
**  Draw(const iArea, const mYt, const dXfirst, const dXstep);
**  
**      iArea 
**          in: int, area index 
**      mYt 
**          in: m x T matrix with m rows of data 
**      dXfirst 
**          in: (optional) double, X-value of first observation, x, default is 1 
**      dXstep 
**          in: (optional) double, gap between X-values, dx, default is 1 
**  
**  No return value. 
**  
**  This function draws m variables against an X variable, where the X variable consists of
**  evenly spaced observations x, x+dx, x+2dx, x+3dx, .... Each variable is drawn by linking up
**  the points. The first line index is 2. 
*/
Draw(const iArea, const mYt, ...)
{
  decl vArgs, mX, i, mlX, mlY, ir;

  vArgs= va_arglist();
  mX= range(1, columns(mYt))';
  if (sizeof(vArgs) == 2)
    mX= (mX-1)*vArgs[1] + vArgs[0];

  ir= s_CheckDimUni(mX, mYt', 0, &mlX, &mlY, "Draw");
  if (ir)
    for (i= 0; i < columns(mlY); ++i)
      // Keys might have been set by setKeys
      s_AppendLine(iArea, mlX[][i], mlY[][i], 0, "lines", 2+i, "");

  s_XYZ_LastPlot= iArea;
}

/*
**  #include <gnudraw.h>
**  DrawAdjust(const iType, ...);
**  
**      iType 
**          in: int, type of adjustment 
**      d1 ... d4 
**          in: optional arguments 
**  
**  No return value. 
**  
**  The expected number of arguments depends on the type of adjustment 
*/
DrawAdjust(const iType, ...)
{
  decl vArgs, dMinY, dMaxY, i, sExtra, nLines;
  
  sExtra= "";
  nLines= s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_NLINES];
  if ((s_XYZ_LastPlot >= 0) && 
      (s_XYZ_LastPlot < s_XYZ_Figure[F_NPLOTS]))
    {
      vArgs= va_arglist();
      if ((iType== ADJ_AREA_X) && (sizeof(vArgs) >= 2))
        {
          if (isnan(vArgs[0]))
            sExtra= sprint("set xrange [*:");
          else
            sExtra= sprint("set xrange [", "%g", vArgs[0], ":");
          if (isnan(vArgs[1]))
            sExtra= sprint(sExtra, "*]");
          else
            sExtra= sprint(sExtra, "%g", vArgs[1], "]");
          s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_XRANGE]=
            {sExtra, "set xrange [*:*]"};
        }
      else if ((iType== ADJ_AREA_Y) && (sizeof(vArgs) >= 2))
        {
          if (isnan(vArgs[0]))
            sExtra= sprint("set yrange [*:");
          else
            sExtra= sprint("set yrange [", "%g", vArgs[0], ":");
          if (isnan(vArgs[1]))
            sExtra= sprint(sExtra, "*]");
          else
            sExtra= sprint(sExtra, "%g", vArgs[1], "]");
          s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_YRANGE]=
            {sExtra, "set yrange [*:*]"};
        }
      else if ((iType== ADJ_AXISHIDE) && (sizeof(vArgs) >= 1))
        {
          if (vArgs[0])
            s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_ZEROAXIS]=
              {"set zeroaxis"};
          else
            s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_ZEROAXIS]=
              {"set nozeroaxis", "set zeroaxis"};
        }
      else if ((iType== ADJ_MINMAX) && (sizeof(vArgs) == 2))
        { 
          dMinY= vArgs[0]; 
          dMaxY= vArgs[1];
          for (i= 0; i < s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][1]; ++i)
            {
              dMinY= min(dMinY,          
                         s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_LINE][i][L_XYZ][1]);
              dMaxY= max(dMaxY, 
                         s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_LINE][i][L_XYZ][1]);
            }
          sExtra= sprint("set yrange [", "%g", dMinY, ":", "%g", dMaxY, "]");
          s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_YRANGE]=
            {sExtra, "set yrange [*:*]"};
        }
      else if (iType== ADJ_LABEL)
        {
          if ((sizeof(vArgs) > 0) && isstring(vArgs[0]))
            {
              s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_XLABEL]= 
                {sprint("set xlabel '", vArgs[0], "'"), 
                 "set xlabel ''"};
            }
          if ((sizeof(vArgs) > 1) && isstring(vArgs[1]))
            {
              s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_YLABEL]= 
                {sprint("set ylabel '", vArgs[1], "'"),
                 "set ylabel ''"};
            }
          if ((sizeof(vArgs) > 2) && isstring(vArgs[2]))
            {
              s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_ZLABEL]= 
                {sprint("set zlabel '", vArgs[2], "'"),
                 "set zlabel ''"};
            }
        }
      else if ((iType== ADJ_COLOR) && (sizeof(vArgs) == 2))
        {
          if (vArgs[0] > -1)
            s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_LINE][nLines-1][L_TYPE][1]=
              vArgs[0];
          if (vArgs[1] > -1)
            if (vArgs[1] == TP_SOLID)
              s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_LINE][nLines-1][L_TYPE][0]=
                "lines";
            else
              s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_LINE][nLines-1][L_TYPE][0]=
                "dots";
        }
      else if ((iType == ADJ_ALIGN) && (sizeof(vArgs) > 0))
        s_XYZ_Align= round(vArgs[0]);
      else if ((iType == ADJ_AXES2ND) && (sizeof(vArgs) == 2))
        {
          sExtra= sprint("x", (vArgs[0]==1)+1, "y", (vArgs[1]==1)+1);
          s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_LINE][nLines-1][L_AXES]=
            sExtra;

          // Turn the correct borders on
          if (vArgs[0])
            s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_BORDER][2]=TRUE;
          else
            s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_BORDER][0]=TRUE;
          if (vArgs[1])
            s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_BORDER][3]=TRUE;
          else
            s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_BORDER][1]=TRUE;
        }      
      else
        println("DrawAdjust: This option is not implemented yet...");
    }
}

/*
**  DrawAxis, DrawAxisAuto 
**  
**  #include <gnudraw.h>
**  DrawAxis(const iArea, const fIsXaxis, const dAnchor,
**      const dAxmin, const dAxmax, const dFirstLarge,
**      const dLargeStep, const dSmallStep, const iFreq);
**  DrawAxisAuto(const iArea, const fIsXaxis, ...);
**  DrawAxisAuto(const iArea, const fIsXaxis, bool fShow,
**      int iAnchor, double dAnchor);
**  
**      iArea 
**          in: int, area index 
**      fIsXaxis 
**          in: int, TRUE: is X axis, else Y axis 
**      dAnchor 
**          in: int, anchor of the axis (e.g. Y location of X axis) 
**      dAxmin 
**          in: axis minimum 
**      dAxmax 
**          in: axis maximum 
**      dFirstLarge 
**          in: location of first large tick 
**      dLargeStep 
**          in: step size between large ticks 
**      dSmallStep 
**          in: step size between small ticks 
**      iFreq 
**          in: int, frequency (for time series axis) 
**      fShow 
**          in: int, TRUE: show the axis 
**      iAnchor 
**          in: int, axis anchor location, ANCHOR_MIN: at minimum, ANCHOR_MAX: at
**          maximum, ANCHOR_USER: at dAnchor 
**  
**  No return value. 
*/
DrawAxis(const iArea, const fIsXaxis, const dAnchor,
         const dAxmin, const dAxmax, const dFirstLarge,
         const dLargeStep, const dSmallStep, const iFreq)
{
  println("DrawAxis: Not implemented yet...");
}
DrawAxisAuto(const iArea, const fIsXaxis, ...)
{
  decl va, fShow, iAnchor, iBorder, dAnchor;
  
  va= va_arglist();
  fShow= TRUE;
  if (sizeof(va) > 0)
    fShow= va[0];
  iAnchor= ANCHOR_MIN;
  if (sizeof(va) > 1)
    iAnchor= va[1];
  dAnchor= 0;
  if (sizeof(va) > 2)
    dAnchor= va[2];
  if (iAnchor == ANCHOR_USER)
    {
      if (dAnchor != 0)
        println("Warning: DrawAxisAuto at ANCHOR_USER only implemented with dAnchor= 0");
      dAnchor= 0;
    }

  if (fIsXaxis)
    {
      if (fShow)
        {
          s_XYZ_Figure[F_PLOT][iArea][P_BORDER][0]= (iAnchor == ANCHOR_MIN);
          s_XYZ_Figure[F_PLOT][iArea][P_BORDER][2]= (iAnchor == ANCHOR_MAX);
          if (iAnchor == ANCHOR_USER)
            {
              s_XYZ_Figure[F_PLOT][iArea][P_SET][S_XZEROAXIS]= 
                {"set xzeroaxis lt -1", "set noxzeroaxis"};
              s_XYZ_Figure[F_PLOT][iArea][P_SET][S_XTICS]= 
                {"set xtics axis", "set xtics border"};
            }
        }
      else
        {
          s_XYZ_Figure[F_PLOT][iArea][P_BORDER][0]= FALSE;
          s_XYZ_Figure[F_PLOT][iArea][P_BORDER][2]= FALSE;
        }       
    }
  else
    {
      if (fShow)
        {
          s_XYZ_Figure[F_PLOT][iArea][P_BORDER][1]= (iAnchor == ANCHOR_MIN);
          s_XYZ_Figure[F_PLOT][iArea][P_BORDER][3]= (iAnchor == ANCHOR_MAX);
          if (iAnchor == ANCHOR_USER)
            {
              s_XYZ_Figure[F_PLOT][iArea][P_SET][S_YZEROAXIS]= 
                {"set yzeroaxis lt -1", "set noyzeroaxis"};
              s_XYZ_Figure[F_PLOT][iArea][P_SET][S_YTICS]= 
                {"set ytics axis", "set ytics border"};
            }
        }
      else
        {
          s_XYZ_Figure[F_PLOT][iArea][P_BORDER][1]= FALSE;
          s_XYZ_Figure[F_PLOT][iArea][P_BORDER][3]= FALSE;
        }       
    }
}

/*  
**  #include <gnudraw.h>
**  DrawBoxPlot(const iArea, const vY, const sY);
**  
**      iArea 
**          in: int, area index 
**      mY 
**          in: k x T matrix, each row is a new plot 
**      asY 
**          in: string, variable name, or array of strings (k > 1) 
**  
**  No return value. 
*/
DrawBoxPlot(const iArea, const mY, const asY)
{
  decl ir, sKey, vQ, i, vY, nY, dIQR, vBX, vBY, vSY;

  // These lines may be removed eventually
  while ((sizeof(s_XYZ_Figure) < 4) || (s_XYZ_Figure[F_NPLOTS] <= iArea))
    s_InitPlot();

  ir= s_CheckDimUni(0, mY, 0, &vBX, &vBY, "DrawBoxPlot");
  sKey= "";
  if (isstring(asY))
    sKey= asY;
  for (i= 0; i < rows(mY); ++i)
    {
      vY= deletec(mY[i][]);
      nY= columns(vY);
      if (isarray(asY) && (i < sizeof(asY)))
        sKey= asY[i];
      if (nY > 0)
        {
          vQ= quantiler(vY, <.25, .5, .75>);
          dIQR= 1.5*(vQ[2]-vQ[0]);

          vBX= <-1, -1, 1, 1, -1>*dIQR;
          vBY= vQ[<0, 2, 2, 0, 0>];
          DrawXMatrix(iArea+i, vBY, sKey, vBX, "");

          vBX= <-1, 1>*dIQR;
          vBY= vQ[<1, 1>];
          DrawXMatrix(iArea+i, vBY, "", vBX, "");
      
          vBY= (vQ[0]-dIQR)~vQ[0]~vQ[2]~(vQ[2]+dIQR);
          vBX= <0, 0>;
          DrawXMatrix(iArea+i, vBY[:1], "", vBX, "");
          DrawXMatrix(iArea+i, vBY[2:], "", vBX, "");

          // Plot points outside Q1-IQR, plus two points inside   
          vSY= sortr(vY);
          vBX= limits(vSY' .> vQ[0]-dIQR)[3][];
          vBX= min(vBX+1, nY-1);
          vBY= vSY[:vBX];
          DrawX(iArea+i, vBY, zeros(vBY));
          DrawAdjust(ADJ_COLOR, 2, -1);

          // Plot points outside Q3+IQR, plus two points inside   
          vBX= limits(vSY' .> vQ[2]+dIQR)[3][];
          vBX= max(vBX-2, 0);
          vBY= vSY[vBX:];
          DrawX(iArea+i, vBY, zeros(vBY));
          DrawAdjust(ADJ_COLOR, 2, -1);

          // Get the graph to be wide enough          
          DrawAdjust(ADJ_AREA_X, -1.3*dIQR, 1.3*dIQR);

          s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_SET][S_XZEROAXIS]=
            {"set xzeroaxis -1", "set noxzeroaxis"};
          s_XYZ_Figure[F_PLOT][s_XYZ_LastPlot][P_BORDER]=
            <FALSE, TRUE, FALSE, FALSE, FALSE>;
        }
    }
}

/*  
**  #include <gnudraw.h>
**  DrawCorrelogram(const iArea, const vY, const asY,
**                  const cLag);
**  
**      iArea 
**          in: int, area index 
**      vY 
**          in: k x T matrix, each row is a new plot 
**      sY 
**          in: string, variable name, or array of strings (k > 1) 
**      cLag 
**          in: int, highest lag to be used in correlogram 
**  
**  No return value. 
**  
**  Draws a correlogram (the ACF is computed using acf). 
*/
DrawCorrelogram(const iArea, const vY, const asY, const cLag)
{
  decl i, sKey, mACF, ir, mlX, mlY;

  mACF= acf(vY', cLag);
  ir= s_CheckDimUni(0, mACF[1:][], 0, &mlX, &mlY, "DrawCorrelogram");

  sKey= "";
  if (isstring(asY))
    sKey= asY;
  for (i= 0; i < columns(mlY); ++i)
    {
      if (isarray(asY) && (i < sizeof(asY)))
        sKey= asY[i];
      ir= ir &&
          s_AppendLine(iArea, mlX[][i], mlY[][i], 0, "impulses", -1, sKey);
      sKey= "";
    }
  s_XYZ_Figure[F_PLOT][iArea][P_SET][S_XZEROAXIS]=
    {"set xzeroaxis lt -1", "set noxzeroaxis"};
  s_XYZ_LastPlot= iArea;

  DrawAdjust(ADJ_AREA_Y, -1, 1);
}
  
/*  
**  DrawDensity
**  
**  #include <gnudraw.h>
**  DrawDensity(const iArea, const vY, const asY, const fDens,
**              const fHist, const fNormal, ...)
**  
**      iArea 
**          in: int, area index 
**      vY 
**          in: k x T matrix, each row is a new plot 
**      asY 
**          in: string, variable name, or array of strings (k > 1) 
**      fDens 
**          in: int, TRUE: draw estimated density 
**      fHist 
**          in: int, TRUE: draw histogram 
**      fNormal 
**          in: int, TRUE: add normal density for reference 
**      iK (optional)
**          in: int, number of cells for histogram or support points for
**          approximating normal density
**  
**  No return value. 
**  
**  Draws the histogram and/or density of the data in the specified area. When fNormal is
**  TRUE, a normal density with the same mean and variance as the data will be drawn. 
*/
DrawDensity(const iArea, const vY, const asY, const fDens,
            const fHist, const fNormal, ...)
{
  decl ir, iK, i, vX, vlY, vYK, sKey, dM, dS, mRet, 
       dMinX, dMaxX, dStep, ilArea;

  ir= 1;
  ilArea= iArea;
  sKey= "";
  if (isstring(asY))
    sKey= asY;

  for (i= 0; i < rows(vY); ++i)
    {
      if (isarray(asY) && (i < sizeof(asY)))
        sKey= asY[i];

      vlY= deletec(vY[i][]);
      iK= min(max(5, columns(vlY)/20), 20);
      if (sizeof(va_arglist()) > 0)
        iK= max(3, va_arglist()[0]);

      dMinX= min(vlY);
      dMaxX= max(vlY);
      dStep= (dMaxX - dMinX)/(iK-1);
      vX= range(0, iK-1)*dStep + dMinX;

      if (ir && (sizeof(vlY) > 0))
        {
          if (fDens)
            {
              mRet= DensEst(vlY', 0, 0, 0, 128);
              ir= s_CheckDimUni(mRet[0][]', mRet[2][]', 0, 0, 0, "DrawDensity-Dens") &&
                  s_AppendLine(ilArea, mRet[0][]', mRet[2][]', 0, "lines", -1, sKey);
              sKey= "";
            }

          if (fHist)
            {
              vYK = discretize(vlY, dMinX, dMaxX, iK, 0);
              vYK /= (columns(vlY) * dStep);
              ir= s_CheckDimUni(vX', vYK', 0, 0, 0, "DrawDensity-Hist") &&
                  s_AppendLine(ilArea, vX', vYK', 0, "boxes", -1, sKey);
              sKey= "";
            }

          if (fNormal)
            {
              dM= meanr(vlY);
              dS= sqrt(varr(vlY));
              vYK= densn((vX - dM)/dS)/dS;
              if (sKey=="")
                sKey= sprint("N(s=", "%.3g", double(dS), ")");
              ir= s_CheckDimUni(vX', vYK', 0, 0, 0, "DrawDensity-Norm") &&
                  s_AppendLine(ilArea, vX', vYK', 0, "lines", -1, sKey);
              sKey= "";
            }
        }
      ++ilArea;
    }  
  s_XYZ_LastPlot= ilArea-1;
}

/*  
**  DrawBivDensity
**  
**  #include <gnudraw.h>
**  DrawBivDensity(const iArea, const mY, const asY, const fDens,
**                 const fHist, const fNormal);
**  DrawBivDensity(const iArea, const mY, const asY, const fDens,
**                 const fHist, const fNormal, const iMode);
**
**      iArea 
**          in: int, area index 
**      vY 
**          in: k x T matrix, each row is a new plot 
**      asY 
**          in: string, variable name, or array of strings (k > 1) 
**      fDens 
**          in: int, TRUE: draw estimated density 
**      fHist 
**          in: int, TRUE: draw histogram 
**      fNormal 
**          in: int, TRUE: add normal density for reference 
**      iMode 
**          in: int, type of plot: 
**          0 - surface plot only
**          1 - surface plot with contours on ground level
**          2 - contour plot
**  
**  No return value. 
**  
**  Draws the bivariate histogram and/or density of the data in the 
**  specified area. When fNormal is TRUE, a normal density with the 
**  same mean and variance as the data will be drawn. 
*/
DrawBivDensity(const iArea, const mY, const asY, const fDens,
               const fHist, const fNormal, ...)
{
  decl ir, iK, nDim, i, j, ilArea, mX, mXX, mlY, vlX, vlY, sKeyX, sKeyY, 
       mRet, vMMX, vStep, iMode, vArgs, vM, mS, ddetS, miS, dLogDet, dSignDet;

  vArgs= va_arglist();
  iMode= 0;
  if (sizeof(vArgs) > 0)
    iMode= vArgs[0];

  ir= 1;
  ilArea= iArea;
  nDim= rows(mY);
  sKeyX= sKeyY= "";

  for (i= 0; i < nDim-1; ++i)
    {
      if (isarray(asY) && (sizeof(asY) > i))
        sKeyX= asY[i];
      for (j= i+1; j < nDim; ++j)
        {
          if (isarray(asY) && (sizeof(asY) > j))
            sKeyY= asY[j];

          // Only bivariate plots catered for
          mlY= deletec(mY[i|j][]);
          iK= min(max(5, columns(mlY)/20), 20);

          vMMX= limits(mlY')[:1][]';
          vStep= (vMMX[][1] - vMMX[][0])/(iK-1);
          mX= range(0, iK-1) .* vStep + vMMX[][0];
          mXX= (mX[0][] ** ones(1, iK)) |
               (ones(1, iK) ** mX[1][]);

          if (ir && (sizeof(mlY) > 0))
            {
              if (fDens)
                {
                  ir= s_KernelMult(mXX, mlY, &mRet);
                  mRet= shape(mRet, iK, iK);
      
                  ir= s_CheckDimBiv(mX[0][]', mX[1][]', mRet, &vlX, &vlY, "DrawDensity-Dens") &&
                      s_AppendLine(ilArea, vlX, vlY, mRet, "lines", -1, "");
                }
            if (fHist)
              {
                println("DrawBivDensity: Histogram plotting not implemented");
    /*
                mYK = discretize(mlY, dMinX, dMaxX, iK, 0);
                mYK /= (columns(mlY) * dStep);
                ir= s_CheckDimUni(mX', mYK', 0, 0, 0, "DrawDensity-Hist") &&
                    s_AppendLine(iArea, mX', mYK', 0, "boxes", -1, "");
    */
              }
              if (fNormal)
                {
                  vM= meanr(mlY);
                  mS= choleski(variance(mlY'));
                  miS=invert(mS, &dLogDet, &dSignDet);
                  ddetS=exp(dLogDet)*dSignDet;
                  mRet= prodc(densn(miS*(mXX - vM)))/ddetS;
                  mRet= shape(mRet, iK, iK);

                  ir= s_CheckDimBiv(mX[0][]', mX[1][]', mRet, &vlX, &vlY, "DrawDensity-Norm") &&
                      s_AppendLine(ilArea, vlX, vlY, mRet, "lines", -1, "");
                }

              if (iMode == 0)
                {
                  // Plot xyz-plot, no contour
                  s_XYZ_Figure[F_PLOT][ilArea][P_TYPE][2]= TRUE;
                  s_XYZ_Figure[F_PLOT][ilArea][P_TYPE][3]= FALSE;

                  // Set and reset (if necessary) 3-d settings
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_TICSLEVEL]= 
                    {"set ticslevel 0"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_HIDDEN3D]= 
                    {"set hidden3d"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_VIEW]= 
                    {"set view 60, 120"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_SURFACE]= 
                    {"set surface"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_CLABEL]= 
                    {"set noclabel", "set clabel"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_CONTOUR]= 
                    {"set nocontour"};
                }
              else if (iMode == 1)
                {
                  // Plot xyz-plot and contour
                  s_XYZ_Figure[F_PLOT][ilArea][P_TYPE][2]= TRUE;
                  s_XYZ_Figure[F_PLOT][ilArea][P_TYPE][3]= TRUE;

                  // Set and reset (if necessary) 3-d settings
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_TICSLEVEL]= 
                    {"set ticslevel 0"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_HIDDEN3D]= 
                    {"set hidden3d"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_VIEW]= 
                    {"set view 60, 120"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_SURFACE]= 
                    {"set surface"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_CLABEL]= 
                    {"set noclabel", "set clabel"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_CONTOUR]= 
                    {"set contour"};
                }
              else if (iMode == 2)
                {
                  // Plot no xyz-plot, only contour
                  s_XYZ_Figure[F_PLOT][ilArea][P_TYPE][2]= FALSE;
                  s_XYZ_Figure[F_PLOT][ilArea][P_TYPE][3]= TRUE;

                  // Set and reset (if necessary) 3-d settings
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_SURFACE]= 
                    {"set nosurface"};
                  s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_CONTOUR]= 
                    {"set contour"};
                }

              // Set the labels
              s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_XLABEL]= 
                {sprint("set xlabel '", sKeyX, "'"), 
                 "set xlabel ''"};
              s_XYZ_Figure[F_PLOT][ilArea][P_SET][S_YLABEL]= 
                {sprint("set ylabel '", sKeyY, "'"), 
                 "set ylabel ''"};
            }

          ++ilArea;
        }
    }
  s_XYZ_LastPlot= ilArea-1;
}

/*  
**  DrawLegend
**  
**  #include <gnudraw.h>
**  DrawLegend(const iArea, const iOffsX, const iOffsY,
**             const fHidden);
**  
**      iArea 
**          in: int, area index 
**      iOffsX 
**          in: int, X pixel offset from top left 
**      iOffsY 
**          in: int, Y pixel offset from top left 
**      fHidden 
**          in: int, TRUE: hide the legend 
**  
**  No return value. 
*/
DrawLegend(const iArea, const iOffsX, const iOffsY,
           const fHidden)
{
  while ((sizeof(s_XYZ_Figure) < 4) || (s_XYZ_Figure[F_NPLOTS] <= iArea))
    s_InitPlot();

  if (fHidden)  // Note extra spaces: Will not get overwritten
    s_XYZ_Figure[F_PLOT][iArea][P_SET][S_KEY]= {"set nokey  "};
  else 
    {
      if (max(iOffsX, iOffsY) > 1)
        s_XYZ_Figure[F_PLOT][iArea][P_SET][S_KEY]= 
          {sprint("set key screen ", "%4.2f,", double(iOffsX)/1000,
           "%4.2f", 1-double(iOffsY)/1000),
           "set nokey"};
      else
        s_XYZ_Figure[F_PLOT][iArea][P_SET][S_KEY]= 
          {sprint("set key screen ", "%4.2f,", iOffsX, "%4.2f",
           1-iOffsY), "set nokey"};
    }
}

/*
**  #include <gnudraw.h>
**  DrawLine(const iArea, const dX1, const dY1, const dX2,
**           const dY2, ...);
**  
**      iArea 
**          in: int, area index 
**      dX1, dY1 
**          in: int, real-world coordinates of starting point 
**      dX2, dY2 
**          in: int, Y real-world coordinates of end point 
**      iIndex 
**          in: int, line index for first row, see default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index. 
**  
**  No return value. 
*/
DrawLine(const iArea, const dX1, const dY1, const dX2,
         const dY2, ...)
{
  decl mlX, mlY, ir, va, iIndex;

  va= va_arglist();
  iIndex= 2;
  if (sizeof(va) == 1)
    iIndex= va[0];

  ir= s_CheckDimUni(dX1|dX2, dY1|dY2, 0, &mlX, &mlY, "DrawLine") &&
      s_AppendLine(iArea, mlX, mlY, 0, "lines", iIndex, "");

  s_XYZ_LastPlot= iArea;
}

/*  
**  #include <gnudraw.h>
**  DrawMatrix(const iArea, const mYt, const asY,
**      const dXfirst, const dXstep, ...);
**  DrawMatrix(const iArea, const mYt, const asY,
**      const dXfirst, const dXstep, iSymbol, iIndex);
**  
**      iArea 
**          in: int, area index 
**      mYt 
**          in: m x T matrix with m rows of data 
**      asY 
**          in: array of strings (holds variable names), or 0 (no names) 
**      dXfirst 
**          in: double, X-value of first observation, x 
**      dXstep 
**          in: double, gap between X-values, dx 
**      iSymbol 
**          in: int, 0: draw line, 1: draw symbols, 2: draw both (optional argument, default
**          is 0). 
**      iIndex 
**          in: int, line index for first row, see default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index. 
**  
**  No return value. 
**  
**  This is a more flexible version of the Draw() function. DrawMatrix draws the m
**  variables in the rows of mYt. The X variable consists of evenly spaced observations x,
**  x+dx, x+2dx, x+3dx, .... The following table gives the default settings for each line index.
**  Note that index 0 is the background colour, and 1 the foreground colour. 
*/
DrawMatrix(const iArea, const mYt, const asY,
           const dXfirst, const dXstep, ...)
{
  decl vArgs, mX, i, mlX, mlY, ir, iSymbol, iIndex, sStyle, sKey;

  vArgs= va_arglist();
  iSymbol= 0;
  if (sizeof(vArgs) >= 1)
    iSymbol= vArgs[0];
  iIndex= 2;
  if (sizeof(vArgs) >= 2)
    iIndex= vArgs[1];
  mX= range(0, columns(mYt)-1)'*dXstep + dXfirst;

  sStyle= "lines";
  if (iSymbol == 1)
    sStyle= "points";
  else if (iSymbol == 2)
    sStyle= "linespoints";

  sKey= "";
  if (isstring(asY))
    sKey= asY;

  ir= s_CheckDimUni(mX, mYt', 0, &mlX, &mlY, "DrawMatrix");
  for (i= 0; i < columns(mlY); ++i)
    {
      if (isarray(asY) && (sizeof(asY) > i))
        sKey= asY[i];
      ir= ir && s_AppendLine(iArea, mlX[][i], mlY[][i], 0, 
                             sStyle, iIndex + i, sKey);
      sKey= "";
    }

  s_XYZ_LastPlot= iArea;
}

/*  
**  DrawPLine, DrawPSymbol, DrawPText 
**  
**  #include <gnudraw.h>
**  DrawPLine(const iArea, const iX1, const iY1, const iX2,
**      const iY2, const iIndex);
**  DrawPSymbol(const iArea, const iX1, const iY1, const iX2,
**      const iY2, const iSymType, const iIndex);
**  DrawPText(const iArea, const sText, const iPx1,
**      const iPy1, ...);
**  DrawPText(const iArea, const sText,const iPx1,const iPy1,
**      const iFontNo, const iFontSize, const fTitle);
**  
**  No return value. 
**  Pixel coordinate versions of DrawLine, DrawSymbol DrawText. 
*/
DrawPLine(const iArea, const iX1, const iY1, const iX2,
          const iY2, const iIndex)
{
  println("DrawPLine: Not implemented yet");
}

DrawPSymbol(const iArea, const iX1, const iY1, const iX2,
            const iY2, const iSymType, const iIndex)
{
  println("DrawPSymbol: Not implemented yet");
}

DrawPText(const iArea, const sText, const iPx1,
          const iPy1, ...)
{
  println("DrawPText: Not implemented yet");
}

/*  
**  #include <gnudraw.h>
**  DrawQQ(const iArea, const vY, const asY, const iDens,
**         const df1, const df2);
**  
**      iArea 
**          in: int, area index 
**      vY 
**          in: k x T matrix, each row is a new plot 
**      asY 
**          in: string, variable name, or array of strings (k > 1) 
**      iDens 
**          in: int, one of: QQ_CHI, QQ_F, QQ_N, QQ_T, QQ_U 
**      df1 
**          in: int, first parameter for distribution 
**      df2 
**          in: int, second parameter for distribution 
**  
**  No return value. 
**  
**  Draws a QQ plot. The following distributions are supported: 
**  
**      QQ_CHI : chi^2(df1), 
**      QQ_F : F(df1, df2), 
**      QQ_N : N(0, 1), 
**      QQ_T : t(df1), 
**      QQ_U : Uniform(0,1), resulting in a Quantile plot. 
*/
DrawQQ(const iArea, const vY, const asY, const iDens, const df1, const df2)
{
  decl i, vqY, vqF, vX, v45, sKey, ir, sKey2;

  vqF= range(.01, .99, .01);
  if (iDens == QQ_CHI)
    {
      vX= quanchi(vqF, df1);
      sKey2= sprint(" x Chi^2(", df1, ")");
    }
  else if (iDens == QQ_F)
    {
      vX= quanf(vqF, df1, df2);
      sKey2= sprint(" x F(", df1, ",", df2, ")");
    }
  else if (iDens == QQ_N)
    {
      vX= quann(vqF);
      sKey2= " x normal";
    }
  else if (iDens == QQ_T)
    {
      vX= quant(vqF, df1);
      sKey2= sprint(" x t(", df1, ")");
    }
  else if (iDens == QQ_U)
    {
      vX= vqF;
      sKey2= " x uniform";
    }
  vqY= quantiler(vY, vqF);
  sKey= "";
  if (isstring(asY))
    sKey= asY;

  v45= min(vX)|max(vX);
  ir= s_AppendLine(iArea, v45, v45, 0, "lines", 0, "");

  for (i= 0; i < rows(vqY); ++i)
    {
      if (isarray(asY) && (sizeof(asY) > i))
        sKey= asY[i];
      if (sKey != "")
        sKey= sprint(sKey, sKey2);
      ir= ir && s_CheckDimUni(vX', vqY[i][]', 0, 0, 0, "DrawQQ") &&
          s_AppendLine(iArea, vX', vqY[i][]', 0, "lines", -1, sKey);
      sKey= "";
    }
  s_XYZ_Figure[F_PLOT][iArea][P_SET][S_ZEROAXIS]= 
    {"set nozeroaxis"};
    
  s_XYZ_LastPlot= iArea;
}

/*  
**  #include <gnudraw.h>
**  DrawSpectrum(const iArea, const vY, const asY,
**      const iOrder);
**  
**      iArea 
**          in: int, area index 
**      vY 
**          in: k x T matrix, each row is a new plot 
**      asY 
**          in: string, variable name, or array of strings (k > 1) 
**      iOrder 
**          in: int, lag truncation parameter m 
**  
**  No return value. 
**  
**  Draws the estimated spectral density. 
*/
DrawSpectrum(const iArea, const vY, const asY, const iOrder)
{
  decl ir, vX, mY, sKey, i, cp= 128;

  vX= range(0, 1, 1/(cp-1))';
  mY= periodogram(vY', iOrder, cp, 2);
  ir= s_CheckDimUni(vX, mY, 0, 0, 0, "DrawSpectrum");

  sKey= "";
  if (isstring(asY))
    sKey= asY;

  for (i= 0; i < rows(mY); ++i)
    {
      if (isarray(asY) && (i < sizeof(asY)))
        sKey= asY[i];
      ir= ir && 
          s_AppendLine(iArea, vX, mY[][i], 0, "lines", -1, sKey);
    }
  s_XYZ_LastPlot= iArea;
}

/*  
**  #include <gnudraw.h>
**  DrawSymbol(const iArea, const dX1, const dY1, const dX2,
**             const dY2, const iSymType, const iIndex);
**  
**      iArea 
**          in: int, area index 
**      dX1, dY1 
**          in: int, real-world coordinates, lower-left corner of bounding box 
**      dX2, dY2 
**          in: int, real-world coordinates, upper-right corner of bounding box 
**      iSymType 
**          in: int, symbol type 
**      iIndex 
**          in: int, line index for first row, see default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index. 
**  
**  No return value. 
*/  
DrawSymbol(const iArea, const dX1, const dY1, const dX2,
           const dY2, const iSymType, const iIndex)
{
  println("DrawSymbol: Not implemented yet");
}

/*  
**  DrawT
**  
**  #include <gnudraw.h>
**  DrawT(const iArea, const mYt, const mnYear,
**        const mnPeriod, const iFreq);
**  DrawT(const iArea, const mYt, const mMY, const dDum, const dDum);
**  
**      iArea 
**          in: int, area index 
**      mYt 
**          in: m x T matrix with m y variables 
**      mnYear 
**          in: int, year of first observation 
**      mnPeriod 
**          in: int, period of first observation 
**      iFreq 
**          in: int, frequency of observations 
**      mDMY
**          in: 2 x T matrix with month and year or
**              3 x T matrix with day, month and year
**      dDum
**          in: not used, placeholder
**  
**  No return value. 
**  
**  Draws m variables in the specified area against time. Each variable is drawn by linking up the
**  points. The first line index is 2. Time can be specified as a
**  combination of the year and period of the first observation together
**  with the frequency, as a 2 x T matrix with months and years or as a 3 x
**  T matrix with day, month and years
*/
DrawT(const iArea, const mYt, const mnYear,
      const mnPeriod, const iFreq)
{
  DrawTMatrix(iArea, mYt, "", mnYear, mnPeriod, iFreq);
}

/*  
**  DrawText, DrawTitle 
**  
**  #include <gnudraw.h>
**  DrawText(const iArea, const sText, const dX1,
**      const dY1, ...);
**  DrawText(const iArea, const sText, const dX1, const dY1,
**      const iFontNo, const iFontSize, const fTitle);
**  DrawTitle(const iArea, const sText);
**  
**      iArea 
**          in: int, area index 
**      dX1, dY1 
**          in: int, real-world coordinates, anchor of text 
**      dX2 
**          in: int, real-world coordinates, upper-right corner of bounding box 
**      iFontNo (not implemented)
**          in: int, font number (0 for first font; use -1 for the default font) 
**      iFontSize (not implemented)
**          in: int, font size (e.g. 300; use -1 for the default size) 
**      fTitle 
**          in: int, TRUE: is graph title (coordinates are ignored) 
**  
**  No return value. 
*/
DrawText(const iArea, const sText, const dX1,
         const dY1, ...)
{
  decl va, dZ1;

  while ((sizeof(s_XYZ_Figure) < 4) || (s_XYZ_Figure[F_NPLOTS] <= iArea))
    s_InitPlot();

  va= va_arglist;
  dZ1= 0;
  if ((sizeof(va) > 0) && (sizeof(s_XYZ_Figure[F_PLOT][iArea][P_LINE][0][L_XYZ]) == 3))
    {
      dZ1= va[0];
      s_XYZ_Figure[F_PLOT][iArea][P_SET][S_LABEL][0]~= 
        sprint("set label '", sText, "' at ", dX1, ",", dY1, ",", dZ1, "\n");
    }
  else
    s_XYZ_Figure[F_PLOT][iArea][P_SET][S_LABEL][0]~= 
      sprint("set label '", sText, "' at ", dX1, ",", dY1, "\n");

  s_XYZ_Figure[F_PLOT][iArea][P_SET][S_LABEL]= 
    {s_XYZ_Figure[F_PLOT][iArea][P_SET][S_LABEL][0], 
     "set nolabel"};
}

DrawTitle(const iArea, const sText)
{
  while ((sizeof(s_XYZ_Figure) < 4) || (s_XYZ_Figure[F_NPLOTS] <= iArea))
    s_InitPlot();

  s_XYZ_Figure[F_PLOT][iArea][P_SET][S_TITLE]= 
    {sprint("set title '", sText, "'"), "set title ''"};
}

/*  
**  #include <gnudraw.h>
**  DrawTMatrix(const iArea, const mYt, const asY,
**      const mnYear, const mnPeriod, const iFreq, ...);
**  DrawTMatrix(const iArea, const mYt, const asY,
**      const mMY, const dDum, const dDum, ...);
**  DrawTMatrix(const iArea, const mYt, const asY,
**      const mnYear, const mnPeriod, const iFreq, iSymbol,
**      iIndex);
**  DrawTMatrix(const iArea, const mYt, const asY,
**      const mMY, const dDum, const dDum, iSymbol, iIndex);
**  
**      iArea 
**          in: int, area index 
**      mYt 
**          in: m x T matrix with m y variables 
**      asY 
**          in: array of strings (holds variable names), or 0 (no names) 
**      mnYear 
**          in: int, year of first observation 
**      mnPeriod 
**          in: int, period of first observation 
**      iFreq 
**          in: int, frequency of observations 
**      mMY
**          in: 2 x T matrix with month and year or
**              3 x T matrix with day, month and year
**      dDum
**          in: not used, only placeholder
**      iSymbol 
**          in: int, 0: draw line, 1: draw symbols, 2: draw both (optional argument, default
**          is 0). 
**      iIndex 
**          in: int, line index for first row, see Default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index. 
**  
**  No return value. 
**  
**  This is a more flexible version of the DrawT() function. Draws m variables in the specified
**  area against time. See under DrawMatrix for the default settings for each line index. 
*/
DrawTMatrix(const iArea, const mYt, const asY, const mnYear,
      const mnPeriod, const iFreq, ...)
{
  decl mMY, iT, ir, mlY, vm, i, vArgs, iSymbol, 
       iIndex, sStyle, siStyle, sKey;

  vArgs= va_arglist();
  if (sizerc(mnYear) > 1)
    mMY= mnYear;
  else
    {
      iT= columns(mYt);
      vm= range(0, iT-1)*12/iFreq + mnPeriod - 1;
      mMY= (vm-12*trunc(vm/12)+1)|(trunc(vm/12)+mnYear);
    }
  if (mMY[rows(mMY)-1][]< 1000)
    mMY[rows(mMY)-1][] += 1900;

  iSymbol= 0;
  if (sizeof(vArgs) >= 1)
    iSymbol= vArgs[0];
  iIndex= 2;
  if (sizeof(vArgs) >= 2)
    iIndex= vArgs[1];

  sStyle= "lines";
  if (iSymbol == 1)
    sStyle= "points";
  else if (iSymbol == 2)
    sStyle= "linespoints";

  sKey= "";
  if (isstring(asY))
    sKey= asY;

  ir= s_CheckDimUni(mMY[0][]', mYt', 0, 0, 0, "DrawT");

  if (ir)
    {
      for (i= 0; i < rows(mYt); ++i)
        {
          if (isarray(asY) && (sizeof(asY) > i))
            sKey= asY[i];
          ir= s_AppendLine(iArea, mMY', mYt[i][]', 0, sStyle, iIndex + i, sKey);
          sKey= "";
        }

      s_XYZ_Figure[F_PLOT][iArea][P_SET][S_XDATA]= 
        {"set xdata time", "set xdata"};
      if ((rows(mMY) == 2) || (columns(mYt) > 20))
        s_XYZ_Figure[F_PLOT][iArea][P_SET][S_FORMAT]= 
           {"set format x '%m/%y'", "set format '%g'"};
      else
        s_XYZ_Figure[F_PLOT][iArea][P_SET][S_FORMAT]= 
           {"set format x '%d/%m/%y'", "set format '%g'"};

      if (rows(mMY) == 2)
        s_XYZ_Figure[F_PLOT][iArea][P_SET][S_TIMEFMT]= 
          {"set timefmt '%m %Y'"};
      else
        s_XYZ_Figure[F_PLOT][iArea][P_SET][S_TIMEFMT]= 
          {"set timefmt '%d %m %Y'"};

      // Indicate that date format is used
      s_XYZ_Figure[F_PLOT][iArea][P_TYPE][1]= TRUE;
    }

  s_XYZ_LastPlot= iArea;
}

/*  
**  #include <gnudraw.h>
**  DrawX(const iArea, const mYt, const vX);
**  
**      iArea 
**          in: int, area index 
**      mYt 
**          in: m x T matrix with m y variables 
**      vX 
**          in: 1 x T matrix with x variable 
**  
**  No return value. 
**  
**  Draws m y variables in the specified area against an x variable. Each point is marked, but the
**  points are not linked, resulting in a cross plot. The first line index is 2. 
*/
DrawX(const iArea, const mYt, const vX)
{
  decl mlX, mlY, ir, i;

  ir= s_CheckDimUni(vX', mYt', 0, &mlX, &mlY, "DrawX");
  for (i= 0; i < columns(mlY); ++i)
    ir= ir &&
        s_AppendLine(iArea, mlX[][i], mlY[][i], 0, "points", -1, "");
}

/*  
**  #include <gnudraw.h>
**  DrawXMatrix(const iArea, const mYt, const asY, const vX,
**              const sX, ...);
**  DrawXMatrix(const iArea, const mYt, const asY, const vX,
**              const sX, iSymbol, iIndex);
**  
**      iArea 
**          in: int, area index 
**      vX 
**          in: 1 x T matrix with x variable 
**      iSymbol 
**          in: int, 0: draw line, 1: draw symbols, 2: draw both (optional argument, default
**          is 0). 
**      iIndex 
**          in: int, line index for first row, see Default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index. 
**  
**  No return value. 
**  
**  This is a more flexible version of the DrawX() function. Draws m variables in the specified
**  area against an x variable See under DrawMatrix for the default settings for each line
**  index. 
*/
DrawXMatrix(const iArea, const mYt, const asY, const vX,
            const sX, ...)
{
  decl mlX, mlY, ir, i, vArgs, iIndex, sStyle, sKey;

  vArgs= va_arglist();
  
  iIndex= 2;
  if (sizeof(vArgs) >= 2)
    iIndex= vArgs[1];

  sStyle= "lines";
  if (sizeof(vArgs) >= 1)
    if (vArgs[0] == 1)
      sStyle= "points";
    else if (vArgs[0] == 2)
      sStyle= "linespoints";

  sKey= "";
  if (isstring(asY))
    sKey= asY;

  ir= s_CheckDimUni(vX', mYt', 0, &mlX, &mlY, "DrawX");
  for (i= 0; i < columns(mlY); ++i)
    {
      if (isarray(asY) && (sizeof(asY) > i))
        sKey= asY[i];
      ir= ir &&
          s_AppendLine(iArea, mlX[][i], mlY[][i], 0, sStyle, iIndex + i,
                       sKey);
      sKey= "";
    }

  s_XYZ_Figure[F_PLOT][iArea][P_SET][S_XLABEL]= 
    {sprint("set xlabel '", sX, "'"), "set xlabel ''"};

  s_XYZ_LastPlot= iArea;
}

/*
**  #include <gnudraw.h>
**  DrawZ(const vZ, ...);
**  DrawZ(const vZ, const sZ, const iMode, const dFac,
**      const iIndex);
**  
**      vZ 
**          in: 1 x T matrix with Z variable 
**      sZ 
**          in: string, 1 name of Z variable 
**      iMode 
**          in: int, type of Z variable (optional argument): 
**          ZMODE_SYMBOL - use values as symbol size, 
**          ZMODE_VALUE - draw value as text, 
**          ZMODE_BAR - draw error bars (the default), 
**          ZMODE_BAND - draw error bands. 
**      dFac 
**          in: double, bar/band factor (optional argument, 2.0 is default) 
**      iIndex 
**          in: int, line index for first row, see Default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index. 
**  
**  No return value. 
**  
**  DrawZ adds a Z component to the most recent graphics object. DrawZ should be a
**  should be used immediately after a call to one of the draw functions Draw,
**  DrawMatrix, DrawX, etc.). 
*/
DrawZ(const vZ, ...)
{
  decl va, sZ, iMode, dFac, iIndex, iArea, iLine, avXY, 
       ir, mlX, mlY;

  va= va_arglist();
  sZ= "";
  iMode= ZMODE_BAR;
  dFac= 2;
  iIndex= 2;
  if (sizeof(va) > 0)
    sZ= va[0];
  if (isarray(sZ))
    sZ= sZ[0];
  if (sizeof(va) > 1)
    {
      iMode= va[1];
      println ("DrawZ does not implement iMode yet");
    }
  if (sizeof(va) > 2)
    dFac= va[2];
  if (sizeof(va) > 3)
    iIndex= va[0];
  iArea= s_XYZ_LastPlot;
  iLine= s_XYZ_Figure[F_PLOT][iArea][P_NLINES]-1;
  avXY= s_XYZ_Figure[F_PLOT][iArea][P_LINE][iLine][L_XYZ][0:1];
  ir= s_CheckDimUni(avXY[0], avXY[1], vZ', &mlX, &mlY, "DrawZ") &&
      s_AppendLine(0, avXY[0], avXY[1]~(dFac*vZ'), 0, "yerrorbars", iIndex, sZ);
}

/*
**  #include <gnudraw.h>
**  DrawXYZ(const iArea, const vX, const vY, const mZ, ...);
**  
**      iArea 
**          in: int, area index 
**      vX 
**          in: 1 x k matrix with X variable 
**      vY 
**          in: 1 x n matrix with Y variable 
**      mZ 
**          in: n x k matrix with Z variable, heights above XY plane
**                or
**          in: 1 x n = 1 x k matrix with Z coordinates for points 
**              (X, Y, Z), rough approximation of surface is constructed
**      sX 
**          in: string, 1 name of X variable (optional argument)
**      sY 
**          in: string, 1 name of Y variable (optional argument)
**      iMode 
**          in: int, type of plot: 
**          0 - surface plot only
**          1 - surface plot with contours on ground level
**          2 - contour plot
**  
**  No return value. 
*/
DrawXYZ(const iArea, const vX, const vY, const mZ, ...)
{
  decl vlX, vlY, vArgs, sKeyX, sKeyY, nSize, iMode, ir, mlZ;

  vArgs= va_arglist();
  sKeyX= "";
  if (sizeof(vArgs) > 0)
    sKeyX= vArgs[0];
  sKeyY= "";
  if (sizeof(vArgs) > 1)
    sKeyY= vArgs[1];
  iMode= 0;
  if (sizeof(vArgs) > 2)
    iMode= vArgs[2];
  mlZ= mZ;
  if (rows(mlZ) == 1)
    mlZ= mZ';

  ir= s_CheckDimBiv(vX', vY', mlZ, &vlX, &vlY, "DrawXYZ");
  if (ir)
    {
      s_AppendLine(iArea, vlX, vlY, mlZ, "lines", -1, "");

      if (iMode == 0)
        {
          // Plot xyz-plot, no contour
          s_XYZ_Figure[F_PLOT][iArea][P_TYPE][2]= TRUE;
          s_XYZ_Figure[F_PLOT][iArea][P_TYPE][3]= FALSE;

          // Set and reset (if necessary) 3-d settings
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_TICSLEVEL]= 
            {"set ticslevel 0"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_HIDDEN3D]= 
            {"set hidden3d"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_VIEW]= 
            {"set view 60, 120"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_SURFACE]= 
            {"set surface"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_CLABEL]= 
            {"set noclabel", "set clabel"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_CONTOUR]= 
            {"set nocontour"};
        }
      else if (iMode == 1)
        {
          // Plot xyz-plot and contour
          s_XYZ_Figure[F_PLOT][iArea][P_TYPE][2]= TRUE;
          s_XYZ_Figure[F_PLOT][iArea][P_TYPE][3]= TRUE;

          // Set and reset (if necessary) 3-d settings
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_TICSLEVEL]= 
            {"set ticslevel 0"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_HIDDEN3D]= 
            {"set hidden3d"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_VIEW]= 
            {"set view 60, 120"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_SURFACE]= 
            {"set surface"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_CLABEL]= 
            {"set noclabel", "set clabel"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_CONTOUR]= 
            {"set contour"};
        }
      else if (iMode == 2)
        {
          // Plot no xyz-plot, only contour
          s_XYZ_Figure[F_PLOT][iArea][P_TYPE][2]= FALSE;
          s_XYZ_Figure[F_PLOT][iArea][P_TYPE][3]= TRUE;

          // Set and reset (if necessary) 3-d settings
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_SURFACE]= 
            {"set nosurface"};
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_CONTOUR]= 
            {"set contour"};
        }

      if (columns(mlZ) == 1)
        {
          // Approximate grid using quadratic weighting scheme, 
          // and on average 5 items per cell
          nSize= min(floor(sqrt(rows(mlZ)/5)), 25);
          s_XYZ_Figure[F_PLOT][iArea][P_SET][S_DGRID3D]= 
            {sprint("set dgrid3d ", nSize, ", ", nSize, ", 2")};
        }

      // Set the labels
      s_XYZ_Figure[F_PLOT][iArea][P_SET][S_XLABEL]= 
        {sprint("set xlabel '", sKeyX, "'"), 
         "set xlabel ''"};
      s_XYZ_Figure[F_PLOT][iArea][P_SET][S_YLABEL]= 
        {sprint("set ylabel '", sKeyY, "'"), 
         "set ylabel ''"};
    }

  s_XYZ_LastPlot= iArea;
}

/*  
**  #include <gnudraw.h>
**  SetDraw(const iOption, ...);
**  
**      iType 
**          in: int, type of adjustment 
**      i1 ... i5 
**          in: optional arguments 
**  
**  No return value. 
**  
**  The expected number of arguments depends on the type of adjustment: 
**  
**  option          i1           i2             i3            i4           i5 
**  
**  SET_AXIS        fontsize     step           tick       
**  SET_AXISLINE    no X-line    no Y-line      center dates  no small Y      
**  SET_BOX         box:0--1     X-grid:0--1    Y-grid:0--1       
**  SET_COLOR       lineno:0--15 red:0--255     green:0--255  blue:0--255     
**  SET_COLORMODEL  model:0--3          
**  SET_FONT        fontno:0--3  fontsize         
**  SET_GRID        color:0--15  type:0--15         
**  SET_HISTOGRAM   inside:0--15 outside:0--15         
**  SET_LEGEND      boxed:0--1   columns         
**  SET_LEGENDHIDE  hide:0--1          
**  SET_LINE        lineno:0--15 linetype:0--4  width         on           off 
**  SET_MARGIN      left t top         
**  SET_PRINTPAGE   orient:0--1  papertype:0--2 X-size        Y-size     
**  SET_SYMBOL      lineno:0--15 symtype:0--4   size            
*/
SetDraw(const iOption, ...)
{
  println ("SetDraw not implemented yet");
}

/*  
**  #include <gnudraw.h>
**  SetDrawWindow(const sTitle);
**  
**      sTitle 
**          in: string, name of window 
**  
**  No return value. 
**  
**  This function is only relevant when interacting with GiveWin otherwise it does nothing. It sets
**  the name of the GiveWin window in which the graphs of the Ox program appear to
**  sTitle.
*/
SetDrawWindow(const sTitle)
{
  // No effect
}

/*  
**  #include <gnudraw.h>
**  SetTextWindow(const sTitle);
**  
**      sTitle 
**          in: string, name of window 
**  
**  No return value. 
**  
**  This function is only relevant when interacting with GiveWin otherwise it does nothing. It sets
**  the name of the GiveWin window in which the output (from the print() function) of the
**  Ox program appears to sTitle. 
*/
SetTextWindow(const sTitle)
{
  // No effect
}

/*  
**  #include <gnudraw.h>
**  ShowDrawWindow();
**  
**  No return value. 
**  
**  Shows the drawing. Note that in some implementations the graphs cannot be displayed.
**  Then a message is printed (SaveDrawWindow() will still work in that case!). A call to
**  ShowDrawWindow also clears the drawing buffer, so that subsequent graphing starts
**  from an empty sheet. 
*/
ShowDrawWindow()
{
  decl sCommand;

#ifdef EXECSYS_INCLUDED
  // Save the file, run gnuplot and remove the file again
  s_XYZ_TempFile= sprint("g", "%.0f", imod(today(), 1e6));
  SaveDrawWindow(s_XYZ_TempFile~".scr");
  sCommand= sprint("execgnu . ", s_XYZ_TempFile, " remove");
  #ifdef OX_Windows  // On windows, spawn a new process
    sCommand= sprint("start command /c ", sCommand);
  #endif
  FnExecSys(sCommand);
#else
  println ("ShowDrawWindow not implemented yet");
#endif
  CloseDrawWindow();
}

#endif
