/*
**  Program HedRat.Ox
**
**  Purpose:
**    Calculate the hedging ratios, based on the results
**    of CalcDRet.ox. When this program is started with an extra
**    command line argument, no recalculation of the hedging results
**    is done.
**
**  Version 7:
**    Based on hedrat6b
**
**  Author:
**    Charles Bos
**
**  Date:
**    6/11/2000
*/
#include <oxstd.h>
#include "include/gnudraw_jae.h"
#include "include/info.ox"
#include "include/size.ox"
#include "include/printmat.ox"

#include "simox.dec"

// Function declarations
LoadDens(const amDens, const avHist, const avX, const amPMV, const anHist, 
         const anPeriods, const sDRetFile);
PrepareHedgeMat(const amHedge, const amHSharpe, const amHVaR, 
                const vGamma, const aMatNames, const vIHF, 
                const dAlpha, const bRecalc, const sSimbase);
CalcOptHedge(const vGamma, const vS, const mDens, const vIHF);
CalcUtil(const vH, const vGamma, const vS, const mDens, const vIHF);
CalcReturn(const mHedge, const vY, const vdIHF);
CalcSharpe(const vS, const mDens, const vIHF);
CalcVaR(const vS, const mDens, const vAlpha, const vIHF);
GiveUtils(const mH, const vGamma, const vYper, const vIHF, const asModel,
          const sType);
PlotUtils(const mU, const mUSharpe, const mUVaR, 
          const vGamma, const mDMY, const asModels, const sPlbFile);
PlotHedgeRatio(const mHedgeRW, const amHedge, const vGamma, 
               const asModels, const mDMY, const sPlbbase);
PlotEvolGamma(const mXOpt, const vGamma, const sSimbase,
              const asModel);

main()
{
  decl sSimbase, sOutFile, sPlbFile, mYt, mInter, mDMY, nPeriods, 
       vGamma, vPlotGamma, viGamma, vHist, nHist, nG, nM, vS, 
       vYper, vIHF, mDens, mHedgeRW, i, j, dAlpha, 
       aMatNames, asMat, asModel, sInd, vOrder,  
       amHedge, mXOpt, mUOpt, 
       amHSharpe, mXSh, mUSh, 
       amHVaR, mXVaR, mUVaR, dTime;

  dTime= timer();
  vOrder= <0, 1, 2, 3, 4, 5, 6>;
  asModel= {"WN", "LL", "GLL", "GLLGa", "GLLSV", "GLLSt",
            "GLLGaSt"};
  asMat= {"ma", "mb", "mc", "md", "me", "mf", "mg"};
  
  sInd= g_OutDir[sizeof(g_OutDir)-2:];
  sSimbase= sprint("excl/hedret/h", sInd);
  for (i= 0, aMatNames= {}; i < 7; ++i)
    aMatNames~= {sprint("excl/", asMat[i], sInd, "/", asMat[i])};

  vGamma= range(-20, -12, 2)~range(-10, 0)~0.5;
  vPlotGamma= <-10, 0>;     // Subset of gamma's used in plotting
  dAlpha= 0.95;
  nG= sizerc(vGamma);
  nM= sizeof(aMatNames);

  vHist= g_Kalm_vHist;
  if (vHist == 0)
    vHist= range(4173, 4695);
  nHist= sizerc(vHist);

  PdfInit(&g_InitVP, &g_nData);
  GetData(&mYt, &mDMY, &mInter);
  mYt~= M_NAN;
  mDMY= mDMY~<1; 1; 2000>;

  vIHF= (mInter[0][]|mInter[1][])/(360*100);  
  vIHF= vIHF[][vHist-1];
  vYper= mYt[0][vHist];

  // Calculate, or reload, the hedge ratio's (in the order vOrder)
  PrepareHedgeMat(&amHedge, &amHSharpe, &amHVaR, vGamma, aMatNames[vOrder], 
                  vIHF, dAlpha, (sizeof(arglist()) == 1), sSimbase);

  sOutFile= sprint(sSimbase, "hedrat.out");
  fopen(sOutFile, "l");
  println("Output concerning hedging ratio's, returns and utilities");
  println("-------------------------------------------------");
  println("Printing results concerning the optimal decisions");
  [mXOpt, mUOpt]= 
    GiveUtils(amHedge, vGamma, vYper, vIHF, asModel[vOrder], "Optimal");
  println("-------------------------------------------------");
  println("Printing results concerning the Sharpe decisions");
  [mXSh, mUSh]= 
    GiveUtils(amHSharpe, vGamma, vYper, vIHF, asModel[vOrder], "Sharpe");
  println("-------------------------------------------------");
  println("Printing results concerning the VaR decisions");
  [mXVaR, mUVaR]= 
    GiveUtils(amHVaR, vGamma, vYper, vIHF, asModel[vOrder], "VaR");

  // Plot the evolution of the utility over time, for gamma= vPlotGamma
  viGamma= vecindex(vGamma, vPlotGamma);
  if (sizeof(viGamma) == 0)
    viGamma= <0>;
  sPlbFile= sprint(sSimbase, "u");
  PlotUtils(mUOpt[viGamma], mUSh[viGamma], mUVaR[viGamma], 
            vGamma[viGamma], mDMY[][vHist[:nHist-2]], 
            asModel[vOrder], sPlbFile);

  mHedgeRW= (lag0(vYper', 1)' .<= 0);
  sPlbFile= sprint(sSimbase, "hg");
  PlotHedgeRatio(mHedgeRW, amHedge[viGamma], vGamma[viGamma], 
                 asModel[vOrder], mDMY[][vHist], sPlbFile);

  // Plot the evolution of losses, gains, returns, and utilities
  //   as a function of Gamma
  PlotEvolGamma(mXOpt, vGamma, sSimbase, asModel[vOrder]);

  println("Time elapsed:\n ", "%10s", timespan(dTime));
}

/*
**  LoadDens
**
**  Outputs:
**    vS        nS x 1 column vector of returns at which density is evaluated
**    mDens     nS x nPeriods matrix of densities of S at the time periods
**    vHist     1 x nHist vector of indices of history
**    mPMV      2 x nHist matrix of mean and variances
*/
LoadDens(const amDens, const avHist, const avS, const amPMV, const anHist, 
         const anPeriods, const sDRetFile)
{
  amDens[0]= loadmat(sDRetFile);
  avHist[0]= amDens[0][0][1:];
  amPMV[0]= amDens[0][1:2][1:];
  avS[0]= amDens[0][3:][0];
  amDens[0]= amDens[0][3:][1:];
  anHist[0]= columns(avHist[0]);
  anPeriods[0]= columns(amDens[0]);
}

/*
**  PrepareHedgeMat(const amHedge, const amHSharpe, const amHVaR, 
**                  const vGamma, const aMatNames, const vIHF, 
**                  const dAlpha, const bRecalc, const sSimbase)
**
*/
PrepareHedgeMat(const amHedge, const amHSharpe, const amHVaR, 
                const vGamma, const aMatNames, const vIHF, 
                const dAlpha, const bRecalc, const sSimbase)
{
  decl nG, nM, i, j, mDens, vHist, vS, mPMV, nHist, nPeriods, sDRetFile,
       mSharpe, mVaR, vLimSharpe, vLimVaR; 
  
  nG= sizerc(vGamma);
  nM= sizeof(aMatNames);
  amHedge[0]= amHSharpe[0]= amHVaR[0]= vLimSharpe= vLimVaR= new array[nG];
  for (j= 0; j < nG; ++j)
    amHedge[0][j]= <>;
  mSharpe= mVaR= <>;             

  for (i= 0; i < nM; ++i)
    {
      sDRetFile= sprint(aMatNames[i], "dret.fmt");

      LoadDens(&mDens, &vHist, &vS, &mPMV, &nHist, &nPeriods, sDRetFile);
      if (!(vHist == range(4173, 4695)))
        print ("Warning: vHists not equal?");

      if (bRecalc)
        for (j= 0; j < nG; ++j)
          {
            info(i*nG+j, nG*nM);
            amHedge[0][j]|= CalcOptHedge(vGamma[j], vS, mDens, vIHF);
          }

      // Calculate Sharpe ratio's and the Value-at-Risk     
      mSharpe|= CalcSharpe(vS, mDens, vIHF);
      mVaR|= CalcVaR(vS, mDens, dAlpha, vIHF);
    }

  for (j= 0; j < nG; ++j)
    {
      sDRetFile= sprint(sSimbase, "h", double(10*vGamma[j]), ".mat");
      if (bRecalc)
        savemat(sDRetFile, amHedge[0][j]);
      else
        amHedge[0][j]= loadmat(sDRetFile);

      vLimVaR[j]= diagonal(quantiler(sortr(mVaR), 1-meanr(amHedge[0][j])'));
      amHVaR[0][j]= (mVaR .> vLimVaR[j]');

      vLimSharpe[j]= 
        diagonal(quantiler(sortr(mSharpe), 1-meanr(amHedge[0][j])'));
      amHSharpe[0][j]= (mSharpe .> vLimSharpe[j]');
    }
}

/*
**  CalcOptHedge()
**
**  Purpose:
**    Calculate the optimal hedging ratio, based on the density as
**    calculated for the return on the exchange rate
**
**  Inputs:
**    vGamma    1 x nG vector of gamma's
**    vS        nS x 1 vector of returns at which density is evaluated
**    mDens     nS x nPeriods matrix of densities of S at the time periods
**    vIHF      2 x nPeriods matrix with the interest rate in the home and the
**              foreign country at the time periods
**
**  Return value:
**    mHedge    nG x nPeriods matrix with optimal hedge ratio's
*/
CalcOptHedge(const vGamma, const vS, const mDens, const vIHF)
{
  decl mHedge, vH, mU, nG, nPeriods, i;

  vH= range(0, 1, 0.01);
  nG= sizec(vGamma);
  nPeriods= sizec(mDens);
  mHedge= new matrix [nG][nPeriods]; 

  for (i= 0; i < nPeriods; ++i)
    {
      mU= CalcUtil(vH, vGamma, vS, mDens[][i], vIHF[][i]);
      mHedge[][i]= vH[][limits(mU')[3][]]';
    }

  return mHedge;
}

/*
**  CalcUtil(const vH, const vGamma, const vS, const mDens, const vIHF)
**
**  Purpose:
**    Calculate the utility of a certain hedging decision, using
**    original utility function, specialized version
**
**  Inputs:
**    vH     nG x nPeriods matrix of hedging decisions
**    vGamma 1 x nG vector or scalar of risk aversion parameters
**    vS     nS x nPeriods vector of percentage returns
**    mDens  nS x nPeriods matrix of corresponding probabilities of S
**    vIHF   2 x nPeriods vector with home and foreign interest rate, in perunages
**
**  Output:
**    mU     nG x nPeriods matrix of utilities
*/
CalcUtil(const vH, const vGamma, const vS, const mDens, const vIHF)
{
  decl nH, nS, dS, nG, viGN0, viG0, i, mW, mU, dI;

  nH= max(columns(vH), 1);
  nS= max(rows(vS), 1);
  nG= max(sizerc(vGamma), 1);

  viGN0= vecindex((vGamma .!= 0));
  viG0= vecindex((vGamma .== 0));
  dI= vIHF[0][]-vIHF[1][];
  dS= 1;
  if (nS > 1)
    dS= vS[1][]-vS[0][];
  mU= 0;
  for (i= 0; i < nS; ++i)
    {
      mW= CalcReturn(vH, vS[i][], dI);

      if (sizeof(viGN0) > 0)
        mW[viGN0][]= (mW[viGN0][] .^ 
         (vGamma[][viGN0]' .* ones(1, nH)) - 1) ./ vGamma[][viGN0]';
      if (sizeof(viG0) > 0)
        mW[viG0][]= log(mW[viG0][]);

      mU+= mW .* mDens[i][];
    }
  mU= mU .* dS;
 
  return mU;
}

/*
**  CalcReturn(const mHedge, const vY, const vdIHF)
**
**  Purpose:
**    Calculate the return based on the hedging decision
**
**  Inputs:
**    mHedge      nG x nPeriods matrix with optimal hedge ratios
**    vY          1 x nPeriods matrix with realized return on exch rate
**    vdIHF       1 x nPeriods matrix with differential between home and 
**                foreign interest rate, on daily basis, as perunage
**
**  Output:
**    mReturn     nG x nPeriods matrix with returns
*/
CalcReturn(const mHedge, const vY, const vdIHF)
{
  decl mReturn;

  // Calculate return in excess of home interest rate
  mReturn= (1 - mHedge) .* exp(vY/100) + mHedge .* exp(vdIHF);

  return mReturn;
}

/*
**  CalcSharpe()
**
**  Inputs:
**    vS     nS x nPeriods vector of percentage returns
**    mDens  nS x nPeriods matrix of corresponding probabilities of S
**    vIHF   2 x nPeriods vector with home and foreign interest rate, in perunages
*/
CalcSharpe(const vS, const mDens, const vIHF)
{
  decl vM1, vM2, vSDev, vSharpe;

  vM1= sumc(mDens .* vS) .* (vS[1][] - vS[0][]);
  vM2= sumc(mDens .* vS .* vS) .* (vS[1][] - vS[0][]);
  vSDev= sqrt(vM2 - vM1 .* vM1);

  // vIHF is perunage, get it into percentage points again
  vSharpe= (vM1+100*(vIHF[0][]-vIHF[1][]))./vSDev;

  return vSharpe;
}

/*
**  CalcVaR()
**
**  Purpose:
**    Calculate the (interpolated) value-at-risk
**
**  Inputs:
**    vS          nS x 1 vector of returns
**    mCDens      nS x nPeriods matrix of cumulative densities
**    vAlpha      1 x nA row of alpha's
**    vIHF        2 x nPeriods matrix with home and foreign interest
**                rates, in daily perunages.
**
**  Output:
**    mVaR  nA x nPeriods matrix of value at risks
*/
CalcVaR(const vS, const mDens, const vAlpha, const vIHF)
{
  decl nA, nS, nHist, vI, mVaR, i, j, dS, mCDens;

nA= sizec(vAlpha);
nS= sizer(vS);
nHist= sizec(mDens);
dS= vS[1][]-vS[0][];

// Calculate cumulative density
mCDens= cumulate(mDens).*dS;
if (!isfeq(mCDens[nS-1][], 1))
  println("Warning: Scaling mCDens up by ", limits(1.0 ./ mCDens[nS-1][]')[:1][]');
mCDens= mCDens ./ mCDens[nS-1][];

mCDens= zeros(1, nHist)|mCDens|ones(1, nHist);
mVaR= new matrix [nA][nHist];
for (i= 0; i < nA; ++i)
  {
    vI= limits(mCDens .> (1-vAlpha[i]))[3][]; 
    for (j= 0; j < nHist; ++j)
      mVaR[i][j]= 
        vS[vI[j]-1][]+ dS.*((1-vAlpha[i])-mCDens[vI[j]-1][j])/
                           (mCDens[vI[j]][j]-mCDens[vI[j]-1][j]);
  }
 mVaR= mVaR + 100*(vIHF[0][] - vIHF[1][]);

return mVaR;
}

/*
**  GiveUtils(const mH, const vGamma, const vYper, 
**            const vIHF, const asModel, const sType)
**
**  Inputs:
**    mH          Array of size nG with nM x nHist hedging decisions
**    vGamma      1 x nG row vector
**    vYper       1 x nHist vector with observations
**    vIHF        2 x nHist vector with home and foreign interest rates
**    asModel     nM array with model descriptions
**    sType       String, indicating type of results (optimal, Sharpe,
**                VaR)
**
**  Outputs:
**    mXOut       nG array with 3+nM x 10 matrix with 
**                  avH, H0, H1, dH, 3ML, 3MG, r, U, r1, U1
**    mU          nG array with nM x nHist matrix of utils.
*/
GiveUtils(const mH, const vGamma, const vYper, 
          const vIHF, const asModel, const sType)
{
  decl nG, nM, nHist, asKeys, acLabs, aPrFmt, i, j, k, 
       mHLoc, mHedgeDet, vG, mU, mX, mCorr, mXOut, 
       mRet, mCRet, mCRet3M;

  nG= columns(vGamma);
  nM= rows(mH[0]);
  nHist= columns(vYper);
  asKeys= {"Full hedge", "No hedge", "RW"};
  mU= mXOut= new array [nG];

  // Start with deterministic hedge decisions.
  mHedgeDet= ones(1, nHist)|zeros(1, nHist)|
             (lag0(vYper', 1)' .<= 0);

  for (j= 0; j < nG; ++j)
    {
      mHLoc= mHedgeDet|mH[j];
      mRet= CalcReturn(mHLoc, vYper, vIHF[0][]-vIHF[1][]);
      mCRet= log(cumprod(mRet')');
      mCRet3M= diff0(mCRet', 65)';
      
      vG= ones(1, nM+3)*vGamma[j];
      mU[j]= CalcUtil(mHLoc[][:nHist-2], vG, 
                      vYper[:nHist-2], ones(1, nHist-1),
                      vIHF[][:nHist-2]);

      // Results on average H, H=0, H=1, dH, 3M-loss, 3M-gain,
      // 2-year return and utility, and 1-year return and utility
      mX= meanr(mHLoc)~sumr(mHLoc.<=1e-3)~sumr(mHLoc.>=1-1e-3)~
          meanr(fabs(diff0(mHLoc', 1)'[][1:]))~
          100*(limits(mCRet3M[][66:columns(mCRet)-2]')[:1][]')~
          100*(mCRet[][columns(mCRet)-2])~
          100*sumr(mU[j])~
          100*(mCRet[][columns(mCRet)/2])~
          100*sumr(mU[j][][:columns(mU[j])/2]);
      mXOut[j]= mX;

      acLabs= {"\\overline H", "\\#(H=0)", "\\#(H=1)", 
               "\\overline{|\\Delta H|}", "3M-L", "3M-G", "C", "U x 100"};
      aPrFmt= {"%6.2f", "%4.0f", "%4.0f", "%.3f", 
               "%6.2f", "%6.2f", "%6.2f", "%6.2f"};
      
      println ("\n", sType, " results: ");
      PrintMatrix(0, sprint("Gamma= ", double(vGamma[j])), aPrFmt, 
                  acLabs, asKeys~asModel, mX[][:5], FALSE);
      println("\\hline");
 
      println("Results in first year and both years jointly: "); 
      PrintMatrix(0, sprint("Gamma= ", double(vGamma[j])), aPrFmt[4:], 
                  {"C", "U x 100", "C", "U x 100"}, 
                  asKeys~asModel, mX[][<8, 9, 6, 7>], FALSE);
      println("\\hline");
 
      mX= variance(mHLoc[2:][]');
      mCorr= M_NAN*zeros(mX);
      for (i= 0; i < columns(mX); ++i)
        for (k= 0; k < i; ++k)
          mCorr[i][k]= mX[i][k]/sqrt(mX[i][i]*mX[k][k]);

      aPrFmt= {"%6.2f"};
      println("Gamma= ", double(vGamma[j]));
      PrintMatrix(0, "Model", aPrFmt, {"RW"}~asModel, 
                  {"RW"}~asModel, mCorr[][:columns(mX)-2], FALSE);
   }

  return {mXOut, mU};
}

/*
**  PlotUtils(const mUOpt, const mUSh, const mUVaR, 
**            const dGamma, const mDMY, const asModels, const sPlbbase)
**
**  Purpose:
**    Plot the evolution of the utility over time
*/
PlotUtils(const mUOpt, const mUSh, const mUVaR, 
          const vGamma, const mDMY, const asModels, const sPlbbase)
{
  decl mUC, asKeys, mU, nM, nG, i, j;

  asKeys= {"Full hedge", "No hedge", "RW hedge"}~asModels;
  nM= sizeof(asModels);
  nG= sizerc(vGamma);  
  for (j= 0; j < nG; ++j)
    {
      mUC= cumulate(mUOpt[j]')';

      DrawTMatrix(j, mUC*100, asKeys, mDMY, 0, 0);
      DrawTitle(j, sprint("Utilities, Gamma= ", vGamma[j]));
    }
  SaveDrawWindow(sprint(sPlbbase, "all.plb"));
  CloseDrawWindow();

  asKeys= {"Full hedge", "No hedge", "RW hedge", "Optimal", 
           "Sharpe", "VaR"};
  for (i= 0; i < nM; ++i)
    {
      for (j= 0; j < nG; ++j)
        {
          mU= mUOpt[j][:2][]|mUOpt[j][3+i][]|
              mUSh[j][3+i][]|mUVaR[j][3+i][];
          mUC= cumulate(mU')';

          DrawTMatrix(j, mUC*100, asKeys, mDMY, 0, 0);
          DrawTitle(j, sprint(asModels[i], ", gamma= ", vGamma[j]));
        }
      SaveDrawWindow(sprint(sPlbbase, strlwr(asModels[i]), ".plb"));
      CloseDrawWindow();
    }
}

/*
**  PlotHedgeRatio(const mHedgeRW, const amHedge, const vGamma, 
**                 const asModels, const mDMY, const sPlbbase)
**
**  Purpose:
**    Plot the evolution of the hedging ratio's
*/
PlotHedgeRatio(const mHedgeRW, const amHedge, const vGamma, 
               const asModels, const mDMY, const sPlbbase)
{
  decl nG, nM, i, j, sPlbFile;
  
  nG= sizerc(vGamma);
  nM= sizeof(asModels);
  for (j= 0; j < nG; ++j)
    {
      DrawTMatrix(0, mHedgeRW, "RW", mDMY, 0, 0);
//      DrawAxisAuto(0, TRUE, FALSE);
      for (i= 0; i < nM; ++i)
        {
          DrawTMatrix(i+1, amHedge[j][i][], asModels[i], mDMY, 0, 0);
//          DrawAxisAuto(i+1, TRUE, FALSE);
        }
      sPlbFile= sprint(sPlbbase, 10*vGamma[j], ".plb");
      DrawAdjust(ADJ_ALIGN, 1);
      SaveDrawWindow(sPlbFile);
      CloseDrawWindow();
    }
}

/*
**  PlotEvolGamma(const mXOpt, const vGamma, const sSimbase,
**                const asModel)
**
**  Purpose:
**    Plot the evolution of losses, gains, returns and utilities as
**    a function of gamma.
*/
PlotEvolGamma(const mXOpt, const vGamma, const sSimbase,
              const asModel)
{
decl nM, nG, i, mGamH, mGamL, mGamG, mGamC, mGamU, asKeys;

nM= sizeof(asModel);
nG= sizerc(vGamma);
mGamH= mGamL= mGamG= mGamC= mGamU= zeros(nM+1, nG);
for (i= 0; i < nG; ++i)
  {
    mGamH[][i]= mXOpt[i][range(2, nM+2)][0];
    mGamL[][i]= mXOpt[i][range(2, nM+2)][4];
    mGamG[][i]= mXOpt[i][range(2, nM+2)][5];
    mGamC[][i]= mXOpt[i][range(2, nM+2)][6];
    mGamU[][i]= mXOpt[i][range(2, nM+2)][7];
  }
  
asKeys= {"RW"}~asModel;
DrawXMatrix(0, mGamH, asKeys, vGamma, "", 2, 2);
DrawAdjust(ADJ_LABEL, "", "H");
DrawAdjust(ADJ_AREA_X, min(vGamma), max(vGamma));
SaveDrawWindow(sprint(sSimbase, "gamh.plb"));
CloseDrawWindow();

DrawXMatrix(0, mGamL, asKeys, vGamma, "", 2, 2);
DrawAdjust(ADJ_LABEL, "", "3M-L");
DrawAdjust(ADJ_AREA_X, min(vGamma), max(vGamma));
SaveDrawWindow(sprint(sSimbase, "gaml.plb"));
CloseDrawWindow();

DrawXMatrix(0, mGamG, asKeys, vGamma, "", 2, 2);
DrawAdjust(ADJ_LABEL, "", "3M-G");
DrawAdjust(ADJ_AREA_X, min(vGamma), max(vGamma));
SaveDrawWindow(sprint(sSimbase, "gamg.plb"));
CloseDrawWindow();

DrawXMatrix(0, mGamC, asKeys, vGamma, "", 2, 2);
DrawAdjust(ADJ_LABEL, "", "C");
DrawAdjust(ADJ_AREA_X, min(vGamma), max(vGamma));
SaveDrawWindow(sprint(sSimbase, "gamc.plb"));
CloseDrawWindow();

DrawXMatrix(0, mGamU, asKeys, vGamma, "", 2, 2);
DrawAdjust(ADJ_LABEL, "", "U");
DrawAdjust(ADJ_AREA_X, min(vGamma), max(vGamma));
SaveDrawWindow(sprint(sSimbase, "gamu.plb"));
CloseDrawWindow();
}
