/*
**  Program GibSV.ox
**
**  Purpose:
**    Run Gibbs sampling on the local level-stochastic volatility
**    model. 
**
**  Author:
**    Charles S. Bos
**
**  Date:
**    13 April 1999
**
**  Version:
**   16     Using normal prior on Rho and Phi
**   18     Using sample routines from incsv4
**   19     Including UIP
**
**  The model:
**    We sample from the model
**          y(t)= mu(t) + eps(t)e^(h(t)/2)
**          mu(t)= rho mu(t-1) + eta(t)
**          h(t)= muh + phi (h(t) - muh) + xi(t)
**          eps(t) ~ norm(0, 1)
**          eta(t) ~ norm(0, s2eta)
**          xi(t)  ~ norm(0, s2xi)
**
**    The SV is approximated as in Kim, Shephard and Chib:
**          ln(y-mu)^2 = z(t) + h(t)
**    with z(t) being a mixture of 7 normals,
**          z(t)= ln(eps(t)^2)
**          f(z(t)) approx sum qi f_Norm(z|mi-1.2704, vi^2)
**                i= 1, .., 7
*/
#include <oxstd.h>
#import <maximize>      // The optimization routines 
#include <oxfloat.h>    // M_INF_NEG
#include <oxprob.h>     // RanGamma
#include <ssfpack.h>
#include "include/info.ox"      // Information on lenght of calc
#include "include/setseed.ox"   // Reset the seed?
#include "include/gnudraw_jae.h"
#include "include/oxprobig.ox"
#include "include/size.ox"
#include "include/tractime.ox"

#include "simox.dec"    // Declaration of the model

// Function declarations
RunChain(const sSimFile, dRho, dSEta, dMuH, 
         dPhi, dSXi, vh, const mYt, const vdR, const iTEst, const nReps, const nBurnin, 
         const nSkip, const vHist, const vS);
FiltPred(const vh, const dRho, const dSEta, const dPhi, const dMuH, 
         const dSXi, const mYt, const nPeriods, const nRepSamp);
InitOutFile(const sOutFile, const Vers, const VarNames, 
            const vPInit, const nReps, const nBurnin);
GiveResults(const sOutFile, const sSimFile, const nReps, const nBurnin, 
            const nLag, const dTime);
InitGibSV(const vInitVP, const amYt, const amDMY, const amInter, 
          const adRho, const adSEta, 
          const adMuH, const adPhi, const adSXi, const avh);
LnPdfSV_Shell(const vP, const adFunc, const avScore, const amHessian);

/*
**  Static declarations, on dimension of problem, index of present
**  parameter, and the full vector of parameters
*/
static decl s_GS_mYStar,
            s_GS_MaxPars, s_GS_MaxLL= M_INF_NEG,
            s_GS_nOut= 1000, 
            s_SV_Filter= TRUE;

main()                // function main is the starting point
{
  decl 
    sSimbase, sSimFile, sOutFile, sDRetFile, 
    nReps, nRot, nHist, nPeriods, iT, ir,  
    dRho, dSEta, dMuH, dPhi, dSXi, vh, vS, vHist, dTime,
    mPMVS, mDens, mYt, mInter, mDMY, vdR, mDum, vIndex;

  if (g_Kalm_UseModel != "GLLSV")
    {
      print ("Error: Incorrect declarations file?");
      exit(1);
    }

  /* Initialize */
  dTime= timer();
  if (g_Seed != 0)
    SetSeed(g_Seed);

  InitGibSV(g_InitVP, &mYt, &mDMY, &mInter, &dRho, &dSEta, &dMuH, &dPhi, &dSXi, &vh);
  iT= columns(mYt);

  // Set a vector with the interest rate differential, for usage in UIP
  vdR= zeros(mYt);
  if (g_Kalm_UseUIP == 1)
    vdR= (mInter[0][]-mInter[1][])/360;
  vdR~= vdR[iT-1];

  nRot= sizerc(g_Flex_nMH);
  sSimbase= sprint(g_OutDir, "/", g_VersFile);
  sSimFile= sprint(sSimbase, ".fmt");
  sOutFile= sprint(sSimbase, "gs.out");
  sDRetFile= sprint(sSimbase, "dret.fmt");

  println ("Writing output to ", sSimFile, ", ", sOutFile, " and ", sDRetFile);

  nReps= g_Flex_nMH[nRot-1];
  vHist= g_Kalm_vHist;
  if (vHist == 0)
    vHist= range(4173, 4695);
  nHist= sizerc(vHist);
  nPeriods= iT-min(vHist)+1;

  InitOutFile(sOutFile, g_Vers, g_VarNames, 
              dRho|dSEta|dMuH|dPhi|dSXi, nReps, g_Flex_nBurnin[nRot-1]);

  vS= range(g_Kalm_vSRegion[0], g_Kalm_vSRegion[1], g_Kalm_vSRegion[2]);
  [mPMVS, mDens]= 
    RunChain(sSimFile, dRho, dSEta, dMuH, dPhi, dSXi, 
             vh, mYt, vdR, g_Kalm_FracEst, nReps, g_Flex_nBurnin[nRot-1],
             g_Flex_nSkip, vHist, vS);
  mDum= (zeros(3, 1)~(vHist|mPMVS[:1][]))|(vS'~mDens);
  ir= savemat(sDRetFile, mDum);

  GiveResults(sOutFile, sSimFile, nReps, g_Flex_nBurnin[nRot-1], 20, dTime);

  mDMY~= <1; 1; 2000>;
  mYt~= M_NAN;

  // Draw a graph with predictive density
  DrawTMatrix(0, mPMVS[0][], "E(s)", mDMY[][vHist], 0, 0);
  DrawTMatrix(1, mPMVS[1][], "s(s)", mDMY[][vHist], 0, 0);

  mDum= mYt[][vHist]|(mPMVS[0][]+mPMVS[1][].*<-1; 0; 1>);
  DrawTMatrix(2, mDum[:2][], {"Y", "-+s(s)", "E(s)"}, mDMY[][vHist], 0, 0);
  DrawTMatrix(2, mDum[3][], "", mDMY[][vHist], 0, 0, 0, 3);

  vIndex= range(0, nHist-1, max(nHist/60, 1));
  DrawXMatrix(3, mDens[][vIndex]', "p(s)", vS, "");
  SaveDrawWindow(sSimbase~"mv.plb");
  CloseDrawWindow();
}

/*
**  Function RunChain(const sSimFile, const vP, const nReps)
**
**  Purpose:
**    Run the chain of the Gibbs sampler. 
**
**  Inputs:
**    sSimFile     string, name of simulation file
**    vP          vector, starting point of the chain, containing the 
**                free parameters from the sequence Rho, SEta, MuH,
**                Phi, SXi
**    iTEst       scalar, number of observations used in estimation
**    nReps       scalar, number of repetitions
**    nBurnin     scalar, number of skipped repetitions
**
**  Outputs:
**    written in sSimFile
**
**  Return value:
**    mPredMVS    3 x nPeriods matrix with mean predicted mean, mean
**                standard deviation, and mean sample
*/
RunChain(const sSimFile, dRho, dSEta, dMuH, 
         dPhi, dSXi, vh, const mYt, const vdR, const iTEst, const nReps, const nBurnin, 
         const nSkip, const vHist, const vS)
{
  decl
    fhSim, nChain, nS, nHist, nPeriods, i, j, 
    vMu, dPars, mYRt, vYStar, vis, dS2Eta, dS2Xi,
    ir, sum_vP, mOut, iOut, mDens, miDens, vSd, 
    vHPred, vS2State, mMSdS, mPMVS, iPMVS, ve, iT, vXi, nXi, vdXi;

  // Introduce the interest rate differential
  iT= columns(mYt);
  mYRt= mYt + vdR[:iT-1];

  // Save the sampled parameters in the sSimFile
  fhSim= fopen(sSimFile, "wbf");

  TrackTime({"Mu", "Rho", "SEta", "iMixt", "vh", "MuH", "Phi", "SXi",
            "Samp", "Write"});

  // Run the chain
  println ("Running the chain...");
  nChain= nReps+nBurnin;
  sum_vP= zeros(sizerc(g_FreePars), 1); 

  nS= columns(vS);
  nHist= columns(vHist);
  mDens= zeros(nS, nHist);

  nXi= 20;
  vXi= range(-4, 4, 8/(nXi-1));

  // There are iT+1 possible predictions to be made.
  nPeriods= iT-min(vHist)+1;
  iOut= iPMVS= 0;
  mOut= zeros(s_GS_nOut, sizerc(g_FreePars));
  mPMVS= zeros(3, nHist);

  // Fix vh
//  vh= ones(mYt)*2*log(0.67);
  for (i= 0; i < nChain; ++i)
    {
    if (imod(i, g_Flex_InfoRep) .== 0)
      {
        info(i, nChain);
        TrackReport();
        print ("Mean parameters: ", sum_vP'/i);
      }

    TrackTime(0);
    vMu= SampleMu(vh, dRho, dSEta, mYRt);

    TrackTime(1);
    if (!(g_FreePars != 0))
      SampleRho(&dRho, vMu[:iTEst], dSEta, g_Kalm_PriorRho);

    TrackTime(2);
    if (!(g_FreePars != 1))
      {
        ve= (vMu - dRho * lag0(vMu', 1)')[1:];
        dS2Eta= SampleS2(ve[:iTEst], g_Kalm_PriorS2EtaAB);
        dSEta= sqrt(dS2Eta);
      }

    TrackTime(3);
    vYStar= log(sqr(mYRt-vMu));
    vis= SampleS(vYStar, vh);

    TrackTime(4);
    vh= SampleH(vYStar, vis, dPhi, dMuH, dSXi);

    TrackTime(5);
    if (!(g_FreePars != 2))
      dMuH= SampleMuH(vh[:iTEst], dPhi, dSXi, g_Kalm_PriorMuH);

    TrackTime(6);
    if (!(g_FreePars != 3))
      dPhi= SamplePhi(vh[:iTEst], dMuH, dSXi, g_Kalm_PriorPhi);

    TrackTime(7);
    if (!(g_FreePars != 4))
      {
        dS2Xi= SampleS2Xi(dPhi, dMuH, vh[:iTEst], g_Kalm_PriorS2XiAB);
        dSXi= sqrt(dS2Xi);
      }
    TrackTime(-1);

    dPars= dRho|dSEta|dMuH|dPhi|dSXi;

    sum_vP = sum_vP + dPars[g_FreePars];
    if ((i >= nBurnin) && (imod(i, nSkip+1) == 0))
      {
        TrackTime(8);
        mOut[iOut][]= dPars[g_FreePars]';
        ++iOut;

        if (s_SV_Filter)
          {
            // Sample future returns
            mMSdS= FiltPred(vh, dRho, dSEta, dPhi, dMuH, 
                            dSXi, mYRt, nPeriods, 1);
            mMSdS= mMSdS[][vHist-min(vHist)];

            // Take out the interest rate differential again, to get a
            // prediction of change in exchange rate.
            mMSdS[0][]-= vdR[vHist];

            mPMVS+= mMSdS[:2][];

            vHPred= mMSdS[3][];
            vS2State= mMSdS[4][];
			vdXi= densn(vXi)/dSXi*((vXi[1]-vXi[0])*dSXi);
            for (j= 0; j < nXi; ++j)
              {
                // Calculate the predictive density at the grid vS
                vSd= sqrt(vS2State + exp(vHPred + dSXi * vXi[][j]));
                // Calculate the normal density, adapting for density of H
                miDens= densn((vS'-mMSdS[0][])./vSd)./vSd *
                          vdXi[j];
                mDens += miDens;
              }
            ++iPMVS;
          }
        TrackTime(-1);
      }

    if (((iOut == s_GS_nOut) || (i == nChain-1)) && (iOut > 0))
      {
        TrackTime(9);
        ir= fwrite(fhSim, mOut[:iOut-1][]);

        iOut= 0;
        TrackTime(-1);
      }
    }
  fhSim= fclose(fhSim);

  mPMVS /= iPMVS;
  mDens /= iPMVS;

  println("Mean of parameters sampled: ", sum_vP'/nChain);

  return {mPMVS, mDens};
}

/*
**  FiltPred(const vh, const dRho, const dSEta, const dPhi, const dMuH, 
**           const dSXi, const mYt)
**
**  Purpose:
**    Calculate a predicted mean and variance of the observation,
**    together with a sample and predicted means of H and variance of state
**
**  Inputs:
**    vh          Vector of variance elements
**    dRho        Value of AR parameter
**    dSEta       SDev in state evolution
**    dPhi        Value of AR parameter in SV model
**    dMuH        Mean h in SV model
**    dSXi        SDev in SV model
**    mYt         Row vector of data
**
**  Outputs:
**    Return value      4 x nPeriods row with predicted mean and sdev of
**                      observation, prediction of H and predicted state variance
*/
FiltPred(const vh, const dRho, const dSEta, const dPhi, const dMuH, 
         const dSXi, const mYt, const nPeriods, const nRepSamp)
{
  decl
    mPhi, mOmega, mSigma, mDelta, mJ_Phi, mJ_Omega, mJ_Delta, vS2Eps,
    mState, mPred, vPredMean, vPredSd, hSamp, vSampSd, hPred, iT, 
    vSamp, vS2State;

  iT= columns(mYt);
  mPhi= dRho|1;
  mOmega= diag(dSEta*dSEta|1);
  mSigma= <-1; 0>;
  if (dRho < 1)
    mSigma= dSEta*dSEta/(1-dRho*dRho)|0;
  mJ_Phi= mDelta= mJ_Delta= <>;

  // Indicate that the variance of epsilon is time varying
  mJ_Omega= <-1, -1; -1, 0>;

  // Create a vector with the variances of epsilon
  vS2Eps= exp(vh);
  mState= SsfMomentEst(ST_PRED, &mPred, mYt, mPhi, mOmega, mSigma, 
                       mDelta, mJ_Phi, mJ_Omega, mJ_Delta, vS2Eps);

  // Predicted mean equals predicted state
  vPredMean= mState[1][0];
  vS2State= mState[0][0];
  if (nPeriods > 1)
    {
      vPredMean= mPred[1][iT-nPeriods+1:]~vPredMean;
      vS2State= mPred[2][iT-nPeriods+1:]~vS2State;
    }

  // Predicted volatility equals predicted volatility of state plus
  // predicted volatility from h.
  hPred= dMuH + dPhi*(vh[iT-nPeriods:] - dMuH);

  // h(t+1) is not observable, therefore should be sampled along with
  //   the return.
  hSamp= hPred + dSXi*rann(nRepSamp, nPeriods);
  vSampSd= sqrt(vS2State + exp(hSamp));
  vPredSd= sqrt(vS2State + exp(hPred+dSXi*dSXi/2));
  vSamp= vPredMean + vSampSd .* rann(nRepSamp, nPeriods);

  return vPredMean|vPredSd|vSamp|hPred|vS2State;
}

/*
**  InitOutFile
**
**  Purpose:
**    Do the general initializations
*/
InitOutFile(const sOutFile, const Vers, const VarNames, 
            const vPInit, const nReps, const nBurnin)
{
    decl fh, i;

    fh= fopen(sOutFile, "w");    
    fprintln(fh, "Gibbs sampling on LL-SV model");
    fprintln(fh, " ");
    fprintln(fh, "Version ");
    fprintln(fh, "  ", Vers);
    fprintln(fh, "Free parameters ");
    for (i= 0; i < rows(VarNames); ++i)
      fprint(fh, "  ", VarNames[i]);
    fprintln(fh, " ");
    fprintln(fh, "Length of burn-in period: ");
    fprintln(fh, "%12i", nBurnin);
    fprintln(fh, "Length of sample: ");
    fprintln(fh, "%12i", nReps);
    fprint(fh, "Initial Mu: ");
    fprintln(fh, "%12.4f", vPInit');
    fprintln(fh, "Date of run: ");
    fprintln(fh, "%12s", date());
    fprintln(fh, "Time of run: ");
    fprintln(fh, "%12s", time());

    fh= fclose(fh);
}


/*
**  GiveResults
**
**  Purpose:
**    Give the results of this rotation
**
**  Inputs:
**    _Mix_FnEval Global, count of number of function evaluations
**    Mu, AutoCov, nLag, sOutFile
*/
GiveResults(const sOutFile, const sSimFile, const nReps, const nBurnin, 
            const nLag, const dTime)
{
    decl fh, nDim, Theta, i, fmt;
    
    fh= fopen(sOutFile, "a");    
    fprintln(fh, "-----------------------------------------------------");
    fprintln(fh, "Results of Gibbs sampling on LL-SV model:");
    
    Theta= loadmat(sSimFile);
    nDim = columns(Theta);
    fprint(fh, "Mean of sampled Theta's: ");
    fprintln(fh, "%12.4f", meanc(Theta));
    fprint(fh, "Covariance of sampled Theta's: ");
    fprintln(fh, "%12.6f", variance(Theta));
    fprint(fh, "Autocorrelations: ");
    fmt= new array[nDim+1];
    fmt[0]= "  AC %2.0f: ";
    for (i= 1; i <= nDim; ++i)
      fmt[i]= "%13.5f";

    fprintln(fh, "%cf", fmt, range(0,nLag)'~acf(Theta, nLag));
    fprintln(fh, "Ended at time: ");
    fprintln(fh, "%12s", time());

    fprintln(fh, "Time elapsed:\n ", "%10s", timespan(dTime));

    fh= fclose(fh);
}


/*
**  InitGibSV
**
**  Purpose:
**    Initialize the static variables in this module, using settings in
**    GibSV.Dec
**
**  Inputs:
**    InitVP      Initial values for vP. Might have been
**                initialized in Kalman.DEC
**    amY         Address to matrix with data
**
**  Outputs:
**    amY         Row vector of data
*/
InitGibSV(const vInitVP, const amYt, const amDMY, const amInter,
          const adRho, const adSEta, 
          const adMuH, const adPhi, const adSXi, const avh)
{
  decl 
    dLnPdf, vP, mPhi, mSigma, mDelta, mOmega, nData, ir, 
    vMuHat, vhHat, dFrac;

  println("Gibbs sampling over Local level-Stochastic Volatility model");
  println("Parameters Rho, SEta, MuH, Phi, SXi are used");
  adRho[0]= vInitVP[0];
  adSEta[0]= vInitVP[1];
  adMuH[0]= vInitVP[2];
  adPhi[0]= vInitVP[3];
  adSXi[0]= vInitVP[4];

  amYt[0]= 0; 
  vP= vInitVP;
  PdfInit(&vP, &g_nData);
  GetData(amYt, amDMY, amInter);

  DrawT(0, amYt[0], amDMY[0], 0, 0);
  SaveDrawWindow(g_OutDir~"/"~g_VersFile~"my.plb");
  CloseDrawWindow();

  // Look for starting values 
  TrackTime({"Opt LL", "Opt SV"});
  TrackTime(0);
  vP= adRho[0]|adSEta[0]|.6;

  println("\nOptimizing the local level model");
  println("Starting values Rho, S(eta) and s(eps) used: ", vP');
  ir= MaxBFGS(LnPdf_Shell, &vP, &dLnPdf, 0, TRUE);

  nData= g_Kalm_FracEst;
  println("\n", MaxConvergenceMsg(ir),
          " using numerical derivatives",
          "\nLog-likelihood = ", 
          "%.8g", double(dLnPdf*nData),
                  "; n = ", nData);
  print("Parameters found:", "%12.5g", vP');

  adRho[0]= vP[0];
  adSEta[0]= vP[1];

  // Calculate the smoothed mean of the ll model
  mPhi= adRho[0]|1;
  mOmega= diag(adSEta[0]*adSEta[0]~vP[2]*vP[2]);
  /* Choose a diffuse initial state */
  mSigma=<-1; 0>;
  if (adRho[0] < 1)
    mSigma= adSEta[0]*adSEta[0]/(1-adRho[0]*adRho[0])|0;
  vMuHat= SsfCondDens(ST_SMO, amYt[0], mPhi, mOmega, mSigma);

  vMuHat= vMuHat[0][];
  s_GS_mYStar= log(sqr(amYt[0]-vMuHat));

  // Look for starting values of the SV part
  TrackTime(1);

  vP= adMuH[0] | adPhi[0] | 1;
       // | meanr(s_GS_mYStar)|sqrt(varr(s_GS_mYStar));
  println("\nOptimizing the SV part of the model");
  println("Starting values MuH, Phi and SXi used: ", vP');
  ir= MaxBFGS(LnPdfSV_Shell, &vP, &dLnPdf, 0, TRUE);

  println("\n", MaxConvergenceMsg(ir),
          " using numerical derivatives",
          "\nLog-likelihood = ", 
          "%.8g", double(dLnPdf*nData),
                  "; n = ", nData);
  print("Parameters found:", "%12.5g", vP');
  TrackReport();

  adMuH[0]= vP[0];
  adPhi[0]= vP[1];
  adSXi[0]= vP[2];

  // Calculate the smoothed mean of the SV model
  mPhi= adPhi[0]|1;
  mDelta= adMuH[0]*(1-adPhi[0])| -1.2704; // vP[3];
  mOmega= diag(adSXi[0]*adSXi[0]~(M_PI^2/2)); //vP[4]*vP[4]);
  /* Choose a diffuse initial state */
  mSigma= -1|(M_PI^2)/2;
  if (adPhi[0] < 1)
    mSigma= adSXi[0]*adSXi[0]/(1-adPhi[0]*adPhi[0])|(M_PI^2)/2;

  vhHat= SsfCondDens(ST_SMO, s_GS_mYStar, mPhi, mOmega, mSigma, mDelta);

  avh[0]= vhHat[0][];
}

/*
**  LnPdfSV_Shell(const vP, const adFunc, const avScore, const amHessian)
**
**  Purpose:
**    Calculate the MEAN LogLikelihood for the StSp model for the SV part, using the 
**    global data variables
**
**  Inputs:
**    vP          nDim vector of parameters, containing Phi, MuH, s(Xi), 
**                mu(ln(eps2)), s(ln(eps2)). 
**    adFunc      Address for function value
**    avScore     0 or address
**    amHessian   0 or address
**    s_GS_mYStar static, matrix with the data
**
**  Outputs:
**    adFunc      Function value at parameter vector(s)
**    avScore     Not changed
**    amHessian   Not changed
**    Return-value      1 if succeeded, 0 otherwise
*/
LnPdfSV_Shell(const vP, const adFunc, const avScore, const amHessian)
{
  decl ir, mPhi, mOmega, mSigma, mDelta, dVar, dPhi, dMuH, dSXi, dMuEps, dSEps;

  dMuH= vP[0];
  dPhi= vP[1];
  dSXi= vP[2];
  dMuEps= -1.2704; // vP[3];  // Expectation of ln(eps^2)
  dSEps= sqrt(M_PI^2/2); // vP[4];   // Standard deviation of ln(eps^2)

  mPhi= dPhi|1;
  mDelta= dMuH*(1-dPhi)|dMuEps;
  mOmega= diag(dSXi*dSXi~dSEps*dSEps);
  /* Choose a diffuse initial state */
  mSigma= -1|(M_PI^2)/2;
  if (dPhi < 1)
    mSigma= dSXi*dSXi/(1-dPhi^2)|(M_PI^2)/2;

  ir= SsfLik(adFunc, &dVar, s_GS_mYStar, mPhi, mOmega, mSigma, mDelta);
  adFunc[0] /= columns(s_GS_mYStar);

  return ir; 
}


