/*
**
**  Program GibGAS.ox
**
**  Purpose:
**    Run Gibbs sampling on the generalized local level-GARCH Student t
**    model. 
**
**  Author:
**    Charles S. Bos
**
**  Date:
**    30/6/2000
**
**  Version:
**   13     Using C
**   15     Writing DRetFile instead of FiltFile
**   17     Using sample routines from incglgs4.ox
**
**  The model:
**    We sample from the model
**          y(t)= mu(t) + sigma(eps) eps(t)
**          mu(t)= rho mu(t-1) + sigma(eta) eta(t)
**          h(t)= delta h(t-1) + (1-delta-alpha) + alpha eps(t-1)^2
**          eps(t) ~ t(0, (nu-2)/nu h(t), nu)   (with var(eps(t))= h(t),
**                                               var(eps)= E(h)= 1)
**          eta(t) ~ norm(0, 1)
**    To incorporate student t errors, effectively we sample
**          eps(t) ~ norm(0, h(t) * z(t))
**          z(t) ~ IG(alpha= nu/2, beta= 2/(nu-2))
**
*/
#include <oxstd.h>
#import <maximize>      // The optimization routines 
#include <oxfloat.h>    // M_INF_NEG
#include <oxprob.h>     // RanGamma
#include <arma.h>       // armagen
#include <ssfpack.h>
#include "include/info.ox"      // Information on lenght of calc
#include "include/setseed.ox"   // Reset the seed?
#include "include/gnudraw_jae.h"
#include "include/oxprobig.ox"
#include "include/tractime.ox"
#include "include/size.ox"

#include "simox.dec"    // Declaration of the model

// Function declarations
RunChain(const sSimFile, dRho, dSEta, dSEps, dDelta, 
         dAlpha, dNu, vz, vMu, const mYt, const vdR, const iTEst, const nReps, const nBurnin, 
         const nSkip, const vHist, const vS);
SampPred(const dRho, const dSEta, const dSEps, const dDelta, 
         const dAlpha, const dNu, const vz, const mYt, const nPeriods);
InitOutFile(const sOutFile, const Vers, const VarNames, 
            const vPInit, const nReps, const nSkip, const nBurnin);
GiveResults(const sOutFile, const sSimFile, const nReps, const nBurnin, 
            const nLag, const dTime);
InitGibStGa(const vInitVP, const amYt, const amDMY, const amInter, const adRho, const adSEta,
            const adSEps, const adDelta, const adAlpha, const adNu, const avz,
            const avMu);
MaxStdErr(const fFunc, const vP, const nData);
LnPdf_Loc(const vP, const adLnPdf, const avScore, const amHessian);

/*
**  Static declarations, on dimension of problem, index of present
**  parameter, and the full vector of parameters
*/
static decl s_GS_Pred= 1, 
            s_GS_GiveSub= 0, s_GS_nOut= 1000, 
            s_GS_CandGARCH,
            s_GS_Garch_DrawnAcc= <0; 0>,
            s_GS_Nu_DrawnAcc= <0; 0>;


main()                // function main is the starting point
{
  decl 
    sSimbase, sSimFile, sOutFile, sDRetFile, 
    nReps, nPeriods, nHist, iT, nRot,
    dRho, dSEta, dSEps, dDelta, dAlpha, dNu, vIndex, dTime,
    vMu, vz, mYt, mDMY, mInter, vdR, vS, vHist, mPMV, mDens, mDum, ir;

  /* Initialize */
  if ((g_Kalm_UseModel != "GLLGAS") && (g_Kalm_UseModel != "GLLGA") &&
      (g_Kalm_UseModel != "GLLGS") && (g_Kalm_UseModel != "GLL"))
    {
      print ("Error: Incorrect declarations file?");
      exit(1);
    }

  dTime= timer();
  InitGibStGa(g_InitVP, &mYt, &mDMY, &mInter, &dRho, &dSEta, &dSEps, 
              &dDelta, &dAlpha, &dNu, &vz, &vMu);
  iT= columns(mYt);

  // Set a vector with the interest rate differential, for usage in UIP
  vdR= zeros(mYt);
  if (g_Kalm_UseUIP == 1)
    vdR= (mInter[0][]-mInter[1][])/360;
  vdR~= vdR[iT-1];
   
  nRot= sizerc(g_Flex_nMH);
  nReps= g_Flex_nMH[nRot-1];
  vHist= g_Kalm_vHist;
  if (vHist == 0)
    vHist= range(4173, 4695);
  nHist= sizerc(vHist);
  nPeriods= iT-min(vHist)+1;

  sSimbase= sprint(g_OutDir, "/", g_VersFile);
  sSimFile= sprint(sSimbase, ".fmt");
  sOutFile= sprint(sSimbase, "gt.out");
  sDRetFile= sprint(sSimbase, "dret.fmt");
    
  println ("Writing output to ", sSimFile, ", ", sOutFile, " and ", sDRetFile);
    
  InitOutFile(sOutFile, g_Vers, g_VarNames,
              dRho|dSEta|dSEps|dDelta|dAlpha, nReps, g_Flex_nSkip,
              g_Flex_nBurnin[nRot-1]);
    
  vS= range(g_Kalm_vSRegion[0], g_Kalm_vSRegion[1], g_Kalm_vSRegion[2]);
  [mPMV, mDens]=
    RunChain(sSimFile, dRho, dSEta, dSEps, dDelta, dAlpha, 
             dNu, vz, vMu, mYt, vdR, g_Kalm_FracEst, nReps, g_Flex_nBurnin[nRot-1], 
             g_Flex_nSkip, vHist, vS);
  mDum= (zeros(3, 1)~(vHist|mPMV[:1][]))|(vS'~mDens);
  ir= savemat(sDRetFile, mDum);
    
  GiveResults(sOutFile, sSimFile, nReps, g_Flex_nBurnin[nRot-1], 20, dTime);

  mDMY~= <1; 1; 2000>;
  mYt~= M_NAN;

  // Draw a graph with predictive density
  DrawTMatrix(0, mPMV[0][], "E(s)", mDMY[][vHist], 0, 0);
  DrawTMatrix(1, mPMV[1][], "s(s)", mDMY[][vHist], 0, 0);

  mDum= mYt[][vHist]|(mPMV[0][]+mPMV[1][].*<-1; 0; 1>);
  DrawTMatrix(2, mDum[:2][], {"Y", "-+s(s)", "E(s)"}, mDMY[][vHist], 0, 0);
  DrawTMatrix(2, mDum[3][], "", mDMY[][vHist], 0, 0, 0, 3);

  vIndex= range(0, nHist-1, max(nHist/60, 1));
  DrawXMatrix(3, mDens[][vIndex]', "p(s)", vS, "");
  SaveDrawWindow(sSimbase~"mv.plb");
  CloseDrawWindow();
}

/*
**
**  Function RunChain(const sSimFile, const vP, const nRepTh)
**
**  Purpose:
**    Run the chain of the Gibbs sampler. 
**
**  Inputs:
**    sSimFile    string, name of simulation file
**    vP          vector, starting point of the chain, containing the 
**                free parameters from the sequence Rho, S2Eta, MuH,
**                Phi, S2Xi
**    nRepTh      scalar, number of repetitions of Theta
**    nBurnin     scalar, number of skipped repetitions
**
**  Outputs:
**    written in sSimFile
**    mPMV        2 x nPeriods matrix with mean predicted mean and mean
**                standard deviation
**    mDens       nS x nPeriods matrix with predictive density estimate
**
*/
RunChain(const sSimFile, dRho, dSEta, dSEps, dDelta, 
         dAlpha, dNu, vz, vMu, const mYt, const vdR, const iTEst, 
         const nRepTh, const nBurnin, 
         const nSkip, const vHist, const vS)
{
  decl
    fhSim, nChain, nS, nHist, nPeriods, i, iT, ve, dPars, dS2,
    ir, sum_vP, mOut, iOut, vv2, mDens, miDens, 
    mPMV, mMSdS, iPMV, vSd, mYRt;

  // Introduce the interest rate differential
  iT= columns(mYt);
  mYRt= mYt + vdR[:iT-1];

  /* Save the sampled parameters in the sSimFile */
  fhSim= fopen(sSimFile, "wbf");

  /* Run the chain */
  println ("Running the chain...");
  TrackTime({"Mu", "Rho", "S2Eta", "GARCH", "Z", "Nu", "DensPred", 
             "Write"});

  nChain= nRepTh+nBurnin;
  sum_vP= zeros(sizerc(g_FreePars), 1);
  iOut= 0;

  nS= columns(vS);
  nHist= columns(vHist);
  mDens= zeros(nS, nHist);

  // There are iT+1 possible predictions to be made.
  nPeriods= iT-min(vHist)+1;
  mOut= zeros(s_GS_nOut, sizerc(g_FreePars));
  mPMV= 0;
  iPMV= 0;
  for (i= 0; i < nChain; ++i)
    {
    if (imod(i, g_Flex_InfoRep) .== 0)
      {
        info(i, nChain);
        if (s_GS_Garch_DrawnAcc[0] > 0)
          println("Garch accepted, drawn, fraction: ", "%5.0i", 
                  s_GS_Garch_DrawnAcc[1], "%5.0i", s_GS_Garch_DrawnAcc[0], "%6.2f",
                  s_GS_Garch_DrawnAcc[1]/s_GS_Garch_DrawnAcc[0]);
        if (s_GS_Nu_DrawnAcc[0] > 0)
            println("Nu accepted, drawn, fraction:    ", "%5.0i", 
                    s_GS_Nu_DrawnAcc[1], "%5.0i", s_GS_Nu_DrawnAcc[0], "%6.2f",
                    s_GS_Nu_DrawnAcc[1]/s_GS_Nu_DrawnAcc[0]);
        TrackReport();
        print ("Mean parameters: ", sum_vP'/i);
      }

    TrackTime(0);
    vMu= SampleMu(mYRt, dRho, dSEta, dSEps, dDelta, dAlpha, vz);

    TrackTime(1);
    if (!(g_FreePars != 0))
      SampleRho(&dRho, vMu[:iTEst], dSEta, g_Kalm_PriorRho);

    TrackTime(2);
    if (!(g_FreePars != 1))
      {
        ve= (vMu - dRho * lag0(vMu', 1)')[1:];
        dS2= SampleS2(ve[:iTEst], g_Kalm_PriorS2EtaAB);
        dSEta= sqrt(dS2);
      }

    TrackTime(3);
    vv2= sqr(mYRt - vMu);
    if (!(g_FreePars != 2) && !(g_FreePars != 3) && !(g_FreePars != 4))
      {
        if (i == 0)
          println ("Using GARCH sampling");
        ++s_GS_Garch_DrawnAcc[0];
        s_GS_Garch_DrawnAcc[1]+= 
          SampleGARCH(&dSEps, &dDelta, &dAlpha, vz[:iTEst], vv2[:iTEst], 
                      dSEps|dDelta|dAlpha, .5*s_GS_CandGARCH[][1],
                      g_Kalm_PriorS2EpsAB);
      }
    else if (!(g_FreePars != 2) && (g_FreePars != 3) && (g_FreePars != 4))
      {
        if (i == 0)
          println ("Using S2Eps sampling");
        ve= (mYRt - vMu);
        dS2= SampleS2(ve[:iTEst], g_Kalm_PriorS2EpsAB);
        dSEps= sqrt(dS2);
      }
    else if (i == 0)
      println ("Using neither GARCH nor S2Eps sampling");

    if (!(g_FreePars != 5))
      {
        TrackTime(4);
        SampleZ(&vz, dSEps, dDelta, dAlpha, dNu, vv2);

        TrackTime(5);
        ++s_GS_Nu_DrawnAcc[0];
        s_GS_Nu_DrawnAcc[1]+= 
          SampleNu(&dNu, vz[:iTEst], g_Kalm_CandNuRW);
      }

    TrackTime(-1);

    dPars= dRho|dSEta|dSEps|dDelta|dAlpha|dNu;
    sum_vP = sum_vP + dPars[g_FreePars];

    if ((i >= nBurnin) && (imod(i, nSkip+1) == 0))
      {
        TrackTime(6);
        // Predict future volatility
        mMSdS= SampPred(dRho, dSEta, dSEps, dDelta, dAlpha, dNu, 
                        vz, mYRt, nPeriods);

        mOut[iOut][]= dPars[g_FreePars]';
        ++iOut;

        mMSdS= mMSdS[][vHist-min(vHist)];

        // Take out the interest rate differential again, to get a
        // prediction of change in exchange rate.
        mMSdS[0][]-= vdR[vHist];

        mPMV+= mMSdS;

        // Calculate the predictive density at the grid vS
        vSd= mMSdS[1][];
        if (dNu == 0)
          miDens= densn((vS'-mMSdS[0][])./vSd)./vSd;
        else
          {
            vSd= vSd .* sqrt((dNu-2)/dNu);
            miDens= denst((vS'-mMSdS[0][])./vSd, dNu)./vSd;
          }
        mDens += miDens;
        ++iPMV;

        if (isnan(miDens))
          {
            print("Warning: Missing density in RunChain, dNu= ", dNu, 
                  limits(miDens'), vS);
            if (isnan(mMSdS))
              print("mMSdS: ", mMSdS, limits(mMSdS'));
            exit(1);
          }
        TrackTime(-1);
      }

    if (((iOut == s_GS_nOut) || (i == nChain-1)) && (iOut > 0))
      {
        TrackTime(7);
        ir= fwrite(fhSim, mOut[:iOut-1][]);
        iOut= 0;
        TrackTime(-1);
      }
    }
  fhSim= fclose(fhSim);

  mPMV /= iPMV;
  mDens /= iPMV;

  println("Mean of parameters sampled: ", sum_vP/nChain);

  return {mPMV, mDens};
}

/*
**
**  SampPred(const dRho, const dSEta, const dSEps, const dDelta, 
**           const dAlpha, const dNu, const vz, const mYt, const vdR, const nPeriods);
**
**  Purpose:
**    Calculate a predicted mean and variance of the observation,
**    and save it in a separate file for use in calculating hedge
**    ratios
**
**  Inputs:
**    dRho        Value of AR parameter
**    dSEta       SDev in state evolution
**    dSEps       Unconditional sdev in observation equation
**    vGamma      3 x 1 vector with GARCH parameters
**    mYt         Row vector of data, possibly adjusted for the UIP
**    nPeriods    Number of periods into the future to be used in
**                calculations
**
**  Outputs:
**    Return value      2 x nPeriods matrix with predicted mean and sdev of
**                      observation
**
**  Remark:
**    The volatility of the future values is predicted as the
**    predicted variance of the state plus the variance of the errorterm,
**    being S2Eps for the t-density. This is the variance
**    CONDITIONAL on vz and Nu. What I should do is to iterate over this
**    routine, generating new z's and nu's, for each vector of parameters, to
**    integrate out the conditioning variables. However, as I calculate the
**    variance prediction for a large set of parameter vectors already, I
**    assume it averages out as it is.
**
*/
SampPred(const dRho, const dSEta, const dSEps, const dDelta, 
         const dAlpha, const dNu, const vz, const mYt, const nPeriods)
{
  decl mSigma, mStateH, mPredH, vPredMean, vPredSd, iT, iOld;

  iT= columns(mYt);
  mSigma= 1e6|0;
  if (dRho < 1)
    mSigma= dSEta*dSEta/(1-dRho*dRho)|0;
  mStateH= SsfMomentEstPred_GarchSt(&mPredH, mYt, dRho, dSEta,
                                    dSEps, dDelta, dAlpha, vz);

  // Predicted mean equals predicted state
  vPredMean= mStateH[1][0];

  // Predicted volatility equals predicted volatility of state plus
  // predicted volatility from t-density. Note that the t-density is
  // transformed to have variance 1
  vPredSd= sqrt(mStateH[0][0] + dSEps*dSEps * mStateH[2][0]);
  if (nPeriods > 1)
    { 
      // Predictions are made not only for time T+1, but also T, .. T-
      // nPeriods+2, or in Ox: Y[iT-nPeriods+1].
      vPredMean= mPredH[1][iT-nPeriods+1:]~vPredMean;

      // V(Y) is the predicted volatility of the state plus the volatility 
      // of the student t disturbance * h(t)
      vPredSd= sqrt(mPredH[2][iT-nPeriods+1:] + 
                    dSEps * dSEps * mPredH[4][iT-nPeriods+1:])~vPredSd;
    }

  return vPredMean|vPredSd;
}

/*
**  InitOutFile
**
**  Purpose:
**    Do the general initializations
*/
InitOutFile(const sOutFile, const Vers, const VarNames, 
            const vPInit, const nRepTh, const nSkip, const nBurnin)
{
    decl fh, i;

    fh= fopen(sOutFile, "w");    
    fprintln(fh, "Gibbs sampling on GLL-GARCH Stud model");
    fprintln(fh, " ");
    fprintln(fh, "Version ");
    fprintln(fh, "  ", Vers);
    fprintln(fh, "Free parameters ");
    for (i= 0; i < rows(VarNames); ++i)
      fprint(fh, "  ", VarNames[i]);
    fprintln(fh, " ");
    fprintln(fh, "Length of burn-in period: ");
    fprintln(fh, "%12i", nBurnin);
    fprintln(fh, "Length of sample: ");
    fprintln(fh, "%12i", nRepTh);
    fprintln(fh, "Saving one parameter vector out of: ");
    fprintln(fh, "%12i", nSkip);
    fprint(fh, "Initial Mu: ");
    fprintln(fh, "%12.4f", vPInit');
    fprintln(fh, "Date of run: ");
    fprintln(fh, "%12s", date());
    fprintln(fh, "Time of run: ");
    fprintln(fh, "%12s", time());

    fh= fclose(fh);
}

/*
**
**  Procedure GiveResults
**
**  Purpose:
**    Give the results of this rotation
**
**  Inputs:
**    _Mix_FnEval Global, count of number of function evaluations
**    Mu, AutoCov, nLag, sOutFile, dTime
**
*/
GiveResults(const sOutFile, const sSimFile, const nRepTh, const nBurnin, 
            const nLag, const dTime)
{
    decl fh, nDim, Theta, i, fmt;
    
    fh= fopen(sOutFile, "a");    
    fprintln(fh, "-----------------------------------------------------");
    fprintln(fh, "Results of Gibbs sampling on GLL-GARCH Stud model:");
    
    Theta= loadmat(sSimFile);
    nDim = columns(Theta);
    fprint(fh, "Mean of sampled Theta's: ");
    fprintln(fh, "%12.4f", meanc(Theta));
    fprint(fh, "Covariance of sampled Theta's: ");
    fprintln(fh, "%12.6f", variance(Theta));
    fprint(fh, "Autocorrelations: ");
    fmt= new array[nDim+1];
    fmt[0]= "  AC %2.0f: ";
    for (i= 1; i <= nDim; ++i)
      fmt[i]= "%13.5f";

    fprintln(fh, "%cf", fmt, range(0,nLag)'~acf(Theta, nLag));
    fprintln(fh, "Garch accepted, drawn, fraction: ", 
                s_GS_Garch_DrawnAcc[1], " ", s_GS_Garch_DrawnAcc[0], " ",
                s_GS_Garch_DrawnAcc[1]/s_GS_Garch_DrawnAcc[0]);
    fprintln(fh, "Nu accepted, drawn, fraction: ", 
                s_GS_Nu_DrawnAcc[1], " ", s_GS_Nu_DrawnAcc[0], " ",
                s_GS_Nu_DrawnAcc[1]/s_GS_Nu_DrawnAcc[0]);
    fprintln(fh, "Ended at time: ");
    fprintln(fh, "%12s", time());

    fprintln(fh, "Time elapsed:\n ", "%12s", timespan(dTime));

    fh= fclose(fh);
}


/*
**
**  InitGibStGa(const vInitVP, const amYt, const amDMY, const amInter, 
**              const adRho, const adSEta, 
**              const adSEps, const adDelta, const adAlpha, const adNu,
**              const avz, const avMu)
**
**  Purpose:
**    Initialize the static variables in this module, using settings in
**    simox.dec
**
**  Inputs:
**    InitVP      Initial values for vP. Might have been
**                initialized in Kalman.DEC
**    amY         Address to matrix with data
**
**  Outputs:
**    amY         Row vector of data
**
*/
InitGibStGa(const vInitVP, const amYt, const amDMY, const amInter, const adRho, const adSEta, 
            const adSEps, const adDelta, const adAlpha, const adNu, const avz, const avMu)
{
  decl 
    dLnPdf, vP, vPG, vS, ir, iT, mPhi, mOmega, mSigma, vv2;

  SetSeed(g_Seed);

  println("Gibbs sampling over Local level-GARCH Student t model");
  println("Internally parameters Rho, SEta, SEps, Delta, Alpha, dNu are used");
  vP= vInitVP;     // Rho, sEta, sEps, Delta, Alpha, dNu
  PdfInit(&vP, &iT);

  // Look for starting values 
  GetData(amYt, amDMY, amInter);

  println("\nOptimizing the local level-GARCH model");
  vPG= vP[g_FreePars];
  if (!(g_FreePars != 5))
    vPG= vPG[:sizerc(vPG)-2];
  println("Starting values Rho, s(Eta), s(eps), Delta and Alpha used: ", vPG');
  if (g_Optim)
    {
      ir= MaxBFGS(LnPdf_Loc, &vPG, &dLnPdf, 0, TRUE);
      println("\n", MaxConvergenceMsg(ir),
              " using numerical derivatives",
              "\nLog-likelihood = ", 
              "%.8g", double(dLnPdf*iT),
                      "; n = ", iT);
    }
  vS= MaxStdErr(LnPdf_Loc, vPG, iT);
  print("Parameters with standard errors:",
        "%cf", {"%12.5g", "  (%7.5f)"}, 
        vPG ~ vS);

  // Candidate density for GARCH parameters 
  s_GS_CandGARCH= M_NAN;
  if (!(g_FreePars != 3))
    {
      // Check for restrictions on rho and sEps
      ir= !(g_FreePars != 0) + !(g_FreePars != 1);
      s_GS_CandGARCH= vPG[ir:ir+2]~vS[ir:ir+2];
    }

  if (!(g_FreePars != 5))
    vPG= vPG|vP[sizerc(vP)-1];
  vP[g_FreePars]= vPG;
  adRho[0]= vP[0];
  adSEta[0]= vP[1];
  adSEps[0]= vP[2];
  adDelta[0]= vP[3];
  adAlpha[0]= vP[4];
  adNu[0]= vP[5];

  // Calculate the smoothed mean of the ll model
  mPhi= adRho[0]|1;
  mOmega= diag(adSEta[0]*adSEta[0]~adSEps[0]*adSEps[0]);

  /* Choose a diffuse initial state */
  mSigma=<-1; 0>;
  if (adRho[0] < 1)
    mSigma= adSEta[0]*adSEta[0]/(1-adRho[0]*adRho[0])|0;
  avMu[0]= SsfCondDens(ST_SMO, amYt[0], mPhi, mOmega, mSigma)[0][];
  avz[0]= ones(amYt[0]);
  if (!(g_FreePars != 5))
    {
      vv2= sqr(amYt[0]-avMu[0]);
      SampleZ(avz, adSEps[0], adDelta[0], adAlpha[0], adNu[0], vv2);
    }
}

/*
**  MaxStdErr(const fFunc, const vP, nData)
**
**  Purpose:
**    Calculate the numerical standard errors
*/
MaxStdErr(const fFunc, const vP, const nData)
{
  decl covar, invcov, result;

  result= Num2Derivative(fFunc, vP, &covar);
  if (!result)
    {
      println("Covar() failed in numerical second derivatives");
      return zeros(vP);
    }
  invcov= invertgen(-covar, 30);
  return sqrt(diagonal(invcov) /nData)';
}

/*
**
**  Function LnPdf_Loc
**
**  Purpose:
**    Enlarge the parameter vector with the elements
**    that are not free. Do not change likelihood value (so return
**    the MEAN loglikelihood.
**
*/
LnPdf_Loc(const vP, const adLnPdf, const avScore, const amHessian)
{
  decl vP_Loc, Succ;

  vP_Loc= g_InitVP;
  vP_Loc[g_FreePars]= vP;

  Succ= ( // (vP >= g_Bounds[g_FreePars][0]) && 
          // (vP <= g_Bounds[g_FreePars][1]) && 
         LnPdf_Shell(vP_Loc, adLnPdf, avScore, amHessian));

  return Succ;
}
