/*
**  Program ForeRisk.Ox
**
**  Purpose:
**    Calculate the forcasting capabilities of the hedging models
**    This program uses HPD, the two-sided HPD interval. 
**
**  Version:
**    1     Based on varisk.ox
**    2     For Clements
**
**  Author:
**    Charles Bos
**
**  Date:
**    28/7/2000
*/
#include <oxstd.h>
#include "include/gnudraw_jae.h"
#include "include/info.ox"
#include "include/printmat.ox"
#include "include/size.ox"
#include "include/inchpd.ox"

#include "simox.dec"

// Function declarations
LoadDens(const amDens, const avHist, const avX, const amPMV, const anHist, 
         const anPeriods, const sDRetFile);
TestUncondCov(const mInRegion, const vp);
TestIndCov(const mInRegion, const vp);

main()
{
  decl bRepInverse, bPlotDouble, sSimbase, sOutFile, fh, 
       mYt, vYtPer, mInter, mDMY, nPeriods, nA, nM, nHist, ir,
       vHist, vS, vIHF, mDens, 
       mPMV, i, j, iInd, aPrFmt, arLabs, acLabs, asKeys, vAlpha,
       mR, vXL, vXU, mXL, mXU, vXB, mXB, vCov, mCov, 
       mInHPD, mUncond, mIndep, mUncondAllD, mIndepAllD, 
       mUncondAllS, mIndepAllS, aMatNames, asModel, vOrder, dTime;

  dTime= timer();
  vAlpha= <.95; .975; .99>;
  bRepInverse= TRUE;    // Report 1-alpha in tables?
  bPlotDouble= FALSE;   // Plot double-sided risk?

  PdfInit(&g_InitVP, &g_nData);
  GetData(&mYt, &mDMY, &mInter);

  mYt~= M_NAN;
  mDMY= mDMY~<1; 1; 2000>;

  aMatNames= {"excl/madu/ma", "excl/mbdu/mb", "excl/mcdu/mc",
              "excl/mddu/md", "excl/medu/me", "excl/mfdu/mf", 
              "excl/mgdu/mg"};
  asModel= {"WN", "LL", "GLL", "GLLGA", "GLLSV", "GLLSt",
            "GLLGASt"};
  vOrder= <0, 1, 2, 3, 4, 5, 6>;
  nM= sizeof(aMatNames);
  nA= sizerc(vAlpha);
  mUncondAllD= mIndepAllD= mUncondAllS= mIndepAllS= <>;
  for (j= 0; j < nM; ++j)
    {
      iInd= vOrder[j];
      sSimbase= aMatNames[iInd];
      println("Starting with model ", asModel[iInd]);

      LoadDens(&mDens, &vHist, &vS, &mPMV, &nHist, &nPeriods, 
               sprint(sSimbase, "dret.fmt"));
      vIHF= mInter[][vHist-1]/360;
      vYtPer= mYt[0][vHist];

      mXL= mXU= mXB= mCov= <>;
      for (i= 0; i < nHist; ++i)
        {
          ir= CalcHPD(&mR, &vXL, &vXU, &vCov, vS', mDens[][i]', 
                      vAlpha, TRUE);
          mXL~= vXL;
          mXU~= vXU;
          mCov~= vCov;

          ir= CalcHPDOnesided(&mR, &vXB, vS', mDens[][i]', 
                              vAlpha, TRUE, TRUE);
          mXB~= vXB;
        }
      if (bPlotDouble)
        {
          DrawT(j, mXL, mDMY[][vHist], 0, 0);
          DrawT(j, mXU|vYtPer, mDMY[][vHist], 0, 0);
        }
      else
        DrawT(j, mXB|vYtPer, mDMY[][vHist], 0, 0);

      // Double sided testing
      mInHPD= ((vYtPer .>= mXL) .&& (vYtPer .<= mXU));
      mUncond= TestUncondCov(mInHPD, vAlpha);
      mIndep= TestIndCov(mInHPD, vAlpha);

      mUncondAllD|= vecr(mUncond[][<0,2>])';
      mIndepAllD|= vecr(mIndep)';

      // Single sided testing
      mInHPD= (vYtPer .>= mXB);
      mUncond= TestUncondCov(mInHPD, vAlpha);
      mIndep= TestIndCov(mInHPD, vAlpha);

      mUncondAllS|= vecr(mUncond[][<0,2>])';
      mIndepAllS|= vecr(mIndep)';
    }
  DrawAdjust(ADJ_ALIGN, 2);
  SaveDrawWindow("excl/alfr.plb");
  ShowDrawWindow();

  if (bRepInverse)
    {
      vAlpha= 1-vAlpha;
      mUncondAllD[][2*range(0, nA-1)]= 
        1-mUncondAllD[][2*range(0, nA-1)];
      mUncondAllS[][2*range(0, nA-1)]= 
        1-mUncondAllS[][2*range(0, nA-1)];
    }

  aPrFmt= new array [nM*2];
  for (i= 0; i < nM; ++i)
    {
      aPrFmt[2*i]= "%6.3f";
      aPrFmt[2*i+1]= "%6.2f";
    }
  acLabs= new array [2*nA];
  for (i= 0; i < nA; ++i)
    {
      acLabs[2*i]= sprint("%5.3f", vAlpha[i]);
      acLabs[2*i+1]= "";
    }

  fh= fopen("excl/alfr.out", "l");
  println("Double sided coverage tests:");
  PrintMatrix(0, "Unconditional", aPrFmt, acLabs, asModel,
              mUncondAllD, FALSE);
  println("\\hline");
  PrintMatrix(0, "Independence", aPrFmt, acLabs, asModel,
              mIndepAllD, FALSE);
  println("\\hline");

  println("Single sided coverage tests:");
  println("Single sided:");
  PrintMatrix(0, "Unconditional", aPrFmt, acLabs, asModel,
              mUncondAllS, FALSE);
  println("\\hline");
  PrintMatrix(0, "Independence", aPrFmt, acLabs, asModel,
              mIndepAllS, FALSE);
  println("\\hline");

  println("Time elapsed:\n ", "%10s", timespan(dTime));
}

/*
**  LoadDens
**
**  Outputs:
**    vS        nS x 1 column vector of returns at which density is evaluated
**    mDens     nS x nPeriods matrix of densities of S at the time periods
**    vHist     1 x nHist vector of indices of history
**    mPMV      2 x nHist matrix of mean and variances
*/
LoadDens(const amDens, const avHist, const avS, const amPMV, const anHist, 
         const anPeriods, const sDRetFile)
{
  amDens[0]= loadmat(sDRetFile);
  avHist[0]= amDens[0][0][1:];
  amPMV[0]= amDens[0][1:2][1:];
  avS[0]= amDens[0][3:][0];
  amDens[0]= amDens[0][3:][1:];
  anHist[0]= columns(avHist[0]);
  anPeriods[0]= columns(amDens[0]);
}

/*
**  TestUncondCov(const mInRegion, const vp)
**
**  Purpose:
**    Calculate the unconditional probability test of Christoffersen,
**    for a coverage of Alpha
**
**  Inputs:
**    mInRegion   nA x n vector with 0/1 values indicating if observation
**                lies within region
**    vp          nA x 1 vector with coverage probabilities
**
**  Output:
**    mRet        nA x 3 vector with observed probabilities, LR statistic
**                and corresponding p-value
*/
TestUncondCov(const mInRegion, const vp)
{
  decl iN0, iN1, vpHat, dLRuc, vpLRuc;
  
  iN1= sumr(mInRegion);
  iN0= sumr(1-mInRegion);
  vpHat= iN1./(iN0+iN1);
  
  // Unconditional coverage
  dLRuc= -2*((iN0.*log(1-vp)+iN1.*log(vp)) - 
             (iN0.*log(1-vpHat) + iN1 .* log(vpHat)));
  vpLRuc= 1-probchi(dLRuc, 1);
  
  return vpHat~dLRuc~vpLRuc;
}

/*
**  TestIndCov(const mInRegion, const vp)
**
**  Purpose:
**	Calculate the independence test of Christoffersen
**
**  Inputs:
**    mInRegion   nA x n matrix with 0/1 values indicating if observation
**                lies within region
**    vp          nA x 1 vector of coverage probabilities
**
**  Output:
**    vRet        nA x 2 matrix with LR statistic of independence test
**                and corresponding p-values
*/
TestIndCov(const mInRegion, const vp)
{
  decl iN, iN00, iN01, iN10, iN11, 
       miN1, mPi1, dL1, dPi2, dL2, dLRind, vpLRind;
  
  iN= columns(mInRegion);
  iN00= sumr((mInRegion[][1:] .== 0) .&& (mInRegion[][:iN-2] .== 0));
  iN01= sumr((mInRegion[][1:] .== 1) .&& (mInRegion[][:iN-2] .== 0));
  iN10= sumr((mInRegion[][1:] .== 0) .&& (mInRegion[][:iN-2] .== 1));
  iN11= sumr((mInRegion[][1:] .== 1) .&& (mInRegion[][:iN-2] .== 1));

  miN1= (iN00~iN01~iN10~iN11);
  mPi1= ((iN00./(iN00+iN01)~iN01./(iN00+iN01))~(iN10./(iN10+iN11)~iN11./(iN10+iN11)));
  dL1= sumr(miN1.*log(mPi1));
  
  dPi2= (iN01+iN11)./iN;
  dL2= (iN00+iN10).*log(1-dPi2)+(iN01+iN11).*log(dPi2);

  // Independence test
  dLRind= -2*(dL2-dL1);
  vpLRind= 1-probchi(dLRind, 1);
  vpLRind= isdotnan(dLRind) .? M_NAN .: vpLRind;

  return dLRind~vpLRind;
}
