/*
**
**  Program CalcDRet.Ox
**
**  Purpose:
**    Calculate the density of the returns on exchange rate, meant to
**    be used instead of a sample from this density.
**
**  Author:
**    Charles Bos
**
**  Date:
**    2/5/2000
**
**  Version:
**    1, based on CalcSamp.ox
**    3, including UIP
**
*/
#include <oxstd.h>
#include "include/info.ox"
#include "include/setseed.ox"
#include "include/size.ox"
#include "include/gnudraw_jae.h"
#include "simox.dec"

// Function declarations
CalcDens(const sSimbase, const mY, const mdR, 
         const nPeriods, const vS, dFrac);

main()
{
  decl dFrac, sSimbase, sDRetFile, i, nPeriods, nHist, mYt, mInter, mdR, iT, 
       vS, mX, mDMY, mPMV, mDens, vHist, asKeys, ir, vIndex, dTime;

  dTime= timer();
  dFrac= 1;

  SetSeed(g_Seed);
  PdfInit(&g_InitVP, &g_nData);
  GetData(&mYt, &mDMY, &mInter);

  iT= columns(mYt);
  sSimbase= sprint(g_OutDir, "/", g_VersFile);
  sDRetFile= sprint(sSimbase, "dret.fmt");
  mDMY= mDMY~<1; 1; 2000>;
  mYt~= M_NAN;
  vHist= g_Kalm_vHist;
  if (vHist == 0)
    vHist= range(4173, 4695);
  nHist= columns(vHist);
  nPeriods= iT-min(vHist)+1;

  println("CalcDRet");
  println("--------");
  println("Calculating the Kalman Filter on the basis of ");
  println("a fraction ", dFrac, " of the sampled points.");
  println("Calculating the density of ", nPeriods, " into the future at");
  print("timepoints ", "%cf", {"%02.0f/", "%02.0f/", "%4.0f"}, mDMY[][vHist]');
  println("A dataset of ", iT, " observations is used");
  mdR= zeros(1, iT+1);
  if (g_Kalm_UseUIP == 1)
    {
      println ("Using uncovered interest rate parity-equation...");
      mdR= (mInter[0][]-mInter[1][])/360;
      mdR~= mdR[][iT-1];
    }
  println ("Writing output to ", sSimbase);

  vS= range(g_Kalm_vSRegion[0], g_Kalm_vSRegion[1], g_Kalm_vSRegion[2]);

  [mPMV, mDens]= CalcDens(sSimbase, mYt, mdR, vHist, vS, dFrac);
  mX= (zeros(3, 1)~(vHist|mPMV[:1][]))|(vS'~mDens);
  ir= savemat(sDRetFile, mX);

  // Draw a graph with predictive density
  DrawTMatrix(0, mPMV[0][], "E(s)", mDMY[][vHist], 0, 0);
  DrawTMatrix(1, mPMV[1][], "s(s)", mDMY[][vHist], 0, 0);

  mX= mYt[][vHist]|(mPMV[0][]+mPMV[1][].*<-1; 0; 1>);
  DrawTMatrix(2, mX[:2][], {"Y", "-+s(s)", "E(s)"}, mDMY[][vHist], 0, 0);
  DrawTMatrix(2, mX[3][], "", mDMY[][vHist], 0, 0, 0, 3);

  vIndex= range(0, nHist-1, max(nHist/60, 1));
  DrawXMatrix(3, mDens[][vIndex]', "p(s)", vS, "");
  SaveDrawWindow(sSimbase~"mv.plb");
  CloseDrawWindow();

  println("Time elapsed:\n ", "%10s", timespan(dTime));
}

/*
**
**  CalcDens(const sSimbase, const mYt, const mdR, const vHist, const vS, 
**           dFrac)
**
**  Purpose:
**    Calculate the filter over the data set, and save the 
**    density of the one-step ahead prediction
**
**  Inputs:
**    sSimbase    Basis of simulation filename
**    mYt         1 x n vector with changes in exchange rate
**                (daily percentages)
**    mdR         1 x n vector with interest rate differential, in daily
**                percentages. Put to zero if UIP should not be used.
**    vHist       1 x nHist vector with indices of moments at which the
**                predictive density should be calculated
**    vS          1 x nS Grid for evaluation of density of return
**    dFrac       double, fraction of dataset to be used
**
**  Output:
**    Return value      3 x nPeriods matrix with mean predicted mean and
**                      mean predicted standard deviation, mean sample values.
**
*/
CalcDens(const sSimbase, const mYt, const mdR, const vHist, const vS, 
         dFrac)
{
    decl i, j, ir, sSimFile, mTheta, nRepTh, vP, iT,
         mYR, mPMV, mPMSdS, mDens, miDens, nS, nHist, nPeriods;
    
    sSimFile = sprint(sSimbase, ".fmt");   
    mTheta = loadmat(sSimFile);
    if (dFrac <= 1)
      dFrac *= rows(mTheta);
    mTheta= thinr(mTheta, dFrac);

    nRepTh= rows(mTheta);
    nS= columns(vS);
    nHist= columns(vHist);
    iT= columns(mYt);
    // There are iT+1 possible predictions to be made.
    nPeriods= iT-min(vHist)+1;
    mPMV= zeros(2, nHist);
    mDens= zeros(nS, nHist);

    // Include interest rate differential
    mYR= mYt + mdR[][:iT-1];

    for (i= 0; i < nRepTh; ++i)
      {
        if (imod(i, 50) .== 0)
          info(i, nRepTh);

        vP= g_InitVP;
        vP[g_FreePars]= mTheta[i][]';
        ir= CalcKalmanMVS(vP, mYR, nPeriods, &mPMSdS, 1);
        if (ir)
          {
            mPMSdS= mPMSdS[][vHist-min(vHist)];

            // Take out the interest rate differential again, to get a
            // prediction of change in exchange rate.
            mPMSdS[0][]-= mdR[][vHist];
            mPMV += mPMSdS[:1][];
            miDens= densn((vS'-mPMSdS[0][])./mPMSdS[1][])./mPMSdS[1][];
            mDens += miDens;
          }
        else
          println("Warning: Filtering dit not work, Theta=", vP');
      } /* Next i */
    info(nRepTh, nRepTh);

    mDens/= nRepTh;
    mPMV /= nRepTh;

    return {mPMV, mDens};
}

