################################################################################
# replicate results with R/Synth and R/MSCMT
# import results obtained with Matlab and Stata
################################################################################

################################################################################
# load packages and prepare the data
################################################################################

library(readstata13)
library(MSCMT)
library(Synth)

# dataset.dta has to be extracted from zip-file accessible via:
# http://onlinelibrary.wiley.com/doi/10.1111/ecoj.12235/abstract
mafia <- read.dta13("dataset.dta")                                              # data for package Synth
Mafia <- listFromLong(mafia,"reg","year","region")                              # data for package MSCMT


################################################################################
# setup original specification of Pinotti for use with package MSCMT
################################################################################
treated.unit   <- "NEW"
control.units  <- c(colnames(Mafia[[1]])[c(1:14,20)])
dependent      <- "gdppercap"
predictors     <- c("gdppercap","invrate","shvain","shvaag","shvams","shvanms",  
                    "shskill","density")
times.dep      <- matrix(c("1951","1960"),nrow=2,ncol=length(dependent))
colnames(times.dep) <- dependent                   
times.pred     <- matrix(c("1951","1960"),nrow=2,ncol=length(predictors))
colnames(times.pred) <- predictors
agg.fns        <- rep("mean",length(predictors))
names(agg.fns) <- predictors

spec_paper     <- list(data=Mafia,treated.unit=treated.unit,
                       control.units=control.units,times.pred=times.pred,
                       times.dep=times.dep,agg.fns=agg.fns)

################################################################################
# helper function for estimating main specification with package MSCMT
################################################################################
main <- function(spec=spec_paper,Options=mscmtOptions) {
	data          <- spec$data
	treated.unit  <- spec$treated.unit
	control.units <- spec$control.units
	times.dep     <- spec$times.dep
	times.pred    <- spec$times.pred
	agg.fns       <- spec$agg.fns
	
	res  <- do.call("mscmt",
					c(list(data,treated.unit,control.units,times.dep,times.pred,agg.fns),
						Options))
	list(main=res)
}	

################################################################################
# helper function for importing results from other software for package MSCMT
################################################################################
main_import <- function(spec=spec_paper,w,v=NULL,lb=0) {
	data          <- spec$data
	treated.unit  <- spec$treated.unit
	control.units <- spec$control.units
	times.dep     <- spec$times.dep
	times.pred    <- spec$times.pred
	agg.fns       <- spec$agg.fns

  # fix names
  if (is.null(names(w))||(length(w)==length(control.units)))
    names(w) <- control.units
  
  w <- w[w>lb]
      	
	res  <- do.call("mscmt",
					c(list(data,treated.unit,control.units,times.dep,times.pred,agg.fns),
						list(outer.optim="none",outer.opar=list(w=w,v=v))))

  # correct row names of v matrix
  rownames(res$v) <- res$agg.pred
  list(main=res)
}	


################################################################################
# solve the SCM task with package MSCMT
################################################################################
mscmtOptions <- list(outer.optim="DEoptC",
                     outer.opar=list(nG=10000,nP=1000,waitgen=1000))

res_narrow   <- main(spec_paper,mscmtOptions)


################################################################################
# import the results of other implementations
################################################################################

################################################################################
# EJ/Stata
################################################################################

# Pinotti's results as reported in EJ:
# weights_paper_w <- list(main= c("ABR"=0.624,"MOL"=0.376))

# Use results obtained with (modified) Stata code instead 
# (same as EJ results, but with more decimal places):                        
w_paper_w <- c(5.74e-14,9.41e-15,1.35e-13,6.56e-14,5.47e-12,1.86e-13,3.77e-13,
               1.74e-13,8.68e-14,2.80e-13,9.60e-14,4.84e-15,.6244443,.3755557,
               1.39e-13)
w_paper_v <- c(.9426886,.0006125,.0377382,.0018849,.0007506,.0088968,.0019189,
               .0055096)                

res_paper <- main_import(spec_paper,w=w_paper_w,v=w_paper_v)   

################################################################################
# Matlab
################################################################################
w_matlab_w <- c(3.4317009e-21, 2.9611612e-21, 8.5853981e-21, 1.4444506e-16, 
                1.8502409e-17, 3.64767e-20, 8.9469018e-21, 1.4624114e-20, 
                6.2387125e-21, 1.0643165e-20, 8.2225436e-21, 2.2541706e-21, 
                0.62489406, 0.37510594, 1.6466871e-19)
w_matlab_v <- c(1, 2513.2068, 260173.65, 5199.1084, 6910.1629, 10386.918, 
                14232.37, 2452.1541)
res_matlab <- main_import(spec_paper,w=w_matlab_w,v=w_matlab_v)

################################################################################
# R/Synth
################################################################################

# prepare dataprep object
dataprep.out <- dataprep(
  foo=mafia,
  predictors = c("gdppercap","invrate","shvain","shvaag","shvams","shvanms",  
                 "shskill","density"),
  predictors.op = "mean", 
  special.predictors = NULL,
  dependent = "gdppercap", 
  unit.variable = "reg",
  time.variable = "year",
  treatment.identifier = "NEW",
  controls.identifier = c("PIE", "VDA", "LOM", "TAA", "VEN", "FVG", "LIG", 
                          "EMR", "TOS", "UMB", "MAR", "LAZ", "ABR", "MOL", 
                          "SAR"),
  time.predictors.prior = 1951:1960,
  time.optimize.ssr = 1951:1960, 
  time.plot = 1951:1960,
  unit.names.variable = "region"
)

set.seed(1)
synth.out <- synth(dataprep.out)
w_synth_w <- as.numeric(synth.out$solution.w)
w_synth_v <- as.numeric(synth.out$solution.v)

res_synth <- main_import(spec_paper,w=w_synth_w,v=w_synth_v)


################################################################################
# save everything
################################################################################
save(res_paper,res_matlab,res_synth,res_narrow,file="Replication.RData")
