/* (C) Ole E. Barndorff-Nielsen and Neil Shephard (2000). Written for
"Econometric analysis of realised volatility and its use in estimating
Levy based non-Gaussian OU type stochastic volatility models,"
Nuffield College, Oxford.

This code uses the Ox code version of SsfPack. This is fully documented in:

Siem Jan Koopman, Neil Shephard and Jurgen A Doornik 1999 Statistical algorithms
for models in state space form using SsfPack 2.2 (with discussion),
Econometrics Journal 2, 107-160. 
*/

#include <oxstd.h>
#include <oxfloat.h>
#include <oxdraw.h>
//#include <bessel.h>
#import <maximize>
#include <arma.h>

#include "d:\program files\ox//lib//densest.ox"
#include "d:\program files\ox//lib//acffft.ox"
#include "e:\ox200 code\ssfpack//ssfpack.h"
#include "e:\ox200 code\ssfpack//ssfcombine.ox"

/* Simulate from a \Gamma(v,a) OU process. First routine sets up noise, second
computes OU and INTOU process */

decl igraph, igraph3;

Gamma_OU_process(const n, const v, const a, const lambda, const delta)
{
// need to check v,a,lambda,delta>0
// n is an integer
	 decl x = ranpoisson(1,n,v*delta*lambda);
	 decl i,mY = zeros(2,n);
	 decl noise = mY;
	 decl ru,rgam;
	 decl phi = exp(-lambda*delta);

	 mY[0][0] = rangamma(1,1,v,a);
	 mY[1][0] = 0.0;
	 for (i=1; i<n; i++)
	 {
	     mY[0][i] = phi*mY[0][i-1];
	     if (x[0][i] >0)
		 {
	         rgam     = -log(ranu(1,x[0][i]));
	         ru       = delta.*ranu(1,x[0][i]);
			 noise[0][i] = phi*sumr(rgam.*exp(lambda.*ru))/a;
			 noise[1][i] = sumr(rgam)/a;
			 mY[0][i] += phi*sumr(rgam.*exp(lambda.*ru))/a;
			 mY[1][i]  = sumr(rgam)/a;
		 }
	 }

	 return noise;
}

OU_int(const n, const lambda, const sstart, const Delta, const noise)
{
     decl i;
	 decl y = new matrix[2][n];
	 decl aneil = sstart;
	 decl aneil_new = aneil;
	 decl phi = exp(-Delta*lambda);
	 
	 for (i=0; i<n; i++)
	 {
	      aneil_new = phi*aneil + noise[0][i];
		  y[0][i] = (noise[1][i] - (aneil_new-aneil))/lambda;	 // INTOU process
		  y[1][i] = aneil_new;									 // OU process
		  aneil = aneil_new;
	 }

	 return y;
}

/* computes the moving average parameter for the actual volatility */

decl phi_star;

first_acf(const phi, const theta)	 // computes r_1 for ARMA(1,1)
{
   	decl autocov = armavar(phi~theta,1,1,1.0,2);
	return autocov[0][1]/autocov[0][0];
}

arma_lik(const vP, const adFunc, const avScore, const amHessian)
{
    decl phi = phi_star;
	decl theta = fabs(vP[0][0])/(1.0+fabs(vP[0][0]));

 	decl d = 0.5*sqr(1.0-phi)/(phi-1.0-log(phi));
	decl autocor = first_acf(phi,theta);
	decl ss = -100.0*fabs(d-autocor);
	adFunc[0] = ss;

//	print(phi~theta~d~autocor~ss);

	if (avScore)
	{
	}

	return 1;
}

solve_arma(const phi)				 // solves for MA parameter numerically //
{
    decl theta;
	phi_star = phi;

 	decl d = 0.5*sqr(1.0-phi)/(phi-1.0-log(phi));
	decl dfunc,vp = 0.3*ones(1,1);
    decl ir = MaxBFGS(arma_lik,&vp,&dfunc,0,TRUE);
	theta = fabs(vp)/(1.0+fabs(vp));


	return theta~d;
}

/* Sets up Linear SSF using notation of SsfPack.  Things which need working out our
MA root
Variance of noise for ARMA process
Variance of measurement error
*/

sigma_n_arma(const omega, const lambda, const Delta)
{
    decl phi = exp(-lambda*Delta);
    decl theta = solve_arma(phi)[0][0];
	decl sigma = 2.0*omega*(phi-1.0-log(phi))/sqr(lambda);
	decl sigma_eta = sigma/armavar(phi~theta,1,1,1.0,2)[0][0];

//	print(sigma~omega*sqr(Delta)~lambda);

	return phi~theta~sigma~sigma_eta;
}

measure_error_var(const psi, const omega, const lambda, const Delta, const M)
{
    decl phi_M = exp(-lambda.*(Delta/M));
	decl sigma_M = 2.0.*omega.*(phi_M-1.0-log(phi_M))./(lambda.*lambda);
	decl var_u = 2.0*M*(sumr(sigma_M) + sqr(columns(lambda)*psi*Delta/M));

	return phi_M~sigma_M~var_u;
}

setup_arma_sup(const psi, const omega, const lambda, const Delta, const M,
                  mPhi1,  mOmega1, mSigma1, mDelta1)
{
    decl i_sup = columns(lambda);

    decl i,j,k,x = sigma_n_arma(omega[0][0],lambda[0][0],Delta);
	for (i=1; i<i_sup; i++)
	    x = x|sigma_n_arma(omega[0][i],lambda[0][i],Delta);
		
	decl phi = x[][0], theta = x[][1], sigma = x[][2], sigma_eta = x[][3];

//	print(sumc(sigma));								

	x = measure_error_var(psi,omega,lambda,Delta,M);
//	print(sigma[0][0]'+x[0][2]);
	decl phi_M = x[0][0], sigma_M = x[0][1], var_u = x[0][columns(x)-1];

	decl mDelta,mDelta0 = zeros(2,1);
//	mDelta0[0][0] = Delta*psi[0][0]*(1.0-phi[0][0]);
	decl vAR = new matrix[i_sup][1];  vAR[][0]= phi[][0];
	decl vMA = new matrix[i_sup][1];  vMA[][0] = theta[][0];
		
	decl mPhi,mOmega,mSigma;
	decl mPhi0,mOmega0,mSigma0;
	
	GetSsfArma(vAR[0][],vMA[0][],sqrt(sigma_eta[0][0]),&mPhi,&mOmega,&mSigma);
//	mSigma[2][0] = Delta*psi[0][0];
//	mDelta = mDelta0;

	for (k=1; k<i_sup; k++)
	{
	     GetSsfArma(vAR[k][],vMA[k][],sqrt(sigma_eta[k][0]),&mPhi0,&mOmega0,&mSigma0);
//		 mSigma0[2][0] = Delta*psi[0][k];
		 mPhi = SsfCombine(mPhi,mPhi0,0);
		 mSigma = SsfCombine(mSigma,mSigma0,0);
		 mOmega = SsfCombineSym(mOmega,2*k,mOmega0,0);
//		 mDelta0[0][0] = Delta*psi[0][k]*(1.0-phi[k][0]);
//		 mDelta = mDelta|mDelta0;
	}
	
	mOmega[rows(mOmega)-1][rows(mOmega)-1] = var_u;
//	mDelta = mDelta|0.0;
    mDelta = zeros(2*i_sup+1,1);

//	if (i_sup==3)
//	   print(mPhi,mOmega,mSigma,mDelta);

	

	mPhi1[0] = mPhi;
	mOmega1[0] = mOmega;
	mSigma1[0] = mSigma;
	mDelta1[0] = mDelta;

//	print(mSigma);

	return;
	
}

setup_arma(const psi, const omega, const lambda, const Delta, const M,
                  mPhi1,  mOmega1, mSigma1, mDelta1)
{
    decl x = sigma_n_arma(omega,lambda,Delta);
		
	decl phi = x[0][0], theta = x[0][1], sigma = x[0][2], sigma_eta = x[0][3];

	x = measure_error_var(psi,omega,lambda,Delta,M);
//	print(sigma[0][0]'+x[0][2]);
	decl phi_M = x[0][0], sigma_M = x[0][1], var_u = x[0][2];

	decl mDelta = zeros(3,1);
	mDelta[0][0] = Delta*psi*(1.0-phi);
	decl vAR = zeros(1,1);  vAR[][0]= phi;
	decl vMA = zeros(1,1);  vMA[][0] = theta;
		
	decl mPhi,mOmega,mSigma;
	decl mPhi0,mOmega0,mSigma0;

	GetSsfArma(vAR,vMA,sqrt(sigma_eta),&mPhi,&mOmega,&mSigma);	
	
	mSigma[2][0] = Delta*psi;
	mOmega[2][2] = var_u;

	mPhi1[0] = mPhi;
	mOmega1[0] = mOmega;
	mSigma1[0] = mSigma;
	mDelta1[0] = mDelta;

//	print(mSigma);

	return;
	
}

smooth_setup_arma(const yobs, const psi, const omega, const lambda, const Delta, const M)
{					// computes one step ahead predictions and smoother

    decl mPhi,mOmega,mSigma,mDelta;
    setup_arma_sup(psi,omega,lambda,Delta,M,&mPhi,&mOmega,&mSigma,&mDelta);
	
	decl mStSmo,mStSmo1;
	decl mSmo = SsfMomentEst(ST_SMO,&mStSmo,yobs,mPhi,mOmega,mSigma,mDelta);
	     mSmo = SsfMomentEst(ST_PRED,&mStSmo1,yobs,mPhi,mOmega,mSigma,mDelta);

/*	Draw(0,mStSmo[3][],0,1);
	Draw(1,mStSmo1[3][],0,1);
	Draw(2,(mStSmo[3][]|mStSmo1[3][])[][5:columns(mStSmo)-10],0,1);
	SetDrawWindow("variance");
	
	ShowDrawWindow();
	print(mOmega);
*/	return mStSmo[2][]|mStSmo1[2][];	
}

log_densn(const x, const mean1, const var1)
{
    return log(densn((x-mean1)./sqrt(var1))) - 0.5.*log(var1);
}

log_densig(const x, const y)
{
	decl mu = meanr(y);
	decl lambda = 1.0/meanr((1.0./y) - (1.0/mu));
  	decl con1 = 0.5.*log(lambda./(M_2PI.*x.*x.*x));

	return con1 - lambda.*(x-mu).*(x-mu)./(2.0.*sqr(mu).*x);
}

compare_ABDL_BNS(const yobs, const psi, const omega, const lambda, const Delta, const M)
{

    decl mPhi,mOmega,mSigma,mDelta;
    setup_arma_sup(psi,omega,lambda,Delta,M,&mPhi,&mOmega,&mSigma,&mDelta);
	
	decl mStSmo,mStSmo1;
	decl mSmo = SsfMomentEst(ST_SMO,&mStSmo,yobs,mPhi,mOmega,mSigma,mDelta);
	     mSmo = SsfMomentEst(ST_PRED,&mStSmo1,yobs,mPhi,mOmega,mSigma,mDelta);
   return (mStSmo[3][columns(yobs)/2])~(mStSmo1[3][columns(yobs)/2])~mOmega[2][2];
}

decl yobs_o, Delta_o, M_o, i_sup_o, y_data_o;
decl acf_store,acf_res;
like_QV(const yobs, const psi, const omega, const lambda, const Delta, const M)
{                   // computes quasi-likelihood
    decl mPhi,mOmega,mSigma,mDelta,dLogLik,dVar;
    setup_arma_sup(psi,omega,lambda,Delta,M,&mPhi,&mOmega,&mSigma,&mDelta);
	SsfLik(&dLogLik,&dVar,yobs_o-sumr(psi),mPhi,mOmega,mSigma,mDelta);

	decl mKF,mD,mPred,mState,n,x_std,acf1;
	decl smooth_vol,sm_mean,sm_var,x1,x2,x,resx;
	decl vol_n; 
	
	if (igraph == igraph3)
	{
	   mKF = KalmanFil(yobs_o-sumr(psi),mPhi,mOmega,mSigma,mDelta);
	n = columns(y_data_o);
	x_std = mKF[0][].*sqrt(mKF[rows(mKF)-1][]);
	acf1 = acf(x_std',20);
	print(columns(yobs)*sumc(acf1[1:][].*acf1[1:][])~20~M_o);
	   mD = SsfCondDens(ST_SMO,yobs_o-sumr(psi),mPhi,mOmega,mSigma,mDelta);
	   mState = SsfMomentEst(ST_PRED,&mPred,yobs_o-sumr(psi),mPhi,mOmega,mSigma,mDelta);

//	   vol_n = log(sumr(psi)+mD[2*i_sup_o][]);
//       x1 = DensEst(vol_n', quantilec(vol_n',0.0005), quantilec(vol_n',0.9995), -1.0,128);
//	   DrawXMatrix(0,x1[2][]|(exp(log_densig(exp(x1[0][]),exp(vol_n))).*exp(x1[0][])),
//	               {"Empirical density","IG fit"},x1[0][],"",0);   //
////	   Draw(1,vol_n,0,1);
//	   SetDrawWindow("IG graph");
//	   ShowDrawWindow();

//	DrawMatrix(0,meanr(psi)+mD[2*i_sup_o][],"Prediction",0,1);
//	DrawTitle(0,"Prediction of next actual vol");
	   DrawMatrix(0,sumr(psi)+mD[2*i_sup_o][]|yobs_o,{"Smoother","Realised vol"},0,1);
	   DrawTitle(0,"Smoothed and realised vol");
	   smooth_vol = log(sumr(psi)+mD[2*i_sup_o][]);
	   sm_mean = meanr(smooth_vol);
	   sm_var  = varr(smooth_vol);
//	   sm_mean = meanr(yobs_o);
//	   sm_var  = varr(yobs_o);
	   x1 = DensEst(smooth_vol', quantilec(smooth_vol',0.0005), quantilec(smooth_vol',0.9995), -1.0,128);
	   x2 = DensEst(log(yobs_o)', quantilec(smooth_vol',0.0005), quantilec(smooth_vol',0.9995), -1.0,128);
	   DrawXMatrix(1,log(x1[2][]|x2[2][])|log_densn(x1[0][],sm_mean,sm_var)
	   |log(exp(log_densig(exp(x1[0][]),exp(smooth_vol))).*exp(x1[0][])),
		  {"Smoothed","Realised vol","log-normal fit","IG fit"},x1[0][],"",0);

//	   DrawXMatrix(1,log(x1[2][]|x2[2][])|log_densn(x1[0][],sm_mean,sm_var)
//	   |log(exp(log_densig(exp(x1[0][]),yobs_o)).*exp(x1[0][])),
//		  {"Smoothed","Realised vol","log-normal fit","IG fit"},x1[0][],"",0);
		  
	   x = (0.5+range(1,n,1))./(1.0+n);
	   resx = y_data_o./sqrt(sumr(psi)+mD[2*i_sup_o][])|
	   y_data_o./sqrt(sumr(psi)+mPred[2*i_sup_o][])|
	   y_data_o./sqrt(yobs_o);
	   DrawXMatrix(2,sortr(resx)|quann(x),{"Smooth","Prediction","Realised vol","45 degree line"},quann(x),0);
	   DrawTitle(2,"QQ plot. Y: observed, X: expected");
	   if (i_sup_o>1)
	      DrawMatrix(3,acf(yobs_o',120)'[][1:]|acf_res|acf_store,{"Empirical","OU","2 OUs","3 OUs","4 OUs"},1,1);
	   else
	      DrawMatrix(3,acf(yobs_o',120)'[][1:]|acf_store,"",1,1);
	   DrawTitle(3,"Acf of actual vol and fitted acf from SV model");
	//
	   SetDrawWindow("Diagnostics");
	   ShowDrawWindow();
	   igraph=0;
	}
	
	igraph += 1;
	   
	return dLogLik;
	
}




vp_par(const vP, const weight)						  // unconstrained vP used by numerical optimisation 
{	// decl v = fabs(vP[0][0]);			  // maps into meaningful parameters
  //   decl a = fabs(vP[1][0]);
//     decl psi,omega,lambda;
//	 psi = v/a;

     decl psi = 20.0*fabs(vP[0][0])/(1.0+fabs(vP[0][0]))*meanr(yobs_o);
	 decl omega = fabs(vP[1][0]);
	 decl lambda=300.0.*fabs(vP[2:rows(vP)-1][0])./(1.0+fabs(vP[2:rows(vP)-1][0]));
	 decl a = 0.0, v=0.0;
	 
//	 decl psi = meanr(yobs_o);
//	 decl omega = fabs(vP[0][0]);
//	 decl lambda=fabs(vP[1:rows(vP)-1][0]);
	 if (i_sup_o>1)
	 {
//	    psi = (psi*weight)~(psi*(1.0-weight));
//		omega=(omega*weight)~(omega*(1.0-weight));
		psi = psi.*weight;
		omega=omega.*weight;
	 }
	 return psi~omega~lambda'~psi~omega;
}

draw_fit(const psi, const omega, const lambda, const Delta)
{
	 decl i, x = sigma_n_arma(omega[0][0],lambda[0][0],Delta);
	 for (i=1; i<i_sup_o; i++)
	    x = x|sigma_n_arma(omega[0][i],lambda[0][i],Delta);
	
	 decl phi = x[][0], theta = x[][1], sigma_eta = x[][3];
    decl xstore = measure_error_var(psi,omega,lambda,Delta,M_o);
	decl var_u = xstore[0][columns(xstore)-1];
	decl iacf = 120;

	decl acf1 = armavar(phi[0][0]~theta[0][0],1,1,sigma_eta[0][0],iacf+1);
	for (i=1; i<i_sup_o; i++)
	    acf1 = acf1|armavar(phi[i][]~theta[i][],1,1,sigma_eta[i][],iacf+1);
		
	acf1 = sumc(acf1);
//	print(var_u~var_u);	
//	print(acf1');
	acf1[0][0] += var_u;
//	Draw(0,acf(yobs_o',iacf)'[][1:]|(acf1./acf1[0][0])[][1:],0,1);
//	if (i_sup_o==1) SetDrawWindow("1 OU processes");
//	if (i_sup_o==2) SetDrawWindow("2 OU processes");
//	if (i_sup_o==3) SetDrawWindow("3 OU processes");
//	ShowDrawWindow();

	acf_store = (acf1./acf1[0][0])[][1:];
	return ;
//	print(meanr(yobs_o)~varr(yobs_o));
}

flike(const vP, const adFunc, const avScore, const amHessian)		// used by optimisation
{
 //    print(vP); // routine. 
     decl psi,omega,lambda;
	 decl weight=0.0,x;
	 if (i_sup_o > 1)
	 {
	     weight = fabs(vP[:i_sup_o-2][0]) / (1.0+sumc(fabs(vP[:i_sup_o-2][0])));
		 weight = (1.0-sumc(weight))~weight';
	     x = vp_par(vP[i_sup_o-1:][],weight);
		 psi = x[0][0:i_sup_o-1];
	     omega=x[0][i_sup_o:2*i_sup_o-1];
	     lambda=x[0][2*i_sup_o:3*i_sup_o-1];
	 }
	 else
	 {
	     x = vp_par(vP,weight);
		 psi = x[0][0];		 
	     omega=x[0][1];
	     lambda=x[0][2:1+i_sup_o];
     }

	 decl y = yobs_o;
	 decl i,M = M_o;
	 decl Delta = Delta_o;
     
     adFunc[0] = like_QV(y,psi,omega,lambda,Delta,M)/sqrt(columns(y));

	 if (avScore)
	 {
	 }
//	 if (igraph == igraph3)
//	{
//	 if (i_sup_o>1)
//    	 print(sumr(psi)~sumr(omega)~lambda~weight~adFunc[0]);
//	 if (i_sup_o==1)
//	     print(psi~omega~lambda~adFunc[0]);
//	 }	 
     draw_fit(psi,omega,lambda,Delta);

	 return 1;
}

flike_eval(const vP)
{
//    print(vP); // routine. 
     decl psi,omega,lambda;
	 decl weight=0.0,x;
	 if (i_sup_o > 1)
	 {
	     weight = fabs(vP[:i_sup_o-2][0]) / (1.0+sumc(fabs(vP[:i_sup_o-2][0])));
		 weight = (1.0-sumc(weight))~weight';
	     x = vp_par(vP[i_sup_o-1:][],weight);
		 psi = x[0][0:i_sup_o-1];
	     omega=x[0][i_sup_o:2*i_sup_o-1];
	     lambda=x[0][2*i_sup_o:3*i_sup_o-1];
	 }
	 else
	 {
	     x = vp_par(vP,weight);
		 psi = x[0][0];		 
	     omega=x[0][1];
	     lambda=x[0][2:1+i_sup_o];
     }

	 decl y = yobs_o;
	 decl i,M = M_o;
	 decl Delta = Delta_o;
     
     like_QV(y,psi,omega,lambda,Delta,M)/sqrt(columns(y));

	 if (i_sup_o>1)
    	 print(sumr(psi)~sumr(omega)~lambda~weight);
	 if (i_sup_o==1)
	     print(psi~omega~lambda);

	 return 1;
}
