program gmm_compute
  version 9.2
  syntax, b(name) [j(name) g(name) useoffset(integer 0)]

  *-- conditional on the quasi-differencing specified by b, the IV 
  *     estimator solves
  *     min_beta J(beta;b)=[Z'(y-Xbeta)-offset]'W[Z'(y-Xbeta)-offset]
  *     so beta=[X'Z*W*Z'X]^-1 * (X'Z)*W*[Z'y-offset]
  *-- as standard in stata, however, we return a row vector that is the
  *     transpose of this formula -- it goes in MY_gmmcoeff
  *-- offset should be a row vector in $MY_gmmoffset
  *   and is used only if useoffset~=0

  *if j is specified, we calculate J(b)

  *if g is specified, we calculate the gradient dj/db

  *-- note: z should not contain constants or time dummies;
  *     we create time dummies here for speed
  *-- note: x SHOULD contain time dummies or constants if you want them
  *-- note: w should be symmetric and positive definite
  *-- these requirements are NOT checked!

  #delim ;
  mata:
    calc("${MY_tbreaks}","`b'",${MY_N},${MY_T},${MY_K_x},${MY_K_z},
         "${MY_xnames}","${MY_yname}","${MY_znames}",
         "${MY_gmmcoeffs}","${MY_gmmw}","`j'","`g'",
         "${MY_gmmoffset}",`useoffset');
  #delim cr

end


version 9.2
mata:
void calc(string scalar tbreaks, string scalar b,
  real scalar N, real scalar T, real scalar Kx, real scalar Kz,
  string xnames, string scalar yname, string znames,
  string scalar coeffs, string scalar w, string scalar j,
  string scalar g, string scalar offset, real scalar useoffset)
  {

  breaks=st_matrix(tbreaks)
  bb=st_matrix(b)

  xtokens=tokens(xnames)
  Lvec="L"
  for(i=2;i<=Kx;i++) {
    Lvec=(Lvec,"L")
    }
  st_view(xvars,.,xtokens)
  st_view(Lxvars,.,Lvec+xtokens)

  st_view(zvars,.,st_varindex(tokens(znames)))

  st_view(yvar,.,yname)
  st_view(Lyvar,.,"L"+yname)

  Zpy=J((Kz+1)*T,1,0)
  ZpX=J((Kz+1)*T,Kx,0)

  for(t=1;t<=T;t++) {
    pos1=(Kz+1)*(t-1)+1
    pos2=(Kz+1)*t
    delx=panelsubmatrix(xvars,t,breaks)-bb[1,t]*panelsubmatrix(Lxvars,t,breaks)
    dely=panelsubmatrix(yvar,t,breaks)-bb[1,t]*panelsubmatrix(Lyvar,t,breaks)
    zblock=panelsubmatrix(zvars,t,breaks)
    Zpy[|pos1\pos2|]=cross(zblock,1,dely,0)
    ZpX[|pos1,1\pos2,.|]=cross(zblock,1,delx,0)
    }

  if(useoffset) {
    ooffset=st_matrix(offset)
    ypZ=ypZ-ooffset
    }

  ww=st_matrix(w)
  ccoeffs=ZpX'*ww
  m2=ccoeffs*ZpX
  ccoeffs=ccoeffs*Zpy
  _lusolve(m2,ccoeffs)
  st_matrix(coeffs,ccoeffs')

  if(j~="") {
    diff=Zpy-ZpX*ccoeffs
    diff=diff'*ww*diff
    st_numscalar(j,diff[1,1]/N)    
    }

  if(g~="") {
    /* the gradient is d/db J(b). we can ignore beta by the envelope 
       theorem. */
    /* the derivative wrt the moment conditions is
       2 * ww * (Z'dely - Z'delX*ccoeffs - offset) */
    /* note we have already subtracted offset from Z'dely if necessary */
    /* the derivative of any given moment condition wrt b_t is 
     - [  (y_s-1)'Z -ccoeffs*[Z'(X_s-1)]'] if s=t and 0 o/w */
    /* so the matrix we want can be constructed in blocks
       -- then we just have to
       sum over the moment conditions at t. */
    /* actually, return the negative of this b/c j is the negative of
       the objective function in use */
    v1=2*ww*(Zpy-ZpX*ccoeffs)/N
    v2=J(rows(v1),T,.)
    for(t=1;t<=T;t++) {
      zblock=panelsubmatrix(zvars,t,breaks)
      u=cross(zblock,1,panelsubmatrix(Lyvar,t,breaks),0) - 
        cross(zblock,1,panelsubmatrix(Lxvars,t,breaks),0) * ccoeffs
      for(zi=1;zi<=Kz+1;zi++) {
        k=(t-1)*(Kz+1)+zi 
        v2[k,t]=v1[k,1]*u[zi,1] 
        } 
      }
    st_matrix(g,colsum(v2))
    }

  }
end


