cap log close
log using output/logs/step1.log, replace

clear all
set more off
version 10.1
set mem 1g
set maxvar 11000

include dofiles/defs.do.h


*---------------------
*  READ RAW PSID DATA AND COMPUTE SUMMARY STATS (table a.2, internet appendix table f.1 in SW11) 
*---------------------



*---------------------
* CPI
*---------------------

insheet using data/raw/cpiu/cpiu.txt
rename year datayear
sort datayear
save data/loaded/cpiu.dta, replace
clear
insheet using data/raw/cpiu/cpiuf.txt
rename year datayear
sort datayear
merge datayear using data/loaded/cpiu.dta, unique
tabulate _merge
drop if _merge~=3
drop _merge
sort datayear
save data/loaded/cpiu.dta, replace
describe
list, clean



*---------------------
* CONSUMPTION/INCOME/LEISURE DATA FROM PSID
*---------------------

* begin code from PSID download

#delimit ;
*  PSID DATA CENTER *****************************************************
   JOBID            : 54374                             
   DATA_DOMAIN      : PSID                              
   USER_WHERE       : None                              
   FILE_TYPE        : All Individuals Data              
   OUTPUT_DATA_TYPE : ASCII Data File                   
   STATEMENTS       : STATA Statements                  
   CODEBOOK_TYPE    : PDF                               
   N_OF_VARIABLES   : 3503                              
   N_OF_OBSERVATIONS: 59872                             
   MAX_REC_LENGTH   : 15968                             
   DATE & TIME      : July 13, 2006 @ 21:26:00
*************************************************************************
;

insheet 
   ER30001 ER30002 ER32000 ER30003 ER30004 ER30005 ER30006 ER30019 V2 V30
   V31 V32 V33 V34 V35 V36 V37 V38 V39 V40
   V41 V42 V43 V44 V45 V46 V47 V48 V49 V50
   V51 V52 V53 V54 V74 V75 V81 V92 V99 V117
   V118 V119 V163 V165 V166 V167 V168 V170 V196 V197
   V198 V239 V243 V251 V256 V257 V260 V262 V268 V269
   V334 V337 V338 V360 V439 ER30020 ER30021 ER30022 ER30023 ER30042
   V442 V465 V466 V467 V468 V469 V470 V471 V472 V473
   V474 V475 V476 V477 V478 V479 V480 V493 V494 V495
   V496 V497 V498 V499 V500 V501 V502 V503 V504 V505
   V506 V507 V508 V509 V510 V511 V512 V513 V514 V515
   V516 V517 V523 V524 V529 V542 V549 V553 V607 V608
   V609 V625 V628 V639 V640 V641 V667 V839 V863 V871
   V873 V1008 V1010 V1011 V1014 V1015 ER30043 ER30044 ER30045 ER30046
   ER30066 V1102 V1106 V1109 V1138 V1139 V1140 V1141 V1142 V1143
   V1144 V1145 V1146 V1147 V1148 V1149 V1150 V1151 V1152 V1153
   V1167 V1169 V1170 V1171 V1172 V1173 V1174 V1175 V1176 V1177
   V1178 V1179 V1180 V1181 V1182 V1183 V1184 V1185 V1186 V1187
   V1188 V1196 V1197 V1198 V1199 V1209 V1210 V1211 V1214 V1216
   V1236 V1238 V1239 V1240 V1241 V1242 V1278 V1279 V1280 V1296
   V1297 V1365 V1366 V1367 V1514 V1534 V1559 V1567 V1569 V1605
   V1609 V1764 V1765 V1766 ER30067 ER30068 ER30069 ER30070 ER30090 V1802
   V1806 V1809 V1839 V1840 V1841 V1842 V1843 V1844 V1845 V1846
   V1847 V1848 V1849 V1850 V1851 V1852 V1853 V1854 V1868 V1870
   V1871 V1872 V1873 V1874 V1875 V1876 V1877 V1878 V1879 V1880
   V1881 V1882 V1883 V1884 V1885 V1886 V1887 V1888 V1889 V1897
   V1898 V1899 V1900 V1911 V1912 V1913 V1916 V1939 V1941 V1942
   V1943 V1944 V1945 V1983 V1984 V1985 V1986 V2002 V2003 V2072
   V2073 V2074 V2226 V2271 V2279 V2281 V2317 V2321 V2345 ER30091
   ER30092 ER30093 ER30094 ER30116 V2402 V2407 V2410 V2439 V2440 V2441
   V2442 V2443 V2444 V2445 V2446 V2447 V2448 V2449 V2450 V2451
   V2452 V2453 V2454 V2468 V2470 V2471 V2472 V2473 V2474 V2475
   V2476 V2477 V2478 V2479 V2480 V2481 V2482 V2483 V2484 V2485
   V2486 V2487 V2488 V2489 V2490 V2498 V2499 V2500 V2501 V2512
   V2513 V2514 V2517 V2519 V2539 V2541 V2542 V2543 V2544 V2545
   V2581 V2582 V2583 V2584 V2600 V2601 V2670 V2671 V2672 V2852
   V2872 V2892 V2898 V2906 V2908 V2968 V2978 V2979 ER30117 ER30118
   ER30119 ER30120 ER30137 V3002 V3007 V3010 V3114 V3301 V3310 ER30138
   ER30139 ER30140 ER30141 ER30159 V3402 V3407 V3410 V3423 V3424 V3425
   V3426 V3427 V3428 V3429 V3430 V3431 V3432 V3437 V3439 V3440
   V3441 V3442 V3443 V3444 V3445 V3446 V3447 V3463 V3464 V3465
   V3466 V3477 V3478 V3479 V3482 V3484 V3505 V3507 V3508 V3509
   V3510 V3511 V3528 V3529 V3530 V3531 V3532 V3548 V3549 V3598
   V3599 V3600 V3601 V3602 V3676 V3694 V3695 V3697 V3721 V3730
   ER30160 ER30161 ER30162 ER30163 ER30187 V3802 V3807 V3810 V3823 V3824
   V3825 V3826 V3827 V3828 V3829 V3830 V3831 V3832 V3833 V3834
   V3835 V3836 V3837 V3839 V3840 V3841 V3842 V3849 V3850 V3851
   V3852 V3853 V3854 V3863 V3864 V3865 V3866 V3877 V3878 V3879
   V3882 V3884 V3885 V3896 V3897 V3900 V3902 V3903 V3918 V3920
   V3921 V3922 V3923 V3924 V3967 V3968 V3969 V3970 V4001 V4002
   V4003 V4053 V4054 V4055 V4056 V4154 V4173 V4174 V4176 V4198
   V4199 V4206 V4224 V4231 ER30188 ER30189 ER30190 ER30191 ER30216 V4302
   V4307 V4310 V4332 V4333 V4334 V4335 V4336 V4337 V4338 V4339
   V4340 V4341 V4342 V4343 V4344 V4345 V4346 V4348 V4349 V4354
   V4355 V4362 V4363 V4364 V4365 V4368 V4369 V4392 V4393 V4394
   V4397 V4399 V4400 V4410 V4411 V4414 V4416 V4417 V4433 V4435
   V4436 V4437 V4438 V4439 V4458 V4459 V4461 V4509 V4510 V4513
   V4603 V4604 V4605 V4708 V4719 V4720 V4721 V4722 V4723 V4724
   V4725 V4726 V4727 V4728 V4729 V4730 V4741 V4743 V4841 V4842
   V4844 V4892 V4893 V4896 V4897 V5029 V5031 V5049 V5050 V5052
   V5074 V5075 V5098 V5099 V5113 ER30217 ER30218 ER30219 ER30220 V5202
   V5207 V5210 V5232 V5233 V5234 V5235 V5236 V5237 V5238 V5239
   V5240 V5241 V5242 V5243 V5244 V5245 V5246 V5247 V5248 V5249
   V5250 V5251 V5252 V5253 V5254 V5256 V5257 V5267 V5268 V5269
   V5270 V5271 V5272 V5273 V5274 V5275 V5276 V5277 V5278 V5303
   V5304 V5305 V5308 V5309 V5311 V5312 V5322 V5323 V5326 V5327
   V5329 V5330 V5347 V5349 V5350 V5351 V5352 V5353 V5373 V5376
   V5420 V5421 V5424 V5477 V5479 V5502 V5506 V5626 V5627 V5630
   V5631 V5632 V5647 V5648 V5650 V5664 V5665 V5672 V5673 V5674
   V5675 V5676 V5677 V5678 V5679 V5680 ER30246 ER30247 ER30248 ER30249
   V5702 V5707 V5710 V5731 V5732 V5733 V5734 V5735 V5736 V5737
   V5738 V5739 V5740 V5741 V5742 V5743 V5744 V5745 V5746 V5747
   V5748 V5749 V5750 V5751 V5752 V5753 V5754 V5755 V5757 V5758
   V5766 V5767 V5768 V5769 V5770 V5771 V5772 V5773 V5774 V5775
   V5776 V5777 V5802 V5803 V5804 V5805 V5808 V5809 V5811 V5821
   V5822 V5823 V5826 V5827 V5829 V5847 V5849 V5850 V5851 V5852
   V5853 V5872 V5875 V5907 V5908 V5911 V6002 V6003 V6017 V6022
   V6034 V6038 V6173 V6174 V6177 V6178 V6179 V6194 V6195 V6197
   V6211 V6212 V6219 ER30283 ER30284 ER30285 ER30286 V6302 V6307 V6310
   V6336 V6337 V6338 V6339 V6340 V6341 V6342 V6343 V6344 V6345
   V6346 V6347 V6348 V6349 V6350 V6351 V6352 V6353 V6354 V6355
   V6356 V6357 V6358 V6359 V6361 V6363 V6364 V6372 V6373 V6374
   V6375 V6376 V6377 V6378 V6379 V6380 V6381 V6382 V6383 V6385
   V6413 V6414 V6415 V6416 V6419 V6420 V6422 V6432 V6433 V6434
   V6437 V6438 V6440 V6459 V6461 V6462 V6463 V6464 V6465 V6492
   V6493 V6518 V6519 V6522 V6575 V6576 V6577 V6591 V6592 V6614
   V6615 V6616 V6647 V6648 V6649 V6659 V6766 V6767 V6770 V6771
   V6772 V6787 V6788 V6790 V6804 V6805 V6812 ER30313 ER30314 ER30315
   ER30316 V6902 V6907 V6910 V6934 V6935 V6936 V6937 V6938 V6939
   V6940 V6941 V6942 V6943 V6944 V6945 V6946 V6947 V6948 V6949
   V6950 V6951 V6952 V6953 V6954 V6955 V6956 V6957 V6959 V6961
   V6962 V6970 V6971 V6972 V6973 V6974 V6975 V6976 V6977 V7003
   V7004 V7005 V7006 V7009 V7010 V7012 V7036 V7037 V7038 V7041
   V7042 V7044 V7064 V7066 V7067 V7068 V7069 V7070 V7095 V7096
   V7121 V7122 V7125 V7177 V7178 V7179 V7193 V7194 V7216 V7217
   V7218 V7249 V7250 V7251 V7261 V7412 V7413 V7416 V7417 V7418
   V7433 V7434 V7435 V7450 V7451 V7455 ER30343 ER30344 ER30345 ER30346
   V7502 V7507 V7510 V7530 V7531 V7532 V7533 V7534 V7535 V7536
   V7537 V7538 V7539 V7540 V7541 V7542 V7543 V7544 V7545 V7546
   V7547 V7548 V7549 V7551 V7553 V7554 V7562 V7563 V7564 V7565
   V7566 V7567 V7568 V7569 V7595 V7596 V7597 V7598 V7601 V7602
   V7604 V7628 V7629 V7630 V7633 V7634 V7636 V7655 V7657 V7658
   V7659 V7660 V7661 V7706 V7707 V7714 V7715 V7718 V7863 V7864
   V7865 V7879 V7880 V7887 V7888 V7889 V7940 V7941 V7942 V7952
   V8065 V8066 V8068 V8069 V8070 V8085 V8086 V8087 V8102 V8103
   V8107 ER30373 ER30374 ER30375 ER30376 V8202 V8207 V8210 V8228 V8229
   V8230 V8231 V8232 V8233 V8234 V8235 V8236 V8237 V8238 V8239
   V8240 V8241 V8242 V8243 V8244 V8245 V8246 V8247 V8249 V8251
   V8252 V8254 V8255 V8256 V8257 V8258 V8259 V8260 V8261 V8288
   V8289 V8290 V8291 V8294 V8295 V8297 V8321 V8322 V8323 V8326
   V8327 V8329 V8349 V8351 V8352 V8353 V8354 V8355 V8374 V8375
   V8382 V8383 V8386 V8523 V8524 V8525 V8538 V8539 V8546 V8547
   V8548 V8591 V8592 V8593 V8689 V8690 V8692 V8693 V8694 V8709
   V8710 V8711 V8726 V8727 V8731 ER30399 ER30400 ER30401 ER30402 ER30403
   ER30404 ER30425 V8802 V8807 V8810 V8830 V8831 V8832 V8833 V8834
   V8835 V8836 V8837 V8838 V8839 V8840 V8841 V8842 V8843 V8844
   V8845 V8846 V8847 V8848 V8849 V8851 V8853 V8854 V8862 V8863
   V8864 V8865 V8866 V8867 V8868 V8869 V8896 V8897 V8898 V8899
   V8902 V8903 V8905 V8929 V8930 V8931 V8934 V8935 V8937 V8958
   V8961 V8962 V8963 V8964 V9005 V9006 V9013 V9014 V9017 V9173
   V9174 V9188 V9189 V9196 V9197 V9198 V9264 V9265 V9266 V9276
   V9375 V9376 V9378 V9379 V9380 V9395 V9396 V9418 V9419 V9420
   V9421 V9433 ER30429 ER30430 ER30431 ER30432 ER30433 ER30434 ER30459 V10002
   V10007 V10010 V10037 V10038 V10039 V10040 V10041 V10042 V10043 V10044
   V10045 V10046 V10047 V10048 V10049 V10050 V10051 V10052 V10053 V10054
   V10055 V10056 V10057 V10058 V10059 V10060 V10061 V10062 V10063 V10064
   V10065 V10066 V10067 V10068 V10069 V10070 V10071 V10072 V10073 V10074
   V10075 V10076 V10077 V10078 V10079 V10080 V10081 V10082 V10083 V10084
   V10085 V10086 V10087 V10088 V10089 V10090 V10091 V10092 V10093 V10094
   V10095 V10096 V10097 V10098 V10099 V10100 V10101 V10102 V10103 V10104
   V10105 V10106 V10107 V10108 V10109 V10110 V10111 V10112 V10113 V10114
   V10115 V10116 V10117 V10118 V10119 V10120 V10121 V10122 V10123 V10124
   V10125 V10126 V10131 V10132 V10133 V10134 V10135 V10136 V10137 V10138
   V10139 V10140 V10141 V10142 V10143 V10144 V10145 V10146 V10147 V10148
   V10149 V10150 V10151 V10152 V10153 V10154 V10155 V10156 V10157 V10158
   V10159 V10160 V10161 V10162 V10163 V10164 V10165 V10166 V10167 V10168
   V10169 V10170 V10171 V10172 V10173 V10174 V10175 V10176 V10177 V10178
   V10179 V10180 V10181 V10182 V10183 V10184 V10185 V10186 V10187 V10188
   V10189 V10190 V10191 V10192 V10193 V10194 V10195 V10196 V10197 V10198
   V10199 V10200 V10201 V10202 V10203 V10204 V10205 V10206 V10207 V10208
   V10209 V10210 V10211 V10212 V10213 V10214 V10215 V10216 V10217 V10218
   V10219 V10220 V10222 V10224 V10225 V10233 V10234 V10235 V10236 V10237
   V10238 V10239 V10240 V10282 V10283 V10285 V10286 V10295 V10296 V10300
   V10385 V10386 V10387 V10391 V10392 V10394 V10416 V10418 V10419 V10420
   V10421 V10422 V10426 V10453 V10454 V10455 V10456 V10462 V10463 V10466
   V10612 V10671 V11022 V11023 V11025 V11026 V11027 V11042 V11043 V11064
   V11065 V11066 V11067 V11079 ER30463 ER30464 ER30465 ER30466 ER30467 ER30468
   ER30494 V11102 V11107 V11112 V11146 V11147 V11148 V11149 V11150 V11151
   V11152 V11153 V11154 V11155 V11156 V11157 V11158 V11159 V11160 V11161
   V11162 V11163 V11164 V11165 V11166 V11167 V11168 V11169 V11170 V11171
   V11172 V11173 V11174 V11175 V11176 V11177 V11178 V11179 V11180 V11181
   V11182 V11183 V11184 V11185 V11186 V11187 V11188 V11189 V11190 V11191
   V11192 V11193 V11194 V11195 V11196 V11197 V11198 V11199 V11200 V11201
   V11202 V11203 V11204 V11205 V11206 V11207 V11208 V11209 V11210 V11211
   V11212 V11213 V11214 V11215 V11216 V11217 V11218 V11219 V11220 V11221
   V11222 V11223 V11224 V11225 V11226 V11227 V11228 V11229 V11230 V11231
   V11232 V11233 V11234 V11235 V11236 V11237 V11238 V11239 V11240 V11241
   V11242 V11243 V11244 V11245 V11246 V11247 V11248 V11249 V11250 V11251
   V11258 V11259 V11260 V11261 V11262 V11263 V11264 V11265 V11266 V11267
   V11268 V11269 V11270 V11271 V11272 V11273 V11274 V11275 V11276 V11277
   V11278 V11279 V11280 V11281 V11282 V11283 V11284 V11285 V11286 V11287
   V11288 V11289 V11290 V11291 V11292 V11293 V11294 V11295 V11296 V11297
   V11298 V11299 V11300 V11301 V11302 V11303 V11304 V11305 V11306 V11307
   V11308 V11309 V11310 V11311 V11312 V11313 V11314 V11315 V11316 V11317
   V11318 V11319 V11320 V11321 V11322 V11323 V11324 V11325 V11326 V11327
   V11328 V11329 V11330 V11331 V11332 V11333 V11334 V11335 V11336 V11337
   V11338 V11339 V11340 V11341 V11342 V11343 V11344 V11345 V11346 V11347
   V11348 V11349 V11350 V11351 V11352 V11353 V11354 V11355 V11356 V11357
   V11358 V11359 V11360 V11361 V11362 V11364 V11373 V11374 V11375 V11376
   V11377 V11378 V11379 V11380 V11427 V11428 V11430 V11431 V11440 V11441
   V11445 V11452 V11453 V11457 V11565 V11566 V11567 V11571 V11572 V11574
   V11600 V11603 V11605 V11606 V11607 V11608 V11609 V11612 V11637 V11638
   V11639 V11640 V11653 V11654 V11657 V11767 V12000 V12001 V12002 V12003
   V12016 V12017 V12020 V12130 V12371 V12372 V12373 V12374 V12376 V12377
   V12378 V12400 V12401 V12425 V12426 V12427 V12428 V12445 V12446 ER30498
   ER30499 ER30500 ER30501 ER30502 ER30503 ER30531 V12502 V12507 V12510 V12545
   V12546 V12547 V12548 V12549 V12550 V12551 V12552 V12553 V12554 V12555
   V12556 V12557 V12558 V12559 V12560 V12561 V12562 V12563 V12564 V12565
   V12566 V12567 V12568 V12569 V12570 V12571 V12572 V12573 V12574 V12575
   V12576 V12577 V12578 V12579 V12580 V12581 V12582 V12583 V12584 V12585
   V12586 V12587 V12588 V12589 V12590 V12591 V12592 V12593 V12594 V12595
   V12596 V12597 V12598 V12599 V12600 V12601 V12602 V12603 V12604 V12605
   V12606 V12607 V12608 V12609 V12610 V12611 V12612 V12613 V12614 V12615
   V12616 V12617 V12618 V12619 V12620 V12621 V12622 V12623 V12624 V12625
   V12626 V12627 V12628 V12629 V12630 V12631 V12632 V12633 V12634 V12635
   V12636 V12637 V12638 V12639 V12640 V12641 V12642 V12643 V12644 V12645
   V12646 V12647 V12648 V12649 V12650 V12657 V12658 V12659 V12660 V12661
   V12662 V12663 V12664 V12665 V12666 V12667 V12668 V12669 V12670 V12671
   V12672 V12673 V12674 V12675 V12676 V12677 V12678 V12679 V12680 V12681
   V12682 V12683 V12684 V12685 V12686 V12687 V12688 V12689 V12690 V12691
   V12692 V12693 V12694 V12695 V12696 V12697 V12698 V12699 V12700 V12701
   V12702 V12703 V12704 V12705 V12706 V12707 V12708 V12709 V12710 V12711
   V12712 V12713 V12714 V12715 V12716 V12717 V12718 V12719 V12720 V12721
   V12722 V12723 V12724 V12725 V12726 V12727 V12728 V12729 V12730 V12731
   V12732 V12733 V12734 V12735 V12736 V12737 V12738 V12739 V12740 V12741
   V12742 V12743 V12744 V12745 V12746 V12747 V12748 V12749 V12750 V12751
   V12752 V12753 V12754 V12755 V12756 V12757 V12758 V12759 V12760 V12761
   V12763 V12772 V12773 V12774 V12775 V12776 V12777 V12778 V12779 V12826
   V12827 V12829 V12830 V12839 V12840 V12844 V12847 V12848 V12850 V12851
   V12859 V12860 V12864 V12972 V12973 V12974 V12978 V12979 V12981 V13008
   V13010 V13011 V13012 V13013 V13014 V13017 V13046 V13047 V13048 V13049
   V13056 V13057 V13060 V13163 V13225 V13226 V13227 V13228 V13235 V13236
   V13239 V13331 V13623 V13624 V13625 V13626 V13628 V13629 V13630 V13640
   V13641 V13665 V13666 V13667 V13687 ER30535 ER30536 ER30537 ER30538 ER30539
   ER30540 ER30566 V13702 V13707 V13710 V13745 V13746 V13747 V13748 V13749
   V13750 V13751 V13752 V13753 V13754 V13755 V13756 V13757 V13758 V13759
   V13760 V13761 V13762 V13763 V13764 V13765 V13766 V13767 V13768 V13769
   V13770 V13771 V13772 V13773 V13774 V13775 V13776 V13777 V13778 V13779
   V13780 V13781 V13782 V13783 V13784 V13785 V13786 V13787 V13788 V13789
   V13790 V13791 V13792 V13793 V13794 V13795 V13796 V13797 V13798 V13799
   V13800 V13801 V13802 V13809 V13810 V13811 V13812 V13813 V13814 V13815
   V13816 V13817 V13818 V13819 V13820 V13821 V13822 V13823 V13824 V13825
   V13826 V13827 V13828 V13829 V13830 V13831 V13832 V13833 V13834 V13835
   V13836 V13837 V13838 V13839 V13840 V13841 V13842 V13843 V13844 V13845
   V13846 V13847 V13848 V13849 V13850 V13851 V13852 V13853 V13854 V13855
   V13856 V13857 V13858 V13859 V13860 V13861 V13862 V13863 V13864 V13865
   V13867 V13874 V13875 V13876 V13877 V13878 V13879 V13880 V13881 V13928
   V13929 V13931 V13932 V13941 V13942 V13946 V13949 V13950 V13952 V13953
   V13961 V13962 V13966 V14074 V14075 V14076 V14080 V14081 V14083 V14111
   V14113 V14114 V14115 V14116 V14117 V14120 V14146 V14147 V14148 V14149
   V14156 V14157 V14160 V14259 V14321 V14322 V14323 V14324 V14331 V14332
   V14335 V14423 V14670 V14671 V14672 V14673 V14675 V14676 V14677 V14687
   V14688 V14712 V14713 V14714 V14737 ER30570 ER30571 ER30572 ER30573 ER30574
   ER30575 ER30602 V14802 V14807 V14810 V14846 V14847 V14848 V14849 V14850
   V14851 V14852 V14853 V14854 V14855 V14856 V14857 V14858 V14876 V14877
   V14878 V14879 V14880 V14881 V14882 V14883 V14884 V14885 V14886 V14887
   V15127 V15129 V15130 V15131 V15132 V15133 V15136 V15154 V15456 V16161
   V16162 V16187 V16188 V16189 V16208 ER30606 ER30607 ER30608 ER30609 ER30610
   ER30611 ER30638 V16302 V16307 V16310 V16346 V16347 V16348 V16349 V16350
   V16351 V16352 V16353 V16354 V16355 V16356 V16357 V16358 V16376 V16377
   V16378 V16379 V16380 V16381 V16382 V16383 V16384 V16385 V16386 V16387
   V16628 V16630 V16631 V16632 V16633 V16634 V16637 V16655 V16974 V17545
   V17546 V17565 V17566 V17567 V17611 V17612 ER30642 ER30643 ER30644 ER30645
   ER30646 ER30647 ER30680 V17702 V17707 V17710 V17744 V17745 V17746 V17747
   V17748 V17749 V17750 V17751 V17752 V17753 V17754 V17755 V17756 V17757
   V17758 V17759 V17760 V17761 V17762 V17763 V17764 V17765 V17766 V17767
   V17774 V17775 V17776 V17777 V17778 V17779 V17780 V17781 V17782 V17783
   V17784 V17785 V17786 V17787 V17788 V17789 V17790 V17791 V17792 V17793
   V17794 V17795 V17796 V17798 V17805 V17806 V17807 V17808 V17809 V17810
   V17811 V17812 V17859 V17860 V17862 V17863 V17872 V17873 V17877 V17880
   V17881 V17883 V17884 V17892 V17893 V17897 V18005 V18006 V18007 V18011
   V18012 V18014 V18046 V18048 V18049 V18050 V18051 V18052 V18055 V18093
   V18094 V18095 V18096 V18103 V18104 V18107 V18395 V18396 V18397 V18398
   V18405 V18406 V18409 V18875 V18878 V18881 V18882 V18884 V18886 V18887
   V18888 V18898 V18899 V18916 V18917 V18918 V18943 V18944 V18945 ER30689
   ER30690 ER30691 ER30692 ER30693 ER30694 ER30723 V19002 V19007 V19010 V19044
   V19045 V19046 V19047 V19048 V19049 V19050 V19051 V19052 V19053 V19054
   V19055 V19056 V19057 V19058 V19059 V19060 V19061 V19062 V19063 V19064
   V19065 V19066 V19067 V19074 V19075 V19076 V19077 V19078 V19079 V19080
   V19081 V19082 V19083 V19084 V19085 V19086 V19087 V19088 V19089 V19090
   V19091 V19092 V19093 V19094 V19095 V19096 V19098 V19105 V19106 V19107
   V19108 V19109 V19110 V19111 V19112 V19159 V19160 V19162 V19163 V19172
   V19173 V19177 V19180 V19181 V19183 V19184 V19192 V19193 V19197 V19305
   V19306 V19307 V19311 V19312 V19314 V19346 V19348 V19349 V19350 V19351
   V19352 V19355 V19393 V19394 V19395 V19396 V19403 V19404 V19407 V19695
   V19696 V19697 V19698 V19705 V19706 V19709 V20175 V20178 V20181 V20182
   V20184 V20186 V20187 V20188 V20198 V20199 V20216 V20217 V20218 V20243
   V20244 V20245 ER30733 ER30734 ER30735 ER30736 ER30737 ER30738 ER30798 V20302
   V20307 V20310 V20344 V20345 V20346 V20347 V20348 V20349 V20350 V20351
   V20352 V20353 V20354 V20355 V20356 V20357 V20358 V20359 V20360 V20361
   V20362 V20363 V20364 V20365 V20366 V20367 V20374 V20375 V20376 V20377
   V20378 V20379 V20380 V20381 V20382 V20383 V20384 V20385 V20386 V20387
   V20388 V20389 V20390 V20391 V20392 V20393 V20394 V20395 V20396 V20398
   V20405 V20406 V20407 V20408 V20409 V20410 V20411 V20412 V20459 V20460
   V20462 V20463 V20472 V20473 V20477 V20480 V20481 V20483 V20484 V20492
   V20493 V20497 V20605 V20606 V20607 V20611 V20612 V20614 V20648 V20650
   V20651 V20652 V20653 V20654 V20657 V20693 V20694 V20695 V20696 V20703
   V20704 V20707 V20995 V20996 V20997 V20998 V21005 V21006 V21009 V21481
   V21484 V21487 V21488 V21490 V21492 V21493 V21494 V21504 V21505 V21522
   V21523 V21524 V21547 V21548 V21549 ER30806 ER30807 ER30808 ER30809 ER30810
   ER30811 V21602 V21606 V21608 V21634 V21635 V21636 V21637 V21638 V21639
   V21640 V21641 V21642 V21643 V21644 V21645 V21646 V21647 V21648 V21649
   V21650 V21651 V21652 V21653 V21654 V21655 V21656 V21657 V21658 V21659
   V21660 V21661 V21662 V21663 V21664 V21670 V21671 V21672 V21673 V21674
   V21675 V21676 V21677 V21678 V21679 V21680 V21681 V21682 V21683 V21684
   V21685 V21686 V21687 V21688 V21689 V21690 V21691 V21692 V21693 V21694
   V21695 V21696 V21697 V21698 V21699 V21703 V21704 V21705 V21706 V21707
   V21708 V21709 V21710 V21711 V21712 V21713 V21714 V21727 V21728 V21964
   V21965 V21978 V21979 V21980 V21981 V21994 V21995 V21996 V21997 V22010
   V22011 V22094 V22095 V22108 V22109 V22110 V22111 V22124 V22125 V22142
   V22143 V22156 V22157 V22158 V22159 V22172 V22173 V22190 V22191 V22204
   V22205 V22206 V22207 V22220 V22221 V22222 V22223 V22236 V22237 V22238
   V22239 V22252 V22253 V22270 V22271 V22284 V22285 V22319 V22320 V22333
   V22334 V22335 V22336 V22349 V22350 V22374 V22375 V22376 V22377 V22378
   V22379 V22386 V22387 V22388 V22389 V22392 V22393 V22403 V22405 V22406
   V22407 V22408 V22409 V22412 V22448 V22449 V22450 V22451 V22463 V22464
   V22465 V22470 V22801 V22802 V22803 V22804 V22816 V22817 V22818 V22823
   V23322 V23323 V23324 V23325 V23326 V23333 V23334 V23336 V23337 V23338
   V23361 V23362 V23363 ER33101 ER33102 ER33103 ER33104 ER33105 ER33106 ER2002
   ER2005 ER2006 ER2007 ER2008 ER2009 ER2010 ER2014 ER2068 ER2069 ER2070
   ER2071 ER2072 ER2073 ER2074 ER2075 ER2076 ER2081 ER2082 ER2083 ER2088
   ER2407 ER2408 ER2409 ER2410 ER2562 ER2563 ER2564 ER2565 ER2566 ER2567
   ER2568 ER2569 ER2570 ER2575 ER2576 ER2577 ER2582 ER2901 ER2902 ER2903
   ER2904 ER3060 ER3061 ER3076 ER3078 ER3079 ER3081 ER3082 ER3083 ER3084
   ER3085 ER3086 ER3088 ER3089 ER3090 ER3091 ER3263 ER3264 ER3278 ER3279
   ER3293 ER3294 ER3371 ER3372 ER3386 ER3387 ER3431 ER3432 ER3446 ER3447
   ER3475 ER3494 ER3495 ER3509 ER3510 ER3584 ER3585 ER3599 ER3600 ER3629
   ER3630 ER3659 ER3660 ER3674 ER3675 FIPS94ST PSID94ST FAMINC94 TXHW94 TRHW94
   TXOFM94 TROFM94 SSEC94 FRMINC94 BUSINC94 HDBUSY94 HDBUSK94 WFBUSY94 WFBUSK94 BUSNUM94
   HDBNUM94 WFBNUM94 HDEARN94 HDWG94 BNUS94 OVTM94 TIPS94 COMS94 PROF94 MKTG94
   XTRA94 OTHINC94 WFEARN94 HDOCC94 HDIND94 WFOCC94 WFIND94 USDA94 CENSUS94 HDWKS94
   HDAVG94 HDOVT94 HDXTR94 HDTOT94 HDWGE94 WFWKS94 WFAVG94 WFOVT94 WFXTR94 WFTOT94
   WFWGE94 HDILO94 HDILS94 HDVAC94 HDSTR94 HDUMP94 HDOLF94 WFILO94 WFILS94 WFVAC94
   WFSTR94 WFUMP94 WFOLF94 ER33201 ER33202 ER33203 ER33204 ER33205 ER33206 ER5002
   ER5004 ER5005 ER5006 ER5007 ER5008 ER5009 ER5013 ER5067 ER5068 ER5069
   ER5070 ER5071 ER5072 ER5073 ER5074 ER5075 ER5080 ER5081 ER5082 ER5087
   ER5406 ER5407 ER5408 ER5409 ER5561 ER5562 ER5563 ER5564 ER5565 ER5566
   ER5567 ER5568 ER5569 ER5574 ER5575 ER5576 ER5581 ER5900 ER5901 ER5902
   ER5903 ER6059 ER6060 ER6075 ER6076 ER6077 ER6078 ER6080 ER6081 ER6082
   ER6083 ER6084 ER6085 ER6087 ER6088 ER6089 ER6090 ER6091 ER6263 ER6264
   ER6278 ER6279 ER6280 ER6294 ER6295 ER6372 ER6373 ER6387 ER6388 ER6432
   ER6433 ER6447 ER6448 ER6476 ER6495 ER6496 ER6510 ER6511 ER6585 ER6586
   ER6587 ER6601 ER6602 ER6631 ER6632 ER6661 ER6662 ER6676 ER6677 FIPS95ST
   PSID95ST FAMINC95 TXHW95 TRHW95 TXOFM95 TROFM95 SSEC95 FRMINC95 BUSINC95 HDBUSY95
   HDBUSK95 WFBUSY95 WFBUSK95 BUSNUM95 HDBNUM95 WFBNUM95 HDEARN95 HDWG95 BNUS95 OVTM95
   TIPS95 COMS95 PROF95 MKTG95 XTRA95 OTHINC95 WFEARN95 HDOCC95 HDIND95 WFOCC95
   WFIND95 USDA95 CENSUS95 HDWKS95 HDAVG95 HDOVT95 HDXTR95 HDTOT95 HDWGE95 WFWKS95
   WFAVG95 WFOVT95 WFXTR95 WFTOT95 WFWGE95 HDILO95 HDILS95 HDVAC95 HDSTR95 HDUMP95
   HDOLF95 WFILO95 WFILS95 WFVAC95 WFSTR95 WFUMP95 WFOLF95 ER33301 ER33302 ER33303
   ER33304 ER33305 ER33306 ER7002 ER7004 ER7005 ER7006 ER7007 ER7008 ER7009
   ER7012 ER7013 ER7163 ER7164 ER7165 ER7166 ER7167 ER7168 ER7169 ER7170
   ER7171 ER7176 ER7177 ER7178 ER7183 ER7502 ER7503 ER7504 ER7505 ER7657
   ER7658 ER7659 ER7660 ER7661 ER7662 ER7663 ER7664 ER7665 ER7670 ER7671
   ER7672 ER7677 ER7996 ER7997 ER7998 ER7999 ER8156 ER8157 ER8172 ER8174
   ER8175 ER8177 ER8178 ER8179 ER8180 ER8181 ER8182 ER8184 ER8185 ER8186
   ER8187 ER8188 ER8380 ER8381 ER8395 ER8396 ER8397 ER8411 ER8412 ER8489
   ER8490 ER8504 ER8505 ER8549 ER8550 ER8564 ER8565 ER8593 ER8612 ER8613
   ER8627 ER8628 ER8702 ER8703 ER8704 ER8718 ER8719 ER8748 ER8749 ER8778
   ER8779 ER8793 ER8794 ER9103 ER9104 ER9105 ER9106 ER9107 FIPS96ST PSID96ST
   FAMINC96 TXHW96 TRHW96 TXOFM96 TROFM96 SSEC96 FRMINC96 BUSINC96 HDBUSY96 HDBUSK96
   WFBUSY96 WFBUSK96 BUSNUM96 HDBNUM96 WFBNUM96 HDEARN96 HDWG96 BNUS96 OVTM96 TIPS96
   COMS96 PROF96 MKTG96 XTRA96 OTHINC96 WFEARN96 HDOCC96 HDIND96 WFOCC96 WFIND96
   USDA96 CENSUS96 HDWKS96 HDAVG96 HDOVT96 HDXTR96 HDTOT96 HDWGE96 WFWKS96 WFAVG96
   WFOVT96 WFXTR96 WFTOT96 WFWGE96 HDILO96 HDILS96 HDVAC96 HDSTR96 HDUMP96 HDOLF96
   WFILO96 WFILS96 WFVAC96 WFSTR96 WFUMP96 WFOLF96 ER33401 ER33402 ER33403 ER33404
   ER33405 ER33406 ER10002 ER10005 ER10006 ER10007 ER10008 ER10009 ER10010 ER10011
   ER10012 ER10016 ER10081 ER10082 ER10083 ER10084 ER10085 ER10086 ER10091 ER10092
   ER10093 ER10098 ER10417 ER10418 ER10419 ER10420 ER10563 ER10564 ER10565 ER10566
   ER10567 ER10568 ER10573 ER10574 ER10575 ER10580 ER10899 ER10900 ER10901 ER10902
   ER11050 ER11051 ER11066 ER11067 ER11068 ER11069 ER11071 ER11072 ER11073 ER11074
   ER11075 ER11076 ER11077 ER11079 ER11080 ER11081 ER11082 ER11083 ER11273 ER11274
   ER11288 ER11289 ER11290 ER11304 ER11305 ER11383 ER11384 ER11398 ER11399 ER11443
   ER11444 ER11458 ER11459 ER11487 ER11494 ER11495 ER11509 ER11510 ER11584 ER11585
   ER11586 ER11600 ER11601 ER11630 ER11631 ER11660 ER11661 ER11675 ER11676 ER12079
   ER12080 ER12081 ER12082 ER12083 ER12084 FIPS97ST PSID97ST FAMINC97 TXHW97 TRHW97
   TXOFM97 TROFM97 SSEC97 FRMINC97 BUSINC97 HDBUSY97 HDBUSK97 WFBUSY97 WFBUSK97 BUSNUM97
   HDBNUM97 WFBNUM97 HDEARN97 HDWG97 BNUS97 OVTM97 TIPS97 COMS97 PROF97 MKTG97
   XTRA97 OTHINC97 WFEARN97 HDOCC97 HDIND97 WFOCC97 WFIND97 USDA97 CENSUS97 HDWKS97
   HDAVG97 HDOVT97 HDXTR97 HDTOT97 HDWGE97 WFWKS97 WFAVG97 WFOVT97 WFXTR97 WFTOT97
   WFWGE97 HDILO97 HDILS97 HDVAC97 HDSTR97 HDUMP97 HDOLF97 WFILO97 WFILS97 WFVAC97
   WFSTR97 WFUMP97 WFOLF97
using data/raw/job54374/job54374.txt, delim("^") clear 
;
label variable er30001  "1968 INTERVIEW NUMBER"  ;
label variable er30002  "PERSON NUMBER                         68"  ;
label variable er32000  "SEX OF INDIVIDUAL"  ;
label variable er30003  "RELATIONSHIP TO HEAD                  68"  ;
label variable er30004  "AGE OF INDIVIDUAL                     68"  ;
label variable er30005  "MARR PAIRS INDICATOR                  68"  ;
label variable er30006  "WHETHER MOVED IN                      68"  ;
label variable er30019  "INDIVIDUAL WEIGHT                     68"  ;
label variable v2  "INTERVIEW NUMBER 68"  ;
label variable v30  "FAMILY SIZE 2:11"  ;
label variable v31  "YRLY FOOD STD. 2:12-15"  ;
label variable v32  "YRLY NEED STD 2:16-20"  ;
label variable v33  "YRLY ALCOHOL 2:21-23"  ;
label variable v34  "ACCURACY OF ALC 2:24"  ;
label variable v35  "YRLY CIG EXP 2:25-27"  ;
label variable v36  "ACCURCY OF CIG EXP 2:28"  ;
label variable v37  "YRLY FOOD BILL 2:29-32"  ;
label variable v38  "ACCURACY OF FOOD 2:33"  ;
label variable v39  "YRLY FOOD GROWN 2:34-36"  ;
label variable v40  "ACCURACY OF FD GRN 2:37"  ;
label variable v41  "YRLY SAVD SEWING 2:38-40"  ;
label variable v42  "ACCURACY OF SVD SEW 2:41"  ;
label variable v43  "YRLY FOOD ON JOB 2:42-44"  ;
label variable v44  "ACCURACY FOOD JOB 2:45"  ;
label variable v45  "YRLY FOOD STAMPS 2:46-48"  ;
label variable v46  "ACCURACY FOOD STPS 2:49"  ;
label variable v47  "YRLY HEADS HRS 3:11-14"  ;
label variable v48  "ACCURACY:HDS HRS 3:15"  ;
label variable v49  "HRS UNEMPL HEAD 3:16-19"  ;
label variable v50  "ACCURACY UNEMP HEAD 3:20"  ;
label variable v51  "HRS HEAD ILL 3:21-24"  ;
label variable v52  "ACCURACY HEAD ILL 3:25"  ;
label variable v53  "YRLY WIFE HRS 3:26-29"  ;
label variable v54  "ACCURACY WIFE HRS 3:30"  ;
label variable v74  "HDS LABOR INCOME 4:11-15"  ;
label variable v75  "WIFE LBR INCOME 4:16-20"  ;
label variable v81  "FAM MONEY INC 4:46-50"  ;
label variable v92  "SRC PR SAMPL UN 5:11-13"  ;
label variable v99  "DATE OF INTVW 5:32"  ;
label variable v117  "AGE OF HEAD 5:54-55"  ;
label variable v118  "AGE OF WIFE 5:56-57"  ;
label variable v119  "SEX OF HEAD 5:58"  ;
label variable v163  "SPEND EATING OUT 6:43"  ;
label variable v165  "SPEND ON MILK DLVRD 6:45"  ;
label variable v166  "SPEND ON OTHR FOOD 6:46"  ;
label variable v167  "SPEND ON ALCOHOL 6:47"  ;
label variable v168  "ALCHL IN FOOD BILL 6:48"  ;
label variable v170  "CIGS IN FOOD BILL 6:50"  ;
label variable v196  "WORKING NOW 7:11"  ;
label variable v197  "OCCUPATION 7:12"  ;
label variable v198  "SELF-EMPLOYED 7:13"  ;
label variable v239  "MARITAL STATUS 7:60"  ;
label variable v243  "WIFES OCCUPATION 7:65"  ;
label variable v251  "HDS WAGES 8:13"  ;
label variable v256  "ADC& ADCU INC-H 8:18"  ;
label variable v257  "OTHER WELFARE-H 8:19"  ;
label variable v260  "UNEMPL COM& WC 8:22"  ;
label variable v262  "HELP FR RELATIVES 8:24"  ;
label variable v268  "FOOD, CLOTH AS PAY 8:30"  ;
label variable v269  "FOOD, CLOTH FREE 8:31"  ;
label variable v334  "TOTAL FOOD CONS 8V334"  ;
label variable v337  "HEAD HOURLY EARN   8V337"  ;
label variable v338  "WIFE HOURLY EARN   8V338"  ;
label variable v360  "SEX-MARITAL STATUS 8V360"  ;
label variable v439  "1968 WEIGHT 8V439"  ;
label variable er30020  "1969 INTERVIEW NUMBER"  ;
label variable er30021  "SEQUENCE NUMBER                       69"  ;
label variable er30022  "RELATIONSHIP TO HEAD                  69"  ;
label variable er30023  "AGE OF INDIVIDUAL                     69"  ;
label variable er30042  "INDIVIDUAL WEIGHT                     69"  ;
label variable v442  "1969 INT NUMBER   11:6-9"  ;
label variable v465  "HRS HEAD WORKED 12:10-13"  ;
label variable v466  "ACCRCY HRS HD WRK  12:14"  ;
label variable v467  "HRS HEAD ILL    12:15-18"  ;
label variable v468  "ACCRCY HRS HD ILL  12:19"  ;
label variable v469  "HRS HD UNEMPLD  12:20-23"  ;
label variable v470  "ACCRCY HRS UNEMPLD 12:24"  ;
label variable v471  "$  HEAD TRVL WRK 12:25-28"  ;
label variable v472  "ACCRCY $  HEAD TRVL 12:29"  ;
label variable v473  "HRS HD TRVL WRK 12:30-32"  ;
label variable v474  "ACCRCY HRS HD TRVL 12:33"  ;
label variable v475  "HRS WIFE WORKED 12:34-37"  ;
label variable v476  "ACCRCY HRS WIFE WRK12:38"  ;
label variable v477  "$  WIFE TRVL WRK 12:39-42"  ;
label variable v478  "ACCRCY $  WIFE TRVL 12:43"  ;
label variable v479  "HRS WF TRVL WRK 12:44-46"  ;
label variable v480  "ACCRCY HRS WF TRVL 12:47"  ;
label variable v493  "# IN FAMILY (69)13:10-11"  ;
label variable v494  "FU FOOD STANDRD 13:12:15"  ;
label variable v495  "FU NEED STANDRD 13:16-20"  ;
label variable v496  "$  ANN ALCHL EXP 13:21-24"  ;
label variable v497  "ACCRCY $  ANN ALCHL 13:25"  ;
label variable v498  "$  ANN CIG EXPND 13:26-28"  ;
label variable v499  "ACCRCY $  ANN CIG   13:29"  ;
label variable v500  "$  FOOD USD HOME 13:30-33"  ;
label variable v501  "ACCRCY $  FOOD HOME 13:34"  ;
label variable v502  "$  MEALS WRK/SCH 13:35-38"  ;
label variable v503  "ACCRCY $  MEALS W/S 13:39"  ;
label variable v504  "$  SAVD MEALS W/S13:40-43"  ;
label variable v505  "ACCRCY $  SAVD MEALS13:44"  ;
label variable v506  "$  EATING OUT    13:45-48"  ;
label variable v507  "ACCRCY $  EATING OUT13:49"  ;
label variable v508  "$ SAVD GROW FOOD 13:50-53"  ;
label variable v509  "ACCRCY $  SAVD GR FD13:54"  ;
label variable v510  "$ SAVD FOOD STMP 13:55-58"  ;
label variable v511  "ACCRCY $ SVD FD STMP13:59"  ;
label variable v512  "$ SVD FREE FOOD  13:60-63"  ;
label variable v513  "ACCRCY $ SVD FREE FD13:64"  ;
label variable v514  "LABOR INC HEAD  14:10-14"  ;
label variable v515  "ACCRCY LBR INC HD  14:15"  ;
label variable v516  "LABOR INC WIFE  14:16-20"  ;
label variable v517  "ACCRCY LBR INC WF  14:21"  ;
label variable v523  "ADC,AFDC HEAD+W 14:38-42"  ;
label variable v524  "ACCRCY ADC INCOME  14:43"  ;
label variable v529  "TOTAL FU $  INC  14:56-60"  ;
label variable v542  "CHANGE IN FU COMP  15:25"  ;
label variable v549  "TOTAL # IN FU   15:32-33"  ;
label variable v553  "DATE OF IW      1  15:47"  ;
label variable v607  "MARITAL STATUS     16:42"  ;
label variable v608  "DID WIFE WORK      16-43"  ;
label variable v609  "OCCUPATION-WIFE    16-44"  ;
label variable v625  "ALCOH IN FOOD BILL 16-60"  ;
label variable v628  "CIGS IN FOOD BILL  16-63"  ;
label variable v639  "EMPLOYMENT STATUS  17:10"  ;
label variable v640  "OCCUPATION OF HEAD 17:11"  ;
label variable v641  "SELF-EMPLOYED HEAD 17:12"  ;
label variable v667  "MARGINAL WAGE 17:43-45"  ;
label variable v839  "FRE HELP FRM OTHRS 9V399"  ;
label variable v863  "TOTAL FOOD CONS 9V423"  ;
label variable v871  "HEADS HRLY EARN 9V431"  ;
label variable v873  "WIFES HRLY EARN 9V433"  ;
label variable v1008  "AGE OF HEAD 9V1008"  ;
label variable v1010  "SEX OF HEAD 9V1010"  ;
label variable v1011  "AGE OF WIFE 9V1011"  ;
label variable v1014  "1969 FAMILY WEIGHT"  ;
label variable v1015  "HOUSEHOLD ID # 69"  ;
label variable er30043  "1970 INTERVIEW NUMBER"  ;
label variable er30044  "SEQUENCE NUMBER                       70"  ;
label variable er30045  "RELATIONSHIP TO HEAD                  70"  ;
label variable er30046  "AGE OF INDIVIDUAL                     70"  ;
label variable er30066  "INDIVIDUAL WEIGHT                     70"  ;
label variable v1102  "1970 INT #        21:6-9"  ;
label variable v1106  "COVERSHEET COLOR 21:15"  ;
label variable v1109  "CHANGE IN FU COMP 21:18"  ;
label variable v1138  "HRS HEAD WORKED 22:10-13"  ;
label variable v1139  "ACC HRS HEAD WKD 22:14"  ;
label variable v1140  "HRS HEAD ILL    22:15-18"  ;
label variable v1141  "ACC HRS HEAD ILL   22:19"  ;
label variable v1142  "HRS HEAD UNEMP  22:20-23"  ;
label variable v1143  "ACC HRS HEAD UNEMP 22:24"  ;
label variable v1144  "$  HEAD TRVL WR  22:25-28"  ;
label variable v1145  "ACC $  HEAD TRVL 22:29"  ;
label variable v1146  "HRS HEAD TRVL WK22:30-32"  ;
label variable v1147  "ACC HRS HEAD TR WK 22:33"  ;
label variable v1148  "HRS WIFE WORKED 22:34-37"  ;
label variable v1149  "ACC HRS WIFE WK 22:38"  ;
label variable v1150  "$  WIFE TRVL WK  22:39-42"  ;
label variable v1151  "ACC $  WIFE TR WK 22:43"  ;
label variable v1152  "HRS WF TRVL WK  22:44-46"  ;
label variable v1153  "ACC HRS WF TRVL WK 22:47"  ;
label variable v1167  "# IN FU         23:10-11"  ;
label variable v1169  "FU WKLY FOOD STD23:13-17"  ;
label variable v1170  "ANNUAL NEED STD 23:18-22"  ;
label variable v1171  "$  ANNUAL ALCOHOL23:23-26"  ;
label variable v1172  "ACC $  ANN ALCOHOL 23:27"  ;
label variable v1173  "$  ANNUAL CIGS   23:28-30"  ;
label variable v1174  "ACC $  ANN CIGARETTE23:31"  ;
label variable v1175  "$  FOOD USED HOME23:32-35"  ;
label variable v1176  "ACC $  FOOD HOME 23:36"  ;
label variable v1177  "$  MEALS WRK/SCH 23:37-40"  ;
label variable v1178  "ACC $  MEALS W/S 23:41"  ;
label variable v1179  "$  SAVD GROW FOOD23:42-45"  ;
label variable v1180  "ACC $  SAVD GROW FD 23:46"  ;
label variable v1181  "$  SAVD WRK/SCH  23:47-50"  ;
label variable v1182  "ACC $  SAVD WRK/SCH 23:51"  ;
label variable v1183  "$  SAVD FOOD STMP23:52-55"  ;
label variable v1184  "ACC $  SAVD FD STMP 23:56"  ;
label variable v1185  "$  EATING OUT    23:57-60"  ;
label variable v1186  "ACC $  EATING OUT 23:61"  ;
label variable v1187  "$  SAVD OTHER FD 23:62-65"  ;
label variable v1188  "ACC $  SAVD OTHER FD23:66"  ;
label variable v1196  "LABOR INC HEAD  24:21-25"  ;
label variable v1197  "ACC LABOR Y HEAD   24:26"  ;
label variable v1198  "LABOR INC WIFE  24:27-31"  ;
label variable v1199  "ACC LABOR Y WIFE   24:32"  ;
label variable v1209  "ADC,AFDC H& W    25:10-14"  ;
label variable v1210  "ACC ADC H& W     25:15"  ;
label variable v1211  "OTHER WELFARE   25:16-20"  ;
label variable v1214  "UNEMP,WRK.COMP  25:31-35"  ;
label variable v1216  "BKT HELP FROM REL. 25:41"  ;
label variable v1236  "DATE OF IW      26:32"  ;
label variable v1238  "# IN FU         26:36-37"  ;
label variable v1239  "AGE OF HEAD     26:38-39"  ;
label variable v1240  "SEX OF HEAD     26:40"  ;
label variable v1241  "AGE OF WIFE     26:41-42"  ;
label variable v1242  "# CHILDREN IN FU   26:43"  ;
label variable v1278  "EMPLOYMENT STATUS  27:10"  ;
label variable v1279  "OCCUPATION-HEAD(E) 27:11"  ;
label variable v1280  "SELF EMPLOYED HEAD 27:12"  ;
label variable v1296  "WTR REG WAGE (E)   27:35"  ;
label variable v1297  "REG HRLY WAGE(E)27:36-38"  ;
label variable v1365  "MARITAL STATUS     29:10"  ;
label variable v1366  "DID WIFE WORK      29:11"  ;
label variable v1367  "OCCUPATION-WIFE    29:12"  ;
label variable v1514  "TOT FU MON INC OV414"  ;
label variable v1534  "FRE HELP FRM OTHRS 0V434"  ;
label variable v1559  "TOTAL FOOD CONS 0V459"  ;
label variable v1567  "HEADS AVG HRLY ERN 0V467"  ;
label variable v1569  "WIFE AV HRLY EARN 0V469"  ;
label variable v1605  "ANNUAL FOOD STD 0V505"  ;
label variable v1609  "WEIGHT             0V509"  ;
label variable v1764  "AMT PAID FOOD STAMPS"  ;
label variable v1765  "$  FOOD BOUGHT W/ STAMPS"  ;
label variable v1766  "HOUSEHOLD ID # 70"  ;
label variable er30067  "1971 INTERVIEW NUMBER"  ;
label variable er30068  "SEQUENCE NUMBER                       71"  ;
label variable er30069  "RELATIONSHIP TO HEAD                  71"  ;
label variable er30070  "AGE OF INDIVIDUAL                     71"  ;
label variable er30090  "INDIVIDUAL WEIGHT                     71"  ;
label variable v1802  "71 ID NO."  ;
label variable v1806  "COVERSHEET COLOR   33:16"  ;
label variable v1809  "CHANGE IN FU COMP  33:22"  ;
label variable v1839  "HRS HEAD WORKED 34:10-13"  ;
label variable v1840  "ACC HRS HD WKD     34:14"  ;
label variable v1841  "HRS HEAD ILL    34:15-18"  ;
label variable v1842  "ACC HRS HD ILL     34:19"  ;
label variable v1843  "HRS HEAD UNEMP  34:20-23"  ;
label variable v1844  "ACC HRS HEAD UNEMP 34:24"  ;
label variable v1845  "$  HEAD TRVL WK  34:25-28"  ;
label variable v1846  "ACC $  HEAD TRVL WK 34:29"  ;
label variable v1847  "HRS HEAD TRVL WK34:30-32"  ;
label variable v1848  "ACC HRS HEAD TR WK 34:33"  ;
label variable v1849  "HRS WIFE WORKED 34:34-37"  ;
label variable v1850  "ACC HRS WIFE WK    34:38"  ;
label variable v1851  "$  WIFE TRVL WK  34:39-42"  ;
label variable v1852  "ACC WIFE TRVL WK   34:43"  ;
label variable v1853  "HRS WF TRVL WK  34:44-46"  ;
label variable v1854  "ACC HRS WF TRVL WK 34:47"  ;
label variable v1868  "# IN FU         35:10-11"  ;
label variable v1870  "FU WKLY FOOD STD35:13-17"  ;
label variable v1871  "ANNUAL NEED STD 23:18-22"  ;
label variable v1872  "$  ANNUAL ALCOHOL35:23-26"  ;
label variable v1873  "ACC $  ANN ALCOHOL  35:27"  ;
label variable v1874  "$  ANNUAL CIGS   35:28-30"  ;
label variable v1875  "ACC $  ANN CIGARETTE35:31"  ;
label variable v1876  "$  FOOD USED HOME35:32-35"  ;
label variable v1877  "ACC $  FOOD HOME    35:36"  ;
label variable v1878  "$  MEALS WRK/SCH 35:37-40"  ;
label variable v1879  "ACC $  MEALS W/S    35:41"  ;
label variable v1880  "$  SAVD GROW FOOD35:42-45"  ;
label variable v1881  "ACC $  SAVD GROW FD 35:46"  ;
label variable v1882  "$  SAVD WRK/SCH  35:47-50"  ;
label variable v1883  "ACC $  SAVD WRK/SCH 35:51"  ;
label variable v1884  "$  SAVD FOOD STMP35:52-55"  ;
label variable v1885  "ACC $  SAVD FD STMP 35:56"  ;
label variable v1886  "$  EATING OUT    35:57-60"  ;
label variable v1887  "ACC $  EATING OUT   35:61"  ;
label variable v1888  "$  SAVD OTHER FD 35:62-65"  ;
label variable v1889  "ACC $  SAVD OTHER FD35:66"  ;
label variable v1897  "LABOR INC HEAD  36:21-25"  ;
label variable v1898  "ACC LABOR Y HEAD   36:26"  ;
label variable v1899  "LABOR INC WIFE  36:27-31"  ;
label variable v1900  "ACC LABOR Y WIFE   36:32"  ;
label variable v1911  "ADC, AFDC H& W   37:10-14"  ;
label variable v1912  "ACC ADC H& W        37:15"  ;
label variable v1913  "OTHER WELFARE   37:16-20"  ;
label variable v1916  "UNEMP,WRK.COMP  37:31-35"  ;
label variable v1939  "DATE OF IW         38:36"  ;
label variable v1941  "# IN FU         38:40-41"  ;
label variable v1942  "AGE OF HEAD     38:42-43"  ;
label variable v1943  "SEX OF HEAD        38:44"  ;
label variable v1944  "AGE OF WIFE     38:45-46"  ;
label variable v1945  "# CHILDREN IN FU   38:47"  ;
label variable v1983  "EMPLOYMENT STATUS  39:27"  ;
label variable v1984  "OCCUPATION-HEAD(E) 39:28"  ;
label variable v1985  "INDUS-HEAD OCC  39:29-30"  ;
label variable v1986  "SELF EMPLOYED HEAD 39:31"  ;
label variable v2002  "WTR REG WAGE (E)   39:54"  ;
label variable v2003  "REG HRLY WAGE(E)39:55-57"  ;
label variable v2072  "MARITAL STATUS     41:34"  ;
label variable v2073  "DID WIFE WORK      41:35"  ;
label variable v2074  "OCCUPATION-WIFE    41:36"  ;
label variable v2226  "TOT FU MON INC 1V426"  ;
label variable v2271  "TOTAL FOOD CONS 1V471"  ;
label variable v2279  "HEADS AVG HRLY ERN 1V479"  ;
label variable v2281  "WIFE AV HRLY EARN 1V481"  ;
label variable v2317  "ANNUAL FOOD STD 1V517"  ;
label variable v2321  "WEIGHT             1V521"  ;
label variable v2345  "HOUSEHOLD ID # 71"  ;
label variable er30091  "1972 INTERVIEW NUMBER"  ;
label variable er30092  "SEQUENCE NUMBER                       72"  ;
label variable er30093  "RELATIONSHIP TO HEAD                  72"  ;
label variable er30094  "AGE OF INDIVIDUAL                     72"  ;
label variable er30116  "INDIVIDUAL WEIGHT                     72"  ;
label variable v2402  "1972 INT #        46:6-9"  ;
label variable v2407  "COVERSHEET COLOR   46:18"  ;
label variable v2410  "CHANGE IN FU COMP  46:24"  ;
label variable v2439  "HRS HEAD WORKED 47:10-13"  ;
label variable v2440  "ACC HRS HD WKD     47:14"  ;
label variable v2441  "HRS HEAD ILL    47:15-18"  ;
label variable v2442  "ACC HRS HD ILL     47:19"  ;
label variable v2443  "HRS HEAD UNEMP  47:20-23"  ;
label variable v2444  "ACC HRS HEAD UNEMP 47:24"  ;
label variable v2445  "$  HEAD TRVL WK  47:25-28"  ;
label variable v2446  "ACC $  HEAD TRVL WK 47:29"  ;
label variable v2447  "HRS HD TRVL WK  47:30-32"  ;
label variable v2448  "ACC HRS HEAD TR WK 47:33"  ;
label variable v2449  "HRS WIFE WORKED 47:34-37"  ;
label variable v2450  "ACC HRS WIFE WK    47:38"  ;
label variable v2451  "$  WIFE TRVL WK  47:39-42"  ;
label variable v2452  "ACC $  WIFE TRVL WK 47:43"  ;
label variable v2453  "HRS WF TRVL WK  47:44-46"  ;
label variable v2454  "ACC HRS WF TRVL WK 47:47"  ;
label variable v2468  "# IN FU         48:10-11"  ;
label variable v2470  "FU WKLY FOOD STD48:13-17"  ;
label variable v2471  "ANNUAL NEED STD 48:18-22"  ;
label variable v2472  "$  ANNUAL ALCOHOL48:23-26"  ;
label variable v2473  "ACC $  ANA ALCOHOL  48:27"  ;
label variable v2474  "$  ANNUAL CIGS   48:28-30"  ;
label variable v2475  "ACC $  ANN CIGARETTE48:31"  ;
label variable v2476  "$  FOOD USED HOME48:32-35"  ;
label variable v2477  "ACC $  FOOD HOME    48:36"  ;
label variable v2478  "$  SAVD FD STMP  48:37-40"  ;
label variable v2479  "ACC $  SAVD FD STMP 48:41"  ;
label variable v2480  "$  EATING OUT    48:42-45"  ;
label variable v2481  "ACC $  EATING OUT   48:46"  ;
label variable v2482  "STMP SUBSIDY INCL? 48:47"  ;
label variable v2483  "$  MEALS WRK/SCH 48:48-51"  ;
label variable v2484  "ACC $  MEALS W/S    48:52"  ;
label variable v2485  "$  SAVD GROW FOOD48:53-56"  ;
label variable v2486  "ACC $  SAVD GROW FD 48:57"  ;
label variable v2487  "$  SAVD WRK/SCH  48:58-61"  ;
label variable v2488  "$  ACC SAVD WRK/SCH 48:62"  ;
label variable v2489  "$  SAVD OTHER FD 48:63-66"  ;
label variable v2490  "ACC $  SAVD OTHER FD48:67"  ;
label variable v2498  "LABOR INC HEAD  49:21-25"  ;
label variable v2499  "ACC LABOR Y HEAD   49:26"  ;
label variable v2500  "LABOR INC WIFE  49:27-31"  ;
label variable v2501  "ACC LABOR Y WIFE   49:32"  ;
label variable v2512  "ADC, AFDC H& W   50:10-14"  ;
label variable v2513  "ACC ADC H& W        50:15"  ;
label variable v2514  "OTHER WELFARE   50:16-20"  ;
label variable v2517  "UNEMP,WRK.COMP  50:31-35"  ;
label variable v2519  "BKT HELP FROM REL. 50:41"  ;
label variable v2539  "DATE OF IW         51:37"  ;
label variable v2541  "# IN FU         51:41-42"  ;
label variable v2542  "AGE OF HEAD     51:43-44"  ;
label variable v2543  "SEX OF HEAD        51:45"  ;
label variable v2544  "AGE OF WIFE     51:46-47"  ;
label variable v2545  "# CHILDREN IN FU51:48-49"  ;
label variable v2581  "EMPLOYMENT STATUS  52:19"  ;
label variable v2582  "OCCUPATION-HEAD(E) 52:20"  ;
label variable v2583  "INDUS-HEADS OCC 52:21-22"  ;
label variable v2584  "SELF EMPLOYED HEAD 52:23"  ;
label variable v2600  "WTR REG WAGE       52:46"  ;
label variable v2601  "REG HRLY WAGE(E)52:47-49"  ;
label variable v2670  "MARITAL STATUS     54:10"  ;
label variable v2671  "DID WIFE WORK?     54:11"  ;
label variable v2672  "OCCUPATION-WIFE    54:12"  ;
label variable v2852  "TOT FU MON INC     2V452"  ;
label variable v2872  "FRE HELP FRM OTHRS 2V472"  ;
label variable v2892  "ANNUAL FOOD STD    2V492"  ;
label variable v2898  "TOTAL FOOD CONS    2V498"  ;
label variable v2906  "HEADS AVG HRLY ERN 2V506"  ;
label variable v2908  "WIFE AVG HRLY EARN 2V508"  ;
label variable v2968  "1972 WEIGHTS"  ;
label variable v2978  "UNIQUE LINK TO 1968"  ;
label variable v2979  "HOUSEHOLD ID # 72"  ;
label variable er30117  "1973 INTERVIEW NUMBER"  ;
label variable er30118  "SEQUENCE NUMBER                       73"  ;
label variable er30119  "RELATIONSHIP TO HEAD                  73"  ;
label variable er30120  "AGE OF INDIVIDUAL                     73"  ;
label variable er30137  "INDIVIDUAL WEIGHT                     73"  ;
label variable v3002  "1973 INT #        59:6-9"  ;
label variable v3007  "COVERSHEET COLOR   59:18"  ;
label variable v3010  "FAM COMP CHANGE    59:24"  ;
label variable v3114  "EMPLOYMENT STATUS  64:10"  ;
label variable v3301  "1973 FAMILY WEIGHT"  ;
label variable v3310  "HOUSEHOLD ID # 73"  ;
label variable er30138  "1974 INTERVIEW NUMBER"  ;
label variable er30139  "SEQUENCE NUMBER                       74"  ;
label variable er30140  "RELATIONSHIP TO HEAD                  74"  ;
label variable er30141  "AGE OF INDIVIDUAL                     74"  ;
label variable er30159  "INDIVIDUAL WEIGHT                     74"  ;
label variable v3402  "1974 ID NUMBER"  ;
label variable v3407  "COLOR COVERSHEET   69:18"  ;
label variable v3410  "FAM COMP CHG       69:24"  ;
label variable v3423  "ANN WRK HRS-HD  69:42-45"  ;
label variable v3424  "ACC ANN WK HRS     69:46"  ;
label variable v3425  "ANN ILLNESS HRS 69:47-50"  ;
label variable v3426  "ACC ANN ILL HRS    69:51"  ;
label variable v3427  "ANN UNEMPL HRS  69:52-55"  ;
label variable v3428  "ACC ANN UNEMP HRS  69:56"  ;
label variable v3429  "ANN HRS TRAV-WK 69:57-59"  ;
label variable v3430  "ACC HRS TRAV TO WK 69:60"  ;
label variable v3431  "ANN WRK HRS-WIFE69:61-64"  ;
label variable v3432  "ACC ANN WF WK HRS  69:65"  ;
label variable v3437  "FAMILY SIZE     70:16-17"  ;
label variable v3439  "WEEKLY FD NEED  70:19-23"  ;
label variable v3440  "ANN NEED STND   70:24-28"  ;
label variable v3441  "ANN FOOD AT HOME70:29-32"  ;
label variable v3442  "ACC FOOD AT HOME   70:33"  ;
label variable v3443  "AMT SAVD FD STMP70:34-37"  ;
label variable v3444  "ACC AMT SVD FD STMP70:38"  ;
label variable v3445  "ANN $  EATNG OUT 70:39-42"  ;
label variable v3446  "ACC $  EATING OUT   70:43"  ;
label variable v3447  "WTR INCL VAL STAMPS70:44"  ;
label variable v3463  "TOT LABOR INC-HD71:21-25"  ;
label variable v3464  "ACC LABOR INC-HD   71:26"  ;
label variable v3465  "TOT LABOR INC-WF71:27-31"  ;
label variable v3466  "ACC LABOR INC-WF   71:32"  ;
label variable v3477  "ANN ADC,AFDC Y  72:10-14"  ;
label variable v3478  "ACC ADC,AFDC INC   72:15"  ;
label variable v3479  "ANN OTR WELFARE 72:16-20"  ;
label variable v3482  "ANN UNEMP COMP  72:31-35"  ;
label variable v3484  "BKT HELP FR RELATS 72:41"  ;
label variable v3505  "DATE OF INTERVIEW  73:45"  ;
label variable v3507  "# IN FU         73:49-50"  ;
label variable v3508  "AGE OF HEAD     73:51-52"  ;
label variable v3509  "SEX OF HEAD        73:53"  ;
label variable v3510  "AGE OF WIFE     73:54-55"  ;
label variable v3511  "# KIDS IN FU    73:56-57"  ;
label variable v3528  "EMPLOYMENT STATUS  74:10"  ;
label variable v3529  "3-DIGIT OCC (E) 74:11-13"  ;
label variable v3530  "1-DIGIT OCC (E)    74:14"  ;
label variable v3531  "INDUSTRY (E)    74:15-16"  ;
label variable v3532  "WTR SELF-EMPLD (E) 74:17"  ;
label variable v3548  "WTR HRLY WAGE (E)  74:40"  ;
label variable v3549  "HRLY WAGE MAIN-E74:41-43"  ;
label variable v3598  "MARITAL STATUS     76:10"  ;
label variable v3599  "DID WIFE WORK?     76:11"  ;
label variable v3600  "3-DIGIT OCC-WF  76:12-14"  ;
label variable v3601  "1-DIGIT OCC-WIFE   76:15"  ;
label variable v3602  "INDUSTRY-WF     76:16-17"  ;
label variable v3676  "TOT FU MON INC     4V276"  ;
label variable v3694  "ANNUAL FOOD STD    4V294"  ;
label variable v3695  "HDS AVG HRLY EARN  4V295"  ;
label variable v3697  "WIFE AVG HRLY EARN 4V297"  ;
label variable v3721  "1974 FAMILY WEIGHT"  ;
label variable v3730  "HOUSEHOLD ID # 74"  ;
label variable er30160  "1975 INTERVIEW NUMBER"  ;
label variable er30161  "SEQUENCE NUMBER                       75"  ;
label variable er30162  "RELATIONSHIP TO HEAD                  75"  ;
label variable er30163  "AGE OF INDIVIDUAL                     75"  ;
label variable er30187  "INDIVIDUAL WEIGHT                     75"  ;
label variable v3802  "1975 INT #        80:6-9"  ;
label variable v3807  "COVERSHEET COLOR   80:18"  ;
label variable v3810  "CHANGE IN FU COMP  80:24"  ;
label variable v3823  "HRS HD WORKED   80:43-46"  ;
label variable v3824  "ACC HRS HD WORKED  80:47"  ;
label variable v3825  "HRS HD ILL      80:48-51"  ;
label variable v3826  "ACC HRS HD ILL     80:52"  ;
label variable v3827  "HRS HD UNEMP    80:53-56"  ;
label variable v3828  "ACC HRS HD UNEM    80:57"  ;
label variable v3829  "HRS HD TRVL WK  80:58-60"  ;
label variable v3830  "ACC HRS HD TR WK   80:61"  ;
label variable v3831  "HRS WIFE WORKED 81:10-13"  ;
label variable v3832  "ACC HRS WIFE WK    81:14"  ;
label variable v3833  "HRS WIFE UNEMP  81:15-18"  ;
label variable v3834  "ACC HRS WF UNEMP   81:19"  ;
label variable v3835  "HRS WF TRVL WK  81:20-22"  ;
label variable v3836  "ACC HRS WF TR WK   81:23"  ;
label variable v3837  "# IN FU         81:24-25"  ;
label variable v3839  "FU WKLY FOOD STD81:27-31"  ;
label variable v3840  "ANNUAL NEED STD 81:32-36"  ;
label variable v3841  "$  FOOD USED HOME81:37-40"  ;
label variable v3842  "ACC $  FOOD HOME    81:41"  ;
label variable v3849  "$  FD STMP 1974  81:54-57"  ;
label variable v3850  "ACC $  FDSTMP 1974  81:58"  ;
label variable v3851  "$  SAVD FDST 74  81:59-62"  ;
label variable v3852  "ACC $  SAVD FD STMP 81:63"  ;
label variable v3853  "$  EATING OUT    81:64-67"  ;
label variable v3854  "ACC $  EATING OUT   81:68"  ;
label variable v3863  "HEAD LABOR Y    82:21-25"  ;
label variable v3864  "ACC HD LABOR Y     82:26"  ;
label variable v3865  "WIFE LABOR Y    82:27-31"  ;
label variable v3866  "ACC WIFE LABOR Y   82:32"  ;
label variable v3877  "ADC,AFDC H+W    83:10-14"  ;
label variable v3878  "ACC ADC H+W        83:15"  ;
label variable v3879  "OTH WELFARE H+W 83:16-20"  ;
label variable v3882  "HD UNEMP,WKMNCMP83:31-35"  ;
label variable v3884  "H+W SSI         83:41-45"  ;
label variable v3885  "BKT HD HELP RELAT  83:46"  ;
label variable v3896  "ADC,AFDC OTHERS 84:30-34"  ;
label variable v3897  "OTH WELF OTHERS 84:35-39"  ;
label variable v3900  "UNEMP OF OTHERS 84:50-54"  ;
label variable v3902  "SSI OF OTHERS   84:60-64"  ;
label variable v3903  "BKT OTR HELP-RELAT 84:65"  ;
label variable v3918  "DATE OF INT        85:49"  ;
label variable v3920  "# IN FU         85:53-54"  ;
label variable v3921  "AGE OF HEAD     85:55-56"  ;
label variable v3922  "SEX OF HEAD        85:57"  ;
label variable v3923  "AGE OF WIFE     85:58-59"  ;
label variable v3924  "# KIDS IN FU    85:60-61"  ;
label variable v3967  "EMPLOYMENT STATUS  87:10"  ;
label variable v3968  "OCC-HD (E)         87:11"  ;
label variable v3969  "IND-HD (E)      87:12-13"  ;
label variable v3970  "SELF-EMP OR? (E)   87:14"  ;
label variable v4001  "HRLY WAGE-OTR(E)87:53-55"  ;
label variable v4002  "WTR HRLY WAGE REG-E87:56"  ;
label variable v4003  "HRLY WAGE REG(E)87:57-59"  ;
label variable v4053  "MAR STATUS         89:10"  ;
label variable v4054  "WTR WF WKD         89:11"  ;
label variable v4055  "OCC-WF             89:12"  ;
label variable v4056  "IND-WF          89:13-14"  ;
label variable v4154  "TOT FU MON INC     5V354"  ;
label variable v4173  "ANNUAL FOOD STD    5V373"  ;
label variable v4174  "HDS AVG HRLY EARN  5V374"  ;
label variable v4176  "WIFE AVG HRLY EARN 5V376"  ;
label variable v4198  "HEAD EDUCATION BKT 5V398"  ;
label variable v4199  "WIFE EDUCATION BKT 5V399"  ;
label variable v4206  "8 YR FAM COMP CHNG 5V406"  ;
label variable v4224  "1975 FAMILY WEIGHT"  ;
label variable v4231  "HOUSEHOLD ID # 75"  ;
label variable er30188  "1976 INTERVIEW NUMBER"  ;
label variable er30189  "SEQUENCE NUMBER                       76"  ;
label variable er30190  "RELATIONSHIP TO HEAD                  76"  ;
label variable er30191  "AGE OF INDIVIDUAL                     76"  ;
label variable er30216  "INDIVIDUAL WEIGHT                     76"  ;
label variable v4302  "1976 ID NUMBER       6V2"  ;
label variable v4307  "COLOR OF COVERSHEET  6V7"  ;
label variable v4310  "FAM COMP CHANGE     6V10"  ;
label variable v4332  "ANN WORK HOURS      6V32"  ;
label variable v4333  "ACC ANN WK HOURS    6V33"  ;
label variable v4334  "ANN ILL HRS OTHERSD40E21"  ;
label variable v4335  "ACC ANN ILL HRS OT  6V35"  ;
label variable v4336  "ANN ILL HRS SELF  D42E23"  ;
label variable v4337  "ACC ANN ILL HRS SELF6V37"  ;
label variable v4338  "ANN UNEMPLOYMENT  D49E27"  ;
label variable v4339  "ACC ANN UNEMPLOY    6V39"  ;
label variable v4340  "ANN STRIKE HOURS  D47E25"  ;
label variable v4341  "ACC ANN STRIKE HRS  6V41"  ;
label variable v4342  "HD ANN TRAVL TO WKHR6V42"  ;
label variable v4343  "ACC H ANN TRAV WRK  6V43"  ;
label variable v4344  "WIFE ANN WORK HRS   G5G6"  ;
label variable v4345  "ACC WIFE ANN WRK H  6V45"  ;
label variable v4346  "FAMILY SIZE         6V46"  ;
label variable v4348  "WEEKLY FOOD NEED    6V48"  ;
label variable v4349  "ANNUAL NEED STAND   6V49"  ;
label variable v4354  "ANN FOOD EXPENDITRG13G15"  ;
label variable v4355  "ACC ANN FOOD EXPEN  6V55"  ;
label variable v4362  "$  SPENT FDSTA LASTYR G22"  ;
label variable v4363  "ACC $  PAID FS 1975  6V63"  ;
label variable v4364  "$  SAV FDSTAMP 1975G22G23"  ;
label variable v4365  "ACC $  SAV FDST  75  6V65"  ;
label variable v4368  "$  SPENT MEALS OUT 75 G26"  ;
label variable v4369  "ACC $  SPENT MEALS OUTV69"  ;
label variable v4392  "ANNUAL ADC AFDC H& W H11D"  ;
label variable v4393  "ACC ANN ADC AFDC    6V93"  ;
label variable v4394  "ANN OTH WELFARE H& W H11E"  ;
label variable v4397  "HD  UNEMP WRK COMP  H11H"  ;
label variable v4399  "HD  HELP FROM REL   H11J"  ;
label variable v4400  "HD  SUPPLE SECURIT  H11K"  ;
label variable v4410  "ADC AFDC Y OTHERS H28H38"  ;
label variable v4411  "OTH WELFARE Y OTH H28H38"  ;
label variable v4414  "UNEMP Y OTHERS    H28H38"  ;
label variable v4416  "SSI Y OTHERS      H28H38"  ;
label variable v4417  "HELP FRM RELA OTR H28H38"  ;
label variable v4433  "DATE OF INTERVIEW  6V133"  ;
label variable v4435  "NUMBER IN FU       6V135"  ;
label variable v4436  "AGE OF HEAD        6V136"  ;
label variable v4437  "SEX OF HEAD        6V137"  ;
label variable v4438  "AGE OF WIFE        6V138"  ;
label variable v4439  "# OF CHILDREN FU   6V139"  ;
label variable v4458  "HD EMPLOYMENT STATUS  D1"  ;
label variable v4459  "HEAD OCCUPATION       D2"  ;
label variable v4461  "WRK SOMEONEELSE/SELF  D5"  ;
label variable v4509  "HOW PAID MAIN JOB    D55"  ;
label variable v4510  "$ /HOUR  SALARY       D56"  ;
label variable v4513  "$ /HOUR HOURLY PAY    D59"  ;
label variable v4603  "MARITAL STATUS        G1"  ;
label variable v4604  "DID WIFE WRK FOR $  75 G2"  ;
label variable v4605  "OCC WIFE              G3"  ;
label variable v4708  "1976 ID NUMBER-WF"  ;
label variable v4719  "ANN WORK HRS-WIFE"  ;
label variable v4720  "ACC ANN WRK HRS-WF"  ;
label variable v4721  "ANN ILL HRS-OTR  D40,E22"  ;
label variable v4722  "ACC ANN ILL HR OTR"  ;
label variable v4723  "ANN ILL HRS-WF   D42,E24"  ;
label variable v4724  "ACC ANN ILL HR-WF"  ;
label variable v4725  "ANN STRIKE HOURS D47,E26"  ;
label variable v4726  "ACC ANN STRIKE HRS"  ;
label variable v4727  "ANN UNEMP HRS-WF  D49E28"  ;
label variable v4728  "ACC ANN UNEMP HRS-WF"  ;
label variable v4729  "ANN HR TRAVEL TO WRK-WF"  ;
label variable v4730  "ACC HR TRAV WK-WF"  ;
label variable v4741  "DATE OF WF INT"  ;
label variable v4743  "AGE OF WIFE"  ;
label variable v4841  "EMPLOYMENT STATUS-WF  D1"  ;
label variable v4842  "OCCUPATION(E)-WF      D2"  ;
label variable v4844  "WRK FOR SELF OR?(E)-WFD5"  ;
label variable v4892  "SALARY/HRLY OR?(E)-WFD55"  ;
label variable v4893  "PAY/HR-SALARIED (E)  D56"  ;
label variable v4896  "HRLY WAGE-HRLY EMPL  D59"  ;
label variable v4897  "HRLY WAGE-OT HRLY EMPD60"  ;
label variable v5029  "TOT FU MONEY INC   6V729"  ;
label variable v5031  "HEAD TOTAL LABOR Y 6V731"  ;
label variable v5049  "ANNUAL FOOD STD    6V749"  ;
label variable v5050  "HD AVG HOURLY EARN 6V750"  ;
label variable v5052  "WIFE AVG HRLY EARN 6V752"  ;
label variable v5074  "HEAD EDUCATION BKT 6V774"  ;
label variable v5075  "WIFE EDUCATION BKT 6V775"  ;
label variable v5098  "9YR FAM COMP CHNG  6V798"  ;
label variable v5099  "1976 FAMILY WEIGHT"  ;
label variable v5113  "HOUSEHOLD ID # 76"  ;
label variable er30217  "1977 INTERVIEW NUMBER"  ;
label variable er30218  "SEQUENCE NUMBER                       77"  ;
label variable er30219  "RELATIONSHIP TO HEAD                  77"  ;
label variable er30220  "AGE OF INDIVIDUAL                     77"  ;
label variable v5202  "1977 ID"  ;
label variable v5207  "COLOR OF COVERSHEET  77"  ;
label variable v5210  "FAM COMP CHNG 1977"  ;
label variable v5232  "HD ANN WRK HRS IN 76"  ;
label variable v5233  "ACC HD 76 WRK HRS"  ;
label variable v5234  "HD HRS WRK LOST OTH ILL"  ;
label variable v5235  "ACC HD HR LOST OTH ILL"  ;
label variable v5236  "HD HRS WRK LOST OWN ILL"  ;
label variable v5237  "ACC HD HR LOST OWN ILL"  ;
label variable v5238  "HEAD STRIKE HOURS 76"  ;
label variable v5239  "ACC HD STRIKE HRS 76"  ;
label variable v5240  "HEAD UNEMPLMT HRS 76"  ;
label variable v5241  "ACC HD UNEMPLNT HRS 76"  ;
label variable v5242  "HEAD TRAV TO WORK HRS 76"  ;
label variable v5243  "ACC HD TRAV TO WRK HR 76"  ;
label variable v5244  "WIFE ANN WORK HOURS 1976"  ;
label variable v5245  "ACC WIFE WORK HRS  1976"  ;
label variable v5246  "WF HRS WRK LOST OTH ILL"  ;
label variable v5247  "ACC WF HR LOST OTH ILL"  ;
label variable v5248  "WF HRS WRK LOST OWN ILL"  ;
label variable v5249  "ACC WF HR LOST OWN ILL"  ;
label variable v5250  "WIFE STRIKE HOURS 1976"  ;
label variable v5251  "ACC WF STRIKE HRS 1976"  ;
label variable v5252  "WIFE UNEMPLMNT HRS 76"  ;
label variable v5253  "ACC WF UNEMPLMNT HR 76"  ;
label variable v5254  "1977 FAMILY SIZE"  ;
label variable v5256  "1977 WEEKLY FOOD NEED"  ;
label variable v5257  "1977 ANNUAL NEED STD"  ;
label variable v5267  "$  PAID FD STAMP LAST MO"  ;
label variable v5268  "ACC $ PD FD STAMP LAST MO"  ;
label variable v5269  "$  BONUS FD STAMP LAST MO"  ;
label variable v5270  "ACC $ BONUS FDSTMP LASTMO"  ;
label variable v5271  "ANNUAL FOOD$  EXCL FDSTMP"  ;
label variable v5272  "ACC ANN FOOD $"  ;
label variable v5273  "ANNUAL $  EATING OUT"  ;
label variable v5274  "ACC ANN $  EATING OUT"  ;
label variable v5275  "$  PAID FD STAMPS 1976"  ;
label variable v5276  "ACC $  PAID FD STAMPS  76"  ;
label variable v5277  "BONUS $  FD STAMPS 1976"  ;
label variable v5278  "ACC BONUS $  FDSTAMS 1976"  ;
label variable v5303  "H+W 1976 ADC,AFDC"  ;
label variable v5304  "ACC H+W 1976 ADC,AFDC"  ;
label variable v5305  "H+W 1976 OTHER WELFARE"  ;
label variable v5308  "HD 1976 UNEMPLOYMENTCOMP"  ;
label variable v5309  "HD 1976 WRKMANS COMP"  ;
label variable v5311  "HD 1976 HELP FRM RELATIV"  ;
label variable v5312  "H+W 1976 SSI"  ;
label variable v5322  "OTH 76 ADC,AFDC $"  ;
label variable v5323  "OTH 76 OTR WELFARE $"  ;
label variable v5326  "OTH 76 UNEMPLOYMENT $"  ;
label variable v5327  "OTH 76 WORKMAN COMP $"  ;
label variable v5329  "OTH 76 SSI $"  ;
label variable v5330  "OTH 76 HELP RELATIV $"  ;
label variable v5347  "DATE OF INT"  ;
label variable v5349  "# IN FU"  ;
label variable v5350  "AGE OF HEAD"  ;
label variable v5351  "SEX OF HEAD"  ;
label variable v5352  "AGE OF WIFE"  ;
label variable v5353  "# KIDS IN FU"  ;
label variable v5373  "EMPLOYMENT STATUS     D1"  ;
label variable v5376  "WORK FOR SELF/OTR?(E) D5"  ;
label variable v5420  "SALARIED HRLY OR (E) D50"  ;
label variable v5421  "PAY/HR SALARY        D51"  ;
label variable v5424  "PAY/HR HRLY (E)      D54"  ;
label variable v5477  "RET/DIS,HSWF,STUD (R) F1"  ;
label variable v5479  "WRK FOR MONEY? (R)    F3"  ;
label variable v5502  "MARITAL STAT          G1"  ;
label variable v5506  "WTR WF WKD            G5"  ;
label variable v5626  "TOT 1976 FAM MONEY INCOM"  ;
label variable v5627  "TOT 1976 LABOR INCM HEAD"  ;
label variable v5630  "ANNUAL FOOD STANDARD"  ;
label variable v5631  "1976 AVG HRLY ERNGS-HEAD"  ;
label variable v5632  "1976 AVG HRLY ERNGS-WIFE"  ;
label variable v5647  "EDUCATION OF HEAD"  ;
label variable v5648  "EDUCATION OF WIFE"  ;
label variable v5650  "MARITAL STATUS"  ;
label variable v5664  "10 YR CHNGS IN FU COMP"  ;
label variable v5665  "1977 FAMILY WEIGHT"  ;
label variable v5672  "1968-69 CHNG MARITAL STA"  ;
label variable v5673  "1969-70 CHNG MARITAL STA"  ;
label variable v5674  "1970-71 CHNG MARITAL STA"  ;
label variable v5675  "1971-72 CHNG MARITAL STA"  ;
label variable v5676  "1972-73 CHNG MARITAL STA"  ;
label variable v5677  "1973-74 CHNG MARITAL STA"  ;
label variable v5678  "1974-75 CHNG MARITAL STA"  ;
label variable v5679  "1975-76 CHNG MARITAL STA"  ;
label variable v5680  "1976-77 CHNG MARITAL STA"  ;
label variable er30246  "1978 INTERVIEW NUMBER"  ;
label variable er30247  "SEQUENCE NUMBER                       78"  ;
label variable er30248  "RELATIONSHIP TO HEAD                  78"  ;
label variable er30249  "AGE OF INDIVIDUAL                     78"  ;
label variable v5702  "1978 ID"  ;
label variable v5707  "COLOR OF COVERSHEET 78"  ;
label variable v5710  "FAMILY COMP CHNG 1978"  ;
label variable v5731  "HD ANN WRK HRS IN 77"  ;
label variable v5732  "ACC HD 77 WRK HRS"  ;
label variable v5733  "HD HRS WRK LOST OTH ILL"  ;
label variable v5734  "ACC HD HR LOST OTH ILL"  ;
label variable v5735  "HD HRS WRK LOST OWN ILL"  ;
label variable v5736  "ACC HD HR LOST OWN ILL"  ;
label variable v5737  "HD STRIKE HRS 77"  ;
label variable v5738  "ACC HD STRIKE HRS 77"  ;
label variable v5739  "HD UNEMP HRS 77"  ;
label variable v5740  "ACC HD UNEMP HRS 77"  ;
label variable v5741  "HD TRAV TO WRK HRS 77"  ;
label variable v5742  "ACC HD TRAV TO WRK HR 77"  ;
label variable v5743  "WF ANN WRK HRS 1977"  ;
label variable v5744  "ACC WF ANN WRK HRS 1977"  ;
label variable v5745  "WF HRS WRK LOST OTH ILL"  ;
label variable v5746  "ACC WF HR LOST OTH ILL"  ;
label variable v5747  "WF HRS WRK LOST SELF ILL"  ;
label variable v5748  "ACC WF HR LOST SELF ILL"  ;
label variable v5749  "WF STRIKE HRS 1977"  ;
label variable v5750  "ACC WF STRIKE HRS 1977"  ;
label variable v5751  "WF UNEMP HRS 1977"  ;
label variable v5752  "ACC WF UNEMP HRS 1977"  ;
label variable v5753  "WF TRAV TO WRK HRS 77"  ;
label variable v5754  "ACC WF TRAV TO WRK HR 77"  ;
label variable v5755  "1978 FAMILY SIZE"  ;
label variable v5757  "1978 WEEKLY FOOD NEED"  ;
label variable v5758  "1978 ANN NEED STANDARD"  ;
label variable v5766  "$  PAID FD ST LAST MO"  ;
label variable v5767  "ACC $  PAID FD ST LAST MO"  ;
label variable v5768  "$  BONUS PD ST LAST MO"  ;
label variable v5769  "ACC BONUS FDSTMP LST MO"  ;
label variable v5770  "ANN FD $  EXCL FDSTMP"  ;
label variable v5771  "ACC ANN FOOD $"  ;
label variable v5772  "ANN $  EATING OUT"  ;
label variable v5773  "ACC ANN $  EATING OUT"  ;
label variable v5774  "$  PAID FDSTMP 1977"  ;
label variable v5775  "ACC $  PAID FDSTMP 1977"  ;
label variable v5776  "$  BONUS FDSTMP 1977"  ;
label variable v5777  "ACC $  BONUS FDSTMP 1977"  ;
label variable v5802  "H+W 1977 ADC,AFDC"  ;
label variable v5803  "ACC H+W 1977 ADC,AFDC"  ;
label variable v5804  "H+W 1977 SSI"  ;
label variable v5805  "H+W 1977 OTHER WELFARE"  ;
label variable v5808  "HD 1977 UNEMP COMP"  ;
label variable v5809  "HD 1977 WRKMANS COMP"  ;
label variable v5811  "HD 1977 HELP FROM RELAT"  ;
label variable v5821  "OTH 1977 ADC/AFDC"  ;
label variable v5822  "OTH 1977 SSI"  ;
label variable v5823  "OTH 1977 OTHER WELFARE"  ;
label variable v5826  "OTH 1977 UNEPL COMP"  ;
label variable v5827  "OTH 1977 WRKMANS COMP"  ;
label variable v5829  "OTH 1977 HLP FRM RELATIV"  ;
label variable v5847  "DATE OF INT"  ;
label variable v5849  "# IN FU"  ;
label variable v5850  "AGE OF HEAD"  ;
label variable v5851  "SEX OF HEAD"  ;
label variable v5852  "AGE OF WIFE"  ;
label variable v5853  "# KIDS IN FU"  ;
label variable v5872  "D1 EMPLOYMENT STATUS"  ;
label variable v5875  "D5 WORK FOR SELF/OTR (E)"  ;
label variable v5907  "D38 SLRY/HRLY/OTR (E)"  ;
label variable v5908  "D39 PAY/HR SALARY (E)"  ;
label variable v5911  "D42 PAY/HR HRLY (E)"  ;
label variable v6002  "F1 RET/DIS,HSWF,STUD (R)"  ;
label variable v6003  "F2 YEAR RETIRED (R)"  ;
label variable v6017  "F15 WRK FOR MONEY (R)"  ;
label variable v6022  "F20 WORKING NOW (R)"  ;
label variable v6034  "G1 MARRITAL STATUS"  ;
label variable v6038  "G5 WTR WF WRKD"  ;
label variable v6173  "TOT 1977 FAM $  INC"  ;
label variable v6174  "TOT 1977 HEAD LABOR Y"  ;
label variable v6177  "ANNUAL FOOD STD"  ;
label variable v6178  "HEAD 77 AVG HRLY EARNING"  ;
label variable v6179  "WIFE 77 AVG HRLY EARNING"  ;
label variable v6194  "EDUCATION 1978 HEAD"  ;
label variable v6195  "EDUCATION 1978 WIFE"  ;
label variable v6197  "MARITAL STATUS"  ;
label variable v6211  "11YR FAM COMP CHANGE"  ;
label variable v6212  "78 UPDATED FAM WEIGHT"  ;
label variable v6219  "77-78 CNG MARIT STATUS"  ;
label variable er30283  "1979 INTERVIEW NUMBER"  ;
label variable er30284  "SEQUENCE NUMBER                       79"  ;
label variable er30285  "RELATIONSHIP TO HEAD                  79"  ;
label variable er30286  "AGE OF INDIVIDUAL                     79"  ;
label variable v6302  "1979 ID"  ;
label variable v6307  "COLOR OF COVERSHEET 79"  ;
label variable v6310  "FAMILY COMP CHNG 1979"  ;
label variable v6336  "HD ANN WRK HRS IN 78"  ;
label variable v6337  "ACC HD 78 WRK HRS"  ;
label variable v6338  "HD HRS WRK LOST OTH ILL"  ;
label variable v6339  "ACC HD HR LOST OTH ILL"  ;
label variable v6340  "HD HRS WRK LOST OWN ILL"  ;
label variable v6341  "ACC HD HR LOST OWN ILL"  ;
label variable v6342  "HD STRIKE HRS 78"  ;
label variable v6343  "ACC HD STRIKE HRS 78"  ;
label variable v6344  "HD UNEMP HRS 78"  ;
label variable v6345  "ACC HDUNEMP HRS 78"  ;
label variable v6346  "HD TRAV TO WRK HRS 78"  ;
label variable v6347  "ACC HD TRAV TO WK HRS 78"  ;
label variable v6348  "WF ANN WRK HRS IN 78"  ;
label variable v6349  "ACC WF 78 WRK HRS"  ;
label variable v6350  "WF HRS WRK LOST OTH ILL"  ;
label variable v6351  "ACC WF HR LOST OTH ILL"  ;
label variable v6352  "WF HRS WRK LOST OWN ILL"  ;
label variable v6353  "ACC WF HR LOST OWN ILL"  ;
label variable v6354  "WF STRIKE HRS 78"  ;
label variable v6355  "ACC WF STRIKE HRS 78"  ;
label variable v6356  "WF UNEMP HRS 78"  ;
label variable v6357  "ACC WF UNEMP HRS 78"  ;
label variable v6358  "WF TRAV TO WRK HRS 78"  ;
label variable v6359  "ACC WF TRAV TO WK HRS 78"  ;
label variable v6361  "1979 FAMILY SIZE"  ;
label variable v6363  "1979 WEEKLY FOOD NEED"  ;
label variable v6364  "1979 ANN FOOD NEED"  ;
label variable v6372  "$  PD FD ST LAST MO (J14)"  ;
label variable v6373  "ACC $  PAID FD ST LAST MO"  ;
label variable v6374  "$  BONUS PD LAST MO"  ;
label variable v6375  "ACC BONUS FDSTMP LAST MO"  ;
label variable v6376  "ANN FD $  EXCL FDSTMP"  ;
label variable v6377  "ACC ANN FOOD $"  ;
label variable v6378  "ANN $  EATING OUT"  ;
label variable v6379  "ACC ANN $  EATING OUT"  ;
label variable v6380  "$  PAID FDSTMP 1978"  ;
label variable v6381  "ACC $  PAID FDSTMP 1978"  ;
label variable v6382  "$  BONUS FDSTMP 1978"  ;
label variable v6383  "ACC $  BONUS FDSTMP 1978"  ;
label variable v6385  "$  SAVED GRWNG OWN FD"  ;
label variable v6413  "H+W 1978 ADC/AFDC"  ;
label variable v6414  "ACC H+W 1978 ADC/AFDC"  ;
label variable v6415  "H+W 1978 SSI"  ;
label variable v6416  "H+W 1978 OTHER WELFARE"  ;
label variable v6419  "HD 1978 UNEMP COMP"  ;
label variable v6420  "HD 1978 WRKMNS COMP"  ;
label variable v6422  "HD 1978 HELP FROM REL"  ;
label variable v6432  "OTH 1978 ADC/AFDC"  ;
label variable v6433  "OTH 1978 SSI"  ;
label variable v6434  "OTH 1978 OTHER WELFARE"  ;
label variable v6437  "OTH 1978 UNEMP COMP"  ;
label variable v6438  "OTH 1978 WRKMNS COMP"  ;
label variable v6440  "OTH 1978 HLP FRM RLTVS"  ;
label variable v6459  "DATE OF INT"  ;
label variable v6461  "# IN FU"  ;
label variable v6462  "AGE OF HEAD"  ;
label variable v6463  "SEX OF HEAD"  ;
label variable v6464  "AGE OF WIFE"  ;
label variable v6465  "# CHILDREN IN FU"  ;
label variable v6492  "C1 EMP STATUS HEAD"  ;
label variable v6493  "C2 WORK SELF/OTR (HD-E)"  ;
label variable v6518  "C30 SLRY/HRLY/OTR (HD-E)"  ;
label variable v6519  "C31 PAY/HRLY SLRY (HD-E)"  ;
label variable v6522  "C34 PAY/HR HRLY (HD-E)"  ;
label variable v6575  "E1 RET/HWF DIS ST (HD-R)"  ;
label variable v6576  "E2 YR RETIRED (HD-R)"  ;
label variable v6577  "E3 WRK FOR $  (HD-R)"  ;
label variable v6591  "F2 EMP STATUS WIFE"  ;
label variable v6592  "F3 WRK SELF/OTR (WF-E)"  ;
label variable v6614  "F28 SLRY/HRLY/OTR (WF-E)"  ;
label variable v6615  "F29 PAY/HRLY SLRY (WF-E)"  ;
label variable v6616  "F30 PAY/HR HRLY (WF-E)"  ;
label variable v6647  "H1 RET/HWF/DIS/ST (WF-R)"  ;
label variable v6648  "H2 YR RETIRED (WF-R)"  ;
label variable v6649  "H3 WRK FOR $  (WF-R)"  ;
label variable v6659  "J1 MARRITAL STATUS"  ;
label variable v6766  "TOT 1978 FAM $  INC"  ;
label variable v6767  "TOT 1978 HEAD LABOR Y"  ;
label variable v6770  "ANNUAL FOOD STD"  ;
label variable v6771  "HEAD 78 AVG HRLY EARNING"  ;
label variable v6772  "WIFE 78 AVG HRLY EARNING"  ;
label variable v6787  "EDUCATION 1979 HEAD"  ;
label variable v6788  "EDUCATION 1979 WIFE"  ;
label variable v6790  "MARITA;L STATUS 79"  ;
label variable v6804  "12YR FAM COMP CHANGE"  ;
label variable v6805  "79 FAMILY WEIGHT"  ;
label variable v6812  "78-79 CNG MARIT STATUS"  ;
label variable er30313  "1980 INTERVIEW NUMBER"  ;
label variable er30314  "SEQUENCE NUMBER                       80"  ;
label variable er30315  "RELATIONSHIP TO HEAD                  80"  ;
label variable er30316  "AGE OF INDIVIDUAL                     80"  ;
label variable v6902  "1980 INTERVIEW NUMBER"  ;
label variable v6907  "COLOR COVERSHEET 1980"  ;
label variable v6910  "FAM COMP CHANGE 1980"  ;
label variable v6934  "HD ANN WRK HRS IN 79"  ;
label variable v6935  "ACC HD 79 WRK HRS"  ;
label variable v6936  "HD HRS WRK LOST OTH ILL"  ;
label variable v6937  "ACC HD HR LOST OTH ILL"  ;
label variable v6938  "HD HRS WRK LOST OWN ILL"  ;
label variable v6939  "ACC HD HR LOST OWN ILL"  ;
label variable v6940  "HD STRIKE HRS 79"  ;
label variable v6941  "ACC HD STRIKE HRS 79"  ;
label variable v6942  "HD UNEMP HRS 79"  ;
label variable v6943  "ACC HD UNEMP HRS 79"  ;
label variable v6944  "HD TRAV TO WRK HRS 79"  ;
label variable v6945  "ACC HD TRAV TO WRK HR 79"  ;
label variable v6946  "WF ANN WRK HRS IN 79"  ;
label variable v6947  "ACC WF 79 WRK HRS"  ;
label variable v6948  "WF HRS WRK LOST OTH ILL"  ;
label variable v6949  "ACC WF HR LOST OTH ILL"  ;
label variable v6950  "WF HRS WRK LOST OWN ILL"  ;
label variable v6951  "ACC WF HR LOST OWN ILL"  ;
label variable v6952  "WF STRIKE HRS 79"  ;
label variable v6953  "ACC WF STRIK HRS 79"  ;
label variable v6954  "WF UNEMP HRS 79"  ;
label variable v6955  "ACC WF UNEMP HRS 79"  ;
label variable v6956  "WF TRAV TO WRK HRS 79"  ;
label variable v6957  "ACC WF TRAV TO WRK HR 79"  ;
label variable v6959  "1980 FAMILY SIZE"  ;
label variable v6961  "1980 WEEKLY FOOD NEED"  ;
label variable v6962  "1980 ANN NEED STANDARD"  ;
label variable v6970  "$ VLUE FD ST LST MO (J14)"  ;
label variable v6971  "ACC $ VALUE FD ST LAST MO"  ;
label variable v6972  "ANN FD $  EXCL FD STMP"  ;
label variable v6973  "ACC ANN FOOD $"  ;
label variable v6974  "ANN $  EATING OUT"  ;
label variable v6975  "ACC ANN $  EATING OUT"  ;
label variable v6976  "$ VALUE FD ST 1979"  ;
label variable v6977  "ACC $  VALUE FD ST 1979"  ;
label variable v7003  "H+W 1979 ADC/AFDC"  ;
label variable v7004  "ACC H+W 1979 ADC/AFDC"  ;
label variable v7005  "H+W 1979 SSI"  ;
label variable v7006  "H+W 1979 OTHER WELFARE"  ;
label variable v7009  "HD 1979 UNEMP COMP"  ;
label variable v7010  "HD 1979 WRKMNS COMP"  ;
label variable v7012  "HD 1979 HELP FROM REL"  ;
label variable v7036  "OTH 1979 ADC/AFDC"  ;
label variable v7037  "OTH 1979 SSI"  ;
label variable v7038  "OTH 1979 OTHER WELFARE"  ;
label variable v7041  "OTH 1979 UNEMP COMP"  ;
label variable v7042  "OTH 1979 WRKMNS COMP"  ;
label variable v7044  "OTH 1979 HLP FRM RLTVS"  ;
label variable v7064  "DATE OF INT"  ;
label variable v7066  "# IN FU"  ;
label variable v7067  "AGE OF HEAD"  ;
label variable v7068  "SEX OF HEAD"  ;
label variable v7069  "AGE OF WIFE"  ;
label variable v7070  "# CHILDREN IN FU"  ;
label variable v7095  "C1 EMP STATUS HEAD"  ;
label variable v7096  "C2 WORK SELF/OTR (HD-E)"  ;
label variable v7121  "C30 SLRY/HRLY/OTHR(HD-E)"  ;
label variable v7122  "C31 PAY/HRLY SLRY (HD-E)"  ;
label variable v7125  "C34 PAY/HR HRLY (HD-E)"  ;
label variable v7177  "E1 WTR RETIRED (HD-R)"  ;
label variable v7178  "E2 YEAR RETIRED (HD-R)"  ;
label variable v7179  "E3 WRK FOR $  (HD-R)"  ;
label variable v7193  "F2 EMP STATUS WIFE"  ;
label variable v7194  "F3 WRK SELF/OTR (WF-E)"  ;
label variable v7216  "F28 SLRY/HRLY/OTR (WF-E)"  ;
label variable v7217  "F29 PAY/HRLY SLRY (WF-E)"  ;
label variable v7218  "F30 PAY/HR HRLY (WF-E)"  ;
label variable v7249  "H1 WTR RETIRED (WF-R)"  ;
label variable v7250  "H2 YR RETIRED (WF-R)"  ;
label variable v7251  "H3 WRK FOR $  (WF-R)"  ;
label variable v7261  "J1 MARITAL STATUS"  ;
label variable v7412  "TOT FAM $ Y 79"  ;
label variable v7413  "TOT HD LABOR $ Y 79"  ;
label variable v7416  "ANNUAL FOOD STD"  ;
label variable v7417  "HEAD 79 AVG HRLY EARNING"  ;
label variable v7418  "WIFE 79 AVG HRLY EARNING"  ;
label variable v7433  "EDUCATION 1980 HEAD"  ;
label variable v7434  "EDUCATION 1980 WIFE"  ;
label variable v7435  "MARITAL STATUS 1980"  ;
label variable v7450  "13YR FAM COMP CHANGE"  ;
label variable v7451  "1980 FAMILY WEIGHT"  ;
label variable v7455  "79-80 CNG MARITAL STATUS"  ;
label variable er30343  "1981 INTERVIEW NUMBER"  ;
label variable er30344  "SEQUENCE NUMBER                       81"  ;
label variable er30345  "RELATIONSHIP TO HEAD                  81"  ;
label variable er30346  "AGE OF INDIVIDUAL                     81"  ;
label variable v7502  "1981 INTERVIEW NUMBER"  ;
label variable v7507  "COLOR COVERSHEET-1981"  ;
label variable v7510  "FAM COMP CHANGE-1981"  ;
label variable v7530  "HD ANN WRK HRS IN 1980"  ;
label variable v7531  "ACC HD 80 WORK HRS"  ;
label variable v7532  "HD HRS WRK LOST OTR ILL"  ;
label variable v7533  "ACC HD HR LOST OTR ILL"  ;
label variable v7534  "HD HRS WRK LOST OWN ILL"  ;
label variable v7535  "ACC HD HRS LOST OWN ILL"  ;
label variable v7536  "HD STRIKE HOURS 1980"  ;
label variable v7537  "ACC HD STRIKE HRS 1980"  ;
label variable v7538  "HD UNEMP HRS 1980"  ;
label variable v7539  "ACC HD UNEMP HRS 1980"  ;
label variable v7540  "WF ANN WRK HRS IN 1980"  ;
label variable v7541  "ACC WF 80 WRK HRS"  ;
label variable v7542  "WF HRS WRK LOST OTR ILL"  ;
label variable v7543  "ACC WF HR LOST OTR ILL"  ;
label variable v7544  "WF HRS WRK LOST OWN ILL"  ;
label variable v7545  "ACC WF HRS LOST OWN ILL"  ;
label variable v7546  "WF STRIKE HOURS 1980"  ;
label variable v7547  "ACC WF STRIKE HRS 1980"  ;
label variable v7548  "WF UNEMP HRS 1980"  ;
label variable v7549  "ACC WF UNEMP HRS 1980"  ;
label variable v7551  "1981 FAMILY SIZE"  ;
label variable v7553  "1981 WEEKLY FOOD NEEDS"  ;
label variable v7554  "1981 ANNUAL NEED STD"  ;
label variable v7562  "$ $  VALU FD ST LST MO-J14"  ;
label variable v7563  "ACC $ $  VALU FD ST LST MO"  ;
label variable v7564  "ANN FD $ $  EXC FD STMP"  ;
label variable v7565  "ACC ANN FOOD $ $"  ;
label variable v7566  "ANN $ $  EATING OUT"  ;
label variable v7567  "ACC ANN $ $  EATING OUT"  ;
label variable v7568  "$ $  VALUE FOOD STMPS 1980"  ;
label variable v7569  "ACC $ $  VALU FD ST 1980"  ;
label variable v7595  "H+W 1980 ADC/AFDC"  ;
label variable v7596  "ACC H+W 1980 ADC/AFDC"  ;
label variable v7597  "H+W 1980 SSI"  ;
label variable v7598  "H+W 1980 OTR WELFARE"  ;
label variable v7601  "HD 1980 UNEMPLMT COMP"  ;
label variable v7602  "HD 1980 WORKMANS COMP"  ;
label variable v7604  "HD 1980 HELP FROM RELS"  ;
label variable v7628  "OTR 1980 ADC/AFDC"  ;
label variable v7629  "OTR 1980 SSI"  ;
label variable v7630  "OTR 1980 OTHR WELFARE"  ;
label variable v7633  "OTR 1980 UNEMPL COMP"  ;
label variable v7634  "OTR 1980 WORKERS COMP"  ;
label variable v7636  "OTR 1980 HELP FR RELS"  ;
label variable v7655  "DATE OF 1981 INT"  ;
label variable v7657  "# IN FU"  ;
label variable v7658  "AGE OF 1981 HEAD"  ;
label variable v7659  "SEX OF 1981 HEAD"  ;
label variable v7660  "AGE OF 1981 WIFE"  ;
label variable v7661  "# CHILDREN IN FU"  ;
label variable v7706  "C1 EMPLMT STATUS-HEAD"  ;
label variable v7707  "C2 WORK SELF/OTR?(HD-E)"  ;
label variable v7714  "C10 SLRY/HRLY/OTR (H-E)"  ;
label variable v7715  "C11 PAY/HR-SALARY(HD-E)"  ;
label variable v7718  "C14 PAY/HR-HOURLY(HD-E)"  ;
label variable v7863  "E1 WTR RETIRED   (HD-R)"  ;
label variable v7864  "E2 YEAR RETIRED  (HD-R)"  ;
label variable v7865  "E3 WTR WRK FOR $ $  (HD-R)"  ;
label variable v7879  "F2 EMPLOYMENT STAT-WF"  ;
label variable v7880  "F3 WORK FOR SELF?(WF-E)"  ;
label variable v7887  "F11 SLRY/HRLY/OTR(WF-E)"  ;
label variable v7888  "F12 PAY/HR-SALARY(WF-E)"  ;
label variable v7889  "F13 PAY/HR-HOURLY(WF-E)"  ;
label variable v7940  "H1 WTR RETIRED   (WF-R)"  ;
label variable v7941  "H2 YEAR RETIRED  (WF-R)"  ;
label variable v7942  "H3 WTR WRK FOR $ $  (WF-R)"  ;
label variable v7952  "J1 MARITAL STATUS"  ;
label variable v8065  "TOT FAM $ $  Y 80"  ;
label variable v8066  "TOT HD LABOR $ $  Y 80"  ;
label variable v8068  "ANNUAL FOOD STD"  ;
label variable v8069  "HEAD 80 AVG HRLY EARNING"  ;
label variable v8070  "WIFE 80 AVG HRLY EARNING"  ;
label variable v8085  "EDUCATION 1981 HEAD"  ;
label variable v8086  "EDUCATION 1981 WIFE"  ;
label variable v8087  "MARITAL STATUS 1981"  ;
label variable v8102  "14YR FAM COMP CHANGE"  ;
label variable v8103  "1981 FAMILY WEIGHT"  ;
label variable v8107  "80-81 CNG MARITAL STATUS"  ;
label variable er30373  "1982 INTERVIEW NUMBER"  ;
label variable er30374  "SEQUENCE NUMBER                       82"  ;
label variable er30375  "RELATIONSHIP TO HEAD                  82"  ;
label variable er30376  "AGE OF INDIVIDUAL                     82"  ;
label variable v8202  "1982 INTERVIEW NUMBER"  ;
label variable v8207  "COLOR COVERSHEET-1982"  ;
label variable v8210  "FAM COMP CHANGE-1982"  ;
label variable v8228  "HD ANN WRK HRS IN 1981"  ;
label variable v8229  "ACC HD 81 WORK HRS"  ;
label variable v8230  "HD HRS WRK LOST OTR ILL"  ;
label variable v8231  "ACC HD HR LOST OTR ILL"  ;
label variable v8232  "HD HRS WRK LOST OWN ILL"  ;
label variable v8233  "ACC HD HRS LOST OWN ILL"  ;
label variable v8234  "HD STRIKE HOURS 1981"  ;
label variable v8235  "ACC HD STRIKE HRS 1981"  ;
label variable v8236  "HD UNEMP HRS 1981"  ;
label variable v8237  "ACC HD UNEMP HRS 1981"  ;
label variable v8238  "WF ANN WRK HRS IN 1981"  ;
label variable v8239  "ACC WF 81 WRK HRS"  ;
label variable v8240  "WF HRS WRK LOST OTR ILL"  ;
label variable v8241  "ACC WF HR LOST OTR ILL"  ;
label variable v8242  "WF HRS WRK LOST OWN ILL"  ;
label variable v8243  "ACC WF HRS LOST OWN ILL"  ;
label variable v8244  "WF STRIKE HOURS 1981"  ;
label variable v8245  "ACC WF STRIKE HRS 1981"  ;
label variable v8246  "WF UNEMP HRS 1981"  ;
label variable v8247  "ACC WF UNEMP HRS 1981"  ;
label variable v8249  "1982 FAMILY SIZE"  ;
label variable v8251  "1982 WEEKLY FOOD NEEDS"  ;
label variable v8252  "1982 ANNUAL NEED STD"  ;
label variable v8254  "VALUE FD ST LAST MO(J6)"  ;
label variable v8255  "ACC VALUE FD ST LAST MO"  ;
label variable v8256  "ANN FOOD COST EXC FD ST"  ;
label variable v8257  "ACC ANN FOOD COST"  ;
label variable v8258  "ANNUAL EATING OUT COST"  ;
label variable v8259  "ACC ANN EAT OUT COST"  ;
label variable v8260  "VALUE FD ST 1981 (J16)"  ;
label variable v8261  "ACC VALUE FD ST 1981"  ;
label variable v8288  "H+W 1981 ADC/AFDC"  ;
label variable v8289  "ACC H+W 1981 ADC/AFDC"  ;
label variable v8290  "H+W 1981 SSI"  ;
label variable v8291  "H+W 1981 OTR WELFARE"  ;
label variable v8294  "HD 1981 UNEMPLMT COMP"  ;
label variable v8295  "HD 1981 WORKMANS COMP"  ;
label variable v8297  "HD 1981 HELP FROM RELS"  ;
label variable v8321  "OTR 1981 ADC/AFDC"  ;
label variable v8322  "OTR 1981 SSI"  ;
label variable v8323  "OTR 1981 OTHR WELFARE"  ;
label variable v8326  "OTR 1981 UNEMPL COMP"  ;
label variable v8327  "OTR 1981 WORKERS COMP"  ;
label variable v8329  "OTR 1981 HELP FR RELS"  ;
label variable v8349  "DATE OF 1982 INT"  ;
label variable v8351  "# IN FU"  ;
label variable v8352  "AGE OF 1982 HEAD"  ;
label variable v8353  "SEX OF 1982 HEAD"  ;
label variable v8354  "AGE OF 1982 WIFE"  ;
label variable v8355  "# CHILDREN IN FU"  ;
label variable v8374  "C1 EMPLMT STATUS-HEAD"  ;
label variable v8375  "C2 WORK SELF/OTR?(HD-E)"  ;
label variable v8382  "C10 SLRY/HRLY/OTR (H-E)"  ;
label variable v8383  "C11 PAY/HR-SALARY(HD-E)"  ;
label variable v8386  "C14 PAY/HR-HOURLY(HD-E)"  ;
label variable v8523  "E1 WTR RETIRED   (HD-R)"  ;
label variable v8524  "E2 YEAR RETIRED  (HD-R)"  ;
label variable v8525  "E3 WRK FOR PAY-81?(H-R)"  ;
label variable v8538  "F2 EMPLOYMENT STAT-WF"  ;
label variable v8539  "F3 WORK FOR SELF?(WF-E)"  ;
label variable v8546  "F11 SLRY/HRLY/OTR(WF-E)"  ;
label variable v8547  "F12 PAY/HR-SALARY(WF-E)"  ;
label variable v8548  "F13 PAY/HR-HOURLY(WF-E)"  ;
label variable v8591  "H1 WTR RETIRED   (WF-R)"  ;
label variable v8592  "H2 YEAR RETIRED  (WF-R)"  ;
label variable v8593  "H3 WRK FOR PAY-81?(W-R)"  ;
label variable v8689  "TOT FAM $ $  Y 81"  ;
label variable v8690  "TOT HD LABOR $ $  Y 81"  ;
label variable v8692  "ANNUAL FOOD STD"  ;
label variable v8693  "HEAD 81 AVG HRLY EARNING"  ;
label variable v8694  "WIFE 81 AVG HRLY EARNING"  ;
label variable v8709  "EDUCATION 1982 HEAD"  ;
label variable v8710  "EDUCATION 1982 WIFE"  ;
label variable v8711  "MARITAL STATUS 1982"  ;
label variable v8726  "15YR FAM COMP CHANGE"  ;
label variable v8727  "1982 FAMILY WEIGHT"  ;
label variable v8731  "81-82 CNG MARITAL STATUS"  ;
label variable er30399  "1983 INTERVIEW NUMBER"  ;
label variable er30400  "SEQUENCE NUMBER                       83"  ;
label variable er30401  "RELATIONSHIP TO HEAD                  83"  ;
label variable er30402  "AGE OF INDIVIDUAL                     83"  ;
label variable er30403  "MONTH IND BORN                        83"  ;
label variable er30404  "YEAR IND BORN                         83"  ;
label variable er30425  "AGE FROM BIRTH DATE                   83"  ;
label variable v8802  "1983 INTERVIEW NUMBER"  ;
label variable v8807  "COLOR COVERSHEET-1983"  ;
label variable v8810  "FAM COMP CHANGE-1983"  ;
label variable v8830  "HD ANN WRK HRS IN 1982"  ;
label variable v8831  "ACC HD 82 WORK HRS"  ;
label variable v8832  "HD HRS WRK LOST OTR ILL"  ;
label variable v8833  "ACC HD HR LOST OTR ILL"  ;
label variable v8834  "HD HRS WRK LOST OWN ILL"  ;
label variable v8835  "ACC HD HRS LOST OWN ILL"  ;
label variable v8836  "HD STRIKE HOURS 1982"  ;
label variable v8837  "ACC HD STRIKE HRS 1982"  ;
label variable v8838  "HD UNEMP HRS 1982"  ;
label variable v8839  "ACC HD UNEMP HRS 1982"  ;
label variable v8840  "WF ANN WRK HRS IN 1982"  ;
label variable v8841  "ACC WF 82 WRK HRS"  ;
label variable v8842  "WF HRS WRK LOST OTR ILL"  ;
label variable v8843  "ACC WF HR LOST OTR ILL"  ;
label variable v8844  "WF HRS WRK LOST OWN ILL"  ;
label variable v8845  "ACC WF HRS LOST OWN ILL"  ;
label variable v8846  "WF STRIKE HOURS 1982"  ;
label variable v8847  "ACC WF STRIKE HRS 1982"  ;
label variable v8848  "WF UNEMP HRS 1982"  ;
label variable v8849  "ACC WF UNEMP HRS 1982"  ;
label variable v8851  "1983 FAMILY SIZE"  ;
label variable v8853  "1983 WEEKLY FOOD NEEDS"  ;
label variable v8854  "1982 ANNUAL NEED STD"  ;
label variable v8862  "VALUE FD ST LAST MO(J13)"  ;
label variable v8863  "ACC VALUE FD ST LAST MO"  ;
label variable v8864  "ANN FOOD COST EXC FD ST"  ;
label variable v8865  "ACC ANN FOOD COST"  ;
label variable v8866  "ANNUAL EATING OUT COST"  ;
label variable v8867  "ACC ANN EAT OUT COST"  ;
label variable v8868  "VALUE FD ST 1982 (J24)"  ;
label variable v8869  "ACC VALUE FD ST 1982"  ;
label variable v8896  "H+W 1982 ADC/AFDC"  ;
label variable v8897  "ACC H+W 1982 ADC/AFDC"  ;
label variable v8898  "H+W 1982 SSI"  ;
label variable v8899  "H+W 1982 OTHER WELFARE"  ;
label variable v8902  "HD 1982 UNEMPLMT COMP"  ;
label variable v8903  "HD 1982 WORKMANS COMP"  ;
label variable v8905  "HD 1982 HELP FROM RELS"  ;
label variable v8929  "OTR 1982 ADC/AFDC"  ;
label variable v8930  "OTR 1982 SSI"  ;
label variable v8931  "OTR 1982 OTHR WELFARE"  ;
label variable v8934  "OTR 1982 UNEMPL COMP"  ;
label variable v8935  "OTR 1982 WORKERS COMP"  ;
label variable v8937  "OTR 1982 HELP FROM RELS"  ;
label variable v8958  "DATE OF 1983 INT"  ;
label variable v8961  "AGE OF 1983 HEAD"  ;
label variable v8962  "SEX OF 1983 HEAD"  ;
label variable v8963  "AGE OF 1983 WIFE"  ;
label variable v8964  "# CHILDREN IN FU"  ;
label variable v9005  "C1 EMPLMT STATUS-HEAD"  ;
label variable v9006  "C2 WORK SELF/OTR?(HD-E)"  ;
label variable v9013  "C10 SLRY/HRLY/OTR (H-E)"  ;
label variable v9014  "C11 PAY/HR-SALARY(HD-E)"  ;
label variable v9017  "C14 PAY/HR-HOURLY(HD-E)"  ;
label variable v9173  "E1 WTR RETIRED   (HD-R)"  ;
label variable v9174  "E2 YEAR RETIRED  (HD-R)"  ;
label variable v9188  "F2 EMPLOYMENT STAT-WF"  ;
label variable v9189  "F3 WORK FOR SELF?(WF-E)"  ;
label variable v9196  "F11 SLRY/HRLY/OTR(WF-E)"  ;
label variable v9197  "F12 PAY/HR-SALARY(WF-E)"  ;
label variable v9198  "F13 PAY/HR-HOURLY(WF-E)"  ;
label variable v9264  "H1 WTR RETIRED   (WF-R)"  ;
label variable v9265  "H2 YEAR RETIRED  (WF-R)"  ;
label variable v9266  "H3 WRK FOR PAY-82(W-R)"  ;
label variable v9276  "J1 MARITAL STATUS"  ;
label variable v9375  "TOT FAM MONEY Y 82"  ;
label variable v9376  "TOTAL HEAD LABOR Y 82"  ;
label variable v9378  "ANNUAL FOOD STD"  ;
label variable v9379  "HEAD 82 AVG HRLY EARNING"  ;
label variable v9380  "WIFE 82 AVG HRLY EARNING"  ;
label variable v9395  "EDUCATION 1983 HEAD"  ;
label variable v9396  "EDUCATION 1983 WIFE"  ;
label variable v9418  "16YR FAM COMP CHANGE"  ;
label variable v9419  "MARITAL STATUS 1983"  ;
label variable v9420  "82-83 CNG MARITAL STATUS"  ;
label variable v9421  "COUPLE STATUS OF HEAD"  ;
label variable v9433  "1983 FAMILY WEIGHT"  ;
label variable er30429  "1984 INTERVIEW NUMBER"  ;
label variable er30430  "SEQUENCE NUMBER                       84"  ;
label variable er30431  "RELATIONSHIP TO HEAD                  84"  ;
label variable er30432  "AGE OF INDIVIDUAL                     84"  ;
label variable er30433  "MONTH IND BORN                        84"  ;
label variable er30434  "YEAR IND BORN                         84"  ;
label variable er30459  "AGE FROM BIRTH DATE                   84"  ;
label variable v10002  "1984 INTERVIEW NUMBER"  ;
label variable v10007  "SPLITOFF INDICATOR 1984"  ;
label variable v10010  "FAM COMP CHANGE-1984"  ;
label variable v10037  "HD ANN WRK HRS-1983"  ;
label variable v10038  "ACC HD WORK HOURS 83"  ;
label variable v10039  "HD WRKHRS LOST OTR ILL83"  ;
label variable v10040  "ACC HD HRS LOST OTR ILL"  ;
label variable v10041  "HD WRK HRS LOST OWN ILL"  ;
label variable v10042  "ACC HD HRS LOST OWN ILL"  ;
label variable v10043  "HD STRIKE HOURS 1983"  ;
label variable v10044  "ACC HD STRIKE HRS 1983"  ;
label variable v10045  "HD UNEMP HRS 1983"  ;
label variable v10046  "ACC 83 UNEMP HRS-1983"  ;
label variable v10047  "HD HRS OUT LAB FORCE 83"  ;
label variable v10048  "ACC 83 HD HR OUT LAB FRC"  ;
label variable v10049  "HD WRKD/EMPLOYER JAN 83"  ;
label variable v10050  "HD WRKD/EMPLOYER FEB 83"  ;
label variable v10051  "HD WRKD/EMPLOYER MAR 83"  ;
label variable v10052  "HD WRKD/EMPLOYER APR 83"  ;
label variable v10053  "HD WRKD/EMPLOYER MAY 83"  ;
label variable v10054  "HD WRKD/EMPLOYER JUN 83"  ;
label variable v10055  "HD WRKD/EMPLOYER JUL 83"  ;
label variable v10056  "HD WRKD/EMPLOYER AUG 83"  ;
label variable v10057  "HD WRKD/EMPLOYER SEP 83"  ;
label variable v10058  "HD WRKD/EMPLOYER OCT 83"  ;
label variable v10059  "HD WRKD/EMPLOYER NOV 83"  ;
label variable v10060  "HD WRKD/EMPLOYER DEC 83"  ;
label variable v10061  "HD WRKD/EMPLOYER JAN 84"  ;
label variable v10062  "HD WRKD/EMPLOYER FEB 84"  ;
label variable v10063  "HD WRKD/EMPLOYER MAR 84"  ;
label variable v10064  "HD WRKD/EMPLOYER APR 84"  ;
label variable v10065  "HD WRKD/EMPLOYER MAY 84"  ;
label variable v10066  "HD WRKD/EMPLOYER JUN 84"  ;
label variable v10067  "HD WRKD/EMPLOYER JUL 84"  ;
label variable v10068  "HD WRKD/EMPLOYER AUG 84"  ;
label variable v10069  "HD WRKD/EMPLOYER SEP 84"  ;
label variable v10070  "HD WRKD/EMPLOYER OCT 84"  ;
label variable v10071  "HD START/END POS JAN 83"  ;
label variable v10072  "HD START/END POS FEB 83"  ;
label variable v10073  "HD START/END POS MAR 83"  ;
label variable v10074  "HD START/END POS APR 83"  ;
label variable v10075  "HD START/END POS MAY 83"  ;
label variable v10076  "HD START/END POS JUN 83"  ;
label variable v10077  "HD START/END POS JUL 83"  ;
label variable v10078  "HD START/END POS AUG 83"  ;
label variable v10079  "HD START/END POS SEP 83"  ;
label variable v10080  "HD START/END POS OCT 83"  ;
label variable v10081  "HD START/END POS NOV 83"  ;
label variable v10082  "HD START/END POS DEC 83"  ;
label variable v10083  "HD START/END POS JAN 84"  ;
label variable v10084  "HD START/END POS FEB 84"  ;
label variable v10085  "HD START/END POS MAR 84"  ;
label variable v10086  "HD START/END POS APR 84"  ;
label variable v10087  "HD START/END POS MAY 84"  ;
label variable v10088  "HD START/END POS JUN 84"  ;
label variable v10089  "HD START/END POS JUL 84"  ;
label variable v10090  "HD START/END POS AUG 84"  ;
label variable v10091  "HD START/END POS SEP 84"  ;
label variable v10092  "HD START/END POS OCT 84"  ;
label variable v10093  "HD UNEMP/OUT LAB JAN 83"  ;
label variable v10094  "HD UNEMP/OUT LAB FEB 83"  ;
label variable v10095  "HD UNEMP/OUT LAB MAR 83"  ;
label variable v10096  "HD UNEMP/OUT LAB APR 83"  ;
label variable v10097  "HD UNEMP/OUT LAB MAY 83"  ;
label variable v10098  "HD UNEMP/OUT LAB JUN 83"  ;
label variable v10099  "HD UNEMP/OUT LAB JUL 83"  ;
label variable v10100  "HD UNEMP/OUT LAB AUG 83"  ;
label variable v10101  "HD UNEMP/OUT LAB SEP 83"  ;
label variable v10102  "HD UNEMP/OUT LAB OCT 83"  ;
label variable v10103  "HD UNEMP/OUT LAB NOV 83"  ;
label variable v10104  "HD UNEMP/OUT LAB DEC 83"  ;
label variable v10105  "HD UNEMP/OUT LAB JAN 84"  ;
label variable v10106  "HD UNEMP/OUT LAB FEB 84"  ;
label variable v10107  "HD UNEMP/OUT LAB MAR 84"  ;
label variable v10108  "HD UNEMP/OUT LAB APR 84"  ;
label variable v10109  "HD UNEMP/OUT LAB MAY 84"  ;
label variable v10110  "HD UNEMP/OUT LAB JUN 84"  ;
label variable v10111  "HD UNEMP/OUT LAB JUL 84"  ;
label variable v10112  "HD UNEMP/OUT LAB AUG 84"  ;
label variable v10113  "HD UNEMP/OUT LAB SEP 84"  ;
label variable v10114  "HD UNEMP/OUT LAB OCT 84"  ;
label variable v10115  "HD XTRA JOBS JAN 83"  ;
label variable v10116  "HD XTRA JOBS FEB 83"  ;
label variable v10117  "HD XTRA JOBS MAR 83"  ;
label variable v10118  "HD XTRA JOBS APR 83"  ;
label variable v10119  "HD XTRA JOBS MAY 83"  ;
label variable v10120  "HD XTRA JOBS JUN 83"  ;
label variable v10121  "HD XTRA JOBS JUL 83"  ;
label variable v10122  "HD XTRA JOBS AUG 83"  ;
label variable v10123  "HD XTRA JOBS SEP 83"  ;
label variable v10124  "HD XTRA JOBS OCT 83"  ;
label variable v10125  "HD XTRA JOBS NOV 83"  ;
label variable v10126  "HD XTRA JOBS DEC 83"  ;
label variable v10131  "WF ANN WRK HRS-1983"  ;
label variable v10132  "ACC WF WRK HOURS-1983"  ;
label variable v10133  "WF WRKRS LOST OTR ILL-83"  ;
label variable v10134  "ACC WF HRS LOST OTR ILL"  ;
label variable v10135  "WF WRK HRS LOST OWN ILL"  ;
label variable v10136  "ACC WF HRS LOST OWN ILL"  ;
label variable v10137  "WF STRIKE HOURS 1983"  ;
label variable v10138  "ACC WF STRIKE HRS 1983"  ;
label variable v10139  "WF UNEMP HRS 1983"  ;
label variable v10140  "ACC WF UNEMP HRS 1983"  ;
label variable v10141  "WF HRS OUT LAB FORCE 83"  ;
label variable v10142  "ACC WF 83 HR OUT LAB FRC"  ;
label variable v10143  "WF WRK/EMPLOYER JAN 83"  ;
label variable v10144  "WF WRK/EMPLOYER FEB 83"  ;
label variable v10145  "WF WRK/EMPLOYER MAR 83"  ;
label variable v10146  "WF WRK/EMPLOYER APR 83"  ;
label variable v10147  "WF WRK/EMPLOYER MAY 83"  ;
label variable v10148  "WF WRK/EMPLOYER JUN 83"  ;
label variable v10149  "WF WRK/EMPLOYER JUL 83"  ;
label variable v10150  "WF WRK/EMPLOYER AUG 83"  ;
label variable v10151  "WF WRK/EMPLOYER SEP 83"  ;
label variable v10152  "WF WRK/EMPLOYER OCT 83"  ;
label variable v10153  "WF WRK/EMPLOYER NOV 83"  ;
label variable v10154  "WF WRK/EMPLOYER DEC 83"  ;
label variable v10155  "WF WRK/EMPLOYER JAN 84"  ;
label variable v10156  "WF WRK/EMPLOYER FEB 84"  ;
label variable v10157  "WF WRK/EMPLOYER MAR 84"  ;
label variable v10158  "WF WRK/EMPLOYER APR 84"  ;
label variable v10159  "WF WRK/EMPLOYER MAY 84"  ;
label variable v10160  "WF WRK/EMPLOYER JUN 84"  ;
label variable v10161  "WF WRK/EMPLOYER JUL 84"  ;
label variable v10162  "WF WRK/EMPLOYER AUG 84"  ;
label variable v10163  "WF WRK/EMPLOYER SEP 84"  ;
label variable v10164  "WF WRK/EMPLOYER OCT 84"  ;
label variable v10165  "WF START/END POS JAN 83"  ;
label variable v10166  "WF START/END POS FEB 83"  ;
label variable v10167  "WF START/END POS MAR 83"  ;
label variable v10168  "WF START/END POS APR 83"  ;
label variable v10169  "WF START/END POS MAY 83"  ;
label variable v10170  "WF START/END POS JUN 83"  ;
label variable v10171  "WF START/END POS JUL 83"  ;
label variable v10172  "WF START/END POS AUG 83"  ;
label variable v10173  "WF START/END POS SEP 83"  ;
label variable v10174  "WF START/END POS OCT 83"  ;
label variable v10175  "WF START/END POS NOV 83"  ;
label variable v10176  "WF START/END POS DEC 83"  ;
label variable v10177  "WF START/END POS JAN 84"  ;
label variable v10178  "WF START/END POS FEB 84"  ;
label variable v10179  "WF START/END POS MAR 84"  ;
label variable v10180  "WF START/END POS APR 84"  ;
label variable v10181  "WF START/END POS MAY 84"  ;
label variable v10182  "WF START/END POS JUN 84"  ;
label variable v10183  "WF START/END POS JUL 84"  ;
label variable v10184  "WF START/END POS AUG 84"  ;
label variable v10185  "WF START/END POS SEP 84"  ;
label variable v10186  "WF START/END POS OCT 84"  ;
label variable v10187  "WF UNEMP/OUT LAB JAN 83"  ;
label variable v10188  "WF UNEMP/OUT LAB FEB 83"  ;
label variable v10189  "WF UNEMP/OUT LAB MAR 83"  ;
label variable v10190  "WF UNEMP/OUT LAB APR 83"  ;
label variable v10191  "WF UNEMP/OUT LAB MAY 83"  ;
label variable v10192  "WF UNEMP/OUT LAB JUN 83"  ;
label variable v10193  "WF UNEMP/OUT LAB JUL 83"  ;
label variable v10194  "WF UNEMP/OUT LAB AUG 83"  ;
label variable v10195  "WF UNEMP/OUT LAB SEP 83"  ;
label variable v10196  "WF UNEMP/OUT LAB OCT 83"  ;
label variable v10197  "WF UNEMP/OUT LAB NOV 83"  ;
label variable v10198  "WF UNEMP/OUT LAB DEC 83"  ;
label variable v10199  "WF UNEMP/OUT LAB JAN 84"  ;
label variable v10200  "WF UNEMP/OUT LAB FEB 84"  ;
label variable v10201  "WF UNEMP/OUT LAB MAR 84"  ;
label variable v10202  "WF UNEMP/OUT LAB APR 84"  ;
label variable v10203  "WF UNEMP/OUT LAB MAY 84"  ;
label variable v10204  "WF UNEMP/OUT LAB JUN 84"  ;
label variable v10205  "WF UNEMP/OUT LAB JUL 84"  ;
label variable v10206  "WF UNEMP/OUT LAB AUG 84"  ;
label variable v10207  "WF UNEMP/OUT LAB SEP 84"  ;
label variable v10208  "WF UNEMP/OUT LAB OCT 84"  ;
label variable v10209  "WF XTRA JOBS JAN 83"  ;
label variable v10210  "WF XTRA JOBS FEB 83"  ;
label variable v10211  "WF XTRA JOBS MAR 83"  ;
label variable v10212  "WF XTRA JOBS APR 83"  ;
label variable v10213  "WF XTRA JOBS MAY 83"  ;
label variable v10214  "WF XTRA JOBS JUN 83"  ;
label variable v10215  "WF XTRA JOBS JUL 83"  ;
label variable v10216  "WF XTRA JOBS AUG 83"  ;
label variable v10217  "WF XTRA JOBS SEP 83"  ;
label variable v10218  "WF XTRA JOBS OCT 83"  ;
label variable v10219  "WF XTRA JOBS NOV 83"  ;
label variable v10220  "WF XTRA JOBS DEC 83"  ;
label variable v10222  "# IN FU-1984"  ;
label variable v10224  "WEEKLY FOOD NEEDS-1984"  ;
label variable v10225  "ANNUAL NEEDS STD-1983"  ;
label variable v10233  "VALUE FD ST LAST MO(J10)"  ;
label variable v10234  "ACC VALUE FD ST LAST MO"  ;
label variable v10235  "ANN FOOD COST EXC FD ST"  ;
label variable v10236  "ACC ANN FOOD COST"  ;
label variable v10237  "ANN EAT OUT COST(J15/19)"  ;
label variable v10238  "ACC ANN EAT OUT COST"  ;
label variable v10239  "VALUE FD ST 1983 (J21)"  ;
label variable v10240  "ACC VALUE FD ST 1983"  ;
label variable v10282  "H+W 1983 ADC/AFDC"  ;
label variable v10283  "ACC H+W 1983 ADC/AFDC"  ;
label variable v10285  "H+W 1983 SSI"  ;
label variable v10286  "H+W 1983 OTR WELFARE"  ;
label variable v10295  "HD 1983 UNEMP COMP"  ;
label variable v10296  "HD 1983 WORKERS COMP"  ;
label variable v10300  "HD 1983 HELP FROM RELS"  ;
label variable v10385  "OFUM 1983 ADC/AFDC"  ;
label variable v10386  "OFUM 1983 SSI"  ;
label variable v10387  "OFUM 1983 OTHER WELFARE"  ;
label variable v10391  "OFUM 1983 UNEMPL COMP"  ;
label variable v10392  "OFUM 1983 WORKERS COMP"  ;
label variable v10394  "OFUM 1983 HELP FR RELS"  ;
label variable v10416  "DATE OF 1984 INT"  ;
label variable v10418  "# IN FU"  ;
label variable v10419  "AGE OF 1984 HEAD"  ;
label variable v10420  "SEX OF 1984 HEAD"  ;
label variable v10421  "AGE OF 1984 WIFE"  ;
label variable v10422  "# CHILDREN IN FU"  ;
label variable v10426  "A3 MARITAL STATUS"  ;
label variable v10453  "C1 EMPLOYMENT STATUS-HD"  ;
label variable v10454  "C2 WORK FOR MONEY?(HD-E)"  ;
label variable v10455  "C3 WORK 10+ HRS/WK(HD-E)"  ;
label variable v10456  "C4 WORK SELF/OTR?(HD-E)"  ;
label variable v10462  "C11 SLRY/HRLY/OTR (H-E)"  ;
label variable v10463  "C12 PAY/HR-SALARY(HD-E)"  ;
label variable v10466  "C15 PAY/HR-HOURLY(HD-E)"  ;
label variable v10612  "D16 WAGE LAST JOB(HD-U)"  ;
label variable v10671  "F2 EMPLOYMENT STAT-WF"  ;
label variable v11022  "TOT FAM MONEY Y 83"  ;
label variable v11023  "TOTAL HEAD LABOR Y 83"  ;
label variable v11025  "ANNUAL FOOD STD"  ;
label variable v11026  "HEAD 83 AVG HRLY EARNING"  ;
label variable v11027  "WIFE 83 AVG HRLY EARNING"  ;
label variable v11042  "EDUCATION 1984 HEAD"  ;
label variable v11043  "EDUCATION 1984 WIFE"  ;
label variable v11064  "17YR FAM COMP CHANGE"  ;
label variable v11065  "MARITAL STATUS 1984"  ;
label variable v11066  "83-84 CNG MARITAL STATUS"  ;
label variable v11067  "COUPLE STATUS OF HEAD"  ;
label variable v11079  "1984 FAMILY WEIGHT"  ;
label variable er30463  "1985 INTERVIEW NUMBER"  ;
label variable er30464  "SEQUENCE NUMBER                       85"  ;
label variable er30465  "RELATIONSHIP TO HEAD                  85"  ;
label variable er30466  "AGE OF INDIVIDUAL                     85"  ;
label variable er30467  "MONTH IND BORN                        85"  ;
label variable er30468  "YEAR IND BORN                         85"  ;
label variable er30494  "AGE FROM BIRTH DATE                   85"  ;
label variable v11102  "1985 INTERVIEW NUMBER"  ;
label variable v11107  "SPLITOFF INDICATOR    85"  ;
label variable v11112  "FAM COMP CHANGE       85"  ;
label variable v11146  "HD ANN WRK HRS IN 84"  ;
label variable v11147  "HD HRS WRK LOST OTR ILL"  ;
label variable v11148  "ACC HD HR LOST OTR ILL"  ;
label variable v11149  "HD HRS WRK LOST OWN ILL"  ;
label variable v11150  "ACC HD HRS LOST OWN ILL"  ;
label variable v11151  "HD STRIKE HOURS 84"  ;
label variable v11152  "ACC HD STRIKE HRS 84"  ;
label variable v11153  "HD UNEMP HRS 84"  ;
label variable v11154  "ACC 84 HD UNEMP HRS"  ;
label variable v11155  "HD HRS OUT LBR FORCE 84"  ;
label variable v11156  "ACC 84 HD HR OUT LBR FRC"  ;
label variable v11157  "HD HRS TRAVEL TO WORK 84"  ;
label variable v11158  "ACC HD TRAVEL TO WORK"  ;
label variable v11159  "HD WRKD/EMPLOYER JAN 84"  ;
label variable v11160  "HD WRKD/EMPLOYER FEB 84"  ;
label variable v11161  "HD WRKD/EMPLOYER MAR 84"  ;
label variable v11162  "HD WRKD/EMPLOYER APR 84"  ;
label variable v11163  "HD WRKD/EMPLOYER MAY 84"  ;
label variable v11164  "HD WRKD/EMPLOYER JUN 84"  ;
label variable v11165  "HD WRKD/EMPLOYER JUL 84"  ;
label variable v11166  "HD WRKD/EMPLOYER AUG 84"  ;
label variable v11167  "HD WRKD/EMPLOYER SEP 84"  ;
label variable v11168  "HD WRKD/EMPLOYER OCT 84"  ;
label variable v11169  "HD WRKD/EMPLOYER NOV 84"  ;
label variable v11170  "HD WRKD/EMPLOYER DEC 84"  ;
label variable v11171  "HD WRKD/EMPLOYER JAN 85"  ;
label variable v11172  "HD WRKD/EMPLOYER FEB 85"  ;
label variable v11173  "HD WRKD/EMPLOYER MAR 85"  ;
label variable v11174  "HD WRKD/EMPLOYER APR 85"  ;
label variable v11175  "HD WRKD/EMPLOYER MAY 85"  ;
label variable v11176  "HD WRKD/EMPLOYER JUN 85"  ;
label variable v11177  "HD WRKD/EMPLOYER JUL 85"  ;
label variable v11178  "HD WRKD/EMPLOYER AUG 85"  ;
label variable v11179  "HD WRKD/EMPLOYER SEP 85"  ;
label variable v11180  "HD WRKD/EMPLOYER OCT 85"  ;
label variable v11181  "HD WRK/EMPR QUALITY CODE"  ;
label variable v11182  "HD START/END POS JAN 84"  ;
label variable v11183  "HD START/END POS FEB 84"  ;
label variable v11184  "HD START/END POS MAR 84"  ;
label variable v11185  "HD START/END POS APR 84"  ;
label variable v11186  "HD START/END POS MAY 84"  ;
label variable v11187  "HD START/END POS JUN 84"  ;
label variable v11188  "HD START/END POS JUL 84"  ;
label variable v11189  "HD START/END POS AUG 84"  ;
label variable v11190  "HD START/END POS SEP 84"  ;
label variable v11191  "HD START/END POS OCT 84"  ;
label variable v11192  "HD START/END POS NOV 84"  ;
label variable v11193  "HD START/END POS DEC 84"  ;
label variable v11194  "HD START/END POS JAN 85"  ;
label variable v11195  "HD START/END POS FEB 85"  ;
label variable v11196  "HD START/END POS MAR 85"  ;
label variable v11197  "HD START/END POS APR 85"  ;
label variable v11198  "HD START/END POS MAY 85"  ;
label variable v11199  "HD START/END POS JUN 85"  ;
label variable v11200  "HD START/END POS JUL 85"  ;
label variable v11201  "HD START/END POS AUG 85"  ;
label variable v11202  "HD START/END POS SEP 85"  ;
label variable v11203  "HD START/END POS OCT 85"  ;
label variable v11204  "HD STRT/END QUALITY CODE"  ;
label variable v11205  "HD UNEMP/OUT LBR JAN 84"  ;
label variable v11206  "HD UNEMP/OUT LBR FEB 84"  ;
label variable v11207  "HD UNEMP/OUT LBR MAR 84"  ;
label variable v11208  "HD UNEMP/OUT LBR APR 84"  ;
label variable v11209  "HD UNEMP/OUT LBR MAY 84"  ;
label variable v11210  "HD UNEMP/OUT LBR JUN 84"  ;
label variable v11211  "HD UNEMP/OUT LBR JUL 84"  ;
label variable v11212  "HD UNEMP/OUT LBR AUG 84"  ;
label variable v11213  "HD UNEMP/OUT LBR SEP 84"  ;
label variable v11214  "HD UNEMP/OUT LBR OCT 84"  ;
label variable v11215  "HD UNEMP/OUT LBR NOV 84"  ;
label variable v11216  "HD UNEMP/OUT LBR DEC 84"  ;
label variable v11217  "HD UNEMP/OUT LBR JAN 85"  ;
label variable v11218  "HD UNEMP/OUT LBR FEB 85"  ;
label variable v11219  "HD UNEMP/OUT LBR MAR 85"  ;
label variable v11220  "HD UNEMP/OUT LBR APR 85"  ;
label variable v11221  "HD UNEMP/OUT LBR MAY 85"  ;
label variable v11222  "HD UNEMP/OUT LBR JUN 85"  ;
label variable v11223  "HD UNEMP/OUT LBR JUL 85"  ;
label variable v11224  "HD UNEMP/OUT LBR AUG 85"  ;
label variable v11225  "HD UNEMP/OUT LBR SEP 85"  ;
label variable v11226  "HD UNEMP/OUT LBR OCT 85"  ;
label variable v11227  "HD UNEMPLYD QUALITY CODE"  ;
label variable v11228  "HD XTRA JOBS JAN 84"  ;
label variable v11229  "HD XTRA JOBS FEB 84"  ;
label variable v11230  "HD XTRA JOBS MAR 84"  ;
label variable v11231  "HD XTRA JOBS APR 84"  ;
label variable v11232  "HD XTRA JOBS MAY 84"  ;
label variable v11233  "HD XTRA JOBS JUN 84"  ;
label variable v11234  "HD XTRA JOBS JUL 84"  ;
label variable v11235  "HD XTRA JOBS AUG 84"  ;
label variable v11236  "HD XTRA JOBS SEP 84"  ;
label variable v11237  "HD XTRA JOBS OCT 84"  ;
label variable v11238  "HD XTRA JOBS NOV 84"  ;
label variable v11239  "HD XTRA JOBS DEC 84"  ;
label variable v11240  "HD XTRA JOBS JAN 85"  ;
label variable v11241  "HD XTRA JOBS FEB 85"  ;
label variable v11242  "HD XTRA JOBS MAR 85"  ;
label variable v11243  "HD XTRA JOBS APR 85"  ;
label variable v11244  "HD XTRA JOBS MAY 85"  ;
label variable v11245  "HD XTRA JOBS JUN 85"  ;
label variable v11246  "HD XTRA JOBS JUL 85"  ;
label variable v11247  "HD XTRA JOBS AUG 8S"  ;
label variable v11248  "HD XTRA JOBS SEP 85"  ;
label variable v11249  "HD XTRA JOBS OCT 85"  ;
label variable v11250  "HD XTRA JOB QUALITY CODE"  ;
label variable v11251  "WIFE IN FU?"  ;
label variable v11258  "WF ANN WRK HRS IN 84"  ;
label variable v11259  "WF HRS WRK LOST OTR ILL"  ;
label variable v11260  "ACC WF HR LOST OTR ILL"  ;
label variable v11261  "WF HRS WRK LOST OWN ILL"  ;
label variable v11262  "ACC WF HRS LOST OWN ILL"  ;
label variable v11263  "WF STRIKE HOURS 84"  ;
label variable v11264  "ACC WF STRIKE HRS 84"  ;
label variable v11265  "WF UNEMP HRS 84"  ;
label variable v11266  "ACC WF UNEMP HRS 84"  ;
label variable v11267  "WF HRS OUT LBR FORCE 84"  ;
label variable v11268  "ACC WF 84 HR OUT LBR FRC"  ;
label variable v11269  "WF HRS TRAVEL TO WORK 84"  ;
label variable v11270  "ACC WF TRAVEL TO WORK"  ;
label variable v11271  "WF WRK/EMPLOYER JAN 84"  ;
label variable v11272  "WF WRK/EMPLOYER FEB 84"  ;
label variable v11273  "WF WRK/EMPLOYER MAR 84"  ;
label variable v11274  "WF WRK/EMPLOYER APR 84"  ;
label variable v11275  "WF WRK/EMPLOYER MAY 84"  ;
label variable v11276  "WF WRK/EMPLOYER JUN 84"  ;
label variable v11277  "WF WRK/EMPLOYER JUL 84"  ;
label variable v11278  "WF WRK/EMPLOYER AUG 84"  ;
label variable v11279  "WF WRK/EMPLOYER SEP 84"  ;
label variable v11280  "WF WRK/EMPLOYER OCT 84"  ;
label variable v11281  "WF WRK/EMPLOYER NOV 84"  ;
label variable v11282  "WF WRK/EMPLOYER DEC 84"  ;
label variable v11283  "WF WRK/EMPLOYER JAN 85"  ;
label variable v11284  "WF WRK/EMPLOYER FEB 85"  ;
label variable v11285  "WF WRK/EMPLOYER MAR 85"  ;
label variable v11286  "WF WRK/EMPLOYER APR 85"  ;
label variable v11287  "WF WRK/EMPLOYER MAY 85"  ;
label variable v11288  "WF WRK/EMPLOYER JUN 85"  ;
label variable v11289  "WF WRK/EMPLOYER JUL 85"  ;
label variable v11290  "WF WRK/EMPLOYER AUG 85"  ;
label variable v11291  "WF WRK/EMPLOYER SEP 85"  ;
label variable v11292  "WF WRK/EMPLOYER OCT 85"  ;
label variable v11293  "WF WRK/EMPR QUALITY CODE"  ;
label variable v11294  "WF START/END POS JAN 84"  ;
label variable v11295  "WF START/END POS FEB 84"  ;
label variable v11296  "WF START/END POS MAR 84"  ;
label variable v11297  "WF START/END POS APR 84"  ;
label variable v11298  "WF START/END POS MAY 84"  ;
label variable v11299  "WF START/END POS JUN 84"  ;
label variable v11300  "WF START/END POS JUL 84"  ;
label variable v11301  "WF START/END POS AUG 84"  ;
label variable v11302  "WF START/END POS SEP 84"  ;
label variable v11303  "WF START/END POS OCT 84"  ;
label variable v11304  "WF START/END POS NOV 84"  ;
label variable v11305  "WF START/END POS DEC 84"  ;
label variable v11306  "WF START/END POS JAN 85"  ;
label variable v11307  "WF START/END POS FEB 85"  ;
label variable v11308  "WF START/END POS MAR 85"  ;
label variable v11309  "WF START/END POS APR 85"  ;
label variable v11310  "WF START/END POS MAY 85"  ;
label variable v11311  "WF START/END POS JUN 85"  ;
label variable v11312  "WF START/END POS JUL 85"  ;
label variable v11313  "WF START/END POS AUG 85"  ;
label variable v11314  "WF START/END POS SEP 85"  ;
label variable v11315  "WF START/END POS OCT 85"  ;
label variable v11316  "WF STRT/END QUALITY CODE"  ;
label variable v11317  "WF UNEMP/OUT LBR JAN 84"  ;
label variable v11318  "WF UNEMP/OUT LBR FEB 84"  ;
label variable v11319  "WF UNEMP/OUT LBR MAR 84"  ;
label variable v11320  "WF UNEMP/OUT LBR APR 84"  ;
label variable v11321  "WF UNEMP/OUT LBR MAY 84"  ;
label variable v11322  "WF UNEMP/OUT LBR JUN 84"  ;
label variable v11323  "WF UNEMP/OUT LBR JUL 84"  ;
label variable v11324  "WF UNEMP/OUT LBR AUG 84"  ;
label variable v11325  "WF UNEMP/OUT LBR SEP 84"  ;
label variable v11326  "WF UNEMP/OUT LBR OCT 84"  ;
label variable v11327  "WF UNEMP/OUT LBR NOV 84"  ;
label variable v11328  "WF UNEMP/OUT LBR DEC 84"  ;
label variable v11329  "WF UNEMP/OUT LBR JAN 85"  ;
label variable v11330  "WF UNEMP/OUT LBR FEB 85"  ;
label variable v11331  "WF UNEMP/OUT LBR MAR 85"  ;
label variable v11332  "WF UNEMP/OUT LBR APR 85"  ;
label variable v11333  "WF UNEMP/OUT LBR MAY 85"  ;
label variable v11334  "WF UNEMP/OUT LBR JUN 85"  ;
label variable v11335  "WF UNEMP/OUT LBR JUL 85"  ;
label variable v11336  "WF UNEMP/OUT LBR AUG 85"  ;
label variable v11337  "WF UNEMP/OUT LBR SEP 85"  ;
label variable v11338  "WF UNEMP/OUT LBR OCT 85"  ;
label variable v11339  "WF UNEMPLYD QUALITY CODE"  ;
label variable v11340  "WF XTRA JOBS JAN 84"  ;
label variable v11341  "WF XTRA JOBS FEB 84"  ;
label variable v11342  "WF XTRA JOBS MAR 84"  ;
label variable v11343  "WF XTRA JOBS APR 84"  ;
label variable v11344  "WF XTRA JOBS MAY 84"  ;
label variable v11345  "WF XTRA JOBS JUN 84"  ;
label variable v11346  "WF XTRA JOBS JUL 84"  ;
label variable v11347  "WF XTRA JOBS AUG 84"  ;
label variable v11348  "WF XTRA JOBS SEP 84"  ;
label variable v11349  "WF XTRA JOBS OCT 84"  ;
label variable v11350  "WF XTRA JOBS NOV 84"  ;
label variable v11351  "WF XTRA JOBS DEC 84"  ;
label variable v11352  "WF XTRA JOBS JAN 85"  ;
label variable v11353  "WF XTRA JOBS FEB 85"  ;
label variable v11354  "WF XTRA JOBS MAR 85"  ;
label variable v11355  "WF XTRA JOBS APR 85"  ;
label variable v11356  "WF XTRA JOBS MAY 85"  ;
label variable v11357  "WF XTRA JOBS JUN 85"  ;
label variable v11358  "WF XTRA JOBS JUL 85"  ;
label variable v11359  "WF XTRA JOBS AUG 85"  ;
label variable v11360  "WF XTRA JOBS SEP 85"  ;
label variable v11361  "WF XTRA JOBS OCT 85"  ;
label variable v11362  "WF XTRA JOB QUALITY CODE"  ;
label variable v11364  "1985 FAMILY SIZE"  ;
label variable v11373  "VALUE FD ST LAST MO (H9)"  ;
label variable v11374  "ACC VALUE FD ST LAST MO"  ;
label variable v11375  "ANN FOOD COST EXC FD ST"  ;
label variable v11376  "ACC ANN FOOD COST"  ;
label variable v11377  "ANN EAT OUT COST(H14/18)"  ;
label variable v11378  "ACC ANN EAT OUT COST"  ;
label variable v11379  "VALUE FD ST 84     (H20)"  ;
label variable v11380  "ACC VALUE FD ST 84"  ;
label variable v11427  "H+W 84 ADC/AFDC"  ;
label variable v11428  "ACC H+W 84 ADC/AFDC"  ;
label variable v11430  "H+W 84 SSI"  ;
label variable v11431  "H+W 84 OTR WELFARE"  ;
label variable v11440  "HD 84 UNEMP COMP"  ;
label variable v11441  "HD 84 WORKERS COMP"  ;
label variable v11445  "HD 84 HELP FROM RELS"  ;
label variable v11452  "WF 84 UNEMP COMP"  ;
label variable v11453  "WF 84 WORKERS COMP"  ;
label variable v11457  "WF 84 HELP FROM RELS"  ;
label variable v11565  "OFUM 84 ADC/AFDC"  ;
label variable v11566  "OFUM 84 SSI"  ;
label variable v11567  "OFUM 84 OTHER WELFARE"  ;
label variable v11571  "OFUM 84 UNEMP COMP"  ;
label variable v11572  "OFUM 84 WORKERS COMP"  ;
label variable v11574  "OFUM 84 HELP FROM RELS"  ;
label variable v11600  "DATE OF 1985 IW-HEAD"  ;
label variable v11603  "DATE OF 1985 IW-WIFE"  ;
label variable v11605  "# IN FU"  ;
label variable v11606  "AGE OF 1985 HEAD"  ;
label variable v11607  "SEX OF 1985 HEAD"  ;
label variable v11608  "AGE OF 1985 WIFE"  ;
label variable v11609  "# CHILDREN IN FU"  ;
label variable v11612  "A3 MARITAL STATUS"  ;
label variable v11637  "B1 EMPLMT STATUS-HEAD"  ;
label variable v11638  "B2 YEAR RETIRED   (HD-E)"  ;
label variable v11639  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable v11640  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable v11653  "B18 SLRY/HRLY/OTR  (H-E)"  ;
label variable v11654  "B19 PAY/HR-SALARY (HD-E)"  ;
label variable v11657  "B22 PAY/HR-HOURLY (HD-E)"  ;
label variable v11767  "C27 LAST END WAGE (HD-U)"  ;
label variable v12000  "J1F EMPLMT STATUS-WIFE"  ;
label variable v12001  "J2 YEAR RETIRED   (WF-E)"  ;
label variable v12002  "J3 WORK FOR MONEY?(WF-E)"  ;
label variable v12003  "J4 WORK SELF/OTR? (WF-E)"  ;
label variable v12016  "J18 SLRY/HRLY/OTR  (W-E)"  ;
label variable v12017  "J19 PAY/HR-SALARY (WF-E)"  ;
label variable v12020  "J22 PAY/HR-HOURLY (WF-E)"  ;
label variable v12130  "K27 LAST END WAGE (WF-U)"  ;
label variable v12371  "TOT FAM MONEY Y 84"  ;
label variable v12372  "TOTAL HEAD LABOR Y 84"  ;
label variable v12373  "WEEKLY FOOD NEEDS-1985"  ;
label variable v12374  "ANNUAL NEEDS STD-1984"  ;
label variable v12376  "ANNUAL FOOD STD"  ;
label variable v12377  "HEAD 84 AVG HRLY EARNING"  ;
label variable v12378  "WIFE 84 AVG HRLY EARNING"  ;
label variable v12400  "EDUCATION 1985 HEAD"  ;
label variable v12401  "EDUCATION 1985 WIFE"  ;
label variable v12425  "18YR FAM COMP CHANGE"  ;
label variable v12426  "MARITAL STATUS 1985"  ;
label variable v12427  "84-85 CNG MARITAL STATUS"  ;
label variable v12428  "COUPLE STATUS OF HEAD"  ;
label variable v12445  "1983 OLD FAMILY WEIGHT"  ;
label variable v12446  "1985 FAMILY WEIGHT"  ;
label variable er30498  "1986 INTERVIEW NUMBER"  ;
label variable er30499  "SEQUENCE NUMBER                       86"  ;
label variable er30500  "RELATIONSHIP TO HEAD                  86"  ;
label variable er30501  "AGE OF INDIVIDUAL                     86"  ;
label variable er30502  "MONTH IND BORN                        86"  ;
label variable er30503  "YEAR IND BORN                         86"  ;
label variable er30531  "AGE FROM BIRTH DATE                   86"  ;
label variable v12502  "1986 INTERVIEW NUMBER"  ;
label variable v12507  "SPLITOFF INDICATOR    86"  ;
label variable v12510  "FAM COMP CHANGE       86"  ;
label variable v12545  "HD ANN WRK HRS IN 85"  ;
label variable v12546  "HD HRS WRK LOST OTR ILL"  ;
label variable v12547  "ACC HD HR LOST OTR ILL"  ;
label variable v12548  "HD HRS WRK LOST OWN ILL"  ;
label variable v12549  "ACC HD HRS LOST OWN ILL"  ;
label variable v12550  "HD STRIKE HOURS 85"  ;
label variable v12551  "ACC HD STRIKE HRS 85"  ;
label variable v12552  "HD UNEMP HRS 85"  ;
label variable v12553  "ACC 85 HD UNEMP HRS"  ;
label variable v12554  "HD HRS OUT LBR FORCE 85"  ;
label variable v12555  "ACC 85 HD HR OUT LBR FRC"  ;
label variable v12556  "HD HRS TRAVEL TO WORK 85"  ;
label variable v12557  "ACC 85 HD TRAVEL TO WORK"  ;
label variable v12558  "HD WRKD/EMPLOYER JAN 85"  ;
label variable v12559  "HD WRKD/EMPLOYER FEB 85"  ;
label variable v12560  "HD WRKD/EMPLOYER MAR 85"  ;
label variable v12561  "HD WRKD/EMPLOYER APR 85"  ;
label variable v12562  "HD WRKD/EMPLOYER MAY 85"  ;
label variable v12563  "HD WRKD/EMPLOYER JUN 85"  ;
label variable v12564  "HD WRKD/EMPLOYER JUL 85"  ;
label variable v12565  "HD WRKD/EMPLOYER AUG 85"  ;
label variable v12566  "HD WRKD/EMPLOYER SEP 85"  ;
label variable v12567  "HD WRKD/EMPLOYER OCT 85"  ;
label variable v12568  "HD WRKD/EMPLOYER NOV 85"  ;
label variable v12569  "HD WRKD/EMPLOYER DEC 85"  ;
label variable v12570  "HD WRKD/EMPLOYER JAN 86"  ;
label variable v12571  "HD WRKD/EMPLOYER FEB 86"  ;
label variable v12572  "HD WRKD/EMPLOYER MAR 86"  ;
label variable v12573  "HD WRKD/EMPLOYER APR 86"  ;
label variable v12574  "HD WRKD/EMPLOYER MAY 86"  ;
label variable v12575  "HD WRKD/EMPLOYER JUN 86"  ;
label variable v12576  "HD WRKD/EMPLOYER JUL 86"  ;
label variable v12577  "HD WRKD/EMPLOYER AUG 86"  ;
label variable v12578  "HD WRKD/EMPLOYER SEP 86"  ;
label variable v12579  "HD WRKD/EMPLOYER OCT 86"  ;
label variable v12580  "HD WRK/EMPR QUALITY CODE"  ;
label variable v12581  "HD START/END POS JAN 85"  ;
label variable v12582  "HD START/END POS FEB 85"  ;
label variable v12583  "HD START/END POS MAR 85"  ;
label variable v12584  "HD START/END POS APR 85"  ;
label variable v12585  "HD START/END POS MAY 85"  ;
label variable v12586  "HD START/END POS JUN 85"  ;
label variable v12587  "HD START/END POS JUL 85"  ;
label variable v12588  "HD START/END POS AUG 85"  ;
label variable v12589  "HD START/END POS SEP 85"  ;
label variable v12590  "HD START/END POS OCT 85"  ;
label variable v12591  "HD START/END POS NOV 85"  ;
label variable v12592  "HD START/END POS DEC 85"  ;
label variable v12593  "HD START/END POS JAN 86"  ;
label variable v12594  "HD START/END POS FEB 86"  ;
label variable v12595  "HD START/END POS MAR 86"  ;
label variable v12596  "HD START/END POS APR 86"  ;
label variable v12597  "HD START/END POS MAY 86"  ;
label variable v12598  "HD START/END POS JUN 86"  ;
label variable v12599  "HD START/END POS JUL 86"  ;
label variable v12600  "HD START/END POS AUG 86"  ;
label variable v12601  "HD START/END POS SEP 86"  ;
label variable v12602  "HD START/END POS OCT 86"  ;
label variable v12603  "HD STRT/END QUALITY CODE"  ;
label variable v12604  "HD UNEMP/OUT LBR JAN 85"  ;
label variable v12605  "HD UNEMP/OUT LBR FEB 85"  ;
label variable v12606  "HD UNEMP/OUT LBR MAR 85"  ;
label variable v12607  "HD UNEMP/OUT LBR APR 85"  ;
label variable v12608  "HD UNEMP/OUT LBR MAY 85"  ;
label variable v12609  "HD UNEMP/OUT LBR JUN 85"  ;
label variable v12610  "HD UNEMP/OUT LBR JUL 85"  ;
label variable v12611  "HD UNEMP/OUT LBR AUG 85"  ;
label variable v12612  "HD UNEMP/OUT LBR SEP 85"  ;
label variable v12613  "HD UNEMP/OUT LBR OCT 85"  ;
label variable v12614  "HD UNEMP/OUT LBR NOV 85"  ;
label variable v12615  "HD UNEMP/OUT LBR DEC 85"  ;
label variable v12616  "HD UNEMP/OUT LBR JAN 86"  ;
label variable v12617  "HD UNEMP/OUT LBR FEB 86"  ;
label variable v12618  "HD UNEMP/OUT LBR MAR 86"  ;
label variable v12619  "HD UNEMP/OUT LBR APR 86"  ;
label variable v12620  "HD UNEMP/OUT LBR MAY 86"  ;
label variable v12621  "HD UNEMP/OUT LBR JUN 86"  ;
label variable v12622  "HD UNEMP/OUT LBR JUL 86"  ;
label variable v12623  "HD UNEMP/OUT LBR AUG 86"  ;
label variable v12624  "HD UNEMP/OUT LBR SEP 86"  ;
label variable v12625  "HD UNEMP/OUT LBR OCT 86"  ;
label variable v12626  "HD UNEMPLYD QUALITY CODE"  ;
label variable v12627  "HD XTRA JOBS JAN 85"  ;
label variable v12628  "HD XTRA JOBS FEB 85"  ;
label variable v12629  "HD XTRA JOBS MAR 85"  ;
label variable v12630  "HD XTRA JOBS APR 85"  ;
label variable v12631  "HD XTRA JOBS MAY 85"  ;
label variable v12632  "HD XTRA JOBS JUN 85"  ;
label variable v12633  "HD XTRA JOBS JUL 85"  ;
label variable v12634  "HD XTRA JOBS AUG 85"  ;
label variable v12635  "HD XTRA JOBS SEP 85"  ;
label variable v12636  "HD XTRA JOBS OCT 85"  ;
label variable v12637  "HD XTRA JOBS NOV 85"  ;
label variable v12638  "HD XTRA JOBS DEC 85"  ;
label variable v12639  "HD XTRA JOBS JAN 86"  ;
label variable v12640  "HD XTRA JOBS FEB 86"  ;
label variable v12641  "HD XTRA JOBS MAR 86"  ;
label variable v12642  "HD XTRA JOBS APR 86"  ;
label variable v12643  "HD XTRA JOBS MAY 86"  ;
label variable v12644  "HD XTRA JOBS JUN 86"  ;
label variable v12645  "HD XTRA JOBS JUL 86"  ;
label variable v12646  "HD XTRA JOBS AUG 86"  ;
label variable v12647  "HD XTRA JOBS SEP 86"  ;
label variable v12648  "HD XTRA JOBS OCT 86"  ;
label variable v12649  "HD XTRA JOB QUALITY CODE"  ;
label variable v12650  "WIFE IN FU?"  ;
label variable v12657  "WF ANN WRK HRS IN 85"  ;
label variable v12658  "WF HRS WRK LOST OTR ILL"  ;
label variable v12659  "ACC WF HR LOST OTR ILL"  ;
label variable v12660  "WF HRS WRK LOST OWN ILL"  ;
label variable v12661  "ACC WF HRS LOST OWN ILL"  ;
label variable v12662  "WF STRIKE HOURS 85"  ;
label variable v12663  "ACC WF STRIKE HRS 85"  ;
label variable v12664  "WF UNEMP HRS 85"  ;
label variable v12665  "ACC WF UNEMP HRS 85"  ;
label variable v12666  "WF HRS OUT LBR FORCE 85"  ;
label variable v12667  "ACC WF 85 HR OUT LBR FRC"  ;
label variable v12668  "WF HRS TRAVEL TO WORK 85"  ;
label variable v12669  "ACC WF TRAVEL TO WORK"  ;
label variable v12670  "WF WRK/EMPLOYER JAN 85"  ;
label variable v12671  "WF WRK/EMPLOYER FEB 85"  ;
label variable v12672  "WF WRK/EMPLOYER MAR 85"  ;
label variable v12673  "WF WRK/EMPLOYER APR 85"  ;
label variable v12674  "WF WRK/EMPLOYER MAY 85"  ;
label variable v12675  "WF WRK/EMPLOYER JUN 85"  ;
label variable v12676  "WF WRK/EMPLOYER JUL 85"  ;
label variable v12677  "WF WRK/EMPLOYER AUG 85"  ;
label variable v12678  "WF WRK/EMPLOYER SEP 85"  ;
label variable v12679  "WF WRK/EMPLOYER OCT 85"  ;
label variable v12680  "WF WRK/EMPLOYER NOV 85"  ;
label variable v12681  "WF WRK/EMPLOYER DEC 85"  ;
label variable v12682  "WF WRK/EMPLOYER JAN 86"  ;
label variable v12683  "WF WRK/EMPLOYER FEB 86"  ;
label variable v12684  "WF WRK/EMPLOYER MAR 86"  ;
label variable v12685  "WF WRK/EMPLOYER APR 86"  ;
label variable v12686  "WF WRK/EMPLOYER MAY 86"  ;
label variable v12687  "WF WRK/EMPLOYER JUN 86"  ;
label variable v12688  "WF WRK/EMPLOYER JUL 86"  ;
label variable v12689  "WF WRK/EMPLOYER AUG 86"  ;
label variable v12690  "WF WRK/EMPLOYER SEP 86"  ;
label variable v12691  "WF WRK/EMPLOYER OCT 86"  ;
label variable v12692  "WF WRK/EMPR QUALITY CODE"  ;
label variable v12693  "WF START/END POS JAN 85"  ;
label variable v12694  "WF START/END POS FEB 85"  ;
label variable v12695  "WF START/END POS MAR 85"  ;
label variable v12696  "WF START/END POS APR 85"  ;
label variable v12697  "WF START/END POS MAY 85"  ;
label variable v12698  "WF START/END POS JUN 85"  ;
label variable v12699  "WF START/END POS JUL 85"  ;
label variable v12700  "WF START/END POS AUG 85"  ;
label variable v12701  "WF START/END POS SEP 85"  ;
label variable v12702  "WF START/END POS OCT 85"  ;
label variable v12703  "WF START/END POS NOV 85"  ;
label variable v12704  "WF START/END POS DEC 85"  ;
label variable v12705  "WF START/END POS JAN 86"  ;
label variable v12706  "WF START/END POS FEB 86"  ;
label variable v12707  "WF START/END POS MAR 86"  ;
label variable v12708  "WF START/END POS APR 86"  ;
label variable v12709  "WF START/END POS MAY 86"  ;
label variable v12710  "WF START/END POS JUN 86"  ;
label variable v12711  "WF START/END POS JUL 86"  ;
label variable v12712  "WF START/END POS AUG 86"  ;
label variable v12713  "WF START/END POS SEP 86"  ;
label variable v12714  "WF START/END POS OCT 86"  ;
label variable v12715  "WF STRT/END QUALITY CODE"  ;
label variable v12716  "WF UNEMP/OUT LBR JAN 85"  ;
label variable v12717  "WF UNEMP/OUT LBR FEB 85"  ;
label variable v12718  "WF UNEMP/OUT LBR MAR 85"  ;
label variable v12719  "WF UNEMP/OUT LBR APR 85"  ;
label variable v12720  "WF UNEMP/OUT LBR MAY 85"  ;
label variable v12721  "WF UNEMP/OUT LBR JUN 85"  ;
label variable v12722  "WF UNEMP/OUT LBR JUL 85"  ;
label variable v12723  "WF UNEMP/OUT LBR AUG 85"  ;
label variable v12724  "WF UNEMP/OUT LBR SEP 85"  ;
label variable v12725  "WF UNEMP/OUT LBR OCT 85"  ;
label variable v12726  "WF UNEMP/OUT LBR NOV 85"  ;
label variable v12727  "WF UNEMP/OUT LBR DEC 85"  ;
label variable v12728  "WF UNEMP/OUT LBR JAN 86"  ;
label variable v12729  "WF UNEMP/OUT LBR FEB 86"  ;
label variable v12730  "WF UNEMP/OUT LBR MAR 86"  ;
label variable v12731  "WF UNEMP/OUT LBR APR 86"  ;
label variable v12732  "WF UNEMP/OUT LBR MAY 86"  ;
label variable v12733  "WF UNEMP/OUT LBR JUN 86"  ;
label variable v12734  "WF UNEMP/OUT LBR JUL 86"  ;
label variable v12735  "WF UNEMP/OUT LBR AUG 86"  ;
label variable v12736  "WF UNEMP/OUT LBR SEP 86"  ;
label variable v12737  "WF UNEMP/OUT LBR OCT 86"  ;
label variable v12738  "WF UNEMPLYD QUALITY CODE"  ;
label variable v12739  "WF XTRA JOBS JAN 85"  ;
label variable v12740  "WF XTRA JOBS FEB 85"  ;
label variable v12741  "WF XTRA JOBS MAR 85"  ;
label variable v12742  "WF XTRA JOBS APR 85"  ;
label variable v12743  "WF XTRA JOBS MAY 85"  ;
label variable v12744  "WF XTRA JOBS JUN 85"  ;
label variable v12745  "WF XTRA JOBS JUL 85"  ;
label variable v12746  "WF XTRA JOBS AUG 85"  ;
label variable v12747  "WF XTRA JOBS SEP 85"  ;
label variable v12748  "WF XTRA JOBS OCT 85"  ;
label variable v12749  "WF XTRA JOBS NOV 85"  ;
label variable v12750  "WF XTRA JOBS DEC 85"  ;
label variable v12751  "WF XTRA JOBS JAN 86"  ;
label variable v12752  "WF XTRA JOBS FEB 86"  ;
label variable v12753  "WF XTRA JOBS MAR 86"  ;
label variable v12754  "WF XTRA JOBS APR 86"  ;
label variable v12755  "WF XTRA JOBS MAY 86"  ;
label variable v12756  "WF XTRA JOBS JUN 86"  ;
label variable v12757  "WF XTRA JOBS JUL 86"  ;
label variable v12758  "WF XTRA JOBS AUG 86"  ;
label variable v12759  "WF XTRA JOBS SEP 86"  ;
label variable v12760  "WF XTRA JOBS OCT 86"  ;
label variable v12761  "WF XTRA JOB QUALITY CODE"  ;
label variable v12763  "1986 FAMILY SIZE"  ;
label variable v12772  "VALUE FD ST LAST MO(F11)"  ;
label variable v12773  "ACC VALUE FD ST LAST MO"  ;
label variable v12774  "ANN FOOD COST EXC FD ST"  ;
label variable v12775  "ACC ANN FOOD COST"  ;
label variable v12776  "ANN EAT OUT COST(F16/2O)"  ;
label variable v12777  "ACC ANN EAT OUT COST"  ;
label variable v12778  "VALUE FD ST 85     (F22)"  ;
label variable v12779  "ACC VALUE FD ST 85"  ;
label variable v12826  "HD 85 ADC/AFDC"  ;
label variable v12827  "ACC HD 85 ADC/AFDC"  ;
label variable v12829  "HD 85 SSI"  ;
label variable v12830  "HD 85 OTR WELFARE"  ;
label variable v12839  "HD 85 UNEMP COMP"  ;
label variable v12840  "HD 85 WORKERS COMP"  ;
label variable v12844  "HD 85 HELP FROM RELS"  ;
label variable v12847  "WF 85 ADC/AFDC"  ;
label variable v12848  "ACC WF 85 ADC/AFDC"  ;
label variable v12850  "WF 85 SSI"  ;
label variable v12851  "WF 85 OTR WELFARE"  ;
label variable v12859  "WF 85 UNEMP COMP"  ;
label variable v12860  "WF 85 WORKERS COMP"  ;
label variable v12864  "WF 85 HELP FROM RELS"  ;
label variable v12972  "OFUM 85 ADC/AFDC"  ;
label variable v12973  "OFUM 85 SSI"  ;
label variable v12974  "OFUM 85 OTHER WELFARE"  ;
label variable v12978  "OFUM 85 UNEMP COMP"  ;
label variable v12979  "OFUM 85 WORKERS COMP"  ;
label variable v12981  "OFUM 85 HELP FROM RELS"  ;
label variable v13008  "DATE OF 1986 IW"  ;
label variable v13010  "# IN FU"  ;
label variable v13011  "AGE OF 1986 HEAD"  ;
label variable v13012  "SEX OF 1986 HEAD"  ;
label variable v13013  "AGE OF 1986 WIFE"  ;
label variable v13014  "# CHILDREN IN FU"  ;
label variable v13017  "A3 MARITAL STATUS"  ;
label variable v13046  "B1 EMPLMT STATUS-HEAD"  ;
label variable v13047  "B2 YEAR RETIRED   (HD-R)"  ;
label variable v13048  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable v13049  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable v13056  "B12 SLRY/HRLY/OTR  (H-E)"  ;
label variable v13057  "B13 PAY/HR-SALARY (HD-E)"  ;
label variable v13060  "B16 PAY/HR-HOURLY (HD-E)"  ;
label variable v13163  "C27 LAST END WAGE (HD-U)"  ;
label variable v13225  "D1B EMPLMT STATUS-WIFE"  ;
label variable v13226  "D2 YEAR RETIRED   (WF-E)"  ;
label variable v13227  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable v13228  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable v13235  "D12 SLRY/HRLY/OTR  (W-E)"  ;
label variable v13236  "D13 PAY/HR-SALARY (WF-E)"  ;
label variable v13239  "D16 PAY/HR-HOURLY (WF-E)"  ;
label variable v13331  "E25 LAST END WAGE (WF-U)"  ;
label variable v13623  "TOT FAM MONEY Y 85"  ;
label variable v13624  "TOTAL HEAD LABOR Y 85"  ;
label variable v13625  "WEEKLY FOOD NEEDS-1986"  ;
label variable v13626  "ANNUAL NEEDS STD-1985"  ;
label variable v13628  "ANNUAL FOOD STD"  ;
label variable v13629  "HEAD 85 AVG HRLY EARNING"  ;
label variable v13630  "WIFE 85 AVG HRLY EARNING"  ;
label variable v13640  "EDUCATION 1986 HEAD"  ;
label variable v13641  "EDUCATION 1986 WIFE"  ;
label variable v13665  "MARITAL STATUS 1986 HEAD"  ;
label variable v13666  "85-86 CNG MARITAL STATUS"  ;
label variable v13667  "COUPLE STATUS OF HEAD"  ;
label variable v13687  "1986 FAMILY WEIGHT"  ;
label variable er30535  "1987 INTERVIEW NUMBER"  ;
label variable er30536  "SEQUENCE NUMBER                       87"  ;
label variable er30537  "RELATIONSHIP TO HEAD                  87"  ;
label variable er30538  "AGE OF INDIVIDUAL                     87"  ;
label variable er30539  "MONTH IND BORN                        87"  ;
label variable er30540  "YEAR IND BORN                         87"  ;
label variable er30566  "AGE FROM BIRTH DATE                   87"  ;
label variable v13702  "1987 INTERVIEW NUMBER"  ;
label variable v13707  "SPLITOFF INDICATOR    87"  ;
label variable v13710  "FAM COMP CHANGE       87"  ;
label variable v13745  "HD ANN WRK HRS IN 86"  ;
label variable v13746  "HD HRS WRK LOST OTR ILL"  ;
label variable v13747  "ACC HD HR LOST OTR ILL"  ;
label variable v13748  "HD HRS WRK LOST OWN ILL"  ;
label variable v13749  "ACC HD HRS LOST OWN ILL"  ;
label variable v13750  "HD STRIKE HOURS 86"  ;
label variable v13751  "ACC HD STRIKE HRS 86"  ;
label variable v13752  "HD UNEMP HRS 86"  ;
label variable v13753  "ACC 86 HD UNEMP HRS"  ;
label variable v13754  "HD HRS OUT LBR FORCE 86"  ;
label variable v13755  "ACC 86 HD HR OUT LBR FRC"  ;
label variable v13756  "HD UNEMP/OUT LBR JAN 86"  ;
label variable v13757  "HD UNEMP/OUT LBR FEB 86"  ;
label variable v13758  "HD UNEMP/OUT LBR MAR 86"  ;
label variable v13759  "HD UNEMP/OUT LBR APR 86"  ;
label variable v13760  "HD UNEMP/OUT LBR MAY 86"  ;
label variable v13761  "HD UNEMP/OUT LBR JUN 86"  ;
label variable v13762  "HD UNEMP/OUT LBR JUL 86"  ;
label variable v13763  "HD UNEMP/OUT LBR AUG 86"  ;
label variable v13764  "HD UNEMP/OUT LBR SEP 86"  ;
label variable v13765  "HD UNEMP/OUT LBR OCT 86"  ;
label variable v13766  "HD UNEMP/OUT LBR NOV 86"  ;
label variable v13767  "HD UNEMP/OUT LBR DEC 86"  ;
label variable v13768  "HD UNEMP/OUT LBR JAN 87"  ;
label variable v13769  "HD UNEMP/OUT LBR FEB 87"  ;
label variable v13770  "HD UNEMP/OUT LBR MAR 87"  ;
label variable v13771  "HD UNEMP/OUT LBR APR 87"  ;
label variable v13772  "HD UNEMP/OUT LBR MAY 87"  ;
label variable v13773  "HD UNEMP/OUT LBR JUN 87"  ;
label variable v13774  "HD UNEMP/OUT LBR JUL 87"  ;
label variable v13775  "HD UNEMP/OUT LBR AUG 87"  ;
label variable v13776  "HD UNEMP/OUT LBR SEP 87"  ;
label variable v13777  "HD UNEMP/OUT LBR OCT 87"  ;
label variable v13778  "HD UNEMPLYD QUALITY CODE"  ;
label variable v13779  "HD XTRA JOBS JAN 86"  ;
label variable v13780  "HD XTRA JOBS FEB 86"  ;
label variable v13781  "HD XTRA JOBS MAR 86"  ;
label variable v13782  "HD XTRA JOBS APR 86"  ;
label variable v13783  "HD XTRA JOBS MAY 86"  ;
label variable v13784  "HD XTRA JOBS JUN 86"  ;
label variable v13785  "HD XTRA JOBS JUL 86"  ;
label variable v13786  "HD XTRA JOBS AUG 86"  ;
label variable v13787  "HD XTRA JOBS SEP 86"  ;
label variable v13788  "HD XTRA JOBS OCT 86"  ;
label variable v13789  "HD XTRA JOBS NOV 86"  ;
label variable v13790  "HD XTRA JOBS DEC 86"  ;
label variable v13791  "HD XTRA JOBS JAN 87"  ;
label variable v13792  "HD XTRA JOBS FEB 87"  ;
label variable v13793  "HD XTRA JOBS MAR 87"  ;
label variable v13794  "HD XTRA JOBS APR 87"  ;
label variable v13795  "HD XTRA JOBS MAY 87"  ;
label variable v13796  "HD XTRA JOBS JUN 87"  ;
label variable v13797  "HD XTRA JOBS JUL 87"  ;
label variable v13798  "HD XTRA JOBS AUG 87"  ;
label variable v13799  "HD XTRA JOBS SEP 87"  ;
label variable v13800  "HD XTRA JOBS OCT 87"  ;
label variable v13801  "HD XTRA JOB QUALITY CODE"  ;
label variable v13802  "WIFE IN FU?"  ;
label variable v13809  "WF ANN WRK HRS IN 86"  ;
label variable v13810  "WF HRS WRK LOST OTR ILL"  ;
label variable v13811  "ACC WF HR LOST OTR ILL"  ;
label variable v13812  "WF HRS WRK LOST OWN ILL"  ;
label variable v13813  "ACC WF HRS LOST OWN ILL"  ;
label variable v13814  "WF STRIKE HOURS 86"  ;
label variable v13815  "ACC WF STRIKE HRS 86"  ;
label variable v13816  "WF UNEMP HRS 86"  ;
label variable v13817  "ACC WF UNEMP HRS 86"  ;
label variable v13818  "WF HRS OUT LBR FORCE 86"  ;
label variable v13819  "ACC WF 86 HR OUT LBR FRC"  ;
label variable v13820  "WF UNEMP/OUT LBR JAN 86"  ;
label variable v13821  "WF UNEMP/OUT LBR FEB 86"  ;
label variable v13822  "WF UNEMP/OUT LBR MAR 86"  ;
label variable v13823  "WF UNEMP/OUT LBR APR 86"  ;
label variable v13824  "WF UNEMP/OUT LBR MAY 86"  ;
label variable v13825  "WF UNEMP/OUT LBR JUN 86"  ;
label variable v13826  "WF UNEMP/OUT LBR JUL 86"  ;
label variable v13827  "WF UNEMP/OUT LBR AUG 86"  ;
label variable v13828  "WF UNEMP/OUT LBR SEP 86"  ;
label variable v13829  "WF UNEMP/OUT LBR OCT 86"  ;
label variable v13830  "WF UNEMP/OUT LBR NOV 86"  ;
label variable v13831  "WF UNEMP/OUT LBR DEC 86"  ;
label variable v13832  "WF UNEMP/OUT LBR JAN 87"  ;
label variable v13833  "WF UNEMP/OUT LBR FEB 87"  ;
label variable v13834  "WF UNEMP/OUT LBR MAR 87"  ;
label variable v13835  "WF UNEMP/OUT LBR APR 87"  ;
label variable v13836  "WF UNEMP/OUT LBR MAY 87"  ;
label variable v13837  "WF UNEMP/OUT LBR JUN 87"  ;
label variable v13838  "WF UNEMP/OUT LBR JUL 87"  ;
label variable v13839  "WF UNEMP/OUT LBR AUG 87"  ;
label variable v13840  "WF UNEMP/OUT LBR SEP 87"  ;
label variable v13841  "WF UNEMP/OUT LBR OCT 87"  ;
label variable v13842  "WF UNEMPLYD QUALITY CODE"  ;
label variable v13843  "WF XTRA JOBS JAN 86"  ;
label variable v13844  "WF XTRA JOBS FEB 86"  ;
label variable v13845  "WF XTRA JOBS MAR 86"  ;
label variable v13846  "WF XTRA JOBS APR 86"  ;
label variable v13847  "WF XTRA JOBS MAY 86"  ;
label variable v13848  "WF XTRA JOBS JUN 86"  ;
label variable v13849  "WF XTRA JOBS JUL 86"  ;
label variable v13850  "WF XTRA JOBS AUG 86"  ;
label variable v13851  "WF XTRA JOBS SEP 86"  ;
label variable v13852  "WF XTRA JOBS OCT 86"  ;
label variable v13853  "WF XTRA JOBS NOV 86"  ;
label variable v13854  "WF XTRA JOBS DEC 86"  ;
label variable v13855  "WF XTRA JOBS JAN 87"  ;
label variable v13856  "WF XTRA JOBS FEB 87"  ;
label variable v13857  "WF XTRA JOBS MAR 87"  ;
label variable v13858  "WF XTRA JOBS APR 87"  ;
label variable v13859  "WF XTRA JOBS MAY 87"  ;
label variable v13860  "WF XTRA JOBS JUN 87"  ;
label variable v13861  "WF XTRA JOBS JUL 87"  ;
label variable v13862  "WF XTRA JOBS AUG 87"  ;
label variable v13863  "WF XTRA JOBS SEP 87"  ;
label variable v13864  "WF XTRA JOBS OCT 87"  ;
label variable v13865  "WF XTRA JOB QUALITY CODE"  ;
label variable v13867  "1987 FAMILY SIZE"  ;
label variable v13874  "VALUE FD ST LAST MO (F6)"  ;
label variable v13875  "ACC VALUE FD ST LAST MO"  ;
label variable v13876  "ANN FOOD COST EXC FD ST"  ;
label variable v13877  "ACC ANN FD COST EX FD ST"  ;
label variable v13878  "ANN COST EAT OUT(F11/15)"  ;
label variable v13879  "ACC ANN COST EAT OUT"  ;
label variable v13880  "VALUE FD ST 86     (F17)"  ;
label variable v13881  "ACC VALUE FD ST 86"  ;
label variable v13928  "HD 86 ADC/AFDC"  ;
label variable v13929  "ACC HD 86 ADC/AFDC"  ;
label variable v13931  "HD 86 SSI"  ;
label variable v13932  "HD 86 OTR WELFARE"  ;
label variable v13941  "HD 86 UNEMP COMP"  ;
label variable v13942  "HD 86 WORKERS COMP"  ;
label variable v13946  "HD 86 HELP FROM RELS"  ;
label variable v13949  "WF 86 ADC/AFDC"  ;
label variable v13950  "ACC WF 86 ADC/AFDC"  ;
label variable v13952  "WF 86 SSI"  ;
label variable v13953  "WF 86 OTR WELFARE"  ;
label variable v13961  "WF 86 UNEMP COMP"  ;
label variable v13962  "WF 86 WORKERS COMP"  ;
label variable v13966  "WF 86 HELP FROM RELS"  ;
label variable v14074  "OFUM 86 ADC/AFDC"  ;
label variable v14075  "OFUM 86 SSI"  ;
label variable v14076  "OFUM 86 OTHER WELFARE"  ;
label variable v14080  "OFUM 86 UNEMP COMP"  ;
label variable v14081  "OFUM 86 WORKERS COMP"  ;
label variable v14083  "OFUM 86 HELP FROM RELS"  ;
label variable v14111  "DATE OF 1987 IW"  ;
label variable v14113  "# IN FU"  ;
label variable v14114  "AGE OF 1987 HEAD"  ;
label variable v14115  "SEX OF 1987 HEAD"  ;
label variable v14116  "AGE OF 1987 WIFE"  ;
label variable v14117  "# CHILDREN IN FU"  ;
label variable v14120  "A3 MARITAL STATUS"  ;
label variable v14146  "B1 EMPLMT STATUS-HEAD"  ;
label variable v14147  "B2 YEAR RETIRED   (HD-R)"  ;
label variable v14148  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable v14149  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable v14156  "B12 SLRY/HRLY/OTR  (H-E)"  ;
label variable v14157  "B13 PAY/HR-SALARY (HD-E)"  ;
label variable v14160  "B16 PAY/HR-HOURLY (HD-E)"  ;
label variable v14259  "C24 LAST END WAGE (HD-U)"  ;
label variable v14321  "D1B EMPLMT STATUS-WIFE"  ;
label variable v14322  "D2 YEAR RETIRED   (WF-E)"  ;
label variable v14323  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable v14324  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable v14331  "D12 SLRY/HRLY/OTR  (W-E)"  ;
label variable v14332  "D13 PAY/HR-SALARY (WF-E)"  ;
label variable v14335  "D16 PAY/HR-HOURLY (WF-E)"  ;
label variable v14423  "E22 LAST END WAGE (WF-U)"  ;
label variable v14670  "TOT FAM MONEY Y 86"  ;
label variable v14671  "TOTAL HEAD LABOR Y 86"  ;
label variable v14672  "WEEKLY FOOD NEEDS-1987"  ;
label variable v14673  "ANNUAL NEEDS STD-1986"  ;
label variable v14675  "ANNUAL FOOD STD"  ;
label variable v14676  "HEAD 86 AVG HRLY EARNING"  ;
label variable v14677  "WIFE 86 AVG HRLY EARNING"  ;
label variable v14687  "EDUCATION 1987 HEAD"  ;
label variable v14688  "EDUCATION 1987 WIFE"  ;
label variable v14712  "MARITAL STATUS 1987"  ;
label variable v14713  "86-87 CNG MARITAL STATUS"  ;
label variable v14714  "COUPLE STATUS OF HEAD"  ;
label variable v14737  "1987 FAMILY WEIGHT"  ;
label variable er30570  "1988 INTERVIEW NUMBER"  ;
label variable er30571  "SEQUENCE NUMBER                       88"  ;
label variable er30572  "RELATION TO HEAD                      88"  ;
label variable er30573  "AGE OF INDIVIDUAL                     88"  ;
label variable er30574  "MONTH INDIVIDUAL BORN                 88"  ;
label variable er30575  "YEAR INDIVIDUAL BORN                  88"  ;
label variable er30602  "AGE FROM BIRTH DATE                   88"  ;
label variable v14802  "1988 INTERVIEW NUMBER"  ;
label variable v14807  "SPLITOFF INDICATOR    88"  ;
label variable v14810  "FAM COMP CHANGE       88"  ;
label variable v14846  "HD UNEMP/OUT LBR JAN 87"  ;
label variable v14847  "HD UNEMP/OUT LBR FEB 87"  ;
label variable v14848  "HD UNEMP/OUT LBR MAR 87"  ;
label variable v14849  "HD UNEMP/OUT LBR APR 87"  ;
label variable v14850  "HD UNEMP/OUT LBR MAY 87"  ;
label variable v14851  "HD UNEMP/OUT LBR JUN 87"  ;
label variable v14852  "HD UNEMP/OUT LBR JUL 87"  ;
label variable v14853  "HD UNEMP/OUT LBR AUG 87"  ;
label variable v14854  "HD UNEMP/OUT LBR SEP 87"  ;
label variable v14855  "HD UNEMP/OUT LBR OCT 87"  ;
label variable v14856  "HD UNEMP/OUT LBR NOV 87"  ;
label variable v14857  "HD UNEMP/OUT LBR DEC 87"  ;
label variable v14858  "WIFE IN FU?"  ;
label variable v14876  "WF UNEMP/OUT LBR JAN 87"  ;
label variable v14877  "WF UNEMP/OUT LBR FEB 87"  ;
label variable v14878  "WF UNEMP/OUT LBR MAR 87"  ;
label variable v14879  "WF UNEMP/OUT LBR APR 87"  ;
label variable v14880  "WF UNEMP/OUT LBR MAY 87"  ;
label variable v14881  "WF UNEMP/OUT LBR JUN 87"  ;
label variable v14882  "WF UNEMP/OUT LBR JUL 87"  ;
label variable v14883  "WF UNEMP/OUT LBR AUG 87"  ;
label variable v14884  "WF UNEMP/OUT LBR SEP 87"  ;
label variable v14885  "WF UNEMP/OUT LBR OCT 87"  ;
label variable v14886  "WF UNEMP/OUT LBR NOV 87"  ;
label variable v14887  "WF UNEMP/OUT LBR DEC 87"  ;
label variable v15127  "DATE OF 1988 IW"  ;
label variable v15129  "# IN FU"  ;
label variable v15130  "AGE OF 1988 HEAD"  ;
label variable v15131  "SEX OF 1988 HEAD"  ;
label variable v15132  "AGE OF 1988 WIFE"  ;
label variable v15133  "# CHILDREN IN FU"  ;
label variable v15136  "A3 MARITAL STATUS"  ;
label variable v15154  "B1 EMPLOYMENT STATUS-HD"  ;
label variable v15456  "D1A EMPLOYMENT STATUS-WF"  ;
label variable v16161  "EDUCATION 1988 HEAD"  ;
label variable v16162  "EDUCATION 1988 WIFE"  ;
label variable v16187  "MARITAL STATUS 1988"  ;
label variable v16188  "87-88 CNG MARITAL STATUS"  ;
label variable v16189  "COUPLE STATUS OF HEAD"  ;
label variable v16208  "1988 FAMILY WEIGHT"  ;
label variable er30606  "1989 INTERVIEW NUMBER"  ;
label variable er30607  "SEQUENCE NUMBER                       89"  ;
label variable er30608  "RELATION TO HEAD                      89"  ;
label variable er30609  "AGE OF INDIVIDUAL                     89"  ;
label variable er30610  "MONTH INDIVIDUAL BORN                 89"  ;
label variable er30611  "YEAR INDIVIDUAL BORN                  89"  ;
label variable er30638  "AGE FROM BIRTH DATE                   89"  ;
label variable v16302  "1989 INTERVIEW NUMBER"  ;
label variable v16307  "SPLITOFF INDICATOR    89"  ;
label variable v16310  "FAM COMP CHANGE       89"  ;
label variable v16346  "HD UNEMP/OUT LBR JAN 88"  ;
label variable v16347  "HD UNEMP/OUT LBR FEB 88"  ;
label variable v16348  "HD UNEMP/OUT LBR MAR 88"  ;
label variable v16349  "HD UNEMP/OUT LBR APR 88"  ;
label variable v16350  "HD UNEMP/OUT LBR MAY 88"  ;
label variable v16351  "HD UNEMP/OUT LBR JUN 88"  ;
label variable v16352  "HD UNEMP/OUT LBR JUL 88"  ;
label variable v16353  "HD UNEMP/OUT LBR AUG 88"  ;
label variable v16354  "HD UNEMP/OUT LBR SEP 88"  ;
label variable v16355  "HD UNEMP/OUT LBR OCT 88"  ;
label variable v16356  "HD UNEMP/OUT LBR NOV 88"  ;
label variable v16357  "HD UNEMP/OUT LBR DEC 88"  ;
label variable v16358  "WIFE IN FU?"  ;
label variable v16376  "WF UNEMP/OUT LBR JAN 88"  ;
label variable v16377  "WF UNEMP/OUT LBR FEB 88"  ;
label variable v16378  "WF UNEMP/OUT LBR MAR 88"  ;
label variable v16379  "WF UNEMP/OUT LBR APR 88"  ;
label variable v16380  "WF UNEMP/OUT LBR MAY 88"  ;
label variable v16381  "WF UNEMP/OUT LBR JUN 88"  ;
label variable v16382  "WF UNEMP/OUT LBR JUL 88"  ;
label variable v16383  "WF UNEMP/OUT LBR AUG 88"  ;
label variable v16384  "WF UNEMP/OUT LBR SEP 88"  ;
label variable v16385  "WF UNEMP/OUT LBR OCT 88"  ;
label variable v16386  "WF UNEMP/OUT LBR NOV 88"  ;
label variable v16387  "WF UNEMP/OUT LBR DEC 88"  ;
label variable v16628  "DATE OF 1989 IW"  ;
label variable v16630  "# IN FU"  ;
label variable v16631  "AGE OF 1989 HEAD"  ;
label variable v16632  "SEX OF 1989 HEAD"  ;
label variable v16633  "AGE OF 1989 WIFE"  ;
label variable v16634  "# CHILDREN IN FU"  ;
label variable v16637  "A3 MARITAL STATUS"  ;
label variable v16655  "B1 EMPLOYMENT STATUS-HD"  ;
label variable v16974  "D1A EMPLOYMENT STATUS-WF"  ;
label variable v17545  "EDUCATION 1989 HEAD"  ;
label variable v17546  "EDUCATION 1989 WIFE"  ;
label variable v17565  "MARITAL STATUS 1989"  ;
label variable v17566  "88-89 CNG MARITAL STATUS"  ;
label variable v17567  "COUPLE STATUS OF HEAD"  ;
label variable v17611  "FAM COMP CHANGE 1984-89"  ;
label variable v17612  "1989 FAMILY WEIGHT"  ;
label variable er30642  "1990 INTERVIEW NUMBER"  ;
label variable er30643  "SEQUENCE NUMBER                       90"  ;
label variable er30644  "RELATION TO HEAD                      90"  ;
label variable er30645  "AGE OF INDIVIDUAL                     90"  ;
label variable er30646  "MONTH INDIVIDUAL BORN                 90"  ;
label variable er30647  "YEAR INDIVIDUAL BORN                  90"  ;
label variable er30680  "AGE FROM BIRTH DATE                   90"  ;
label variable v17702  "1990 INTERVIEW NUMBER 90"  ;
label variable v17707  "SPLITOFF INDICATOR    90"  ;
label variable v17710  "FAM COMP CHANGE       90"  ;
label variable v17744  "HD ANN WRK HRS IN 89"  ;
label variable v17745  "HD HRS WRK LOST OTR ILL"  ;
label variable v17746  "ACC HD HR LOST OTR ILL"  ;
label variable v17747  "HD HRS WRK LOST OWN ILL"  ;
label variable v17748  "ACC HD HRS LOST OWN ILL"  ;
label variable v17749  "HD STRIKE HOURS 89"  ;
label variable v17750  "ACC HD STRIKE HRS 89"  ;
label variable v17751  "HD UNEMP HRS 89"  ;
label variable v17752  "ACC 89 HD UNEMP HRS"  ;
label variable v17753  "HD HRS OUT LBR FORCE 89"  ;
label variable v17754  "ACC 89 HD HR OUT LBR FRC"  ;
label variable v17755  "HD UNEMP/OUT LBR JAN 89"  ;
label variable v17756  "HD UNEMP/OUT LBR FEB 89"  ;
label variable v17757  "HD UNEMP/OUT LBR MAR 89"  ;
label variable v17758  "HD UNEMP/OUT LBR APR 89"  ;
label variable v17759  "HD UNEMP/OUT LBR MAY 89"  ;
label variable v17760  "HD UNEMP/OUT LBR JUN 89"  ;
label variable v17761  "HD UNEMP/OUT LBR JUL 89"  ;
label variable v17762  "HD UNEMP/OUT LBR AUG 89"  ;
label variable v17763  "HD UNEMP/OUT LBR SEP 89"  ;
label variable v17764  "HD UNEMP/OUT LBR OCT 89"  ;
label variable v17765  "HD UNEMP/OUT LBR NOV 89"  ;
label variable v17766  "HD UNEMP/OUT LBR DEC 89"  ;
label variable v17767  "WIFE IN FU?"  ;
label variable v17774  "WF ANN WRK HRS IN 89"  ;
label variable v17775  "WF HRS WRK LOST OTR ILL"  ;
label variable v17776  "ACC WF HR LOST OTR ILL"  ;
label variable v17777  "WF HRS WRK LOST OWN ILL"  ;
label variable v17778  "ACC WF HRS LOST OWN ILL"  ;
label variable v17779  "WF STRIKE HOURS 89"  ;
label variable v17780  "ACC WF STRIKE HRS 89"  ;
label variable v17781  "WF UNEMP HRS 89"  ;
label variable v17782  "ACC WF UNEMP HRS 89"  ;
label variable v17783  "WF HRS OUT LBR FORCE 89"  ;
label variable v17784  "ACC WF 89 HR OUT LBR FRC"  ;
label variable v17785  "WF UNEMP/OUT LBR JAN 89"  ;
label variable v17786  "WF UNEMP/OUT LBR FEB 89"  ;
label variable v17787  "WF UNEMP/OUT LBR MAR 89"  ;
label variable v17788  "WF UNEMP/OUT LBR APR 89"  ;
label variable v17789  "WF UNEMP/OUT LBR MAY 89"  ;
label variable v17790  "WF UNEMP/OUT LBR JUN 89"  ;
label variable v17791  "WF UNEMP/OUT LBR JUL 89"  ;
label variable v17792  "WF UNEMP/OUT LBR AUG 89"  ;
label variable v17793  "WF UNEMP/OUT LBR SEP 89"  ;
label variable v17794  "WF UNEMP/OUT LBR OCT 89"  ;
label variable v17795  "WF UNEMP/OUT LBR NOV 89"  ;
label variable v17796  "WF UNEMP/OUT LBR DEC 89"  ;
label variable v17798  "1990 FAMILY SIZE"  ;
label variable v17805  "VALUE FD ST LST MO (F10)"  ;
label variable v17806  "ACC VALUE FD ST LST MO"  ;
label variable v17807  "ANN FD COST-EXC FD ST"  ;
label variable v17808  "ACC ANN FD COST EX FD ST"  ;
label variable v17809  "ANN FD COST-EAT OUT"  ;
label variable v17810  "ACC EAT OUT FD COST"  ;
label variable v17811  "VALUE FD ST 89     (F21)"  ;
label variable v17812  "ACC VALUE FD ST 89"  ;
label variable v17859  "HD 89 ADC/AFDC"  ;
label variable v17860  "ACC HD 89 ADC/AFDC"  ;
label variable v17862  "HD 89 SSI"  ;
label variable v17863  "HD 89 OTR WELFARE"  ;
label variable v17872  "HD 89 UNEMP COMP"  ;
label variable v17873  "HD 89 WORKERS COMP"  ;
label variable v17877  "HD 89 HELP FROM RELS"  ;
label variable v17880  "WF 89 ADC/AFDC"  ;
label variable v17881  "ACC WF 89 ADC/AFDC"  ;
label variable v17883  "WF 89 SSI"  ;
label variable v17884  "WF 89 OTR WELFARE"  ;
label variable v17892  "WF 89 UNEMP COMP"  ;
label variable v17893  "WF 89 WORKERS COMP"  ;
label variable v17897  "WF 89 HELP FROM RELS"  ;
label variable v18005  "OFUM 89 ADC/AFDC"  ;
label variable v18006  "OFUM 89 SSI"  ;
label variable v18007  "OFUM 89 OTHER WELFARE"  ;
label variable v18011  "OFUM 89 UNEMP COMP"  ;
label variable v18012  "OFUM 89 WORKERS COMP"  ;
label variable v18014  "OFUM 89 HELP FROM RELS"  ;
label variable v18046  "DATE OF 1990 IW"  ;
label variable v18048  "# IN FU"  ;
label variable v18049  "AGE OF 1990 HEAD"  ;
label variable v18050  "SEX OF 1990 HEAD"  ;
label variable v18051  "AGE OF 1990 WIFE"  ;
label variable v18052  "# CHILDREN IN FU"  ;
label variable v18055  "A3 MARITAL STATUS"  ;
label variable v18093  "B1 EMPLOYMENT STATUS-HD"  ;
label variable v18094  "B2 YEAR RETIRED   (HD-R)"  ;
label variable v18095  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable v18096  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable v18103  "B12 SLRY/HRLY/OTR  (H-E)"  ;
label variable v18104  "B13 PAY/HR-SALARY (HD-E)"  ;
label variable v18107  "B16 PAY/HR-HOURLY (HD-E)"  ;
label variable v18395  "D1A EMPLOYMENT STATUS-WF"  ;
label variable v18396  "D2 YEAR RETIRED   (WF-R)"  ;
label variable v18397  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable v18398  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable v18405  "D12 SLRY/HRLY/OTR  (W-E)"  ;
label variable v18406  "D13 PAY/HR-SALARY (WF-E)"  ;
label variable v18409  "D16 PAY/HR-HOURLY (WF-E)"  ;
label variable v18875  "TOT FAM MONEY Y 89"  ;
label variable v18878  "TOTAL HEAD LABOR Y 89"  ;
label variable v18881  "WEEKLY FOOD NEEDS-1990"  ;
label variable v18882  "ANNUAL NEEDS STD-1989"  ;
label variable v18884  "ANNUAL NEEDS STD-CENSUS"  ;
label variable v18886  "ANNUAL FOOD STD"  ;
label variable v18887  "HEAD 89 AVG HRLY EARNING"  ;
label variable v18888  "WIFE 89 AVG HRLY EARNING"  ;
label variable v18898  "EDUCATION 1990 HEAD"  ;
label variable v18899  "EDUCATION 1990 WIFE"  ;
label variable v18916  "MARITAL STATUS 1990"  ;
label variable v18917  "89-90 CNG MARITAL STATUS"  ;
label variable v18918  "COUPLE STATUS OF HEAD"  ;
label variable v18943  "1990 CORE FAMILY WEIGHT"  ;
label variable v18944  "1990 LATINO FAM WEIGHT"  ;
label variable v18945  "1990 COMBINED FAM WEIGHT"  ;
label variable er30689  "1991 INTERVIEW NUMBER"  ;
label variable er30690  "SEQUENCE NUMBER                       91"  ;
label variable er30691  "RELATION TO HEAD                      91"  ;
label variable er30692  "AGE OF INDIVIDUAL                     91"  ;
label variable er30693  "MONTH INDIVIDUAL BORN                 91"  ;
label variable er30694  "YEAR INDIVIDUAL BORN                  91"  ;
label variable er30723  "AGE FROM BIRTH DATE                   91"  ;
label variable v19002  "1991 INTERVIEW NUMBER"  ;
label variable v19007  "SPLITOFF INDICATOR    91"  ;
label variable v19010  "FAM COMP CHANGE       91"  ;
label variable v19044  "HD ANN WRK HRS IN 90"  ;
label variable v19045  "HD HRS WRK LOST OTR ILL"  ;
label variable v19046  "ACC HD HR LOST OTR ILL"  ;
label variable v19047  "HD HRS WRK LOST OWN ILL"  ;
label variable v19048  "ACC HD HRS LOST OWN ILL"  ;
label variable v19049  "HD STRIKE HOURS 90"  ;
label variable v19050  "ACC HD STRIKE HRS 90"  ;
label variable v19051  "HD UNEMP HRS 90"  ;
label variable v19052  "ACC 90 HD UNEMP HRS"  ;
label variable v19053  "HD HRS OUT LBR FORCE 90"  ;
label variable v19054  "ACC 90 HD HR OUT LBR FRC"  ;
label variable v19055  "HD UNEMP/OUT LBR JAN 90"  ;
label variable v19056  "HD UNEMP/OUT LBR FEB 90"  ;
label variable v19057  "HD UNEMP/OUT LBR MAR 90"  ;
label variable v19058  "HD UNEMP/OUT LBR APR 90"  ;
label variable v19059  "HD UNEMP/OUT LBR MAY 90"  ;
label variable v19060  "HD UNEMP/OUT LBR JUN 90"  ;
label variable v19061  "HD UNEMP/OUT LBR JUL 90"  ;
label variable v19062  "HD UNEMP/OUT LBR AUG 90"  ;
label variable v19063  "HD UNEMP/OUT LBR SEP 90"  ;
label variable v19064  "HD UNEMP/OUT LBR OCT 90"  ;
label variable v19065  "HD UNEMP/OUT LBR NOV 90"  ;
label variable v19066  "HD UNEMP/OUT LBR DEC 90"  ;
label variable v19067  "WIFE IN FU?"  ;
label variable v19074  "WF ANN WRK HRS IN 90"  ;
label variable v19075  "WF HRS WRK LOST OTR ILL"  ;
label variable v19076  "ACC WF HR LOST OTR ILL"  ;
label variable v19077  "WF HRS WRK LOST OWN ILL"  ;
label variable v19078  "ACC WF HRS LOST OWN ILL"  ;
label variable v19079  "WF STRIKE HOURS 90"  ;
label variable v19080  "ACC WF STRIKE HRS 90"  ;
label variable v19081  "WF UNEMP HRS 90"  ;
label variable v19082  "ACC WF UNEMP HRS 90"  ;
label variable v19083  "WF HRS OUT LBR FORCE 90"  ;
label variable v19084  "ACC WF 90 HR OUT LBR FRC"  ;
label variable v19085  "WF UNEMP/OUT LBR JAN 90"  ;
label variable v19086  "WF UNEMP/OUT LBR FEB 90"  ;
label variable v19087  "WF UNEMP/OUT LBR MAR 90"  ;
label variable v19088  "WF UNEMP/OUT LBR APR 90"  ;
label variable v19089  "WF UNEMP/OUT LBR MAY 90"  ;
label variable v19090  "WF UNEMP/OUT LBR JUN 90"  ;
label variable v19091  "WF UNEMP/OUT LBR JUL 90"  ;
label variable v19092  "WF UNEMP/OUT LBR AUG 90"  ;
label variable v19093  "WF UNEMP/OUT LBR SEP 90"  ;
label variable v19094  "WF UNEMP/OUT LBR OCT 90"  ;
label variable v19095  "WF UNEMP/OUT LBR NOV 90"  ;
label variable v19096  "WF UNEMP/OUT LBR DEC 90"  ;
label variable v19098  "1991 FAMILY SIZE"  ;
label variable v19105  "VALUE FD ST LST MO (F10)"  ;
label variable v19106  "ACC VALUE FD ST LST MO"  ;
label variable v19107  "ANN FD COST-HOME"  ;
label variable v19108  "ACC HOME FD COST"  ;
label variable v19109  "ANN FD COST-EAT OUT"  ;
label variable v19110  "ACC EAT OUT FD COST"  ;
label variable v19111  "VALUE FD ST 90     (F21)"  ;
label variable v19112  "ACC VALUE FD ST 90"  ;
label variable v19159  "HD 90 ADC/AFDC"  ;
label variable v19160  "ACC HD 90 ADC/AFDC"  ;
label variable v19162  "HD 90 SSI"  ;
label variable v19163  "HD 90 OTR WELFARE"  ;
label variable v19172  "HD 90 UNEMP COMP"  ;
label variable v19173  "HD 90 WORKERS COMP"  ;
label variable v19177  "HD 90 HELP FROM RELS"  ;
label variable v19180  "WF 90 ADC/AFDC"  ;
label variable v19181  "ACC WF 90 ADC/AFDC"  ;
label variable v19183  "WF 90 SSI"  ;
label variable v19184  "WF 90 OTR WELFARE"  ;
label variable v19192  "WF 90 UNEMP COMP"  ;
label variable v19193  "WF 90 WORKERS COMP"  ;
label variable v19197  "WF 90 HELP FROM RELS"  ;
label variable v19305  "OFUM 90 ADC/AFDC"  ;
label variable v19306  "OFUM 90 SSI"  ;
label variable v19307  "OFUM 90 OTHER WELFARE"  ;
label variable v19311  "OFUM 90 UNEMP COMP"  ;
label variable v19312  "OFUM 90 WORKERS COMP"  ;
label variable v19314  "OFUM 90 HELP FROM RELS"  ;
label variable v19346  "DATE OF 1991 IW"  ;
label variable v19348  "# IN FU"  ;
label variable v19349  "AGE OF 1991 HEAD"  ;
label variable v19350  "SEX OF 1991 HEAD"  ;
label variable v19351  "AGE OF 1991 WIFE"  ;
label variable v19352  "# CHILDREN IN FU"  ;
label variable v19355  "A3 MARITAL STATUS"  ;
label variable v19393  "B1 EMPLOYMENT STATUS-HD"  ;
label variable v19394  "B2 YEAR RETIRED   (HD-R)"  ;
label variable v19395  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable v19396  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable v19403  "B12 SLRY/HRLY/OTR  (H-E)"  ;
label variable v19404  "B13 PAY/HR-SALARY (HD-E)"  ;
label variable v19407  "B16 PAY/HR-HOURLY (HD-E)"  ;
label variable v19695  "D1A EMPLOYMENT STATUS-WF"  ;
label variable v19696  "D2 YEAR RETIRED   (WF-R)"  ;
label variable v19697  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable v19698  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable v19705  "D12 SLRY/HRLY/OTR  (W-E)"  ;
label variable v19706  "D13 PAY/HR-SALARY (WF-E)"  ;
label variable v19709  "D16 PAY/HR-HOURLY (WF-E)"  ;
label variable v20175  "TOT FAM MONEY Y 90"  ;
label variable v20178  "TOTAL HEAD LABOR Y 90"  ;
label variable v20181  "WEEKLY FOOD NEEDS-1991"  ;
label variable v20182  "ANNUAL NEEDS STD-1990"  ;
label variable v20184  "ANNUAL NEEDS STD-CENSUS"  ;
label variable v20186  "ANNUAL FOOD STD"  ;
label variable v20187  "HEAD 90 AVG HRLY EARNING"  ;
label variable v20188  "WIFE 90 AVG HRLY EARNING"  ;
label variable v20198  "COMPLETED ED-HD 91"  ;
label variable v20199  "COMPLETED ED-WF 91"  ;
label variable v20216  "MARITAL STATUS 1991"  ;
label variable v20217  "90-91 CNG MARITAL STATUS"  ;
label variable v20218  "COUPLE STATUS OF HEAD"  ;
label variable v20243  "1991 CORE FAMILY WEIGHT"  ;
label variable v20244  "1991 LATINO FAM WEIGHT"  ;
label variable v20245  "1991 COMBINED FAM WEIGHT"  ;
label variable er30733  "1992 INTERVIEW NUMBER"  ;
label variable er30734  "SEQUENCE NUMBER                       92"  ;
label variable er30735  "RELATION TO HEAD                      92"  ;
label variable er30736  "AGE OF INDIVIDUAL                     92"  ;
label variable er30737  "MONTH IND BORN                        92"  ;
label variable er30738  "YEAR IND BORN                         92"  ;
label variable er30798  "AGE FROM BIRTH DATE                   92"  ;
label variable v20302  "1992 INTERVIEW NUMBER"  ;
label variable v20307  "SPLITOFF INDICATOR    92"  ;
label variable v20310  "FAM COMP CHANGE       92"  ;
label variable v20344  "HD ANN WRK HRS IN 91"  ;
label variable v20345  "HD HRS WRK LOST OTR ILL"  ;
label variable v20346  "ACC HD HR LOST OTR ILL"  ;
label variable v20347  "HD HRS WRK LOST OWN ILL"  ;
label variable v20348  "ACC HD HRS LOST OWN ILL"  ;
label variable v20349  "HD STRIKE HOURS 91"  ;
label variable v20350  "ACC HD STRIKE HRS 91"  ;
label variable v20351  "HD UNEMP HRS 91"  ;
label variable v20352  "ACC 91 HD UNEMP HRS"  ;
label variable v20353  "HD HRS OUT LBR FORCE 91"  ;
label variable v20354  "ACC 91 HD HR OUT LBR FRC"  ;
label variable v20355  "HD UNEMP/OUT LBR JAN 91"  ;
label variable v20356  "HD UNEMP/OUT LBR FEB 91"  ;
label variable v20357  "HD UNEMP/OUT LBR MAR 91"  ;
label variable v20358  "HD UNEMP/OUT LBR APR 91"  ;
label variable v20359  "HD UNEMP/OUT LBR MAY 91"  ;
label variable v20360  "HD UNEMP/OUT LBR JUN 91"  ;
label variable v20361  "HD UNEMP/OUT LBR JUL 91"  ;
label variable v20362  "HD UNEMP/OUT LBR AUG 91"  ;
label variable v20363  "HD UNEMP/OUT LBR SEP 91"  ;
label variable v20364  "HD UNEMP/OUT LBR OCT 91"  ;
label variable v20365  "HD UNEMP/OUT LBR NOV 91"  ;
label variable v20366  "HD UNEMP/OUT LBR DEC 91"  ;
label variable v20367  "WIFE IN FU?"  ;
label variable v20374  "WF ANN WRK HRS IN 91"  ;
label variable v20375  "WF HRS WRK LOST OTR ILL"  ;
label variable v20376  "ACC WF HR LOST OTR ILL"  ;
label variable v20377  "WF HRS WRK LOST OWN ILL"  ;
label variable v20378  "ACC WF HRS LOST OWN ILL"  ;
label variable v20379  "WF STRIKE HOURS 91"  ;
label variable v20380  "ACC WF STRIKE HRS 91"  ;
label variable v20381  "WF UNEMP HRS 91"  ;
label variable v20382  "ACC WF UNEMP HRS 91"  ;
label variable v20383  "WF HRS OUT LBR FORCE 91"  ;
label variable v20384  "ACC WF 91 HR OUT LBR FRC"  ;
label variable v20385  "WF UNEMP/OUT LBR JAN 91"  ;
label variable v20386  "WF UNEMP/OUT LBR FEB 91"  ;
label variable v20387  "WF UNEMP/OUT LBR MAR 91"  ;
label variable v20388  "WF UNEMP/OUT LBR APR 91"  ;
label variable v20389  "WF UNEMP/OUT LBR MAY 91"  ;
label variable v20390  "WF UNEMP/OUT LBR JUN 91"  ;
label variable v20391  "WF UNEMP/OUT LBR JUL 91"  ;
label variable v20392  "WF UNEMP/OUT LBR AUG 91"  ;
label variable v20393  "WF UNEMP/OUT LBR SEP 91"  ;
label variable v20394  "WF UNEMP/OUT LBR OCT 91"  ;
label variable v20395  "WF UNEMP/OUT LBR NOV 91"  ;
label variable v20396  "WF UNEMP/OUT LBR DEC 91"  ;
label variable v20398  "1992 FAMILY SIZE"  ;
label variable v20405  "VALUE FD ST LST MO (F10)"  ;
label variable v20406  "ACC VALUE FD ST LST MO"  ;
label variable v20407  "ANN FD COST-HOME"  ;
label variable v20408  "ACC HOME FD COST"  ;
label variable v20409  "ANN FD COST-EAT OUT"  ;
label variable v20410  "ACC EAT OUT FD COST"  ;
label variable v20411  "VALUE FD ST 91     (F21)"  ;
label variable v20412  "ACC VALUE FD ST 91"  ;
label variable v20459  "HD 91 ADC/AFDC"  ;
label variable v20460  "ACC HD 91 ADC/AFDC"  ;
label variable v20462  "HD 91 SSI"  ;
label variable v20463  "HD 91 OTR WELFARE"  ;
label variable v20472  "HD 91 UNEMP COMP"  ;
label variable v20473  "HD 91 WORKERS COMP"  ;
label variable v20477  "HD 91 HELP FROM RELS"  ;
label variable v20480  "WF 91 ADC/AFDC"  ;
label variable v20481  "ACC WF 91 ADC/AFDC"  ;
label variable v20483  "WF 91 SSI"  ;
label variable v20484  "WF 91 OTR WELFARE"  ;
label variable v20492  "WF 91 UNEMP COMP"  ;
label variable v20493  "WF 91 WORKERS COMP"  ;
label variable v20497  "WF 91 HELP FROM RELS"  ;
label variable v20605  "OFUM 91 ADC/AFDC"  ;
label variable v20606  "OFUM 91 SSI"  ;
label variable v20607  "OFUM 91 OTHER WELFARE"  ;
label variable v20611  "OFUM 91 UNEMP COMP"  ;
label variable v20612  "OFUM 91 WORKERS COMP"  ;
label variable v20614  "OFUM 91 HELP FROM RELS"  ;
label variable v20648  "DATE OF 1992 IW"  ;
label variable v20650  "# IN FU"  ;
label variable v20651  "AGE OF 1992 HEAD"  ;
label variable v20652  "SEX OF 1992 HEAD"  ;
label variable v20653  "AGE OF 1992 WIFE"  ;
label variable v20654  "# CHILDREN IN FU"  ;
label variable v20657  "A3 MARITAL STATUS"  ;
label variable v20693  "B1 EMPLOYMENT STATUS-HD"  ;
label variable v20694  "B2 YEAR RETIRED   (HD-R)"  ;
label variable v20695  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable v20696  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable v20703  "B12 SLRY/HRLY/OTR  (H-E)"  ;
label variable v20704  "B13 PAY/HR-SALARY (HD-E)"  ;
label variable v20707  "B16 PAY/HR-HOURLY (HD-E)"  ;
label variable v20995  "D1A EMPLOYMENT STATUS-WF"  ;
label variable v20996  "D2 YEAR RETIRED   (WF-R)"  ;
label variable v20997  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable v20998  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable v21005  "D12 SLRY/HRLY/OTR  (W-E)"  ;
label variable v21006  "D13 PAY/HR-SALARY (WF-E)"  ;
label variable v21009  "D16 PAY/HR-HOURLY (WF-E)"  ;
label variable v21481  "TOT FAM MONEY Y 91"  ;
label variable v21484  "TOTAL HEAD LABOR Y 91"  ;
label variable v21487  "WEEKLY FOOD NEEDS-1992"  ;
label variable v21488  "ANNUAL NEEDS STD-1991"  ;
label variable v21490  "ANNUAL NEEDS STD-CENSUS"  ;
label variable v21492  "ANNUAL FOOD STD"  ;
label variable v21493  "HEAD 91 AVG HRLY EARNING"  ;
label variable v21494  "WIFE 91 AVG HRLY EARNING"  ;
label variable v21504  "COMPLETED ED-HD 92"  ;
label variable v21505  "COMPLETED ED-WF 92"  ;
label variable v21522  "MARITAL STATUS 1992"  ;
label variable v21523  "91-92 CNG MARITAL STATUS"  ;
label variable v21524  "COUPLE STATUS OF HEAD"  ;
label variable v21547  "1992 CORE FAMILY WEIGHT"  ;
label variable v21548  "1992 LATINO FAM WEIGHT"  ;
label variable v21549  "1992 COMBINED FAM WEIGHT"  ;
label variable er30806  "1993 INTERVIEW NUMBER"  ;
label variable er30807  "SEQUENCE NUMBER                       93"  ;
label variable er30808  "RELATION TO HEAD                      93"  ;
label variable er30809  "AGE OF INDIVIDUAL                     93"  ;
label variable er30810  "MONTH INDIVIDUAL BORN                 93"  ;
label variable er30811  "YEAR INDIVIDUAL BORN                  93"  ;
label variable v21602  "1993 INTERVIEW NUMBER"  ;
label variable v21606  "1993 SPLITOFF INDICATOR"  ;
label variable v21608  "1993 FAMILY COMPOSITION CHANGE"  ;
label variable v21634  "HD 1992 TOTAL WORK HOURS"  ;
label variable v21635  "HD 1992 WORK HOURS LOST-OTHERS ILL"  ;
label variable v21636  "HD 1992 WORK HOURS LOST-OWN ILLNESS"  ;
label variable v21637  "HD 1992 STRIKE HOURS"  ;
label variable v21638  "HD 1992 HOURS UNEMPLOYED"  ;
label variable v21639  "HD 1992 HOURS OUT OF LABOR FORCE"  ;
label variable v21640  "WTR HD UNEMPLOYED JAN 1992"  ;
label variable v21641  "WTR HD UNEMPLOYED FEB 1992"  ;
label variable v21642  "WTR HD UNEMPLOYED MAR 1992"  ;
label variable v21643  "WTR HD UNEMPLOYED APR 1992"  ;
label variable v21644  "WTR HD UNEMPLOYED MAY 1992"  ;
label variable v21645  "WTR HD UNEMPLOYED JUN 1992"  ;
label variable v21646  "WTR HD UNEMPLOYED JUL 1992"  ;
label variable v21647  "WTR HD UNEMPLOYED AUG 1992"  ;
label variable v21648  "WTR HD UNEMPLOYED SEP 1992"  ;
label variable v21649  "WTR HD UNEMPLOYED OCT 1992"  ;
label variable v21650  "WTR HD UNEMPLOYED NOV 1992"  ;
label variable v21651  "WTR HD UNEMPLOYED DEC 1992"  ;
label variable v21652  "WTR HD OUT OF LABOR FORCE JAN 1992"  ;
label variable v21653  "WTR HD OUT OF LABOR FORCE FEB 1992"  ;
label variable v21654  "WTR HD OUT OF LABOR FORCE MAR 1992"  ;
label variable v21655  "WTR HD OUT OF LABOR FORCE APR 1992"  ;
label variable v21656  "WTR HD OUT OF LABOR FORCE MAY 1992"  ;
label variable v21657  "WTR HD OUT OF LABOR FORCE JUN 1992"  ;
label variable v21658  "WTR HD OUT OF LABOR FORCE JUL 1992"  ;
label variable v21659  "WTR HD OUT OF LABOR FORCE AUG 1992"  ;
label variable v21660  "WTR HD OUT OF LABOR FORCE SEP 1992"  ;
label variable v21661  "WTR HD OUT OF LABOR FORCE OCT 1992"  ;
label variable v21662  "WTR HD OUT OF LABOR FORCE NOV 1992"  ;
label variable v21663  "WTR HD OUT OF LABOR FORCE DEC 1992"  ;
label variable v21664  "WTR WIFE IN FU"  ;
label variable v21670  "WF 1992 TOTAL WORK HOURS"  ;
label variable v21671  "WF 1992 WORK HOURS LOST-OTHERS ILL"  ;
label variable v21672  "WF 1992 WORK HOURS LOST-OWN ILLNESS"  ;
label variable v21673  "WF 1992 STRIKE HOURS"  ;
label variable v21674  "WF 1992 HOURS UNEMPLOYED"  ;
label variable v21675  "WF 1992 HOURS OUT OF LABOR FORCE"  ;
label variable v21676  "WTR WF UNEMPLOYED JAN 1992"  ;
label variable v21677  "WTR WF UNEMPLOYED FEB 1992"  ;
label variable v21678  "WTR WF UNEMPLOYED MAR 1992"  ;
label variable v21679  "WTR WF UNEMPLOYED APR 1992"  ;
label variable v21680  "WTR WF UNEMPLOYED MAY 1992"  ;
label variable v21681  "WTR WF UNEMPLOYED JUN 1992"  ;
label variable v21682  "WTR WF UNEMPLOYED JUL 1992"  ;
label variable v21683  "WTR WF UNEMPLOYED AUG 1992"  ;
label variable v21684  "WTR WF UNEMPLOYED SEP 1992"  ;
label variable v21685  "WTR WF UNEMPLOYED OCT 1992"  ;
label variable v21686  "WTR WF UNEMPLOYED NOV 1992"  ;
label variable v21687  "WTR WF UNEMPLOYED DEC 1992"  ;
label variable v21688  "WTR WF OUT OF LABOR FORCE JAN 1992"  ;
label variable v21689  "WTR WF OUT OF LABOR FORCE FEB 1992"  ;
label variable v21690  "WTR WF OUT OF LABOR FORCE MAR 1992"  ;
label variable v21691  "WTR WF OUT OF LABOR FORCE APR 1992"  ;
label variable v21692  "WTR WF OUT OF LABOR FORCE MAY 1992"  ;
label variable v21693  "WTR WF OUT OF LABOR FORCE JUN 1992"  ;
label variable v21694  "WTR WF OUT OF LABOR FORCE JUL 1992"  ;
label variable v21695  "WTR WF OUT OF LABOR FORCE AUG 1992"  ;
label variable v21696  "WTR WF OUT OF LABOR FORCE SEP 1992"  ;
label variable v21697  "WTR WF OUT OF LABOR FORCE OCT 1992"  ;
label variable v21698  "WTR WF OUT OF LABOR FORCE NOV 1992"  ;
label variable v21699  "WTR WF OUT OF LABOR FORCE DEC 1992"  ;
label variable v21703  "F13 VALUE OF FOOD STAMPS LAST MONTH"  ;
label variable v21704  "ACC VALUE OF FOOD STAMPS LAST MONTH"  ;
label variable v21705  "F15/F17/F19/F21 REPORTED COST FOOD-HOME"  ;
label variable v21706  "F15/17/19/21 TIME UNIT-RPRTD FOOD-HOME"  ;
label variable v21707  "ANNUAL COST FOOD-HOME (F15/F17/F19/F21)"  ;
label variable v21708  "ACC ANNUAL COST OF FOOD AT HOME"  ;
label variable v21709  "F18/F22 REPORTED COST OF FOOD EATEN OUT"  ;
label variable v21710  "F18/F22 TIME UNIT-RPRTD COST FD EAT OUT"  ;
label variable v21711  "ANNUAL COST OF FOOD EATEN OUT (F18/F22)"  ;
label variable v21712  "ACC ANNUAL COST OF FOOD EATEN OUT"  ;
label variable v21713  "F9 REPORTED VALUE OF FOOD STAMPS IN 1992"  ;
label variable v21714  "F9 TIME UNIT-REPRTD VALUE FD STAMPS 1992"  ;
label variable v21727  "VALUE OF FOOD STAMPS RECD IN 1992 (F9)"  ;
label variable v21728  "ACC VALUE FOOD STAMPS RECD IN 1992"  ;
label variable v21964  "G26E HD REPORTED ADC/AFDC INCOME"  ;
label variable v21965  "G26E TIME UNIT-HD RPRTD ADC/AFDC INCOME"  ;
label variable v21978  "HD 1992 ADC/AFDC INCOME (G26E)"  ;
label variable v21979  "ACC HD 1992 ADC/AFDC INCOME"  ;
label variable v21980  "G26F HD REPORTED SSI INCOME"  ;
label variable v21981  "G26F TIME UNIT-HD RPRTD SSI INCOME"  ;
label variable v21994  "HD 1992 SSI INCOME (G26F)"  ;
label variable v21995  "ACC HD 1992 SSI INCOME"  ;
label variable v21996  "G26G HD REPORTED OTR WELFARE INCOME"  ;
label variable v21997  "G26G TIME UNIT-HD RPRTD OTHER WELFARE"  ;
label variable v22010  "HD 1992 OTR WELFARE INCOME (G26G)"  ;
label variable v22011  "ACC HD 1992 OTHER WELFARE INCOME"  ;
label variable v22094  "G45A HD REPORTED UNEMPL COMP INCOME"  ;
label variable v22095  "G45A TIME UNIT-HD RPRTD UNEMPL COMP"  ;
label variable v22108  "HD 1992 UNEMP COMPENSATION (G45A)"  ;
label variable v22109  "ACC HD 1992 UNEMP COMPENSATION"  ;
label variable v22110  "G45B HD REPORTED WORKERS COMPENSATION"  ;
label variable v22111  "G45B TIME UNIT-HD RPRTD WORKERS COMP"  ;
label variable v22124  "HD 1992 WORKERS COMPENSATION (G45B)"  ;
label variable v22125  "ACC HD 1992 WORKERS COMPENSATION"  ;
label variable v22142  "G45E HD REPORTED HELP FROM RELATIVES"  ;
label variable v22143  "G45E TIME UNIT-HD REPRTD HELP FROM RELS"  ;
label variable v22156  "HD 1992 HELP FROM RELATIVES (G45E)"  ;
label variable v22157  "ACC HD 1992 HELP FROM RELATIVES"  ;
label variable v22158  "G45F HD REPORTED HELP-NONRELATIVES"  ;
label variable v22159  "G45F TIME UNIT-HD RPRTD HELP NONRELS"  ;
label variable v22172  "HD 1992 HELP-NONRELATIVES (G45F)"  ;
label variable v22173  "ACC HD 1992 HELP FROM NONRELS"  ;
label variable v22190  "G54 WF REPORTED UNEMP COMPENSATION"  ;
label variable v22191  "G54 TIME UNIT-WF REPORTED UNEMP COMP"  ;
label variable v22204  "WF 1992 UNEMP COMPENSATION (G54)"  ;
label variable v22205  "ACC WF 1992 UNEMPL COMPENSATION"  ;
label variable v22206  "G57 WF REPORTED WORKERS COMPENSATION"  ;
label variable v22207  "G57 TIME UNIT-WF RPRTD WORKERS COMP"  ;
label variable v22220  "WF 1992 WORKERS COMPENSATION (G57)"  ;
label variable v22221  "ACC WF 1992 WORKERS COMPENSATION"  ;
label variable v22222  "G60A WF REPORTED SSI INCOME"  ;
label variable v22223  "G60A TIME UNIT-WF 1992 SSI INCOME"  ;
label variable v22236  "WF 1992 SSI INCOME (G60A)"  ;
label variable v22237  "ACC WF 1992 SSI INCOME"  ;
label variable v22238  "G60B WF REPORTED ADC/AFDC INCOME"  ;
label variable v22239  "G60B TIME UNIT-WF RPRTD ADC/AFDC INCOME"  ;
label variable v22252  "WF 1992 ADC/AFDC INCOME (G60B)"  ;
label variable v22253  "ACC WF ADC/AFDC INCOME"  ;
label variable v22270  "G60D WF REPORTED OTHER WELFARE INCOME"  ;
label variable v22271  "G60D TIME UNIT-WF RPRTD OTHER WELFARE"  ;
label variable v22284  "WF 1992 OTHER WELFARE INCOME (G60D)"  ;
label variable v22285  "ACC WF 1992 OTHER WELFARE INCOME"  ;
label variable v22319  "G62A WF REPORTED HELP FROM RELATIVES"  ;
label variable v22320  "G62A TIME UNIT-WF RPRTD HELP FROM RELS"  ;
label variable v22333  "WF 1992 HELP FROM RELATIVES (G62A)"  ;
label variable v22334  "ACC WF 1992 HELP FROM RELATIVES"  ;
label variable v22335  "G62B WF REPORTED HELP FROM FRIENDS"  ;
label variable v22336  "G62B TIME UNIT-WF RPRT HELP FROM FRIENDS"  ;
label variable v22349  "WF 1992 TOTAL HELP FROM FRIENDS (G62B)"  ;
label variable v22350  "ACC WF 1992 HELP FROM FRIENDS"  ;
label variable v22374  "OFUM 1992 ADC/AFDC INCOME"  ;
label variable v22375  "ACC OFUM 1992 ADC/AFDC INCOME"  ;
label variable v22376  "OFUM 1992 SSI INCOME"  ;
label variable v22377  "ACC OFUM 1992 SSI INCOME"  ;
label variable v22378  "OFUM 1992 OTHER WELFARE INCOME"  ;
label variable v22379  "ACC OFUM 1992 OTHER WELFARE INCOME"  ;
label variable v22386  "OFUM 1992 UNEMP COMPENSATION"  ;
label variable v22387  "ACC OFUM 1992 UNEMP COMP"  ;
label variable v22388  "OFUM 1992 WORKERS COMPENSATION"  ;
label variable v22389  "ACC OFUM 1992 WORKERS COMPENSATION"  ;
label variable v22392  "OFUM 1992 HELP FROM RELATIVES"  ;
label variable v22393  "ACC OFUM 1992 HELP FROM RELATIVES"  ;
label variable v22403  "3. DATE OF 1993 INTERVIEW"  ;
label variable v22405  "NUMBER IN FAMILY UNIT"  ;
label variable v22406  "AGE OF 1993 HEAD"  ;
label variable v22407  "SEX OF 1993 HEAD"  ;
label variable v22408  "AGE OF 1993 WIFE"  ;
label variable v22409  "# CHILDREN IN FAMILY UNIT"  ;
label variable v22412  "A3 MARITAL STATUS"  ;
label variable v22448  "B1 HEAD 1993 EMPLOYMENT STATUS"  ;
label variable v22449  "B2 YEAR RETIRED (HD-R)"  ;
label variable v22450  "B3 WTR DOING WORK FOR MONEY (HD-E)"  ;
label variable v22451  "B4 WTR WORK FOR SELF/SOMEONE ELSE (HD-E)"  ;
label variable v22463  "B12 WTR SALARY/HOURLY/OTHER (HD-E)"  ;
label variable v22464  "B13 SALARY-PD SALARY (HD-E)"  ;
label variable v22465  "B13 TIME UNIT-SALARY (HD-E)"  ;
label variable v22470  "B16 PAY/HR-PD HOURLY (HD-E)"  ;
label variable v22801  "D1A WF 1993 EMPLOYMENT STATUS"  ;
label variable v22802  "D2 YEAR RETIRED   (WF-R)"  ;
label variable v22803  "D3 WTR DOING WORK FOR MONEY (WF-E)"  ;
label variable v22804  "D4 WTR WORK FOR SELF/SOMEONE ELSE (WF-E)"  ;
label variable v22816  "D12 WTR SALARY/HOURLY/OTHER  (WF-E)"  ;
label variable v22817  "D13 SALARY (WF-E)"  ;
label variable v22818  "D13 TIME UNIT-SALARY (WF-E)"  ;
label variable v22823  "D16 PAY/HR-HOURLY (WF-E)"  ;
label variable v23322  "TOTAL 1992 FAMILY MONEY INCOME"  ;
label variable v23323  "HD 1992 TOTAL LABOR INCOME"  ;
label variable v23324  "WF 1992 TOTAL LABOR INCOME"  ;
label variable v23325  "1992 ANNUAL NEEDS STANDARD"  ;
label variable v23326  "CENSUS ANNUAL NEEDS STANDARD"  ;
label variable v23333  "COMPLETED ED-HD 1993"  ;
label variable v23334  "COMPLETED ED-WF 1993"  ;
label variable v23336  "1993 MARITAL STATUS"  ;
label variable v23337  "1992-93 CHANGE IN MARITAL STATUS"  ;
label variable v23338  "COUPLE STATUS OF HEAD"  ;
label variable v23361  "1993 LONGITUDINAL CORE FAMILY WEIGHT"  ;
label variable v23362  "1993 LATINO FAMILY WEIGHT"  ;
label variable v23363  "1993 LONGITUDINAL COMBINED FAMILY WEIGHT"  ;
label variable er33101  "1994 INTERVIEW NUMBER"  ;
label variable er33102  "SEQUENCE NUMBER                       94"  ;
label variable er33103  "RELATION TO HEAD                      94"  ;
label variable er33104  "AGE OF INDIVIDUAL                     94"  ;
label variable er33105  "MONTH INDIVIDUAL BORN                 94"  ;
label variable er33106  "YEAR INDIVIDUAL BORN                  94"  ;
label variable er2002  "1994 INTERVIEW #"  ;
label variable er2005  "DATE OF INTERVIEW"  ;
label variable er2006  "# IN FU"  ;
label variable er2007  "AGE OF HEAD"  ;
label variable er2008  "SEX OF HEAD"  ;
label variable er2009  "AGE OF WIFE"  ;
label variable er2010  "# CHILDREN IN FU"  ;
label variable er2014  "HEAD MARITAL STATUS"  ;
label variable er2068  "B1 EMPLOYMENT STATUS-HD"  ;
label variable er2069  "B1 1ST MENTION"  ;
label variable er2070  "B1 2ND MENTION"  ;
label variable er2071  "B1 3RD MENTION"  ;
label variable er2072  "B2 YEAR RETIRED (HD-R)"  ;
label variable er2073  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable er2074  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable er2075  "B4A ONE JOB OR TWO(HD-E)"  ;
label variable er2076  "B4C SELF EMPL OR OTHERS"  ;
label variable er2081  "B12 SLRY/HRLY/OTR (H-E)"  ;
label variable er2082  "B13 SALARY AMOUNT"  ;
label variable er2083  "B13 SALARY PER WHAT"  ;
label variable er2088  "B16 HOURLY REGULAR RATE"  ;
label variable er2407  "C48A FINAL SALARY/WAGE"  ;
label variable er2408  "C48A FINAL SAL/WAGE PER"  ;
label variable er2409  "C49 FINAL SALARY/WAGE"  ;
label variable er2410  "C49 FINAL SAL/WAGE PER"  ;
label variable er2562  "D1A EMPLOYMENT STATUS-WF"  ;
label variable er2563  "D1 1ST MENTION"  ;
label variable er2564  "D1 2ND MENTION"  ;
label variable er2565  "D1 3RD MENTION"  ;
label variable er2566  "D2 YEAR RETIRED (WF-R)"  ;
label variable er2567  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable er2568  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable er2569  "D4A ONE JOB OR TWO(WF-E)"  ;
label variable er2570  "D4C WORK SEL/OTR? (WF-E)"  ;
label variable er2575  "D12 SLRY/HRLY/OTR (W-E)"  ;
label variable er2576  "D13 SALARY AMOUNT"  ;
label variable er2577  "D13 SALARY PER WHAT"  ;
label variable er2582  "D16 HOURLY REGULAR RATE"  ;
label variable er2901  "E48 FINAL SAL/WAGE AMT"  ;
label variable er2902  "E48 FINAL SAL/WAGE PER"  ;
label variable er2903  "E49 FINAL WAGE"  ;
label variable er2904  "E49 FINAL WAGE PER"  ;
label variable er3060  "F9 $ FD STMPS RECD PYR"  ;
label variable er3061  "F9 $ FD STMPS RECD PER"  ;
label variable er3076  "F13 AMT STAMPS LAST MON"  ;
label variable er3078  "F15$SPENT NOT FDSTMP AMT"  ;
label variable er3079  "F15$SPENT NOT FDSTMP PER"  ;
label variable er3081  "F17 AMT FD DELIVERD"  ;
label variable er3082  "F17 AMT FD DELIVERD PER"  ;
label variable er3083  "F18 AMT EATING OUT"  ;
label variable er3084  "F18 AMT EATING OUT PER"  ;
label variable er3085  "F19 AMT SPENT ON FOOD"  ;
label variable er3086  "F19 $ SPENT ON FOOD PER"  ;
label variable er3088  "F21 AMT FD DELIVERED"  ;
label variable er3089  "F21 AMT FD DELIVERED PER"  ;
label variable er3090  "F22 AMT EATING OUT"  ;
label variable er3091  "F22 AMT EATING OUT PER"  ;
label variable er3263  "ADC/AFDC OF HEAD"  ;
label variable er3264  "ADC/AFDC OF HEAD PER"  ;
label variable er3278  "SSI OF HEAD"  ;
label variable er3279  "SSI OF HEAD PER"  ;
label variable er3293  "OTHER WELFARE OF HEAD"  ;
label variable er3294  "OTHER WELFARE OF HD PER"  ;
label variable er3371  "AMT UNEMPL COMP OF HEAD"  ;
label variable er3372  "AMT UNEMPL COMP HD PER"  ;
label variable er3386  "AMT WORKMEN COMP OF HEAD"  ;
label variable er3387  "AMT WORKMEN COMP HD PER"  ;
label variable er3431  "AMT HELP FRM RELATIV HD"  ;
label variable er3432  "AMT HELP FRM RELATIV PER"  ;
label variable er3446  "AMT HELP FRM OTHERS HD"  ;
label variable er3447  "AMT HELP FRM OTHERS PER"  ;
label variable er3475  "G49 WTR WIFE IN FU NOW"  ;
label variable er3494  "UNEMPLOYMNT COMP OF WIFE"  ;
label variable er3495  "UNEMPLOYMNT COMP WFE PER"  ;
label variable er3509  "WORKMEN COMP OF WIFE"  ;
label variable er3510  "WORKMEN COMP OF WIFE PER"  ;
label variable er3584  "SSI OF WIFE AMT"  ;
label variable er3585  "SSI OF WIFE AMT PER"  ;
label variable er3599  "ADC/AFDC OF WIFE AMT"  ;
label variable er3600  "ADC/AFDC OF WIFE AMT PER"  ;
label variable er3629  "OTHER WELFARE OF WFE AMT"  ;
label variable er3630  "OTHER WELFARE OF WFE PER"  ;
label variable er3659  "HELP FRM RELATIV WFE AMT"  ;
label variable er3660  "HELP FRM RELATIV WFE PER"  ;
label variable er3674  "HELP FRM FRIENDS WFE AMT"  ;
label variable er3675  "HELP FRM FRIENDS WFE PER"  ;
label variable fips94st  "FAMINC94-FIPS STATE NUMERIC CODE"  ;
label variable psid94st  "FAMINC94-PSID STATE CODE"  ;
label variable faminc94  "TOTAL FAMILY INCOME 1993"  ;
label variable txhw94  "TAXABLE INCOME HEAD AND WIFE 1993"  ;
label variable trhw94  "TRANSFER INCOME OF THE HEAD AND WIFE 199"  ;
label variable txofm94  "TAXABLE INCOME OTHER FAMILY UNIT MEMBERS"  ;
label variable trofm94  "TRANSFER INCOME OTHER FAMILY UNIT MEMBER"  ;
label variable ssec94  "SOCIAL SECURITY INCOME 1993"  ;
label variable frminc94  "FARM INCOME OF THE HEAD 1993"  ;
label variable businc94  "BUSINESS INCOME 1993"  ;
label variable hdbusy94  "LABOR PART OF BUSINESS INCOME HEAD 1993"  ;
label variable hdbusk94  "ASSET PART OF BUSINESS INCOME HEAD 1993"  ;
label variable wfbusy94  "LABOR PART OF BUSINESS INCOME WIFE 1993"  ;
label variable wfbusk94  "ASSET PART OF BUSINESS INCOME WIFE 1993"  ;
label variable busnum94  "NUMBER OF BUSINESS OF THE FAMILY 1993"  ;
label variable hdbnum94  "NUMBER OF BUSINESS OF THE HEAD 1993"  ;
label variable wfbnum94  "NUMBER OF BUSINESS OF THE WIFE 1993"  ;
label variable hdearn94  "LABOR INCOME OF THE HEAD 1993"  ;
label variable hdwg94  "WAGES AND SALARIES OF THE HEAD 1993"  ;
label variable bnus94  "BONUS INCOME OF THE HEAD 1993"  ;
label variable ovtm94  "OVERTIME INCOME OF THE HEAD 1993"  ;
label variable tips94  "TIPS FOR THE HEAD 1993"  ;
label variable coms94  "COMMISSION INCOME OF THE HEAD 1993"  ;
label variable prof94  "PROFESSIONAL PRACTICE THE HEAD 1993"  ;
label variable mktg94  "MARKET GARDENING INCOME OF THE HEAD 1993"  ;
label variable xtra94  "EXTRA JOB INCOME OF THE HEAD 1993"  ;
label variable othinc94  "UNKNOW JOB INCOME OF THE HEAD 1993"  ;
label variable wfearn94  "LABOR INCOME OF THE WIFE 1993"  ;
label variable hdocc94  "MAIN OCCUPATION OF HEAD 1994"  ;
label variable hdind94  "MAIN INDUSTRY OF HEAD 1994"  ;
label variable wfocc94  "MAIN OCCUPATION OF WIFE 1994"  ;
label variable wfind94  "MAIN INDUSTRY OF WIFE 1994"  ;
label variable usda94  "1993 USDA ANNUAL NEEDS STANDARD"  ;
label variable census94  "1993 CENSUS ANNUAL NEEDS STANDARD"  ;
label variable hdwks94  "WORK WEEKS IN 1993 (HEAD)"  ;
label variable hdavg94  "WORK HOURS PER WEEK IN 1993 (HEAD)"  ;
label variable hdovt94  "OVERTIME WORK HOURS IN 1993 (HEAD)"  ;
label variable hdxtr94  "EXTRA WORK HOURS IN 1993 (HEAD)"  ;
label variable hdtot94  "TOTAL HOURS OF WORK IN 1993 (HEAD)"  ;
label variable hdwge94  "WAGE RATE IN 1993 (HEAD)"  ;
label variable wfwks94  "WORK WEEKS IN 1993 (WIFE)"  ;
label variable wfavg94  "WORK HOURS PER WEEK IN 1993 (WIFE)"  ;
label variable wfovt94  "OVERTIME WORK HOURS IN 1993 (WIFE)"  ;
label variable wfxtr94  "EXTRA WORK HOURS IN 1993 (WIFE)"  ;
label variable wftot94  "TOTAL HOURS OF WORK IN 1993 (WIFE)"  ;
label variable wfwge94  "WAGE RATE IN 1993 (WIFE)"  ;
label variable hdilo94  "WEEKS MISSED: ILL-OTH (HEAD)"  ;
label variable hdils94  "WEEKS MISSED: ILL-SELF (HEAD)"  ;
label variable hdvac94  "WEEKS MISSED: VAC (HEAD)"  ;
label variable hdstr94  "WEEKS MISSED: STRIKE (HEAD)"  ;
label variable hdump94  "WEEKS MISSED: UNEMP (HEAD)"  ;
label variable hdolf94  "WEEKS MISSED: OOLF (HEAD)"  ;
label variable wfilo94  "WEEKS MISSED: ILL-OTH (WIFE)"  ;
label variable wfils94  "WEEKS MISSED: ILL-SELF (WIFE)"  ;
label variable wfvac94  "WEEKS MISSED: VAC (WIFE)"  ;
label variable wfstr94  "WEEKS MISSED: STRIKE (WIFE)"  ;
label variable wfump94  "WEEKS MISSED: UNEMP (WIFE)"  ;
label variable wfolf94  "WEEKS MISSED: OOLF (WIFE)"  ;
label variable er33201  "1995 INTERVIEW NUMBER"  ;
label variable er33202  "SEQUENCE NUMBER                       95"  ;
label variable er33203  "RELATION TO HEAD                      95"  ;
label variable er33204  "AGE OF INDIVIDUAL                     95"  ;
label variable er33205  "MONTH INDIVIDUAL BORN                 95"  ;
label variable er33206  "YEAR INDIVIDUAL BORN                  95"  ;
label variable er5002  "1995 INTERVIEW #"  ;
label variable er5004  "DATE OF INTERVIEW"  ;
label variable er5005  "# IN FU"  ;
label variable er5006  "AGE OF HEAD"  ;
label variable er5007  "SEX OF HEAD"  ;
label variable er5008  "AGE OF WIFE"  ;
label variable er5009  "# CHILDREN IN FU"  ;
label variable er5013  "HEAD MARITAL STATUS"  ;
label variable er5067  "B1 EMPLOYMENT STATUS-HD"  ;
label variable er5068  "B1 1ST MENTION"  ;
label variable er5069  "B1 2ND MENTION"  ;
label variable er5070  "B1 3RD MENTION"  ;
label variable er5071  "B2 YEAR RETIRED (HD-R)"  ;
label variable er5072  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable er5073  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable er5074  "B4A ONE JOB OR TWO(HD-E)"  ;
label variable er5075  "B4C SELF EMPL OR OTHERS"  ;
label variable er5080  "B12 SLRY/HRLY/OTR (H-E)"  ;
label variable er5081  "B13 SALARY AMOUNT"  ;
label variable er5082  "B13 SALARY PER WHAT"  ;
label variable er5087  "B16 HOURLY REGULAR RATE"  ;
label variable er5406  "C48A FINAL SALARY/WAGE"  ;
label variable er5407  "C48A FINAL SAL/WAGE PER"  ;
label variable er5408  "C49 FINAL SALARY/WAGE"  ;
label variable er5409  "C49 FINAL SAL/WAGE PER"  ;
label variable er5561  "D1A EMPLOYMENT STATUS-WF"  ;
label variable er5562  "D1 1ST MENTION"  ;
label variable er5563  "D1 2ND MENTION"  ;
label variable er5564  "D1 3RD MENTION"  ;
label variable er5565  "D2 YEAR RETIRED (WF-R)"  ;
label variable er5566  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable er5567  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable er5568  "D4A ONE JOB OR TWO(WF-E)"  ;
label variable er5569  "D4C WORK SEL/OTR? (WF-E)"  ;
label variable er5574  "D12 SLRY/HRLY/OTR (W-E)"  ;
label variable er5575  "D13 SALARY AMOUNT"  ;
label variable er5576  "D13 SALARY PER WHAT"  ;
label variable er5581  "D16 HOURLY REGULAR RATE"  ;
label variable er5900  "E48 FINAL SAL/WAGE AMT"  ;
label variable er5901  "E48 FINAL SAL/WAGE PER"  ;
label variable er5902  "E49 FINAL WAGE"  ;
label variable er5903  "E49 FINAL WAGE PER"  ;
label variable er6059  "F9 $ FD STMPS RECD PYR"  ;
label variable er6060  "F9 $ FD STMPS RECD PER"  ;
label variable er6075  "F13 AMT STAMPS LAST MON"  ;
label variable er6076  "F14 BUY FOOD NOT FD STMP"  ;
label variable er6077  "F15$SPENT NOT FDSTMP AMT"  ;
label variable er6078  "F15$SPENT NOT FDSTMP PER"  ;
label variable er6080  "F17 AMT FD DELIVERD"  ;
label variable er6081  "F17 AMT FD DELIVERD PER"  ;
label variable er6082  "F18 AMT EATING OUT"  ;
label variable er6083  "F18 AMT EATING OUT PER"  ;
label variable er6084  "F19 AMT SPENT ON FOOD"  ;
label variable er6085  "F19 $ SPENT ON FOOD PER"  ;
label variable er6087  "F21 AMT FD DELIVERED"  ;
label variable er6088  "F21 AMT FD DELIVERED PER"  ;
label variable er6089  "F22 AMT EATING OUT"  ;
label variable er6090  "F22 AMT EATING OUT PER"  ;
label variable er6091  "F23 WTR $ FOR MORE FOOD"  ;
label variable er6263  "ADC/AFDC OF HEAD"  ;
label variable er6264  "ADC/AFDC OF HEAD PER"  ;
label variable er6278  "HEAD SSI SELF/ELSE/BOTH"  ;
label variable er6279  "SSI OF HEAD"  ;
label variable er6280  "SSI OF HEAD PER"  ;
label variable er6294  "OTHER WELFARE OF HEAD"  ;
label variable er6295  "OTHER WELFARE OF HD PER"  ;
label variable er6372  "AMT UNEMPL COMP OF HEAD"  ;
label variable er6373  "AMT UNEMPL COMP HD PER"  ;
label variable er6387  "AMT WORKMEN COMP OF HEAD"  ;
label variable er6388  "AMT WORKMEN COMP HD PER"  ;
label variable er6432  "AMT HELP FRM RELATIV HD"  ;
label variable er6433  "AMT HELP FRM RELATIV PER"  ;
label variable er6447  "AMT HELP FRM OTHERS HD"  ;
label variable er6448  "AMT HELP FRM OTHERS PER"  ;
label variable er6476  "G49 WTR WIFE IN FU NOW"  ;
label variable er6495  "UNEMPLOYMNT COMP OF WIFE"  ;
label variable er6496  "UNEMPLOYMNT COMP WFE PER"  ;
label variable er6510  "WORKMEN COMP OF WIFE"  ;
label variable er6511  "WORKMEN COMP OF WIFE PER"  ;
label variable er6585  "WIFE SSI SELF/ELSE/BOTH"  ;
label variable er6586  "SSI OF WIFE AMT"  ;
label variable er6587  "SSI OF WIFE AMT PER"  ;
label variable er6601  "ADC/AFDC OF WIFE AMT"  ;
label variable er6602  "ADC/AFDC OF WIFE AMT PER"  ;
label variable er6631  "OTHER WELFARE OF WFE AMT"  ;
label variable er6632  "OTHER WELFARE OF WFE PER"  ;
label variable er6661  "HELP FRM RELATIV WFE AMT"  ;
label variable er6662  "HELP FRM RELATIV WFE PER"  ;
label variable er6676  "HELP FRM FRIENDS WFE AMT"  ;
label variable er6677  "HELP FRM FRIENDS WFE PER"  ;
label variable fips95st  "FAMINC95-FIPS STATE NUMERIC CODE"  ;
label variable psid95st  "FAMINC95-PSID STATE CODE"  ;
label variable faminc95  "TOTAL FAMILY INCOME 1994"  ;
label variable txhw95  "TAXABLE INCOME HEAD AND WIFE 1994"  ;
label variable trhw95  "TRANSFER INCOME OF THE HEAD AND WIFE 199"  ;
label variable txofm95  "TAXABLE INCOME OTHER FAMILY UNIT MEMBERS"  ;
label variable trofm95  "TRANSFER INCOME OTHER FAMILY UNIT MEMBER"  ;
label variable ssec95  "SOCIAL SECURITY INCOME 1994"  ;
label variable frminc95  "FARM INCOME OF THE HEAD 1994"  ;
label variable businc95  "BUSINESS INCOME 1994"  ;
label variable hdbusy95  "LABOR PART OF BUSINESS INCOME HEAD 1994"  ;
label variable hdbusk95  "ASSET PART OF BUSINESS INCOME HEAD 1994"  ;
label variable wfbusy95  "LABOR PART OF BUSINESS INCOME WIFE 1994"  ;
label variable wfbusk95  "ASSET PART OF BUSINESS INCOME WIFE 1994"  ;
label variable busnum95  "NUMBER OF BUSINESS OF THE FAMILY 1994"  ;
label variable hdbnum95  "NUMBER OF BUSINESS OF THE HEAD 1994"  ;
label variable wfbnum95  "NUMBER OF BUSINESS OF THE WIFE 1994"  ;
label variable hdearn95  "LABOR INCOME OF THE HEAD 1994"  ;
label variable hdwg95  "WAGES AND SALARIES OF THE HEAD 1994"  ;
label variable bnus95  "BONUS INCOME OF THE HEAD 1994"  ;
label variable ovtm95  "OVERTIME INCOME OF THE HEAD 1994"  ;
label variable tips95  "TIPS FOR THE HEAD 1994"  ;
label variable coms95  "COMMISSION INCOME OF THE HEAD 1994"  ;
label variable prof95  "PROFESSIONAL PRACTICE THE HEAD 1994"  ;
label variable mktg95  "MARKET GARDENING INCOME OF THE HEAD 1994"  ;
label variable xtra95  "EXTRA JOB INCOME OF THE HEAD 1994"  ;
label variable othinc95  "UNKNOW JOB INCOME OF THE HEAD 1994"  ;
label variable wfearn95  "LABOR INCOME OF THE WIFE 1994"  ;
label variable hdocc95  "MAIN OCCUPATION OF HEAD 1995"  ;
label variable hdind95  "MAIN INDUSTRY OF HEAD 1995"  ;
label variable wfocc95  "MAIN OCCUPATION OF WIFE 1995"  ;
label variable wfind95  "MAIN INDUSTRY OF WIFE 1995"  ;
label variable usda95  "1994 USDA ANNUAL NEEDS STANDARD"  ;
label variable census95  "1994 CENSUS ANNUAL NEEDS STANDARD"  ;
label variable hdwks95  "WORK WEEKS IN 1994 (HEAD)"  ;
label variable hdavg95  "WORK HOURS PER WEEK IN 1994 (HEAD)"  ;
label variable hdovt95  "OVERTIME WORK HOURS IN 1994 (HEAD)"  ;
label variable hdxtr95  "EXTRA WORK HOURS IN 1994 (HEAD)"  ;
label variable hdtot95  "TOTAL HOURS OF WORK IN 1994 (HEAD)"  ;
label variable hdwge95  "WAGE RATE IN 1994 (HEAD)"  ;
label variable wfwks95  "WORK WEEKS IN 1994 (WIFE)"  ;
label variable wfavg95  "WORK HOURS PER WEEK IN 1994 (WIFE)"  ;
label variable wfovt95  "OVERTIME WORK HOURS IN 1994 (WIFE)"  ;
label variable wfxtr95  "EXTRA WORK HOURS IN 1994 (WIFE)"  ;
label variable wftot95  "TOTAL HOURS OF WORK IN 1994 (WIFE)"  ;
label variable wfwge95  "WAGE RATE IN 1994 (WIFE)"  ;
label variable hdilo95  "WEEKS MISSED: ILL-OTH (HEAD)"  ;
label variable hdils95  "WEEKS MISSED: ILL-SELF (HEAD)"  ;
label variable hdvac95  "WEEKS MISSED: VAC (HEAD)"  ;
label variable hdstr95  "WEEKS MISSED: STRIKE (HEAD)"  ;
label variable hdump95  "WEEKS MISSED: UNEMP (HEAD)"  ;
label variable hdolf95  "WEEKS MISSED: OOLF (HEAD)"  ;
label variable wfilo95  "WEEKS MISSED: ILL-OTH (WIFE)"  ;
label variable wfils95  "WEEKS MISSED: ILL-SELF (WIFE)"  ;
label variable wfvac95  "WEEKS MISSED: VAC (WIFE)"  ;
label variable wfstr95  "WEEKS MISSED: STRIKE (WIFE)"  ;
label variable wfump95  "WEEKS MISSED: UNEMP (WIFE)"  ;
label variable wfolf95  "WEEKS MISSED: OOLF (WIFE)"  ;
label variable er33301  "1996 INTERVIEW NUMBER"  ;
label variable er33302  "SEQUENCE NUMBER                       96"  ;
label variable er33303  "RELATION TO HEAD                      96"  ;
label variable er33304  "AGE OF INDIVIDUAL                     96"  ;
label variable er33305  "MONTH INDIVIDUAL BORN                 96"  ;
label variable er33306  "YEAR INDIVIDUAL BORN                  96"  ;
label variable er7002  "1996 INTERVIEW #"  ;
label variable er7004  "DATE OF INTERVIEW"  ;
label variable er7005  "# IN FU"  ;
label variable er7006  "AGE OF HEAD"  ;
label variable er7007  "SEX OF HEAD"  ;
label variable er7008  "AGE OF WIFE"  ;
label variable er7009  "# CHILDREN IN FU"  ;
label variable er7012  "WHO WAS RESPONDENT"  ;
label variable er7013  "HEAD MARITAL STATUS"  ;
label variable er7163  "B1 EMPLOYMENT STATUS-HD"  ;
label variable er7164  "B1 1ST MENTION"  ;
label variable er7165  "B1 2ND MENTION"  ;
label variable er7166  "B1 3RD MENTION"  ;
label variable er7167  "B2 YEAR RETIRED (HD-R)"  ;
label variable er7168  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable er7169  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable er7170  "B4A ONE JOB OR TWO(HD-E)"  ;
label variable er7171  "B4C SELF EMPL OR OTHERS"  ;
label variable er7176  "B12 SLRY/HRLY/OTR (H-E)"  ;
label variable er7177  "B13 SALARY AMOUNT"  ;
label variable er7178  "B13 SALARY PER WHAT"  ;
label variable er7183  "B16 HOURLY REGULAR RATE"  ;
label variable er7502  "C48A FINAL SALARY/WAGE"  ;
label variable er7503  "C48A FINAL SAL/WAGE PER"  ;
label variable er7504  "C49 FINAL SALARY/WAGE"  ;
label variable er7505  "C49 FINAL SAL/WAGE PER"  ;
label variable er7657  "D1A EMPLOYMENT STATUS-WF"  ;
label variable er7658  "D1 1ST MENTION"  ;
label variable er7659  "D1 2ND MENTION"  ;
label variable er7660  "D1 3RD MENTION"  ;
label variable er7661  "D2 YEAR RETIRED (WF-R)"  ;
label variable er7662  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable er7663  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable er7664  "D4A ONE JOB OR TWO(WF-E)"  ;
label variable er7665  "D4C WORK SEL/OTR? (WF-E)"  ;
label variable er7670  "D12 SLRY/HRLY/OTR (W-E)"  ;
label variable er7671  "D13 SALARY AMOUNT"  ;
label variable er7672  "D13 SALARY PER WHAT"  ;
label variable er7677  "D16 HOURLY REGULAR RATE"  ;
label variable er7996  "E48 FINAL SAL/WAGE AMT"  ;
label variable er7997  "E48 FINAL SAL/WAGE PER"  ;
label variable er7998  "E49 FINAL WAGE"  ;
label variable er7999  "E49 FINAL WAGE PER"  ;
label variable er8156  "F9 $ FD STMPS RECD PYR"  ;
label variable er8157  "F9 $ FD STMPS RECD PER"  ;
label variable er8172  "F13 AMT STAMPS LAST MON"  ;
label variable er8174  "F15$SPENT NOT FDSTMP AMT"  ;
label variable er8175  "F15$SPENT NOT FDSTMP PER"  ;
label variable er8177  "F17 AMT FD DELIVERD"  ;
label variable er8178  "F17 AMT FD DELIVERD PER"  ;
label variable er8179  "F18 AMT EATING OUT"  ;
label variable er8180  "F18 AMT EATING OUT PER"  ;
label variable er8181  "F19 AMT SPENT ON FOOD"  ;
label variable er8182  "F19 $ SPENT ON FOOD PER"  ;
label variable er8184  "F21 AMT FD DELIVERED"  ;
label variable er8185  "F21 AMT FD DELIVERED PER"  ;
label variable er8186  "F22 AMT EATING OUT"  ;
label variable er8187  "F22 AMT EATING OUT PER"  ;
label variable er8188  "F23 WTR $ FOR MORE FOOD"  ;
label variable er8380  "ADC/AFDC OF HEAD"  ;
label variable er8381  "ADC/AFDC OF HEAD PER"  ;
label variable er8395  "HEAD SSI SELF/ELSE/BOTH"  ;
label variable er8396  "SSI OF HEAD"  ;
label variable er8397  "SSI OF HEAD PER"  ;
label variable er8411  "OTHER WELFARE OF HEAD"  ;
label variable er8412  "OTHER WELFARE OF HD PER"  ;
label variable er8489  "AMT UNEMPL COMP OF HEAD"  ;
label variable er8490  "AMT UNEMPL COMP HD PER"  ;
label variable er8504  "AMT WORKMEN COMP OF HEAD"  ;
label variable er8505  "AMT WORKMEN COMP HD PER"  ;
label variable er8549  "AMT HELP FRM RELATIV HD"  ;
label variable er8550  "AMT HELP FRM RELATIV PER"  ;
label variable er8564  "AMT HELP FRM OTHERS HD"  ;
label variable er8565  "AMT HELP FRM OTHERS PER"  ;
label variable er8593  "G49 WTR WIFE IN FU NOW"  ;
label variable er8612  "UNEMPLOYMNT COMP OF WIFE"  ;
label variable er8613  "UNEMPLOYMNT COMP WFE PER"  ;
label variable er8627  "WORKMEN COMP OF WIFE"  ;
label variable er8628  "WORKMEN COMP OF WIFE PER"  ;
label variable er8702  "WIFE SSI SELF/ELSE/BOTH"  ;
label variable er8703  "SSI OF WIFE AMT"  ;
label variable er8704  "SSI OF WIFE AMT PER"  ;
label variable er8718  "ADC/AFDC OF WIFE AMT"  ;
label variable er8719  "ADC/AFDC OF WIFE AMT PER"  ;
label variable er8748  "OTHER WELFARE OF WF AMT"  ;
label variable er8749  "OTHER WELFARE OF WF PER"  ;
label variable er8778  "HELP FR RELATIVES WF AMT"  ;
label variable er8779  "HELP FR RELS WF PER"  ;
label variable er8793  "HELP FR FRIENDS WF AMT"  ;
label variable er8794  "HELP FR FRIENDS WF PER"  ;
label variable er9103  "M1 ONE THIRD INCOME CUT"  ;
label variable er9104  "M2 ONE HALF INCOME CUT"  ;
label variable er9105  "M3 20 PERCENT INCOME CUT"  ;
label variable er9106  "M4 10 PERCENT INCOME CUT"  ;
label variable er9107  "M5 75 PERCENT INCOME CUT"  ;
label variable fips96st  "FAMINC96-FIPS STATE NUMERIC CODE"  ;
label variable psid96st  "FAMINC96-PSID STATE CODE"  ;
label variable faminc96  "TOTAL FAMILY INCOME 1995"  ;
label variable txhw96  "TAXABLE INCOME HEAD AND WIFE 1995"  ;
label variable trhw96  "TRANSFER INCOME OF THE HEAD AND WIFE 199"  ;
label variable txofm96  "TAXABLE INCOME OTHER FAMILY UNIT MEMBERS"  ;
label variable trofm96  "TRANSFER INCOME OTHER FAMILY UNIT MEMBER"  ;
label variable ssec96  "SOCIAL SECURITY INCOME 1995"  ;
label variable frminc96  "FARM INCOME OF THE HEAD 1995"  ;
label variable businc96  "BUSINESS INCOME 1995"  ;
label variable hdbusy96  "LABOR PART OF BUSINESS INCOME HEAD 1995"  ;
label variable hdbusk96  "ASSET PART OF BUSINESS INCOME HEAD 1995"  ;
label variable wfbusy96  "LABOR PART OF BUSINESS INCOME WIFE 1995"  ;
label variable wfbusk96  "ASSET PART OF BUSINESS INCOME WIFE 1995"  ;
label variable busnum96  "NUMBER OF BUSINESS OF THE FAMILY 1995"  ;
label variable hdbnum96  "NUMBER OF BUSINESS OF THE HEAD 1995"  ;
label variable wfbnum96  "NUMBER OF BUSINESS OF THE WIFE 1995"  ;
label variable hdearn96  "LABOR INCOME OF THE HEAD 1995"  ;
label variable hdwg96  "WAGES AND SALARIES OF THE HEAD 1995"  ;
label variable bnus96  "BONUS INCOME OF THE HEAD 1995"  ;
label variable ovtm96  "OVERTIME INCOME OF THE HEAD 1995"  ;
label variable tips96  "TIPS FOR THE HEAD 1995"  ;
label variable coms96  "COMMISSION INCOME OF THE HEAD 1995"  ;
label variable prof96  "PROFESSIONAL PRACTICE THE HEAD 1995"  ;
label variable mktg96  "MARKET GARDENING INCOME OF THE HEAD 1995"  ;
label variable xtra96  "EXTRA JOB INCOME OF THE HEAD 1995"  ;
label variable othinc96  "UNKNOW JOB INCOME OF THE HEAD 1995"  ;
label variable wfearn96  "LABOR INCOME OF THE WIFE 1995"  ;
label variable hdocc96  "MAIN OCCUPATION OF HEAD 1996"  ;
label variable hdind96  "MAIN INDUSTRY OF HEAD 1996"  ;
label variable wfocc96  "MAIN OCCUPATION OF WIFE 1996"  ;
label variable wfind96  "MAIN INDUSTRY OF WIFE 1996"  ;
label variable usda96  "1995 USDA ANNUAL NEEDS STANDARD"  ;
label variable census96  "1995 CENSUS ANNUAL NEEDS STANDARD"  ;
label variable hdwks96  "WORK WEEKS IN 1995 (HEAD)"  ;
label variable hdavg96  "WORK HOURS PER WEEK IN 1995 (HEAD)"  ;
label variable hdovt96  "OVERTIME WORK HOURS IN 1995 (HEAD)"  ;
label variable hdxtr96  "EXTRA WORK HOURS IN 1995 (HEAD)"  ;
label variable hdtot96  "TOTAL HOURS OF WORK IN 1995 (HEAD)"  ;
label variable hdwge96  "WAGE RATE IN 1995 (HEAD)"  ;
label variable wfwks96  "WORK WEEKS IN 1995 (WIFE)"  ;
label variable wfavg96  "WORK HOURS PER WEEK IN 1995 (WIFE)"  ;
label variable wfovt96  "OVERTIME WORK HOURS IN 1995 (WIFE)"  ;
label variable wfxtr96  "EXTRA WORK HOURS IN 1995 (WIFE)"  ;
label variable wftot96  "TOTAL HOURS OF WORK IN 1995 (WIFE)"  ;
label variable wfwge96  "WAGE RATE IN 1995 (WIFE)"  ;
label variable hdilo96  "WEEKS MISSED: ILL-OTH (HEAD)"  ;
label variable hdils96  "WEEKS MISSED: ILL-SELF (HEAD)"  ;
label variable hdvac96  "WEEKS MISSED: VAC (HEAD)"  ;
label variable hdstr96  "WEEKS MISSED: STRIKE (HEAD)"  ;
label variable hdump96  "WEEKS MISSED: UNEMP (HEAD)"  ;
label variable hdolf96  "WEEKS MISSED: OOLF (HEAD)"  ;
label variable wfilo96  "WEEKS MISSED: ILL-OTH (WIFE)"  ;
label variable wfils96  "WEEKS MISSED: ILL-SELF (WIFE)"  ;
label variable wfvac96  "WEEKS MISSED: VAC (WIFE)"  ;
label variable wfstr96  "WEEKS MISSED: STRIKE (WIFE)"  ;
label variable wfump96  "WEEKS MISSED: UNEMP (WIFE)"  ;
label variable wfolf96  "WEEKS MISSED: OOLF (WIFE)"  ;
label variable er33401  "1997 INTERVIEW NUMBER"  ;
label variable er33402  "SEQUENCE NUMBER                       97"  ;
label variable er33403  "RELATION TO HEAD                      97"  ;
label variable er33404  "AGE OF INDIVIDUAL                     97"  ;
label variable er33405  "MONTH INDIVIDUAL BORN                 97"  ;
label variable er33406  "YEAR INDIVIDUAL BORN                  97"  ;
label variable er10002  "1997 INTERVIEW #"  ;
label variable er10005  "MONTH  CURRENT IW"  ;
label variable er10006  "DAY CURRENT IW"  ;
label variable er10007  "YEAR CURRENT IW"  ;
label variable er10008  "# IN FU"  ;
label variable er10009  "AGE OF HEAD"  ;
label variable er10010  "SEX OF HEAD"  ;
label variable er10011  "AGE OF WIFE"  ;
label variable er10012  "# CHILDREN IN FU"  ;
label variable er10016  "HEAD MARITAL STATUS"  ;
label variable er10081  "B1 1ST MENTION"  ;
label variable er10082  "B1 2ND MENTION"  ;
label variable er10083  "B1 3RD MENTION"  ;
label variable er10084  "B2 YEAR RETIRED (HD-R)"  ;
label variable er10085  "B3 WORK FOR MONEY?(HD-E)"  ;
label variable er10086  "B4 WORK SELF/OTR? (HD-E)"  ;
label variable er10091  "B12 SLRY/HRLY/OTR (H-E)"  ;
label variable er10092  "B13 SALARY AMOUNT"  ;
label variable er10093  "B13 SALARY PER WHAT"  ;
label variable er10098  "B16 HOURLY REGULAR RATE"  ;
label variable er10417  "C48A FINAL SALARY/WAGE"  ;
label variable er10418  "C48A FINAL SAL/WAGE PER"  ;
label variable er10419  "C49 FINAL SALARY/WAGE"  ;
label variable er10420  "C49 FINAL SAL/WAGE PER"  ;
label variable er10563  "D1 1ST MENTION"  ;
label variable er10564  "D1 2ND MENTION"  ;
label variable er10565  "D1 3RD MENTION"  ;
label variable er10566  "D2 YEAR RETIRED (WF-R)"  ;
label variable er10567  "D3 WORK FOR MONEY?(WF-E)"  ;
label variable er10568  "D4 WORK SELF/OTR? (WF-E)"  ;
label variable er10573  "D12 SLRY/HRLY/OTR (W-E)"  ;
label variable er10574  "D13 SALARY AMOUNT"  ;
label variable er10575  "D13 SALARY PER WHAT"  ;
label variable er10580  "D16 HOURLY REGULAR RATE"  ;
label variable er10899  "E48 FINAL SAL/WAGE AMT"  ;
label variable er10900  "E48 FINAL SAL/WAGE PER"  ;
label variable er10901  "E49 FINAL WAGE"  ;
label variable er10902  "E49 FINAL WAGE PER"  ;
label variable er11050  "F9 $ FD STMPS RECD PYR"  ;
label variable er11051  "F9 $ FD STMPS RECD PER"  ;
label variable er11066  "F13 AMT STAMPS LAST MON"  ;
label variable er11067  "F14 BUY FOOD NOT FD STMP"  ;
label variable er11068  "F15$SPENT NOT FDSTMP AMT"  ;
label variable er11069  "F15$SPENT NOT FDSTMP PER"  ;
label variable er11071  "F17 AMT FD DELIVERD"  ;
label variable er11072  "F17 AMT FD DELIVERD PER"  ;
label variable er11073  "F18 AMT EATING OUT"  ;
label variable er11074  "F18 AMT EATING OUT PER"  ;
label variable er11075  "F18A  WTR $ FOR  FOOD"  ;
label variable er11076  "F19 AMT SPENT ON FOOD"  ;
label variable er11077  "F19 $ SPENT ON FOOD PER"  ;
label variable er11079  "F21 AMT FD DELIVERED"  ;
label variable er11080  "F21 AMT FD DELIVERED PER"  ;
label variable er11081  "F22 AMT EATING OUT"  ;
label variable er11082  "F22 AMT EATING OUT PER"  ;
label variable er11083  "F23 WTR $ FOR MORE FOOD"  ;
label variable er11273  "ADC/AFDC OF HEAD"  ;
label variable er11274  "ADC/AFDC OF HEAD PER"  ;
label variable er11288  "HEAD SSI SELF/ELSE/BOTH"  ;
label variable er11289  "SSI OF HEAD"  ;
label variable er11290  "SSI OF HEAD PER"  ;
label variable er11304  "OTHER WELFARE OF HEAD"  ;
label variable er11305  "OTHER WELFARE OF HD PER"  ;
label variable er11383  "AMT UNEMPL COMP OF HEAD"  ;
label variable er11384  "AMT UNEMPL COMP HD PER"  ;
label variable er11398  "AMT WORKMEN COMP OF HEAD"  ;
label variable er11399  "AMT WORKMEN COMP HD PER"  ;
label variable er11443  "AMT HELP FRM RELATIV HD"  ;
label variable er11444  "AMT HELP FRM RELATIV PER"  ;
label variable er11458  "AMT HELP FRM OTHERS HD"  ;
label variable er11459  "AMT HELP FRM OTHERS PER"  ;
label variable er11487  "G49 WTR WIFE IN FU NOW"  ;
label variable er11494  "UNEMPLOYMNT COMP OF WIFE"  ;
label variable er11495  "UNEMPLOYMNT COMP WFE PER"  ;
label variable er11509  "WORKMEN COMP OF WIFE"  ;
label variable er11510  "WORKMEN COMP OF WIFE PER"  ;
label variable er11584  "WIFE SSI SELF/ELSE/BOTH"  ;
label variable er11585  "SSI OF WIFE AMT"  ;
label variable er11586  "SSI OF WIFE AMT PER"  ;
label variable er11600  "ADC/AFDC OF WIFE AMT"  ;
label variable er11601  "ADC/AFDC OF WIFE AMT PER"  ;
label variable er11630  "OTHER WELFARE OF WF AMT"  ;
label variable er11631  "OTHER WELFARE OF WF PER"  ;
label variable er11660  "HELP FR RELATIVES WF AMT"  ;
label variable er11661  "HELP FR RELS WF PER"  ;
label variable er11675  "HELP FR FRIENDS WF AMT"  ;
label variable er11676  "HELP FR FRIENDS WF PER"  ;
label variable er12079  "TOTAL FAMILY INCOME"  ;
label variable er12080  "LABOR INCOME-HEAD"  ;
label variable er12081  "ACC LABOR INCOME-HD"  ;
label variable er12082  "LABOR INCOME-WIFE"  ;
label variable er12083  "ACC LABOR INCOME-WF"  ;
label variable er12084  "FAMILY WEIGHT"  ;
label variable fips97st  "FAMINC97-FIPS STATE NUMERIC CODE"  ;
label variable psid97st  "FAMINC97-PSID STATE CODE"  ;
label variable faminc97  "TOTAL FAMILY INCOME 1996"  ;
label variable txhw97  "TAXABLE INCOME HEAD AND WIFE 1996"  ;
label variable trhw97  "TRANSFER INCOME OF THE HEAD AND WIFE 199"  ;
label variable txofm97  "TAXABLE INCOME OTHER FAMILY UNIT MEMBERS"  ;
label variable trofm97  "TRANSFER INCOME OTHER FAMILY UNIT MEMBER"  ;
label variable ssec97  "SOCIAL SECURITY INCOME 1996"  ;
label variable frminc97  "FARM INCOME OF THE HEAD 1996"  ;
label variable businc97  "BUSINESS INCOME 1996"  ;
label variable hdbusy97  "LABOR PART OF BUSINESS INCOME HEAD 1996"  ;
label variable hdbusk97  "ASSET PART OF BUSINESS INCOME HEAD 1996"  ;
label variable wfbusy97  "LABOR PART OF BUSINESS INCOME WIFE 1996"  ;
label variable wfbusk97  "ASSET PART OF BUSINESS INCOME WIFE 1996"  ;
label variable busnum97  "NUMBER OF BUSINESS OF THE FAMILY 1996"  ;
label variable hdbnum97  "NUMBER OF BUSINESS OF THE HEAD 1996"  ;
label variable wfbnum97  "NUMBER OF BUSINESS OF THE WIFE 1996"  ;
label variable hdearn97  "LABOR INCOME OF THE HEAD 1996"  ;
label variable hdwg97  "WAGES AND SALARIES OF THE HEAD 1996"  ;
label variable bnus97  "BONUS INCOME OF THE HEAD 1996"  ;
label variable ovtm97  "OVERTIME INCOME OF THE HEAD 1996"  ;
label variable tips97  "TIPS FOR THE HEAD 1996"  ;
label variable coms97  "COMMISSION INCOME OF THE HEAD 1996"  ;
label variable prof97  "PROFESSIONAL PRACTICE THE HEAD 1996"  ;
label variable mktg97  "MARKET GARDENING INCOME OF THE HEAD 1996"  ;
label variable xtra97  "EXTRA JOB INCOME OF THE HEAD 1996"  ;
label variable othinc97  "UNKNOW JOB INCOME OF THE HEAD 1996"  ;
label variable wfearn97  "LABOR INCOME OF THE WIFE 1996"  ;
label variable hdocc97  "MAIN OCCUPATION OF HEAD 1997"  ;
label variable hdind97  "MAIN INDUSTRY OF HEAD 1997"  ;
label variable wfocc97  "MAIN OCCUPATION OF WIFE 1997"  ;
label variable wfind97  "MAIN INDUSTRY OF WIFE 1997"  ;
label variable usda97  "1996 USDA ANNUAL NEEDS STANDARD"  ;
label variable census97  "1996 CENSUS ANNUAL NEEDS STANDARD"  ;
label variable hdwks97  "WORK WEEKS IN 1996 (HEAD)"  ;
label variable hdavg97  "WORK HOURS PER WEEK IN 1996 (HEAD)"  ;
label variable hdovt97  "OVERTIME WORK HOURS IN 1996 (HEAD)"  ;
label variable hdxtr97  "EXTRA WORK HOURS IN 1996 (HEAD)"  ;
label variable hdtot97  "TOTAL HOURS OF WORK IN 1996 (HEAD)"  ;
label variable hdwge97  "WAGE RATE IN 1996 (HEAD)"  ;
label variable wfwks97  "WORK WEEKS IN 1996 (WIFE)"  ;
label variable wfavg97  "WORK HOURS PER WEEK IN 1996 (WIFE)"  ;
label variable wfovt97  "OVERTIME WORK HOURS IN 1996 (WIFE)"  ;
label variable wfxtr97  "EXTRA WORK HOURS IN 1996 (WIFE)"  ;
label variable wftot97  "TOTAL HOURS OF WORK IN 1996 (WIFE)"  ;
label variable wfwge97  "WAGE RATE IN 1996 (WIFE)"  ;
label variable hdilo97  "WEEKS MISSED: ILL-OTH (HEAD)"  ;
label variable hdils97  "WEEKS MISSED: ILL-SELF (HEAD)"  ;
label variable hdvac97  "WEEKS MISSED: VAC (HEAD)"  ;
label variable hdstr97  "WEEKS MISSED: STRIKE (HEAD)"  ;
label variable hdump97  "WEEKS MISSED: UNEMP (HEAD)"  ;
label variable hdolf97  "WEEKS MISSED: OOLF (HEAD)"  ;
label variable wfilo97  "WEEKS MISSED: ILL-OTH (WIFE)"  ;
label variable wfils97  "WEEKS MISSED: ILL-SELF (WIFE)"  ;
label variable wfvac97  "WEEKS MISSED: VAC (WIFE)"  ;
label variable wfstr97  "WEEKS MISSED: STRIKE (WIFE)"  ;
label variable wfump97  "WEEKS MISSED: UNEMP (WIFE)"  ;
label variable wfolf97  "WEEKS MISSED: OOLF (WIFE)"  ;

*end code from PSID download;

*the variables we actually need;

keep er32000 er30001 er30002 er30003 er30004 er30020 er30021 er30022
er30023 er30043 er30044 er30045 er30046 er30067 er30068 er30069 er30070
er30091 er30092 er30093 er30094 er30117 er30118 er30119 er30120 er30138
er30139 er30140 er30141 er30160 er30161 er30162 er30163 er30188 er30189
er30190 er30191 er30217 er30218 er30219 er30220 er30246 er30247 er30248
er30249 er30283 er30284 er30285 er30286 er30313 er30314 er30315 er30316
er30343 er30344 er30345 er30346 er30373 er30374 er30375 er30376 er30399
er30400 er30401 er30402 er30429 er30430 er30431 er30432 er30463 er30464
er30465 er30466 er30498 er30499 er30500 er30501 er30535 er30536 er30537
er30538 er30570 er30571 er30572 er30573 er30606 er30607 er30608 er30609
er30642 er30643 er30644 er30645 er30689 er30690 er30691 er30692 er30733
er30734 er30735 er30736 er30806 er30807 er30808 er30809 er33101 er33102
er33103 er33104 er33201 er33202 er33203 er33204 er33302 er33303 er33304
er33401 er33402 er33403 er33404 er33301 v92 v3441 v3442 v3443 v3444 v3445
v3446 v3841 v3842 v3849 v3850 v3851 v3852 v3853 v3854 v4354 v4355 v4362
v4363 v4364 v4365 v4368 v4369 v5271 v5272 v5273 v5274 v5275 v5276 v5277
v5278 v5770 v5771 v5772 v5773 v5774 v5775 v5776 v5777 v6376 v6377 v6378
v6379 v6380 v6381 v6382 v6383 v6972 v6973 v6974 v6975 v6976 v6977 v7564
v7565 v7566 v7567 v7568 v7569 v8256 v8257 v8258 v8259 v8260 v8261 v8864
v8865 v8866 v8867 v8868 v8869 v10235 v10236 v10237 v10238 v10239 v10240
v11375 v11376 v11377 v11378 v11379 v11380 v12774 v12775 v12776 v12777
v12778 v12779 v13876 v13877 v13878 v13879 v13880 v13881 v17807 v17808
v17809 v17810 v17811 v17812 v19107 v19108 v19109 v19110 v19111 v19112
v20407 v20408 v20409 v20410 v20411 v20412 v21707 v21708 v21711 v21712
v21727 v21728 er3060 er3061 er3078 er3079 er3081 er3082 er3083 er3084
er3085 er3086 er3088 er3089 er3090 er3091 er6059 er6060 er6077 er6078
er6080 er6081 er6082 er6083 er6084 er6085 er6087 er6088 er6089 er6090
er8156 er8157 er8174 er8175 er8177 er8178 er8179 er8180 er8181 er8182
er8184 er8185 er8186 er8187 er11050 er11051 er11068 er11069 er11071
er11072 er11073 er11074 er11076 er11077 er11079 er11080 er11081 er11082
er3263 er3264 er3278 er3279 er3293 er3294 er6263 er6264 er6279 er6280
er6294 er6295 er3371 er3372 er3386 er3387 er3431 er3432 er3446 er3447
er3494 er3495 er3509 er3510 er3584 er3585 er3599 er3600 er3629 er3630
er3659 er3660 er3674 er3675 er6372 er6373 er6387 er6388 er6432 er6433
er6447 er6448 er6495 er6496 er6510 er6511 er6586 er6587 er6601 er6602
er6631 er6632 er6661 er6662 er6676 er6677 er8380 er8381 er8396 er8397
er8411 er8412 er11273 er11274 er11289 er11290 er11304 er11305 er8489
er8490 er8504 er8505 er8549 er8550 er8564 er8565 er8612 er8613 er8627
er8628 er8703 er8704 er8718 er8719 er8748 er8749 er8778 er8779 er8793
er8794 er11383 er11384 er11398 er11399 er11443 er11444 er11458 er11459
er11494 er11495 er11509 er11510 er11585 er11586 er11600 er11601 er11630
er11631 er11660 er11661 er11675 er11676 v3676 v3477 v3479 v3482 v4154
v3877 v3879 v3882 v3884 v3896 v3897 v3900 v3902 v5029 v4392 v4394 v4397
v4399 v4400 v4410 v4411 v4414 v4416 v4417 v5626 v5303 v5305 v5308 v5309
v5311 v5312 v5322 v5323 v5326 v5327 v5329 v5330 v6173 v5802 v5804 v5805
v5808 v5809 v5811 v5821 v5822 v5823 v5826 v5827 v5829 v6766 v6413 v6415
v6416 v6419 v6420 v6422 v6432 v6433 v6434 v6437 v6438 v6440 v7412 v7003
v7005 v7006 v7009 v7010 v7012 v7036 v7037 v7038 v7041 v7042 v7044 v8065
v7595 v7597 v7598 v7601 v7602 v7604 v7628 v7629 v7630 v7633 v7634 v7636
v8689 v8288 v8290 v8291 v8294 v8295 v8297 v8321 v8322 v8323 v8326 v8327
v8329 v9375 v8896 v8898 v8899 v8902 v8903 v8905 v8929 v8930 v8931 v8934
v8935 v8937 v11022 v10282 v10285 v10286 v10295 v10296 v10300 v10385 v10386
v10387 v10391 v10392 v10394 v12371 v11427 v11430 v11431 v11440 v11441
v11445 v11452 v11453 v11457 v11565 v11566 v11567 v11571 v11572 v11574
v13623 v12826 v12829 v12830 v12839 v12840 v12844 v12847 v12850 v12851
v12859 v12860 v12864 v12972 v12973 v12974 v12978 v12979 v12981 v14670
v13928 v13931 v13932 v13941 v13942 v13946 v13949 v13952 v13953 v13961
v13962 v13966 v14074 v14075 v14076 v14080 v14081 v14083 v18875 v17859
v17862 v17863 v17872 v17873 v17877 v17880 v17883 v17884 v17892 v17893
v17897 v18005 v18006 v18007 v18011 v18012 v18014 v20175 v19159 v19162
v19163 v19172 v19173 v19177 v19180 v19183 v19184 v19192 v19193 v19197
v19305 v19306 v19307 v19311 v19312 v19314 v21481 v20459 v20462 v20463
v20472 v20473 v20477 v20480 v20483 v20484 v20492 v20493 v20497 v20605
v20606 v20607 v20611 v20612 v20614 v23322 v21978 v21994 v22010 v22108
v22124 v22156 v22172 v22204 v22220 v22236 v22252 v22284 v22333 v22349
v22374 v22376 v22378 v22386 v22388 v22392 faminc94 faminc95 faminc96
faminc97 v3423 v3823 v4332 v5232 v5731 v6336 v6934 v7530 v8228 v8830
v10037 v11146 v12545 v13745 v17744 v19044 v20344 v21634 hdtot94 hdtot95
hdtot96 hdtot97
v3431 v3831 v4344 v5244 v5743 v6348 v6946 v7540 v8238 v8840 v10131
v11258 v12657 v13809 v17774 v19074 v20374 v21670 wftot94 wftot95 wftot96 wftot97
er9103 er9104 er9105 er9106 er9107;

destring, replace ;
compress;
tempfile job54374;
save `job54374', replace;
sum *, sep(0);

#delim cr



*---------------------
* CALCULATIONS ABOUT FAMILY SIZE AND MEMBERSHIP
*---------------------

*for each year, this code makes ordered list of head/spouse and other household members,
*calculates food need, and saves everything in a file called data/loaded/members`fileyear'.dta

members, intvarnum(30001) reltype(1) fileyear(1968) dfile("`job54374'")
members, intvarnum(30020) reltype(1) fileyear(1969) dfile("`job54374'")
members, intvarnum(30043) reltype(1) fileyear(1970) dfile("`job54374'")
members, intvarnum(30067) reltype(1) fileyear(1971) dfile("`job54374'")
members, intvarnum(30091) reltype(1) fileyear(1972) dfile("`job54374'")
members, intvarnum(30117) reltype(1) fileyear(1973) dfile("`job54374'")
members, intvarnum(30138) reltype(1) fileyear(1974) dfile("`job54374'")
members, intvarnum(30160) reltype(1) fileyear(1975) dfile("`job54374'")
members, intvarnum(30188) reltype(1) fileyear(1976) dfile("`job54374'")
members, intvarnum(30217) reltype(1) fileyear(1977) dfile("`job54374'")
members, intvarnum(30246) reltype(1) fileyear(1978) dfile("`job54374'")
members, intvarnum(30283) reltype(1) fileyear(1979) dfile("`job54374'")
members, intvarnum(30313) reltype(1) fileyear(1980) dfile("`job54374'")
members, intvarnum(30343) reltype(1) fileyear(1981) dfile("`job54374'")
members, intvarnum(30373) reltype(1) fileyear(1982) dfile("`job54374'")
members, intvarnum(30399) reltype(2) fileyear(1983) dfile("`job54374'")
members, intvarnum(30429) reltype(2) fileyear(1984) dfile("`job54374'")
members, intvarnum(30463) reltype(2) fileyear(1985) dfile("`job54374'")
members, intvarnum(30498) reltype(2) fileyear(1986) dfile("`job54374'")
members, intvarnum(30535) reltype(2) fileyear(1987) dfile("`job54374'")
members, intvarnum(30570) reltype(2) fileyear(1988) dfile("`job54374'")
members, intvarnum(30606) reltype(2) fileyear(1989) dfile("`job54374'")
members, intvarnum(30642) reltype(2) fileyear(1990) dfile("`job54374'")
members, intvarnum(30689) reltype(2) fileyear(1991) dfile("`job54374'")
members, intvarnum(30733) reltype(2) fileyear(1992) dfile("`job54374'")
members, intvarnum(30806) reltype(2) fileyear(1993) dfile("`job54374'")
forvalues year=1994/1997 {
  local s=`year'-1993
  doit, cmd(`"members, intvarnum(33`s'01) reltype(2) fileyear(`year') dfile("`job54374'")"')
  }

*combine everything
clear
use data/loaded/members1968.dta
local mstr ""
forvalues yr=1969/1997 {
  local mstr "`mstr' data/loaded/members`yr'.dta"
  }
merge headpid using `mstr', unique nosummary
drop _merge
forvalues yr=1968/1997 {
  rm data/loaded/members`yr'.dta
  }

*mark changes in family structure
#delim ;
label def state2 1 "last year matches this year"
  2 "last year and this year exist but do not match"
  3 "this year exists but not last year"
  4 "last year exists but not this year"
  5 "neither last year nor this year exists";
label def state1 1 "next year matches this year"
  2 "next year and this year exist but do not match"
  3 "next year exists but not this year"
  4 "this year exists but not next year"
  5 "neither next year nor this year exists";
#delim cr

*also mark observations that can be used for sequences of consecutive data 
*on households w/ no composition change. consumption data is recorded now. 
*income and leisure data are for last year. so we need now and last year  
*to be same people, i.e., observation is valid only if state2==1 or 
*year==1968.

*also find the sequences

gen byte valid_hw1968=hw1968~=""
gen byte valid_omemb1968=valid_hw1968
gen double seqid_hw1968=100*headpid+1 if valid_hw1968==1
gen double seqid_omemb1968=seqid_hw1968
gen double seqid_hwlast=seqid_hw1968
gen double seqid_omemblast=seqid_omemb1968
forvalues t=1969/1997 {
  local s=`t'-1
  gen byte state2_hw`t'=5
  replace state2_hw`t'=1 if hw`s'==hw`t' & hw`s'~="" & hw`t'~=""
  replace state2_hw`t'=2 if hw`s'~=hw`t' & hw`s'~="" & hw`t'~=""
  replace state2_hw`t'=3 if hw`s'=="" & hw`t'~=""
  replace state2_hw`t'=4 if hw`s'~="" & hw`t'==""
  gen byte state2_omemb`t'=state2_hw`t'
  replace state2_omemb`t'=2 if (hw`s'~=hw`t' | omemb`s'~=omemb`t') & hw`s'~="" & hw`t'~=""
  foreach sfx in `typelist' {
    gen byte valid_`sfx'`t'=(state2_`sfx'`t'==1)
    gen byte state1_`sfx'`s'=state2_`sfx'`t'
    label values state1_`sfx'`s' state1
    label values state2_`sfx'`t' state2
    gen double seqid_`sfx'`t'=seqid_`sfx'`s' if valid_`sfx'`t' & seqid_`sfx'`s'~=.
    replace seqid_`sfx'`t'=seqid_`sfx'last+1 if valid_`sfx'`t' & seqid_`sfx'`s'==. & seqid_`sfx'last~=.
    replace seqid_`sfx'`t'=100*headpid+1 if valid_`sfx'`t' & seqid_`sfx'`s'==. & seqid_`sfx'last==.
    replace seqid_`sfx'last=seqid_`sfx'`t' if seqid_`sfx'`t'~=.
    }
  }
drop seqid_hwlast seqid_omemblast

sort headpid
order headpid
compress
tempfile members_all
save `members_all', replace
sum headpid interview* state* valid* seqid* famsize* need*, sep(0)
describe

*save files of how head id's are assigned to household consecutive data sequences
foreach type in `typelist' {
  use headpid seqid_`type'* famsize* need* using `members_all'
  reshape long seqid_`type' famsize need, i(headpid) j(fileyear)
  keep if seqid_`type'~=.
  sort headpid fileyear
  tempfile seqs_`type'
  save `seqs_`type'', replace
  sum *, sep(0)
  describe
  drop _all
  }



*---------------------
* CONSTRUCT VARIABLES IN PSID
*---------------------

#delim ;

*strings for identifying current heads and spouses;
local head1968 "(er30002>=1 & er30002<=20 & er30003==1)";
local head1969 "(er30021>=1 & er30021<=20 & er30022==1)";
local head1970 "(er30044>=1 & er30044<=20 & er30045==1)";
local head1971 "(er30068>=1 & er30068<=20 & er30069==1)";
local head1972 "(er30092>=1 & er30092<=20 & er30093==1)";
local head1973 "(er30118>=1 & er30118<=20 & er30119==1)";
local head1974 "(er30139>=1 & er30139<=20 & er30140==1)";
local head1975 "(er30161>=1 & er30161<=20 & er30162==1)";
local head1976 "(er30189>=1 & er30189<=20 & er30190==1)";
local head1977 "(er30218>=1 & er30218<=20 & er30219==1)";
local head1978 "(er30247>=1 & er30247<=20 & er30248==1)";
local head1979 "(er30284>=1 & er30284<=20 & er30285==1)";
local head1980 "(er30314>=1 & er30314<=20 & er30315==1)";
local head1981 "(er30344>=1 & er30344<=20 & er30345==1)";
local head1982 "(er30374>=1 & er30374<=20 & er30375==1)";
local head1983 "(er30400>=1 & er30400<=20 & er30401==10)";
local head1984 "(er30430>=1 & er30430<=20 & er30431==10)";
local head1985 "(er30464>=1 & er30464<=20 & er30465==10)";
local head1986 "(er30499>=1 & er30499<=20 & er30500==10)";
local head1987 "(er30536>=1 & er30536<=20 & er30537==10)";
local head1988 "(er30571>=1 & er30571<=20 & er30572==10)";
local head1989 "(er30607>=1 & er30607<=20 & er30608==10)";
local head1990 "(er30643>=1 & er30643<=20 & er30644==10)";
local head1991 "(er30690>=1 & er30690<=20 & er30691==10)";
local head1992 "(er30734>=1 & er30734<=20 & er30735==10)";
local head1993 "(er30807>=1 & er30807<=20 & er30808==10)";
local head1994 "(er33102>=1 & er33102<=20 & er33103==10)";
local head1995 "(er33202>=1 & er33202<=20 & er33203==10)";
local head1996 "(er33302>=1 & er33302<=20 & er33303==10)";
local head1997 "(er33402>=1 & er33402<=20 & er33403==10)";
local wife1968 "(er30002>=1 & er30002<=20 & er30003==2)";
local wife1969 "(er30021>=1 & er30021<=20 & er30022==2)";
local wife1970 "(er30044>=1 & er30044<=20 & er30045==2)";
local wife1971 "(er30068>=1 & er30068<=20 & er30069==2)";
local wife1972 "(er30092>=1 & er30092<=20 & er30093==2)";
local wife1973 "(er30118>=1 & er30118<=20 & er30119==2)";
local wife1974 "(er30139>=1 & er30139<=20 & er30140==2)";
local wife1975 "(er30161>=1 & er30161<=20 & er30162==2)";
local wife1976 "(er30189>=1 & er30189<=20 & er30190==2)";
local wife1977 "(er30218>=1 & er30218<=20 & er30219==2)";
local wife1978 "(er30247>=1 & er30247<=20 & er30248==2)";
local wife1979 "(er30284>=1 & er30284<=20 & er30285==2)";
local wife1980 "(er30314>=1 & er30314<=20 & er30315==2)";
local wife1981 "(er30344>=1 & er30344<=20 & er30345==2)";
local wife1982 "(er30374>=1 & er30374<=20 & er30375==2)";
local wife1983 "(er30400>=1 & er30400<=20 & (er30401==20 | er30401==22))";
local wife1984 "(er30430>=1 & er30430<=20 & (er30431==20 | er30431==22))";
local wife1985 "(er30464>=1 & er30464<=20 & (er30465==20 | er30465==22))";
local wife1986 "(er30499>=1 & er30499<=20 & (er30500==20 | er30500==22))";
local wife1987 "(er30536>=1 & er30536<=20 & (er30537==20 | er30537==22))";
local wife1988 "(er30571>=1 & er30571<=20 & (er30572==20 | er30572==22))";
local wife1989 "(er30607>=1 & er30607<=20 & (er30608==20 | er30608==22))";
local wife1990 "(er30643>=1 & er30643<=20 & (er30644==20 | er30644==22))";
local wife1991 "(er30690>=1 & er30690<=20 & (er30691==20 | er30691==22))";
local wife1992 "(er30734>=1 & er30734<=20 & (er30735==20 | er30735==22))";
local wife1993 "(er30807>=1 & er30807<=20 & (er30808==20 | er30808==22))";
local wife1994 "(er33102>=1 & er33102<=20 & (er33103==20 | er33103==22))";
local wife1995 "(er33202>=1 & er33202<=20 & (er33203==20 | er33203==22))";
local wife1996 "(er33302>=1 & er33302<=20 & (er33303==20 | er33303==22))";
local wife1997 "(er33402>=1 & er33402<=20 & (er33403==20 | er33403==22))";


use `job54374' if
  er30001<=2930 & (`head1968' | `head1969' | `head1970' | `head1971' |
  `head1972' | `head1973' | `head1974' | `head1975' | `head1976' |
  `head1977' | `head1978' | `head1979' | `head1980' | `head1981' |
  `head1982' | `head1983' | `head1984' | `head1985' | `head1986' | 
  `head1987' | `head1988' | `head1989' | `head1990' | `head1991' | 
  `head1992' | `head1993' | `head1994' | `head1995' | `head1996' | 
  `head1997' | `wife1968' | `wife1969' | `wife1970' | `wife1971' |
  `wife1972' | `wife1973' | `wife1974' | `wife1975' | `wife1976' |
  `wife1977' | `wife1978' | `wife1979' | `wife1980' | `wife1981' |
  `wife1982' | `wife1983' | `wife1984' | `wife1985' | `wife1986' | 
  `wife1987' | `wife1988' | `wife1989' | `wife1990' | `wife1991' | 
  `wife1992' | `wife1993' | `wife1994' | `wife1995' | `wife1996' | 
  `wife1997');
count;
duplicates tag er30001 if `head1968' | `wife1968', gen(haswife1968);
duplicates tag er30020 if `head1969' | `wife1969', gen(haswife1969);
duplicates tag er30043 if `head1970' | `wife1970', gen(haswife1970);
duplicates tag er30067 if `head1971' | `wife1971', gen(haswife1971);
duplicates tag er30091 if `head1972' | `wife1972', gen(haswife1972);
duplicates tag er30117 if `head1973' | `wife1973', gen(haswife1973);
duplicates tag er30138 if `head1974' | `wife1974', gen(haswife1974);
duplicates tag er30160 if `head1975' | `wife1975', gen(haswife1975);
duplicates tag er30188 if `head1976' | `wife1976', gen(haswife1976);
duplicates tag er30217 if `head1977' | `wife1977', gen(haswife1977);
duplicates tag er30246 if `head1978' | `wife1978', gen(haswife1978);
duplicates tag er30283 if `head1979' | `wife1979', gen(haswife1979);
duplicates tag er30313 if `head1980' | `wife1980', gen(haswife1980);
duplicates tag er30343 if `head1981' | `wife1981', gen(haswife1981);
duplicates tag er30373 if `head1982' | `wife1982', gen(haswife1982);
duplicates tag er30399 if `head1983' | `wife1983', gen(haswife1983);
duplicates tag er30429 if `head1984' | `wife1984', gen(haswife1984);
duplicates tag er30463 if `head1985' | `wife1985', gen(haswife1985);
duplicates tag er30498 if `head1986' | `wife1986', gen(haswife1986);
duplicates tag er30535 if `head1987' | `wife1987', gen(haswife1987);
duplicates tag er30570 if `head1988' | `wife1988', gen(haswife1988);
duplicates tag er30606 if `head1989' | `wife1989', gen(haswife1989);
duplicates tag er30642 if `head1990' | `wife1990', gen(haswife1990);
duplicates tag er30689 if `head1991' | `wife1991', gen(haswife1991);
duplicates tag er30733 if `head1992' | `wife1992', gen(haswife1992);
duplicates tag er30806 if `head1993' | `wife1993', gen(haswife1993);
duplicates tag er33101 if `head1994' | `wife1994', gen(haswife1994);
duplicates tag er33201 if `head1995' | `wife1995', gen(haswife1995);
duplicates tag er33301 if `head1996' | `wife1996', gen(haswife1996);
duplicates tag er33401 if `head1997' | `wife1997', gen(haswife1997);
sum haswife*;
forvalues year=1968/1997 {;
  replace haswife`year'=1 if haswife`year'>1 & haswife`year'~=.;
  };
keep if (`head1968' | `head1969' | `head1970' | `head1971' |
  `head1972' | `head1973' | `head1974' | `head1975' | `head1976' |
  `head1977' | `head1978' | `head1979' | `head1980' | `head1981' |
  `head1982' | `head1983' | `head1984' | `head1985' | `head1986' | 
  `head1987' | `head1988' | `head1989' | `head1990' | `head1991' | 
  `head1992' | `head1993' | `head1994' | `head1995' | `head1996' | 
  `head1997');
duplicates report er30001 if `head1968';
duplicates report er30020 if `head1969';
duplicates report er30043 if `head1970';
duplicates report er30067 if `head1971';
duplicates report er30091 if `head1972';
duplicates report er30117 if `head1973';
duplicates report er30138 if `head1974';
duplicates report er30160 if `head1975';
duplicates report er30188 if `head1976';
duplicates report er30217 if `head1977';
duplicates report er30246 if `head1978';
duplicates report er30283 if `head1979';
duplicates report er30313 if `head1980';
duplicates report er30343 if `head1981';
duplicates report er30373 if `head1982';
duplicates report er30399 if `head1983';
duplicates report er30429 if `head1984';
duplicates report er30463 if `head1985';
duplicates report er30498 if `head1986';
duplicates report er30535 if `head1987';
duplicates report er30570 if `head1988';
duplicates report er30606 if `head1989';
duplicates report er30642 if `head1990';
duplicates report er30689 if `head1991';
duplicates report er30733 if `head1992';
duplicates report er30806 if `head1993';
duplicates report er33101 if `head1994';
duplicates report er33201 if `head1995';
duplicates report er33301 if `head1996';
duplicates report er33401 if `head1997';

local keepvars "";
local reshapevars "";

*valid codes;
forvalues yr=1967/1996 {;
  local yp=`yr'+1;
  gen byte valid`yr'=`head`yp'';
  };
forvalues year=1967/1996 {;
  local ynext=`year'+1;
  label var valid`year' "=1 if valid in `year' (person is head on `ynext' file)";
  };
local keepvars "`keepvars' valid*";
local reshapevars "`reshapevars' valid";


*identifiers from 1968;
gen double orig_hh=er30001;
label var orig_hh "original household id: er30001";
gen orig_psu=v92;
label var orig_psu "src primary sampling unit: v92";
local keepvars "`keepvars' orig_hh orig_psu";

*identifier for household head;
gen double headpid=10000000+er30001*1000+er30002;
label var headpid "unique id for head: 10000000+er30001*1000+er30002";
local keepvars "`keepvars' headpid";

*food: drop all assignments, and drop if this person
*is not the head in the relevant year;
*no need to calculate pre-1973 since we will not use;
gen food1973=v3441+v3443+v3445
  if v3442==0 & v3444==0 & v3446==0 & `head1974';
gen food1974=v3841+v3849+v3851+v3853
  if v3842==0 & v3850==0 & v3852==0 & v3854==0 & `head1975';
gen food1975=v4354+v4362+v4364+v4368
  if v4355==0 & v4363==0 & v4365==0 & v4369==0 & `head1976';
gen food1976=v5271+v5273+v5275+v5277
  if v5272==0 & v5274==0 & v5276==0 & v5278==0 & `head1977';
gen food1977=v5770+v5772+v5774+v5776
  if v5771==0 & v5773==0 & v5775==0 & v5777==0 & `head1978';
gen food1978=v6376+v6378+v6380+v6382
  if v6377==0 & v6379==0 & v6381==0 & v6383==0 & `head1979';
gen food1979=v6972+v6974+v6976
  if v6973==0 & v6975==0 & v6977==0 & `head1980';
gen food1980=v7564+v7566+v7568
  if v7565==0 & v7567==0 & v7569==0 & `head1981';
gen food1981=v8256+v8258+v8260
  if v8257==0 & v8259==0 & v8261==0 & `head1982';
gen food1982=v8864+v8866+v8868
  if v8865==0 & v8867==0 & v8868==0 & `head1983';
gen food1983=v10235+v10237+v10239
  if v10236==0 & v10238==0 & v10240==0 & `head1984';
gen food1984=v11375+v11377+v11379
  if v11376==0 & v11378==0 & v11380==0 & `head1985';
gen food1985=v12774+v12776+v12778
  if v12775==0 & v12777==0 & v12779==0 & `head1986';
gen food1986=v13876+v13878+v13880
  if v13877==0 & v13879==0 & v13881==0 & `head1987';
gen food1989=v17807+v17809+v17811
  if v17808==0 & v17810==0 & v17812==0 & `head1990';
gen food1990=v19107+v19109+v19111
  if v19108==0 & v19110==0 & v19112==0 & `head1991';
gen food1991=v20407+v20409+v20411
  if v20408==0 & v20410==0 & v20412==0 & `head1992';
gen food1992=v21707+v21711+v21727
  if v21708==0 & v21712==0 & v21728==0 & `head1993';
foreach vn in er3060  er3078  er3081  er3083  er3085  er3088  er3090
         er6059  er6077  er6080  er6082  er6084  er6087  er6089
         er8156  er8174  er8177  er8179  er8181  er8184  er8186
         er11050 er11068 er11071 er11073 er11076 er11079 er11081 {;
  replace `vn'=. if `vn'>=99996;
  *documentation says 999998, but there are some 99996 in the data;
  };
foreach vn in er3060 er3078 er3081 er3083 er3085 er3088 er3090 {;
  sum `vn' if `head1994', det;
  };
foreach vn in er6059  er6077  er6080  er6082  er6084  er6087  er6089 {;
  sum `vn' if `head1995', det;
  };
foreach vn in er8156  er8174  er8177  er8179  er8181  er8184  er8186 {;
  sum `vn' if `head1996', det;
  };
foreach vn in er11050 er11068 er11071 er11073 er11076 er11079 er11081 {;
  sum `vn' if `head1997', det;
  };
*two obviously wrong data points;
replace er8181=. if er8181==34540.06;
replace er11076=. if er11076==80110;
*begin recoding;
replace er3060=er3060*12 if er3061==1;
replace er3060=er3060*26 if er3061==2;
replace er3060=er3060*52 if er3061==3;
replace er3060=. if er3061>=4;
foreach vn in 3078 3081 3083 3085 3088 3090 {;
  local vp=`vn'+1;
  tabstat er`vn' if (er`vp'==1 | er`vp'==2 | er`vp'==3) & `head1994',
    by(er`vp') s(mean n min max p1 p99);
  replace er`vn'=er`vn'*52 if er`vp'==1;
  replace er`vn'=er`vn'*26 if er`vp'==2;
  replace er`vn'=er`vn'*12 if er`vp'==3;
  replace er`vn'=. if er`vp'>=4;
  };
*codebook for 1995 wave (1994 data: er6*) is wrong;
*use 1996-1997 coding instead;
foreach vn in  6059  6077  6080  6082  6084  6087  6089
               8156  8174  8177  8179  8181  8184  8186 
              11050 11068 11071 11073 11076 11079 11081 {;
  local vp=`vn'+1;
  tabstat er`vn' if (er`vp'==1 | er`vp'==2 | er`vp'==3 | er`vp'==4 | er`vp'==5)
    & (`head1996'|`head1997'),
    by(er`vp') s(mean n min max p1 p99);
  replace er`vn'=er`vn'*365 if er`vp'==2;
  replace er`vn'=er`vn'*52 if er`vp'==3;
  replace er`vn'=er`vn'*26 if er`vp'==4;
  replace er`vn'=er`vn'*12 if er`vp'==5;
  replace er`vn'=. if er`vp'>=7 | er`vp'==1;
  };
gen food1993=er3060 +er3078 +er3081 +er3083 +er3085 +er3088 +er3090
  if `head1994';
gen food1994=er6059 +er6077 +er6080 +er6082 +er6084 +er6087 +er6089
  if `head1995';
gen food1995=er8156 +er8174 +er8177 +er8179 +er8181 +er8184 +er8186
  if `head1996';
gen food1996=er11050+er11068+er11071+er11073+er11076+er11079+er11081
  if `head1997';
sum food*, det;
*check out some weird data points;
list food1984 v11375 v11377 v11379 if `head1985' & food1984>40000 & 
  food1984~=., clean;
list food1989 v17807 v17809 v17811 if `head1990' & food1989>50000 & 
 food1989~=., clean;
list food1990 v19107 v19109 v19111 if `head1991' & food1990>50000 & 
  food1990~=., clean;
list food1991 v20407 v20409 v20411 if `head1992' & food1991>50000 & 
  food1991~=., clean;
list food1992 v21707 v21711 v21727 if `head1993' & food1992>50000 & 
  food1992~=., clean;
list food1993 er3060 er3078 er3081 er3083 er3085 er3088 er3090
  if `head1994' & food1993>50000 & food1993~=., clean;
list food1994 er6059  er6077  er6080  er6082  er6084  er6087  er6089 
  if `head1995' & food1994>50000 & food1994~=., clean;
list food1995 er8156  er8174  er8177  er8179  er8181  er8184  er8186 
  if `head1996' & food1995>50000 & food1995~=., clean;
list food1996 er11050 er11068 er11071 er11073 er11076 er11079 er11081 
  if `head1997' & food1996>50000 & food1996~=., clean;
forvalues year=1973/1996 {;
  if `year'~=1987 & `year'~=1988 {;
    di "`year'";
    local ynext=`year'+1;
    *a few crazy numbers;
    replace food`year'=. if food`year'>99999 & food`year'~=.;
    label var food`year' "food consumption in `year', from `ynext' file";
    gen lfood`year'=log(food`year');
    label var lfood`year' "log(food`year')";
    };
  };
sum food* lfood*, sep(0);
local keepvars "`keepvars' food* lfood*";
local reshapevars "`reshapevars' food lfood";

*family money income net of insurance-type transfers;
*no need to calculate pre-1973 since we will not use;
gen fmy1973=v3676 if `head1974';
gen netfmy1973=v3676-v3477-v3479-v3482 if `head1974';
gen fmy1974=v4154 if `head1975';
gen netfmy1974=v4154-v3877-v3879-v3882-v3884
  -v3896-v3897-v3900-v3902 if `head1975';
gen fmy1975=v5029 if `head1976';
gen netfmy1975=v5029-v4392-v4394-v4397-v4399-v4400
  -v4410-v4411-v4414-v4416-v4417 if `head1976';
gen fmy1976=v5626 if `head1977';
gen netfmy1976=v5626-v5303-v5305-v5308-v5309-v5311-v5312
  -v5322-v5323-v5326-v5327-v5329-v5330 if `head1977';
gen fmy1977=v6173 if `head1978';
gen netfmy1977=v6173-v5802-v5804-v5805-v5808-v5809-v5811
  -v5821-v5822-v5823-v5826-v5827-v5829 if `head1978';
gen fmy1978=v6766 if `head1979';
gen netfmy1978=v6766-v6413-v6415-v6416-v6419-v6420-v6422
  -v6432-v6433-v6434-v6437-v6438-v6440 if `head1979';
gen fmy1979=v7412 if `head1980';
gen netfmy1979=v7412-v7003-v7005-v7006-v7009-v7010-v7012
  -v7036-v7037-v7038-v7041-v7042-v7044 if `head1980';
gen fmy1980=v8065 if `head1981';
gen netfmy1980=v8065-v7595-v7597-v7598-v7601-v7602-v7604
  -v7628-v7629-v7630-v7633-v7634-v7636 if `head1981';
gen fmy1981=v8689 if `head1982';
gen netfmy1981=v8689-v8288-v8290-v8291-v8294-v8295-v8297
  -v8321-v8322-v8323-v8326-v8327-v8329 if `head1982';
gen fmy1982=v9375 if `head1983';
gen netfmy1982=v9375-v8896-v8898-v8899-v8902-v8903-v8905
  -v8929-v8930-v8931-v8934-v8935-v8937 if `head1983';
gen fmy1983=v11022 if `head1984';
gen netfmy1983=v11022-v10282-v10285-v10286-v10295-v10296-v10300
  -v10385-v10386-v10387-v10391-v10392-v10394 if `head1984';
gen fmy1984=v12371 if `head1985';
gen netfmy1984=v12371-v11427-v11430-v11431-v11440-v11441-v11445
  -v11452-v11453-v11457
  -v11565-v11566-v11567-v11571-v11572-v11574 if `head1985';
gen fmy1985=v13623 if `head1986';
gen netfmy1985=v13623-v12826-v12829-v12830-v12839-v12840-v12844
  -v12847-v12850-v12851-v12859-v12860-v12864
  -v12972-v12973-v12974-v12978-v12979-v12981 if `head1986';
gen fmy1986=v14670 if `head1987';
gen netfmy1986=v14670-v13928-v13931-v13932-v13941-v13942-v13946
  -v13949-v13952-v13953-v13961-v13962-v13966
  -v14074-v14075-v14076-v14080-v14081-v14083 if `head1987';
gen fmy1989=v18875 if `head1990';
gen netfmy1989=v18875-v17859-v17862-v17863-v17872-v17873-v17877
  -v17880-v17883-v17884-v17892-v17893-v17897
  -v18005-v18006-v18007-v18011-v18012-v18014 if `head1990';
gen fmy1990=v20175 if `head1991';
gen netfmy1990=v20175-v19159-v19162-v19163-v19172-v19173-v19177
  -v19180-v19183-v19184-v19192-v19193-v19197
  -v19305-v19306-v19307-v19311-v19312-v19314 if `head1991';
gen fmy1991=v21481 if `head1992';
gen netfmy1991=v21481-v20459-v20462-v20463-v20472-v20473-v20477
  -v20480-v20483-v20484-v20492-v20493-v20497
  -v20605-v20606-v20607-v20611-v20612-v20614 if `head1992';
gen fmy1992=v23322 if `head1993';
gen netfmy1992=v23322-v21978-v21994-v22010-v22108-v22124-v22156-v22172
  -v22204-v22220-v22236-v22252-v22284-v22333-v22349
  -v22374-v22376-v22378-v22386-v22388-v22392 if `head1993';
forvalues yr=94/97 {;
  local pyr=19`yr'-1;
  gen fmy`pyr'=faminc`yr' if valid`pyr'==1;
  replace fmy`pyr'=1 if fmy`pyr'<=0 & fmy`pyr'~=.;
  };
foreach vn in 3263 3278 3293 6263 6279 6294 {;
  local vp=`vn'+1;
  replace er`vn'=er`vn'*12 if er`vp'==1;
  replace er`vn'=er`vn'*4 if er`vp'==2;
  replace er`vn'=er`vn'*2 if er`vp'==3;
  replace er`vn'=0 if er`vp'>=7 | er`vn'>=999998;
  };
foreach vn in 3371 3386 3431 3446 3494 3509 3584 3599 3629 3659 3674
              6372 6387 6432 6447 6495 6510 6586 6601 6631 6661 6676 {;
  local vp=`vn'+1;
  replace er`vn'=er`vn'*52 if er`vp'==1;
  replace er`vn'=er`vn'*26 if er`vp'==2;
  replace er`vn'=er`vn'*12 if er`vp'==3;
  replace er`vn'=0 if er`vp'>=5 | er`vn'>=99998;
  };
foreach vn in 8380 8396 8411 11273 11289 11304 {;
  local vp=`vn'+1;
  replace er`vn'=er`vn'*52 if er`vp'==3;
  replace er`vn'=er`vn'*26 if er`vp'==4;
  replace er`vn'=er`vn'*12 if er`vp'==5;
  replace er`vn'=0 if er`vp'>=7 | er`vn'>=999998;
  };
foreach vn in 8489  8504  8549  8564  8612  8627  8703  8718  8748  8778  8793
              11383 11398 11443 11458 11494 11509 11585 11600 11630 11660 11675
  {;
  local vp=`vn'+1;
  replace er`vn'=er`vn'*52 if er`vp'==3;
  replace er`vn'=er`vn'*26 if er`vp'==4;
  replace er`vn'=er`vn'*12 if er`vp'==5;
  replace er`vn'=0 if er`vp'>=7 | er`vn'>=99998;
  };
gen netfmy1993=fmy1993-er3263-er3278-er3293-er3371-er3386-er3431-er3446
  -er3494-er3509-er3584-er3599-er3629-er3659-er3674 if `head1994';
gen netfmy1994=fmy1994-er6263-er6279-er6294-er6372-er6387-er6432-er6447
  -er6495-er6510-er6586-er6601-er6631-er6661-er6676 if `head1995';
gen netfmy1995=fmy1995-er8380-er8396-er8411-er8489-er8504-er8549-er8564
  -er8612-er8627-er8703-er8718-er8748-er8778-er8793 if `head1996';
gen netfmy1996=fmy1996-er11273-er11289-er11304-er11383-er11398-er11443-er11458
  -er11494-er11509-er11585-er11600-er11630-er11660-er11675 if `head1997';
forvalues year=1973/1996 {;
  if `year'~=1987 & `year'~=1988 {;
    di "`year'";
    local ynext=`year'+1;
    label var fmy`year' "family money income in `year', from `ynext' file";
    gen lfmy`year'=log(fmy`year');
    label var lfmy`year' "log(fmy`year')";
    label var netfmy`year'
      "family money income net of transfers in `year', from `ynext' file";
    gen lnetfmy`year'=log(netfmy`year');
    label var lnetfmy`year' "log(netfmy`year')";
    };
  };
sum fmy* lfmy* netfmy* lnetfmy*, sep(0);
local keepvars "`keepvars' fmy* lfmy* netfmy* lnetfmy*";
local reshapevars "`reshapevars' fmy lfmy netfmy lnetfmy";

*leisure;
gen h_leisurea1973=8760-v3423 if `head1974';
gen w_leisurea1973=8760-v3431 if `head1974' & haswife1974;
gen h_leisurea1974=8760-v3823 if `head1975';
gen w_leisurea1974=8760-v3831 if `head1975' & haswife1975;
gen h_leisurea1975=8760-v4332 if `head1976';
gen w_leisurea1975=8760-v4344 if `head1976' & haswife1976;
gen h_leisurea1976=8760-v5232 if `head1977';
gen w_leisurea1976=8760-v5244 if `head1977' & haswife1977;
gen h_leisurea1977=8760-v5731 if `head1978';
gen w_leisurea1977=8760-v5743 if `head1978' & haswife1978;
gen h_leisurea1978=8760-v6336 if `head1979';
gen w_leisurea1978=8760-v6348 if `head1979' & haswife1979;
gen h_leisurea1979=8760-v6934 if `head1980';
gen w_leisurea1979=8760-v6946 if `head1980' & haswife1980;
gen h_leisurea1980=8760-v7530 if `head1981';
gen w_leisurea1980=8760-v7540 if `head1981' & haswife1981;
gen h_leisurea1981=8760-v8228 if `head1982';
gen w_leisurea1981=8760-v8238 if `head1982' & haswife1982;
gen h_leisurea1982=8760-v8830 if `head1983';
gen w_leisurea1982=8760-v8840 if `head1983' & haswife1983;
gen h_leisurea1983=8760-v10037 if `head1984';
gen w_leisurea1983=8760-v10131 if `head1984' & haswife1984;
gen h_leisurea1984=8760-v11146 if `head1985';
gen w_leisurea1984=8760-v11258 if `head1985' & haswife1985;
gen h_leisurea1985=8760-v12545 if `head1986';
gen w_leisurea1985=8760-v12657 if `head1986' & haswife1986;
gen h_leisurea1986=8760-v13745 if `head1987';
gen w_leisurea1986=8760-v13809 if `head1987' & haswife1987;
gen h_leisurea1989=8760-v17744 if `head1990';
gen w_leisurea1989=8760-v17774 if `head1990' & haswife1990;
gen h_leisurea1990=8760-v19044 if `head1991';
gen w_leisurea1990=8760-v19074 if `head1991' & haswife1991;
gen h_leisurea1991=8760-v20344 if `head1992';
gen w_leisurea1991=8760-v20374 if `head1992' & haswife1992;
gen h_leisurea1992=8760-v21634 if `head1993';
gen w_leisurea1992=8760-v21670 if `head1993' & haswife1993;
forvalues yr=94/97 {;
  local pyr=19`yr'-1;
  gen h_leisurea`pyr'=8760-hdtot`yr' if valid`pyr'==1;
  gen w_leisurea`pyr'=8760-wftot`yr' if valid`pyr'==1 & haswife19`yr';
  };
forvalues year=1973/1996 {;
  if `year'~=1987 & `year'~=1988 {;
    local ynext=`year'+1;
    label var h_leisurea`year' "8760-head's work hours in `year', from `ynext' file";
    label var h_leisurea`year' "8760-wife's work hours in `year', from `ynext' file";
    gen lh_leisurea`year'=log(h_leisurea`year');
    label var lh_leisurea`year' "log(h_leisurea`year')";
    gen lw_leisurea`year'=log(w_leisurea`year');
    label var lw_leisurea`year' "log(w_leisurea`year')";
    gen t_leisurea`year'=h_leisurea`year'+cond(w_leisurea`year'~=.,w_leisurea`year',0);
    label var t_leisurea`year' "h_leisurea`year'+cond(w_leisurea`year'~=.,w_leisurea`year',0)";
    gen ts_leisurea`year'=cond(w_leisurea`year'~=.,t_leisurea`year'/2,t_leisurea`year');
    label var ts_leisurea`year' "cond(w_leisurea`year'~=.,t_leisurea`year'/2,t_leisurea`year')";
    gen lts_leisurea`year'=log(ts_leisurea`year');
    label var lts_leisurea`year' "log(ts_leisurea`year')";
    };
  };
sum *leisure*, sep(0);
local keepvars "`keepvars' *leisure*";
local reshapevars "`reshapevars' h_leisurea lh_leisurea w_leisurea lw_leisurea t_leisurea ts_leisurea lts_leisurea";

#delim cr

*1996 risk tolerance variables
destring er9103-er9107, replace
gen rtcat=.
replace rtcat=1 if er9106==5 & er9103==5 & er9105==5 
replace rtcat=2 if er9106==1 & er9103==5 & er9105==5
replace rtcat=3 if er9105==1 & er9103==5 
replace rtcat=4 if er9103==1 & er9104==5 
replace rtcat=5 if er9104==1 & er9107==5 & er9103==1 
replace rtcat=6 if er9107==1 & er9104==1 & er9103==1 
replace rtcat=. if !`head1996'
lab def rtcat 1 "reject 10% cut" 2 "take 10% cut, reject 20%" ///
  3 "take 20% cut, reject 1/3" 4 "take 1/3 cut, reject 50%" ///
  5 "take 50% cut, reject 75%" 6 "take 75% cut"
lab values rtcat rtcat
local keepvars "`keepvars' er9103 er9104 er9105 er9106 er9107 rtcat"


di "`keepvars'"
keep `keepvars'
compress
sum *, sep(0)
describe

*long form
doit, cmd(`"reshape long `reshapevars', i(headpid) j(datayear)"')
drop if valid==0
gen fileyear=datayear+1

*cpi, real food, income, etc
sort datayear
merge datayear using data/loaded/cpiu.dta, uniqusing nokeep
tab _merge
drop _merge
foreach v in food fmy netfmy {
  if "`v'"=="food" local defl "cpiuf"
  else local defl "cpiu"
  gen `v'_real=100*`v'/`defl'
  label var `v'_real "100*`v'/`defl'"
  gen l`v'_real=log(`v'_real)
  label var l`v'_real "log(`v'_real)"
  }

*disturbingly low food values (less than 1 dollar per day in real terms)
di "look here for very low food values!"
doit, cmd(`"tabstat lfood_real lfood food_real food if food_real<365, by(datayear) statistics(n min max mean sd) columns(s)"')
doit, cmd(`"tabstat food, by(datayear) statistics(n p5 p25 p50 p75 p95) columns(s)"')

compress
sort headpid fileyear
tempfile load_long
save `load_long', replace
sum *, sep(0)
describe
drop _all

*merge to sequences, family size, need
foreach type in `typelist' {
  di _newline _newline _newline _newline _newline
  use `load_long'
  doit, cmd(`"merge headpid fileyear using `seqs_`type'', unique"')
  tab _merge
  drop if _merge~=3
  drop _merge
  drop if seqid_`type'==.
  count
  foreach v in food fmy netfmy food_real fmy_real netfmy_real {
    quietly {
      gen `v'_need=`v'/need
      label var `v'_need "`v'/need"
      gen l`v'_need=log(`v'_need)
      label var l`v'_need "log(`v'/need)"
      gen `v'_pc=`v'/famsize
      label var `v'_pc "`v'/famsize"
      gen l`v'_pc=log(`v'_pc)
      label var l`v'_pc "log(`v'/famsize)"
      }
    }
  compress
  tempfile load_long_seqs_`type'
  save `load_long_seqs_`type'', replace
  sum *, sep(0)
  describe
  drop _all
  di _newline _newline _newline _newline _newline
  }



*---------------------
* SELECT THE SAMPLES TO USE
*---------------------

foreach type in `typelist' {

  *load data
  #delim ;
  use valid headpid seqid_`type' datayear fileyear orig_hh 
    orig_psu *food* *fmy* *leisurea cpiu famsize need rtcat
    using `load_long_seqs_`type'';
  #delim cr

  *drop missing values
  doit, cmd("count")
  doit, cmd("keep if valid")
  drop valid
  foreach v in food_real lfood_real lfood_real_need lfood_real_pc h_leisurea lh_leisurea fmy netfmy lnetfmy {
    doit, cmd(`"keep if `v'~=."')
    }

  *too few obs before 1973 to be useful
  *this should be superfluous since the relevant variables are all missing pre-1973
  tab datayear
  doit, cmd(`"drop if datayear<1973"')

  *identify runs of observation
  tsset seqid_`type' datayear
  tsspell, f(L.datayear==.)
  sort seqid_`type' _spell
  by seqid_`type' _spell: egen _spell_n=count(_spell)
  by seqid_`type' (_spell): egen _seqid_n=count(seqid_`type')

  *a run must have at least 4 obs to be usable for GMM/EL
  doit, cmd(`"keep if _spell_n>=4"')
  tsset seqid_`type' datayear
  gen byte has3lags=(L3.datayear~=. & L2.datayear~=. & L1.datayear~=.)
  doit, cmd("count")
  doit, cmd("count if has3lags")

  compress
  #delim ;
  order seqid_`type' headpid datayear fileyear _spell _seq _end
    _spell_n _seqid_n orig_hh orig_psu
    lfood_real lfood_real_need lfood_real_pc
    lnetfmy_real lnetfmy_real_need lnetfmy_real_pc lh_leisurea;
  #delim cr
  save data/loaded/data_`type'.dta, replace

  describe
  sum *, sep(0)
  drop _all

  }


*---------------------
* CONSTRUCT BOOTSTRAP SAMPLES
*---------------------

set seed 12345
di "`c(seed)'"

*need to do some extra iterations (head) to match bootstrap sampling in
*previous code
foreach tbase in head `typelist' {
  if "`tbase'"=="head" local type="hw"
  else local type="`tbase'"
  #delim ;
  doit, cmd(`"use seqid_`type' datayear orig_psu lfood_real* lnetfmy_real* lh_leisurea* lts_leisurea* rtcat
    using data/loaded/data_`type'.dta"');
  #delim cr
  forvalues i=1/`nsamples' {
    di `i'
    preserve
    bsample, cluster(orig_psu) idcluster(newcluster)
    egen newid=group(newcluster seqid_`type')
    if "`tbase'"!="head" save data/bsamples/data_`type'_bs`i'.dta, replace
    restore
    }
  drop _all
  }

di "`c(seed)'"




*---------------------
* SUMMARY STATS (TABLE A.2, INTERNET APPENDIX TABLES F.1 AND G.1)
*---------------------


foreach type in `typelist' {

  if "`type'"=="hw" local output "apptable_f1/apptable_f1"
  else local output "table_a2/table_a2"

  use data/loaded/data_`type'.dta

  doit, cmd(`"tabstat food_real lfood_real food_real_need lfood_real_need netfmy_real lnetfmy_real netfmy_real_need lnetfmy_real_need h_leisurea lh_leisurea, statistics(mean sd) columns(statistics) save"')
  mat stats_`type'=r(StatTotal)'
  mat2txt, matrix(stats_`type') saving(output/`output'_stats_`type'.txt) replace

  mat counts_`type'=J(8,1,.)
  count
  mat counts_`type'[1,1]=r(N)
  collapse (count) nobs=food_real, by(seqid_`type') fast
  sum nobs, det
  mat counts_`type'[2,1]=r(N)
  mat counts_`type'[3,1]=r(mean)
  mat counts_`type'[4,1]=r(min)
  mat counts_`type'[5,1]=r(p25)
  mat counts_`type'[6,1]=r(p50)
  mat counts_`type'[7,1]=r(p75)
  mat counts_`type'[8,1]=r(max)
  mat2txt, matrix(counts_`type') saving(output/`output'_counts_`type') replace

  drop _all

  }

foreach type in `typelist' {

  use data/loaded/data_`type'.dta

  doit, cmd(`"tabstat ts_leisurea lts_leisurea, statistics(mean sd) columns(statistics) save"')
  mat stats_g1=r(StatTotal)'
  mat2txt, matrix(stats_g1) saving(output/apptable_g1/apptable_g1_stats_`type'.txt) replace

  mat counts_g1=J(2,1,.)
  count
  mat counts_g1[1,1]=r(N)
  collapse (count) nobs=food_real, by(seqid_`type') fast
  sum nobs, det
  mat counts_g1[2,1]=r(N)
  mat2txt, matrix(counts_g1) saving(output/apptable_g1/apptable_g1_counts_`type') replace

  drop _all

  }

log close
exit
