################################################################################
##
##  Options
##
################################################################################
options(repos = c(CRAN = "http://cran.rstudio.com"))
options(asciiType = "pandoc")   ## Use pandoc
options(stringsAsFactors = FALSE) ## PITA
par(mar = c(2, 3, 1, 1), mgp = c(2, .7, 0), tck = -0.01)

################################################################################
##
##  Load libraries
##
################################################################################
if (!require("pacman")) install.packages("pacman")
library(pacman)

#p_install(xtsExtra, repos="http://R-Forge.R-project.org")
p_load(ascii, fImport, forecast, devtools, dlm, dynlm, ggplot2, 
       xts, numDeriv, knitr, lattice, lmtest, lubridate, sandwich, vars, progress)

p_load_gh("gragusa/Hmisc", "gragusa/KFAS", "gragusa/MARSS")

##############################################################################
##
## Tilting
##
##############################################################################

tilt <- function(model_forecast, model_variance, survey_forecast) {
  ## The function starts from a h-step-ahead density forecast 
  ## N(model_forecast,model_variance) made at time t for the n-dimensional vector
  ## y_t+h and it imposes the constraint that the forecasts of a subset of k 
  ## variables equal a user-specified value available at time t (e.g.,
  ## h-step-ahead forecasts coming from survey data). For simplicity we assume
  ## that the variables have been ordered so that the constraints are imposed on
  ## the first k yields.
  ## 
  ## The function is used at each time t over an out-of-sample period.
  ##
  ## 
  ## INPUTS: model_forecast: nx1 vector containing the h-step-ahead conditional mean forecast 
  ##                         implied by the model 
  ##         model_variance: nxn matrix containing the h-step-ahead conditional variance forecast
  ##                         implied by the model
  ##         survey: a kx1 vector of h-step-ahead forecasts for the first k
  ##                 variables that are available at the time t of forecasting, for example
  ##                 from survey data
  ##
  ## OUTPUTS: tilted_forecast: an nx1 vector containing the h-step-ahead forecasts at time t for
  ##                                all variables, subject to the constraint that that the time-t forecasts for
  ##                                the first k variables equal the survey
  ##                                forecast
  ##
  ## January, 2013
  
  k  <-  NROW(survey_forecast)
  tilted_forecast  <-  array(0, length(model_forecast))
  tilted_forecast[(1:k)]  <- survey_forecast
  S21 <- model_variance[-c(1:k),(1:k)]
  S11 <- model_variance[1:k,1:k]
  smd  <- model_forecast[1:k] - survey_forecast
  tilted_forecast[-c(1:k)]  <-  model_forecast[-(1:k)] - S21 %*% solve(S11) %*% smd
  return(tilted_forecast)
}

##############################################################################
##
## Utils
##
##############################################################################

span <- function(x) {
  idx <- index(x)
  c(head(idx, 1), tail(idx, 1))
}

zeros <- function(n) rep(0,n)
zeros <- function(n, k) array(0, c(n, k))

# extract_beta_from_var <- function(x){
#   invert_beta <- function(u) {
#     cf <- coef(u)
#     lx <- length(cf)
#     c(cf[lx], cf[1:(lx-1)])
#   }
#   
#   out <- t(sapply(x$varresult, invert_beta))
#   list(C = out[,1], B = out[,-1])
# }

tril <- function(A) {
  ## Given an upper triangular matrix
  ## extract non zero value
  idx <- !lower.tri(A)
  A[idx]
}

untril <- function(x) {
  ## given a vector of size m*(m-1)/2 return 
  ## the m x m upper triangular matrix
  k <- length(x)
  m <- .5*(-1 + sqrt(1 + 8*k))
  out <- matrix(0, m, m)
  idx <- !lower.tri(out)
  out[idx] <- x
  out
}

############################################################################
## Utility function
############################################################################

NSextract <- function(maturity, lambda, data) {
  ## data is an array with length eq. to length of maturities
  mat <- maturity
  load_lev    = array(1,length(mat))
  load_slope  = (1 - exp(-lambda*mat))/(lambda*mat)
  load_curv   = (1 - exp(-lambda*mat))/(lambda*mat) - exp(-lambda*mat)
  load        = cbind(load_lev, load_slope, load_curv)
  X = solve(crossprod(load), crossprod(load, data))
  list(betas = X, load = load, fit = load %*% X)
}

getloadings <- function(maturity, lambda) {
  ## data is an array with length eq. to length of maturities
  mat <- maturity
  load_lev    <-  array(1,length(mat))
  load_slope  <-  (1 - exp(-lambda*mat))/(lambda*mat)
  load_curv   <-  (1 - exp(-lambda*mat))/(lambda*mat)-exp(-lambda*mat)
  load        <-  cbind(load_lev, load_slope, load_curv)
  load
}

dns_theta_init <- function(y, beta, lambda = 0.04) {
  s <- ncol(beta)
  betas <- beta[,1:3]
  addfct <- beta[,-c(1:3)]
  if (ncol(beta) == 3) 
    blm <- apply(betas, 2, function(u) dynlm(ts(u) ~ L(ts(u), 1)))
  else 
    blm <- apply(betas, 2, function(u) dynlm(ts(u) ~ L(ts(u), 1) + ts(addfct)))
  bcoef <- sapply(blm, coef)
  B <- diag(bcoef[2,])
  C <- bcoef[1,]
  D <- bcoef[-c(1,2),]
  bresid <- sapply(blm, resid)
  Sigma_V  <- crossprod(bresid)/(length(bresid) - blm[[1]]$rank - 1)
  mu       <- (solve(diag(min(s, 3)) - B, C))[1:3]
  Lambda   <- getloadings(maturities, lambda)
  u        <- y - beta[,1:3] %*% t(Lambda)
  Sigma_U  <- diag(crossprod(u)/(3*NROW(beta)))
  names(Sigma_U) <- paste0("V", 1:length(Sigma_U))
  G <- c(diag(B))
  d <- c(D)

  names(G) <- paste0("G", 1:length(G))
  names(d) <- paste0("gamma", 1:length(d))
  
  W <- tril(chol(Sigma_V))
  names(W) <- paste0("W", 1:(3*(3+1)/2))
  names(mu) <- paste0("mu", 1:min(3,s))
  lambda <- -1*log(-1-0.999/(0.001 - lambda))
  names(lambda) <- "lambda"
  c(G, d, W,  mu, log(Sigma_U), lambda)
}

SSMpredict <- function(object, n.ahead = 12) {
  m <- attr(object, "m")
  p <- attr(object, "p")
  k <- attr(object, "k")
  
  states <- as.integer(1:attr(object, "m"))
  timespan <- 1:attr(object, "n")
  endtime <- end(object$y)
  
  timespan <- attr(object, "n") + 1:n.ahead
  n <- attr(object, "n") <- attr(object, "n") + as.integer(n.ahead)
  endtime <- end(object$y) + c(0, n.ahead)
  object$y <- window(object$y, end = endtime, extend = TRUE)
  
  out <- KFS(model = object, smoothing = c("mean", "state"))

  TT <- drop(out$model$T)
  Z <- drop(out$model$Z)
  H <- drop(out$model$H)
  R <- drop(out$model$R)
  Q <- R %*% drop(out$model$Q) %*% t(R)
  
  a <- array(0, c(n.ahead + 1, m))
  y <- array(0, c(n.ahead + 1, p))
  V <- array(0, c(m, m, n.ahead + 1))
  S <- array(0, c(p, p, n.ahead + 1))

  a[1,] <- out$alphahat[endtime[1],]
  y[1,] <- Z %*% a[1,]
  V[,,1] <- out$V[, , endtime[1]]
  S[,,1] <- Z %*% V[, , 1] %*% t(Z) + H
  for (it in 1:n.ahead) {
    a[it + 1, ] <- TT %*% a[it, ]
    V[, , it + 1] <- TT %*% V[, , it] %*% t(TT) + Q
    y[it, ] <- Z %*% a[it + 1, ]
    S[, , it] <- Z %*% V[, , it + 1] %*% t(Z) + H
  }
  
  list(Yhat = y[1:n.ahead,], Vhat = S[,,1:n.ahead])
} 

DNScondfore_dist <- function(object, Ycond, nsim) {
  ## Ycond is a N x MAT object
  ## with the missing entries denoting maturities to fill
  ## and the nonmissing entries denoting conditions
  Ncond <- nrow(Ycond)
  mat <- ncol(Ycond)
  
  m <- attr(object, "m")
  p <- attr(object, "p")
  k <- attr(object, "k")
  
  states <- as.integer(1:attr(object, "m"))
  timespan <- 1:attr(object, "n")
  endtime <- end(object$y)
  
  timespan <- attr(object, "n") + 1:Ncond
  n <- attr(object, "n") <- attr(object, "n") + as.integer(Ncond)
  endtime <- end(object$y) + c(0, Ncond)
  Y_ <- window(object$y, end = endtime, extend = TRUE)
  
  Y_[(n-Ncond + 1):n, ] <- Ycond
  
  object$y <- Y_
  
  out <- simulateSSM(object = object, filtered = FALSE, nsim = nsim, type = "observations")

  list(Yhat = apply(out[(n - Ncond + 1):n, , ], c(1,2), mean), 
       Vhat = apply(out[(n - Ncond + 1):n, , ], c(1,2), var))
}

DNScondfore_mean <- function(object, Ycond, nsim) {
  ## Ycond is a N x MAT object
  ## with the missing entries denoting maturities to fill
  ## and the nonmissing entries denoting conditions
  Ncond <- nrow(Ycond)
  mat   <- ncol(Ycond)
  
  m <- attr(object, "m")
  p <- attr(object, "p")
  k <- attr(object, "k")
  
  states   <- as.integer(1:attr(object, "m"))
  timespan <- 1:attr(object, "n")
  endtime  <- end(object$y)
  
  timespan <- attr(object, "n") + 1:Ncond
  n <- attr(object, "n") <- attr(object, "n") + as.integer(Ncond)
  endtime <- end(object$y) + c(0, Ncond)
  Y_ <- window(object$y, end = endtime, extend = TRUE)
  
  Y_[(n - Ncond + 1):n, ] <- Ycond
  
  object$y <- Y_
  
  out <- predict(object = object, filtered = FALSE)
  return(out)
}

encomp_rolling <- function(BC, MODEL, mu, lag = 0) {
  T_1  <- length(BC)
  T_2  <- length(MODEL)
  if (T_1 != T_2)
    stop("wrong dimension")
  T = T_1
  m  <- round(mu*T)
  idx <- seq(1,T - m + 1)
  crit  <- c(3.176, 2.938, 2.770, 2.624, 2.475, 2.352, 2.248, 2.080, 1.975)
  k <- crit[round(mu*10)]
  enc_rol <- array(0, T - m)
  
  jj <- 1
  
  for (ii in m:T) {
    cc  <- (MODEL[(ii - m + 1):ii] - BC[(ii - m + 1):ii])*BC[(ii - m + 1):ii]
    V <- lrvar(cc, kernel = "Bartlet", h = lag)
    enc_rol[jj] <- mean(cc)/sqrt(V)
    jj <- jj + 1
  }
  
  list(enc_rol = enc_rol, k = k, idx = idx)
}

