function results = gentobit(y,x,info,z,s_bar)
% PURPOSE: computes Tobit Regression for pooled panel (partial MLE)
%---------------------------------------------------
% USAGE: results = tobit(y,x,info,T,z)
% where: y = censored dependent variable vector (nobs x 1)
%        x = independent variables matrix (nobs x nvar)
%        info = a structure with options:
%        info.trunc = 'left' or 'right' for censoring (default=left)
%        info.limit = value for censoring (default=0, left/right or [left;right])
%        info.meth  = Hessian: ['dfp'], 'bfgs', 'gn', 'marq', 'sd'
%        info.btol  =  tolerance for b convergence (default = 1e-8)
%        info.ftol  = tolerance for FUN convergence (default = 1e-8)
%        info.maxit = maximum # of iterations (default = 500)
%        info.b     = starting values for parameters (default = ols)
%        info.inference    = whether to compute formular based standard errors or not (default = 1, yes.)
%        vec = an index used to identify households in (unbalanced) panel 
%        z = for censored tobit with random coefficients
%---------------------------------------------------
% RETURNS: a structure
%        results.meth  = 'tobit'
%        results.beta  = bhat
%        results.tstat = t-stats
%        results.yhat  = yhat
%        results.resid = residuals
%        results.sige  = e'*e/(n-k)
%        results.rsqr  = rsquared
%        results.rbar  = rbar-squared
%        results.lik   = Log-Likelihood value
%        results.iter  = # of iterations taken
%        results.grad  = gradient at solution
%        results.opt   = method of optimization used
%        results.nobs  = nobs
%        results.nobsc = # of censored observations
%        results.nvar  = nvars
%        results.y     = y data vector
% --------------------------------------------------
% SEE ALSO: maxlik, prt(results), plt(results), logit, probit
%---------------------------------------------------

% written by:
% James P. LeSage, Dept of Economics
% University of Toledo
% changed for censored panel models with individual-specific slopes by:
% Shu Shen, Dept of Economics
% Univeristy of Austin
% 12/12/2009

% Check input and output
error(nargchk(2,6,nargin));
error(nargoutchk(0,1,nargout));

% default options

if nargin==4
    error('z shall be accompanied by s_bar.')
end

in.btol = eps;
in.gtol = eps;
in.ftol = eps; 
in.maxit = 5000;
in.hess = 'bfgs';
in.call = 'other';

if nargin >= 3 % user supplied options
  if ~isstruct(info)
    error('tobit: options should be in a structure variable');
  end;
end

sflag = 0;
tflag = 1;
inference = 1;
panel=0;
% parse options
fields = fieldnames(info);
nf = length(fields); 
  for i=1:nf
      if strcmp(fields{i},'maxit')
          in.maxit = info.maxit; 
      elseif strcmp(fields{i},'btol')
          in.btol = info.btol;
      elseif strcmp(fields{i},'ftol')
          in.ftol = info.ftol;
      elseif strcmp(fields{i},'meth')
          in.hess = info.meth;
      elseif strcmp(fields{i},'b');
           beta = info.b; sflag = 1;
      elseif strcmp(fields{i},'trunc');
          if strcmp(info.trunc,'left');
          tflag = 1;vflag=min(y);
          elseif strcmp(info.trunc,'right')
          tflag = 2;vflag=max(y);
          else tflag = 3;vflag=[min(y);max(y)];
          vflagl=min(y);vflagr=max(y);
          end;
      elseif strcmp(fields{i},'limit');
          vflag = info.limit;
          if length(vflag)==2;
              vflagl=vflag(1);
              vflagr=vflag(2);
          end
      elseif strcmp(fields{i},'inference');
          inference=info.inference;
      elseif strcmp(fields{i},'panelvec');
          vec=info.panelvec;panel=1;
      end;
  end;

% find # of censored observations
if tflag == 1
   results.nobsc = length(find(y <= vflag));
elseif tflag==2
   results.nobsc = length(find(y >= vflag));
else results.nobsc = length(find(y <= vflagl | y >=vflagr));
end;

if nargin>4
    q=size(z,2);p=size(s_bar,2);
    x2=kron(z,ones(1,p)).*kron(ones(1,q),s_bar);
    xx=[x x2];
else
    q=1;
    xx=x;z=[];
end

% decide the starting point if not specified
res = ols(y,xx);
if sflag == 0
    % use ols starting values
    beta = res.beta;
end
sige = res.sige;
baug = [beta; ones(q,1)*sqrt(sige)];
nvar= size(xx,2);
oresult=maxlik_gentobit('ll_gentobit',baug,in,y,xx,tflag,vflag,z);
nobs=length(y);


% maximize the likelihood function
iter = oresult.iter;
llf = -oresult.f;
hess = oresult.hess;
grad = oresult.g; 
time1 = oresult.time;
beta = oresult.b;
%score= oresult.score;

if iter == in.maxit
error(['no convergence in tobit in ' num2str(iter) ' iterations']);
end;

results.nobs = nobs;
results.nvar = nvar;
results.iter = iter;
results.beta = beta(1:nvar,1);
results.sigsq = max(beta(nvar+1:nvar+q,1),0);

% now compute inference results
if inference==1
score = mle_score('ll_gentobit_2',[beta(1:nvar,1);max(beta(nvar+1:nvar+q,1),0)],y,xx,tflag,vflag,z);
sumB=zeros(nvar+q,nvar+q);
if panel==1
    score_sub=mat2cell(score(:,1:nvar+q),vec,nvar+q);
    for i=1:length(vec)
        score_sub2{i}=sum(score_sub{i},1);
        sumB=sumB+score_sub2{i}'*score_sub2{i};
    end
else
    for i=1:nvar+q
        for j=1:nvar+q
            sumB(i,j)=sum(score(:,i)'*score(:,j));
        end
    end
end
cov=inv(nobs*hess(1:nvar+q,1:nvar+q))*sumB*inv(nobs*hess(1:nvar+q,1:nvar+q))';
stdb = sqrt(diag(cov));
stdb2=sqrt(diag(inv(sumB)));
results.betastd=stdb(1:nvar);
results.betastd2=stdb2(1:nvar);
results.sigsqstd=stdb(nvar+1:nvar+q);
results.sigsqstd2=stdb2(nvar+1:nvar+q);
results.betatstat = beta(1:nvar)./stdb(1:nvar);
results.sigtstat = abs(beta(nvar+1:nvar+q))./stdb(nvar+1:nvar+q);
end

results.lik = llf;
results.grad = grad;
results.time1 = time1;
results.tflag=tflag;
results.vflag=vflag;
